package nashid.verify.sdk.utils.helpers

import com.google.gson.Gson
import com.google.gson.GsonBuilder
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonSerializationContext
import com.google.gson.JsonSerializer
import nashid.verify.sdk.model.Artifacts
import nashid.verify.sdk.model.DocumentVerification
import nashid.verify.sdk.model.Liveness
import nashid.verify.sdk.model.MainData
import nashid.verify.sdk.model.NFC
import nashid.verify.sdk.model.Scan
import java.lang.reflect.Type

class DocumentVerificationSerializer : JsonSerializer<DocumentVerification> {
    override fun serialize(
        src: DocumentVerification,
        typeOfSrc: Type,
        context: JsonSerializationContext,
    ): JsonElement {
        val jsonObject = JsonObject()
        src.metaDataModel?.let {
            jsonObject.add("metadata", context.serialize(it))
        }
        if (src.type != 0) {
            jsonObject.addProperty("type", src.type)
        }
        src.data?.let {
            jsonObject.add("data", context.serialize(it))
        }
        src.artifacts?.let {
            jsonObject.add("artifacts", context.serialize(it))
        }
        if (src.status != 0) {
            jsonObject.addProperty("status", src.status)
        }

        return jsonObject
    }
}

class MainDataSerializer : JsonSerializer<MainData> {
    override fun serialize(
        src: MainData,
        typeOfSrc: Type,
        context: JsonSerializationContext,
    ): JsonElement {
        val jsonObject = JsonObject()

        if (src.scan != null && !isScanEmpty(src.scan)) {
            val scanObject = JsonObject()
            with(src.scan) {
                if (mrzText.isNotEmpty()) scanObject.addProperty("mrz_text", mrzText)
                if (documentNo.isNotEmpty()) scanObject.addProperty("document_no", documentNo)
                if (documentType.isNotEmpty()) scanObject.addProperty("document_type", documentType)
                if (expiryDate.isNotEmpty()) scanObject.addProperty("expiry_date", expiryDate)
                if (dateOfBirth.isNotEmpty()) scanObject.addProperty("date_of_birth", dateOfBirth)
                if (fullName.isNotEmpty()) scanObject.addProperty("full_name", fullName)
                if (nationality.isNotEmpty()) scanObject.addProperty("nationality", nationality)
                if (country.isNotEmpty()) scanObject.addProperty("country", country)
                if (gender.isNotEmpty()) scanObject.addProperty("gender", gender)
                if (cardNumber.isNotEmpty()) scanObject.addProperty("id_number", cardNumber)
                if (scanObject.size() > 0) {
                    jsonObject.add("scan", scanObject)
                }
            }
        }
        if (src.barcode != null) {
            val barcodeObject = JsonObject()
            with(src.barcode) {
                if (idNumber.isNotEmpty()) barcodeObject.addProperty("id_number", idNumber)
                if (serialNumber.isNotEmpty()) barcodeObject.addProperty("serial_number", serialNumber)
            }
            if (barcodeObject.size() > 0) {
                jsonObject.add("barcode", barcodeObject)
            }
        }

        if (src.nfc != null) {
            val nfcObject = JsonObject()
            with(src.nfc) {
                if (fullNameEnglish.isNotEmpty()) nfcObject.addProperty("full_name_english", fullNameEnglish)
                if (nationalityEnglish.isNotEmpty()) nfcObject.addProperty("nationality_english", nationalityEnglish)
                if (genderEnglish.isNotEmpty()) nfcObject.addProperty("gender_english", genderEnglish)
                if (genderArabic.isNotEmpty()) nfcObject.addProperty("gender_arabic", genderArabic)
                if (companyNameEnglish.isNotEmpty()) nfcObject.addProperty("company_name_english", companyNameEnglish)
                if (permitType.isNotEmpty()) nfcObject.addProperty("permit_type", permitType)
                if (countryOfBirthArabic.isNotEmpty()) nfcObject.addProperty("country_of_birth_arabic", countryOfBirthArabic)
                if (placeOfIssueArabic.isNotEmpty()) nfcObject.addProperty("place_of_issue_arabic", placeOfIssueArabic)
                if (identityNumber.isNotEmpty()) nfcObject.addProperty("identity_number", identityNumber)
                if (placeOfIssueEnglish.isNotEmpty()) nfcObject.addProperty("place_of_issue_english", placeOfIssueEnglish)
                if (expiryDate.isNotEmpty()) nfcObject.addProperty("expiry_date", expiryDate)
                if (visaNumber.isNotEmpty()) nfcObject.addProperty("visa_number", visaNumber)
                if (issueDate.isNotEmpty()) nfcObject.addProperty("issue_date", issueDate)
                if (fullNameArabic.isNotEmpty()) nfcObject.addProperty("full_name_arabic", fullNameArabic)
                if (companyNameArabic.isNotEmpty()) nfcObject.addProperty("company_name_arabic", companyNameArabic)
                if (countryOfBirthEnglish.isNotEmpty()) nfcObject.addProperty("country_of_birth_english", countryOfBirthEnglish)
                if (nationalityArabic.isNotEmpty()) nfcObject.addProperty("nationality_arabic", nationalityArabic)
                if (companyAddressArabic.isNotEmpty()) nfcObject.addProperty("company_address_arabic", companyAddressArabic)
                if (permitNumber.isNotEmpty()) nfcObject.addProperty("permit_number", permitNumber)
                if (dateOfBirth.isNotEmpty()) nfcObject.addProperty("date_of_birth", dateOfBirth)
                if (useByDate.isNotEmpty()) nfcObject.addProperty("use_by_date", useByDate)

                if (passportDocType.isNotEmpty()) nfcObject.addProperty("document_type", passportDocType)
                if (passportNationality.isNotEmpty()) nfcObject.addProperty("nationality", passportNationality)
                if (passDocNumber.isNotEmpty()) nfcObject.addProperty("passport_number", passDocNumber)
                if (passportIssuingCountry.isNotEmpty()) nfcObject.addProperty("issuing_country", passportIssuingCountry)
                if (dateOfBirth.isNotEmpty()) nfcObject.addProperty("date_of_birth", dateOfBirth)
                if (passportGender.isNotEmpty()) nfcObject.addProperty("gender", passportGender)
                if (passportDocExpiryDate.isNotEmpty()) nfcObject.addProperty("passport_expiry_date", passportDocExpiryDate)
                if (passportPlaceOfBirth.isNotEmpty()) nfcObject.addProperty("place_of_birth", passportPlaceOfBirth)
                if (passportAddress.isNotEmpty()) nfcObject.addProperty("address", passportAddress)
                if (passportPhoneNumber.isNotEmpty()) nfcObject.addProperty("telephone", passportPhoneNumber)
                if (passportNameOfHolder.isNotEmpty()) nfcObject.addProperty("name", passportNameOfHolder)
                if (passportIssuingAuthority.isNotEmpty()) nfcObject.addProperty("issuing_authority", passportIssuingAuthority)
                if (passportPersonalNumber.isNotEmpty()) nfcObject.addProperty("personal_number", passportPersonalNumber)
            }
            if (nfcObject.size() > 0) {
                jsonObject.add("NFC", nfcObject)
            }
        }

        if (src.liveness != null && !isLivenessEmpty(src.liveness)) {
            jsonObject.add("liveness", context.serialize(src.liveness))
        }

        return jsonObject
    }

    private fun isScanEmpty(scan: Scan): Boolean {
        return scan == Scan()
    }

    private fun isNfcEmpty(nfc: NFC): Boolean {
        return nfc == NFC()
    }

    private fun isLivenessEmpty(liveness: Liveness): Boolean {
        return liveness == Liveness()
    }
}

class ArtifactsSerializer : JsonSerializer<Artifacts> {
    override fun serialize(
        src: Artifacts,
        typeOfSrc: Type,
        context: JsonSerializationContext,
    ): JsonElement {
        val jsonObject = JsonObject()

        if (src.backSideImage.isNotEmpty()) {
            jsonObject.addProperty("back_side_image", src.backSideImage)
        }
        if (src.frontSideImage.isNotEmpty()) {
            jsonObject.addProperty("front_side_image", src.frontSideImage)
        }
        if (src.ocrFaceImage.isNotEmpty()) {
            jsonObject.addProperty("ocr_face_image", src.ocrFaceImage)
        }
        if (src.nfcFaceImage.isNotEmpty()) {
            jsonObject.addProperty("nfc_face_image", src.nfcFaceImage)
        }
        if (src.activeLivenessImage.isNotEmpty()) {
            jsonObject.addProperty("active_liveness_image", src.activeLivenessImage)
        }
        if (src.passiveLiveNessImage.isNotEmpty()) {
            jsonObject.addProperty("passive_liveness_image", src.passiveLiveNessImage)
        }
        return jsonObject
    }
}

fun createCustomGson(): Gson {
    return GsonBuilder()
        .registerTypeAdapter(DocumentVerification::class.java, DocumentVerificationSerializer())
        .registerTypeAdapter(MainData::class.java, MainDataSerializer())
        .registerTypeAdapter(Artifacts::class.java, ArtifactsSerializer())
        .create()
}
