package nashid.verify.sdk.viewmodel

import android.app.Application
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import nashid.verify.sdk.data.repository.ScanDocumentRepository
import nashid.verify.sdk.model.Artifacts
import nashid.verify.sdk.model.DocumentType
import nashid.verify.sdk.model.DocumentVerification
import nashid.verify.sdk.model.ScanDocumentResult
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.Loggers
import nashid.verify.sdk.utils.NetWorkResult
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdkNew.R
import okhttp3.MultipartBody
import java.util.*

class ScanDocumentViewModel(
    private val scanDocumentRepository: ScanDocumentRepository,
    private val context: Application,
    private val dispatcher: CoroutineDispatcher = Dispatchers.IO,
) : ViewModel() {
    private var frontSideImagePart: MultipartBody.Part? = null
    private var backSideImagePart: MultipartBody.Part? = null
    private var ocrFaceImagePart: MultipartBody.Part? = null
    private var passiveLiveImagePart: MultipartBody.Part? = null
    private var activeLiveImagePart: MultipartBody.Part? = null
    private val logger = Loggers.withTag("ScanDocumentViewModel")
    private val _result = MutableLiveData<ScanDocumentResult?>()
    val result: LiveData<ScanDocumentResult?> get() = _result

    companion object {
        private val uploadedArtifacts = Collections.synchronizedSet(mutableSetOf<String>())
        private val requiredArtifacts = Collections.synchronizedSet(mutableSetOf<String>())
    }

    private var isVerificationSubmissionInProgress = false
    private val maxRetries = 10
    private val retryDelayMs = 1500L

    init {
        updateRequiredArtifacts()
    }

    fun updateRequiredArtifacts() {
        synchronized(requiredArtifacts) {
            requiredArtifacts.clear()

            // If OCR scan failed, don't add any artifacts
            if (Utility.getInstance().hasOcrScanFailed()) {
                logger.log("OCR scan failed, no artifacts required")
                return
            }

            // Add OCR-related artifacts
            if (SdkConfig.isOcrEnabled || SdkConfig.isMrzEnabled) {
                requiredArtifacts.add(ArtifactType.FRONTSIDE_IMAGE.type)
                if (SdkConfig.viewType != DocumentType.INTERNATIONAL_PASSPORT) {
                    requiredArtifacts.add(ArtifactType.BACKSIDE_IMAGE.type)
                }
                requiredArtifacts.add(ArtifactType.OCR_FACE_IMAGE.type)
            }

            // Add NFC artifact if NFC hasn't failed
            if (SdkConfig.isNfcScanningEnabled &&
                SdkConfig.isNfcSkipable != true &&
                !Utility.getInstance().hasNfcScanFailed()
            ) {
                requiredArtifacts.add(ArtifactType.NFC_IMAGE.type)
            }

            // Handle active and passive liveness separately
            if (!Utility.getInstance().hasLiveNessScanFailed() && SdkConfig.isFaceMatchingEnabled) {
                if (SdkConfig.isActiveLiveNessEnabled) {
                    requiredArtifacts.add(ArtifactType.ACTIVE_LIVENESS_IMAGE.type)
                }
                if (SdkConfig.isPassiveLiveNessEnabled) {
                    requiredArtifacts.add(ArtifactType.PASSIVE_LIVENESS_IMAGE.type)
                }
            }
        }
        logger.log("Required artifacts updated: $requiredArtifacts")
    }

    fun uploadArtifact(
        token: String,
        appKey: String,
        filePart: MultipartBody.Part,
        artifactType: String,
    ) {
        when (artifactType) {
            ArtifactType.FRONTSIDE_IMAGE.type -> frontSideImagePart = filePart
            ArtifactType.BACKSIDE_IMAGE.type -> backSideImagePart = filePart
            ArtifactType.OCR_FACE_IMAGE.type -> ocrFaceImagePart = filePart
            ArtifactType.PASSIVE_LIVENESS_IMAGE.type -> passiveLiveImagePart = filePart
            ArtifactType.ACTIVE_LIVENESS_IMAGE.type -> activeLiveImagePart = filePart
        }
        viewModelScope.launch(dispatcher) {
            try {
                when (val response = scanDocumentRepository.uploadFile(token, appKey, filePart)) {
                    is NetWorkResult.Success -> {
                        if (response.data?.data?.fileKey != null) {
                            val fileKey = response.data.data.fileKey
                            synchronized(uploadedArtifacts) {
                                uploadedArtifacts.add(artifactType)
                                logger.log("Upload successful for $artifactType. FileKey: $fileKey, Current uploads: $uploadedArtifacts")
                            }
                            _result.postValue(ScanDocumentResult.FileKeyResult(fileKey, artifactType))
                            logCurrentState()
                        } else {
                            logger.log("Upload failed for $artifactType: No fileKey returned")
                            _result.postValue(ScanDocumentResult.Error("Upload failed: No fileKey for $artifactType"))
                        }
                    }
                    is NetWorkResult.Error -> {
                        logger.log("Upload failed for $artifactType: ${response.message}")
                        _result.postValue(ScanDocumentResult.Error(response.message ?: context.getString(R.string.upload_failed)))
                    }
                    else -> {}
                }
            } catch (e: Exception) {
                logger.log("Upload failed for $artifactType: ${e.message}")
                _result.postValue(ScanDocumentResult.Error(e.message ?: context.getString(R.string.unexpected_error_occurred)))
            }
        }
    }

    private fun logCurrentState() {
        logger.log("Current state - Required: $requiredArtifacts, Uploaded: $uploadedArtifacts")
    }

    fun submitVerificationWithRetry(
        token: String,
        appKey: String,
        verification: DocumentVerification,
    ) {
        if (isVerificationSubmissionInProgress) {
            logger.log("Verification submission already in progress")
            _result.postValue(ScanDocumentResult.Error(context.getString(R.string.verification_in_progress)))
            return
        }

        viewModelScope.launch(dispatcher) {
            isVerificationSubmissionInProgress = true
            try {
                var retryCount = 0
                while (retryCount < maxRetries && isVerificationSubmissionInProgress) {
                    logger.log("Verification attempt ${retryCount + 1} of $maxRetries")
                    val currentUploads = synchronized(uploadedArtifacts) { uploadedArtifacts.toSet() }
                    val currentRequired = synchronized(requiredArtifacts) { requiredArtifacts.toSet() }
                    val missingArtifacts = currentRequired - currentUploads

                    // Check file keys for required artifacts
                    val artifacts =
                        Artifacts(
                            backSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.BACKSIDE_IMAGE) ?: "",
                            frontSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.FRONTSIDE_IMAGE) ?: "",
                            ocrFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.OCR_FACE_IMAGE) ?: "",
                            nfcFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.NFC_IMAGE) ?: "",
                            activeLivenessImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.ACTIVE_LIVENESS_IMAGE) ?: "",
                            passiveLiveNessImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.PASSIVE_LIVENESS_IMAGE) ?: "",
                        )
                    logger.log("File key states: frontSideImage=${artifacts.frontSideImage}, ocrFaceImage=${artifacts.ocrFaceImage}, backSideImage=${artifacts.backSideImage}, nfcFaceImage=${artifacts.nfcFaceImage}, activeLiveness=${artifacts.activeLivenessImage}, passiveLiveness=${artifacts.passiveLiveNessImage}")

                    val emptyKeys =
                        currentRequired.filter { artifact ->
                            val key =
                                when (artifact) {
                                    ArtifactType.BACKSIDE_IMAGE.type -> artifacts.backSideImage
                                    ArtifactType.FRONTSIDE_IMAGE.type -> artifacts.frontSideImage
                                    ArtifactType.OCR_FACE_IMAGE.type -> artifacts.ocrFaceImage
                                    ArtifactType.NFC_IMAGE.type -> artifacts.nfcFaceImage
                                    ArtifactType.ACTIVE_LIVENESS_IMAGE.type -> artifacts.activeLivenessImage
                                    ArtifactType.PASSIVE_LIVENESS_IMAGE.type -> artifacts.passiveLiveNessImage
                                    else -> ""
                                }
                            key.isEmpty()
                        }

                    if (missingArtifacts.isEmpty() && emptyKeys.isEmpty()) {
                        logger.log("All required artifacts uploaded and file keys present. Proceeding with verification submission.")
                        submitVerification(token, appKey, verification.copy(artifacts = artifacts))
                        break
                    } else {
                        if (missingArtifacts.isNotEmpty()) {
                            logger.log("Missing artifacts: $missingArtifacts. Retrying.")
                            _result.postValue(ScanDocumentResult.Error(context.getString(R.string.waiting_for_artifact) + ": $missingArtifacts"))
                        }
                        if (emptyKeys.isNotEmpty()) {
                            logger.log("Empty file keys for artifacts: $emptyKeys. Retrying.")
                            _result.postValue(ScanDocumentResult.Error("Waiting for file keys: $emptyKeys"))
                        }
                        delay(retryDelayMs)
                        retryCount++
                    }

                    // Attempt re-upload for front side and ocr face if missing and part is available
                    if (missingArtifacts.contains(ArtifactType.FRONTSIDE_IMAGE.type)) {
                        logger.log("Retrying upload for missing front side image")
                        frontSideImagePart?.let { uploadArtifact(token, appKey, it, ArtifactType.FRONTSIDE_IMAGE.type) }
                    }
                    if (missingArtifacts.contains(ArtifactType.BACKSIDE_IMAGE.type)) {
                        logger.log("Retrying upload for missing back side image")
                        backSideImagePart?.let { uploadArtifact(token, appKey, it, ArtifactType.BACKSIDE_IMAGE.type) }
                    }
                    if (missingArtifacts.contains(ArtifactType.OCR_FACE_IMAGE.type)) {
                        logger.log("Retrying upload for missing ocr face image")
                        ocrFaceImagePart?.let { uploadArtifact(token, appKey, it, ArtifactType.OCR_FACE_IMAGE.type) }
                    }
                    if (missingArtifacts.contains(ArtifactType.PASSIVE_LIVENESS_IMAGE.type)) {
                        logger.log("Retrying upload for missing passive image")
                        passiveLiveImagePart?.let { uploadArtifact(token, appKey, it, ArtifactType.PASSIVE_LIVENESS_IMAGE.type) }
                    }
                    if (missingArtifacts.contains(ArtifactType.ACTIVE_LIVENESS_IMAGE.type)) {
                        logger.log("Retrying upload for missing active image")
                        activeLiveImagePart?.let { uploadArtifact(token, appKey, it, ArtifactType.ACTIVE_LIVENESS_IMAGE.type) }
                    }

                    if (retryCount >= maxRetries) {
                        logger.log("Max retries reached. Missing artifacts: $missingArtifacts, Missing files: $emptyKeys")
                        _result.postValue(
                            ScanDocumentResult.ScanDocumentResponse(
                                false,
                                context.getString(R.string.reach_max_try) + ": $missingArtifacts, Missing files: $emptyKeys",
                                "",
                            ),
                        )
                    }
                }
            } catch (e: Exception) {
                logger.log("Verification submission failed with exception: ${e.message}")
                _result.postValue(ScanDocumentResult.Error(context.getString(R.string.verification_failed) + ": ${e.message}"))
            } finally {
                isVerificationSubmissionInProgress = false
            }
        }
    }

    fun resetArtifacts() {
        synchronized(uploadedArtifacts) {
            uploadedArtifacts.clear()
            logger.log("Cleared uploaded artifacts")
        }
        Utility.getInstance().clearArtifactKeys()
        updateRequiredArtifacts()
        viewModelScope.launch(Dispatchers.Main) {
            _result.value = null
            isVerificationSubmissionInProgress = false
        }
        logger.log("Artifacts reset. Required: $requiredArtifacts, Uploaded: $uploadedArtifacts")
    }

    private suspend fun submitVerification(
        token: String,
        appKey: String,
        verification: DocumentVerification,
    ) {
        try {
            when (val response = scanDocumentRepository.submitVerification(token, appKey, verification)) {
                is NetWorkResult.Success -> {
                    if (response.data?.data?.verificationId != null) {
                        val verificationId = response.data.data.verificationId
                        logger.log("SubmitVerification successful for $verificationId: ${response.data.message}")

                        // Post the success result before resetting artifacts
                        withContext(Dispatchers.Main) {
                            _result.value =
                                ScanDocumentResult.ScanDocumentResponse(
                                    true,
                                    response.data.message ?: context.getString(R.string.verification_submit_success),
                                    verificationId,
                                )
                        }
                        // Small delay to ensure the success result is processed before resetting artifacts
                        delay(100)
                        resetArtifacts()
                    } else {
                        withContext(Dispatchers.Main) {
                            _result.value =
                                ScanDocumentResult.ScanDocumentResponse(
                                    false,
                                    response.data?.message ?: context.getString(R.string.verification_submit_fail),
                                    "",
                                )
                        }
                    }
                }
                is NetWorkResult.Error -> {
                    withContext(Dispatchers.Main) {
                        _result.value =
                            ScanDocumentResult.ScanDocumentResponse(
                                false,
                                response.message ?: context.getString(R.string.unexpected_error_occurred),
                                "",
                            )
                    }
                }
                else -> {}
            }
        } catch (e: Exception) {
            logger.log("SubmitVerification failed with exception: ${e.message}")
            withContext(Dispatchers.Main) {
                _result.value =
                    ScanDocumentResult.ScanDocumentResponse(
                        false,
                        e.message ?: context.getString(R.string.unexpected_error_occurred),
                        "",
                    )
            }
        }
    }

    fun getVerification(
        token: String,
        appKey: String,
        verificationId: String,
    ) {
        viewModelScope.launch(dispatcher) {
            when (val response = scanDocumentRepository.getVerification(token, appKey, verificationId)) {
                is NetWorkResult.Success -> {
                    logger.log("GetVerification successful for $verificationId: ${response.data?.message}")
                    _result.postValue(response.data?.let { ScanDocumentResult.GetScanResultResponse(true, it.message, response.data) })
                }
                is NetWorkResult.Error -> {
                    logger.log("GetVerification failed for $verificationId: ${response.message}")
                    _result.postValue(
                        ScanDocumentResult.GetScanResultResponse(
                            false,
                            response.message ?: context.getString(R.string.unexpected_error_occurred),
                            null,
                        ),
                    )
                }
                else -> {}
            }
        }
    }

    fun clearScanDocumentData() {
        _result.value = null
    }
}
