package nashid.verify.sdk.data.repository

import android.content.Context
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import nashid.verify.sdk.data.remote.ApiService
import nashid.verify.sdk.model.AuthResponse
import nashid.verify.sdk.model.RefreshTokenResponse
import nashid.verify.sdk.request.AuthRequest
import nashid.verify.sdk.request.RefreshTokenRequest
import nashid.verify.sdk.utils.NetWorkResult
import nashid.verify.sdkNew.R
import retrofit2.Response

class AuthRepository(
    private val apiService: ApiService,
    private val context: Context,
    private val ioDispatcher: CoroutineDispatcher = Dispatchers.IO,
) {
    suspend fun authenticateUser(authRequest: AuthRequest): NetWorkResult<AuthResponse> {
        return withContext(ioDispatcher) {
            try {
                val response: Response<AuthResponse> = apiService.authenticate(authRequest)
                processAuthenticateUserResponse(response)
            } catch (e: Exception) {
                NetWorkResult.Error(e.localizedMessage ?: context.getString(R.string.error_unknown))
            }
        }
    }

    suspend fun refreshToken(refreshToken: RefreshTokenRequest): NetWorkResult<RefreshTokenResponse> {
        return withContext(ioDispatcher) {
            try {
                val response: Response<RefreshTokenResponse> = apiService.tokenRefresh(refreshToken)
                processRefreshTokenResponse(response)
            } catch (e: Exception) {
                NetWorkResult.Error(e.localizedMessage ?: context.getString(R.string.error_unknown))
            }
        }
    }

    private fun processRefreshTokenResponse(response: Response<RefreshTokenResponse>): NetWorkResult<RefreshTokenResponse> {
        return if (response.isSuccessful) {
            val body = response.body()
            if (body != null) {
                NetWorkResult.Success(body)
            } else {
                NetWorkResult.Error(context.getString(R.string.error_response_body_null))
            }
        } else {
            NetWorkResult.Error(response.errorBody()?.string() ?: context.getString(R.string.error_unknown))
        }
    }

    private fun processAuthenticateUserResponse(response: Response<AuthResponse>): NetWorkResult<AuthResponse> {
        return if (response.isSuccessful) {
            val body = response.body()
            if (body != null) {
                NetWorkResult.Success(body)
            } else {
                NetWorkResult.Error(context.getString(R.string.error_response_body_null))
            }
        } else {
            NetWorkResult.Error(response.errorBody()?.string() ?: context.getString(R.string.error_unknown))
        }
    }
}
