package nashid.verify.sdk.ui

import android.app.Dialog
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.os.Build
import android.os.Bundle
import android.util.TypedValue
import android.view.View
import android.view.Window
import android.view.WindowInsets
import android.view.WindowInsetsController
import android.view.animation.AnimationUtils
import android.widget.AdapterView
import android.widget.LinearLayout
import androidx.activity.addCallback
import androidx.core.content.res.ResourcesCompat
import nashid.verify.sdk.VerifySDKManager
import nashid.verify.sdk.model.VerifySDKExitStep
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.CameraOverlayManager
import nashid.verify.sdk.utils.helpers.ErrorUtility
import nashid.verify.sdk.utils.helpers.LayoutAndTextSizeConfigurator
import nashid.verify.sdk.utils.helpers.TextSizeConverter
import nashid.verify.sdk.viewmodel.CameraXLiveViewModel
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.ActivityCameraXliveAcitivityBinding
import nashid.verify.sdkNew.databinding.DialogEmailPhoneConfirmationBinding
import org.koin.android.ext.android.inject

class CameraXLiveActivity :
    BaseActivity(),
    AdapterView.OnItemSelectedListener {
    private val textSizeConverter: TextSizeConverter by inject()
    private val viewModel: CameraXLiveViewModel by inject()
    private lateinit var binding: ActivityCameraXliveAcitivityBinding
    private lateinit var cameraOverlayManager: CameraOverlayManager

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityCameraXliveAcitivityBinding.inflate(layoutInflater)
        setContentView(binding.root)
        initView()
        backPress()
    }

    private fun backPress() {
        onBackPressedDispatcher.addCallback(this) {
            if (binding.customLoader.root.visibility != View.VISIBLE) openBackPressDialog()
        }
    }

    private fun initView() {
        logger.log("View setting..")
        handleIntentExtras()
        setUpCameraSelector()
        LayoutAndTextSizeConfigurator(this, binding, textSizeConverter).configureLayoutAndTextSize()
        cameraOverlayManager = CameraOverlayManager(this, binding, textSizeConverter, storage, viewModel, isInternetAvailable, logger)
        cameraOverlayManager.setupOverlayAndAnimation()
        viewModel.cameraXLiveData.setIsInternetAvailable(isInternetAvailable)
        observeViewModel()
    }

    private fun handleIntentExtras() {
        if (intent.hasExtra(getString(R.string.doc_key))) {
            val selectedDoc = intent.getStringExtra(getString(R.string.doc_key))
            if (selectedDoc != null) {
                logger.log("selectedDoc from activity x live: $selectedDoc")
                viewModel.setSelectedDoc(selectedDoc, resources)
            }
        }
    }

    private fun setUpCameraSelector() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            window.setDecorFitsSystemWindows(false)
            window.insetsController?.apply {
                hide(WindowInsets.Type.statusBars())
                systemBarsBehavior = WindowInsetsController.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
            }
        } else {
            @Suppress("DEPRECATION")
            window.decorView.systemUiVisibility =
                View.SYSTEM_UI_FLAG_LAYOUT_STABLE or
                View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
        }
        window.statusBarColor = Color.TRANSPARENT
    }

    private fun observeViewModel() {
        viewModel.cameraXLiveData.getIsInternetAvailable().observe(this) { isAvailable: Boolean -> handleInternetConnection(isAvailable) }
        viewModel.cameraXLiveData.getCloseImageProxy().observe(this) { needToClose: Boolean -> cameraOverlayManager.handleCloseImageProxy(needToClose) }
        viewModel.cameraXLiveData.getCaptureAnImage().observe(this) { needToCallCapture: Boolean -> cameraOverlayManager.handleCaptureAnImage(needToCallCapture) }
        viewModel.cameraXLiveData.getCTimer().observe(this) { needToCloseTimer: Boolean -> cameraOverlayManager.handleCTimer(needToCloseTimer) }
        viewModel.cameraXLiveData.getCloseAnalysisUseCase().observe(this) { needToClearAnalyzer: Boolean ->
            cameraOverlayManager.handleAnalysisUseCase(needToClearAnalyzer)
        }
        viewModel.cameraXLiveData.getUpdateCustomLoader().observe(this) { needToUpdateCustomLoader: Boolean ->
            updateCustomLoader(needToUpdateCustomLoader)
        }
        viewModel.cameraXLiveData.getUnbindCameraProvider().observe(this) { isUnbind: Boolean -> cameraOverlayManager.unBindCameraProvider(isUnbind) }
        viewModel.cameraXLiveData.getIsOverlayVisible().observe(this) { visibility: Boolean ->
            logger.log("observeViewModel: overlay")
            binding.overlay.visibility = if (visibility) View.VISIBLE else View.GONE
        }
        viewModel.cameraXLiveData.getIsTransparentVisible().observe(this) { visibility: Boolean ->
            logger.log("observeViewModel: transparent")
            binding.lytTransparent.visibility = if (visibility) View.VISIBLE else View.GONE
        }
        viewModel.cameraXLiveData.getGlareTextVisibility().observe(this) { visibility: Boolean ->
            logger.log("observeViewModel: glare")
            binding.txtGlare.visibility = if (visibility) View.VISIBLE else View.GONE
        }
        viewModel.cameraXLiveData.getIsLayoutBottomVisible().observe(this) { visibility: Boolean ->
            logger.log("observeViewModel: bottom")
            binding.lytBottom.visibility = if (visibility) View.VISIBLE else View.GONE
        }
        viewModel.cameraXLiveData.getIsScanVisible().observe(this) { visibility: Boolean ->
            logger.log("observeViewModel: scan")
            binding.scan.visibility = if (visibility) View.VISIBLE else View.GONE
        }
        viewModel.cameraXLiveData.getIsScanCompleteVisible().observe(this) { visibility: Boolean ->
            logger.log("observeViewModel: scan complete")
            binding.lyoutScanComplete.visibility = if (visibility) View.VISIBLE else View.GONE
        }
        viewModel.cameraXLiveData.getIsScanFailVisible().observe(this) { visibility: Boolean ->
            logger.log("observeViewModel: scan fail")
            binding.lyoutScanFail.visibility = if (visibility) View.VISIBLE else View.GONE
        }
        viewModel.cameraXLiveData.getIsBackCardScanVisible().observe(this) { visibility: Boolean ->
            binding.imgBackcrdScan.visibility = if (visibility) View.VISIBLE else View.INVISIBLE
        }
        viewModel.cameraXLiveData.getStatusText().observe(this) { text: String -> binding.statusTxt.text = text }
        viewModel.cameraXLiveData.getGlareText().observe(this) { text: String -> binding.txtGlare.text = text }
        viewModel.cameraXLiveData.getBottomText().observe(this) { text: String -> binding.txtBottom.text = text }
    }

    private fun handleInternetConnection(isAvailable: Boolean) {
        if (!isAvailable) {
            ErrorUtility.getInstance().showNoInternetDialog(this, !this.isFinishing && !this.isDestroyed)
        }
    }

    private fun updateCustomLoader(needToUpdateCustomLoader: Boolean) {
        if (needToUpdateCustomLoader) {
            binding.customLoader.root.visibility = View.VISIBLE
            binding.customLoader.loaderContainer.setBackgroundColor(Color.WHITE)

            val layoutParams2 = binding.customLoader.imgLoader.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(R.dimen.custom_loader_image_width)
            layoutParams2.height = textSizeConverter.getHeight(R.dimen.custom_loader_image_height)
            binding.customLoader.imgLoader.layoutParams = layoutParams2

            val marginLayoutParam = binding.customLoader.txtValidating.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(0, textSizeConverter.getPaddingOrMarginValue(R.dimen.custom_loader_margin_top), 0, 0)
            binding.customLoader.txtValidating.layoutParams = marginLayoutParam
            binding.customLoader.txtValidating.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(R.dimen.custom_loader_text_size).toFloat(),
            )
            val rotation = AnimationUtils.loadAnimation(this@CameraXLiveActivity, R.anim.rotate)
            rotation.fillAfter = true
            binding.customLoader.imgLoader.startAnimation(rotation)
        }
    }

    private fun openBackPressDialog() {
        val dialog = Dialog(this)
        dialog.requestWindowFeature(Window.FEATURE_NO_TITLE)
        val binding = DialogEmailPhoneConfirmationBinding.inflate(dialog.layoutInflater)
        dialog.setContentView(binding.root)
        var layoutParams = binding.dialogHeader.layoutParams as LinearLayout.LayoutParams
        var padding = textSizeConverter.getPaddingOrMarginValue(16)
        layoutParams.setMargins(padding, padding, padding, 0)
        binding.dialogHeader.layoutParams = layoutParams
        val layoutParams2 = binding.cardConfirmation.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(280)
        binding.cardConfirmation.layoutParams = layoutParams2
        binding.cardConfirmation.radius = textSizeConverter.calculateRadius(14).toFloat()
        binding.dialogHeader.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(18).toFloat(),
        )
        binding.dialogSubtitle.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(14).toFloat(),
        )
        binding.btnNo.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeConverter.getTextSize(16).toFloat())
        binding.btnYes.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeConverter.getTextSize(16).toFloat())
        layoutParams = binding.dialogSubtitle.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(padding, padding, padding, padding)
        binding.dialogSubtitle.layoutParams = layoutParams
        padding = textSizeConverter.getPaddingOrMarginValue(12)
        layoutParams = binding.btnNo.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(0, padding, 0, padding)
        binding.btnNo.layoutParams = layoutParams
        layoutParams = binding.btnYes.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            textSizeConverter.getPaddingOrMarginValue(24),
            padding,
            textSizeConverter.getPaddingOrMarginValue(16),
            padding,
        )
        binding.btnYes.layoutParams = layoutParams
        binding.dialogHeader.typeface = ResourcesCompat.getFont(applicationContext, R.font.pingmedium)
        binding.dialogSubtitle.typeface = ResourcesCompat.getFont(applicationContext, R.font.regular)
        binding.dialogHeader.text = getString(R.string.back_dialog_header)
        binding.dialogSubtitle.text = getString(R.string.back_dialog_title)
        binding.btnYes.setOnClickListener {
            dialog.dismiss()
            VerifySDKManager.getInstance().getCallback()?.onSDKExit(VerifySDKExitStep.OcrCancelled)
            Utility.getInstance().restartApp(this)
        }
        binding.btnNo.setOnClickListener { dialog.dismiss() }
        dialog.show()
        dialog.setCancelable(false)
        dialog.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        dialog.window!!.attributes.windowAnimations = R.style.DialogAnimation
    }

    override fun onItemSelected(
        p0: AdapterView<*>?,
        p1: View?,
        p2: Int,
        p3: Long,
    ) {
        cameraOverlayManager.bindAnalysisUseCase()
    }

    override fun onNothingSelected(p0: AdapterView<*>?) {}

    override fun onDestroy() {
        super.onDestroy()
        viewModel.cameraXLiveData.resetState()
    }
}
