package nashid.verify.sdk.di

import nashid.verify.sdk.data.remote.ApiService
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdkNew.BuildConfig
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.Response
import okhttp3.ResponseBody.Companion.toResponseBody
import okhttp3.logging.HttpLoggingInterceptor
import org.json.JSONObject
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import java.io.IOException
import java.util.concurrent.TimeUnit

fun provideHttpClient(): OkHttpClient {
    val clientBuilder =
        OkHttpClient.Builder()
            .readTimeout(60, TimeUnit.SECONDS)
            .connectTimeout(60, TimeUnit.SECONDS)
            .addInterceptor(RetryInterceptor())
    if (BuildConfig.DEBUG) {
        val loggingInterceptor =
            HttpLoggingInterceptor().apply {
                level = HttpLoggingInterceptor.Level.BODY
            }
        clientBuilder.addInterceptor(loggingInterceptor)
    }
    return clientBuilder.build()
}

fun provideConverterFactory(): GsonConverterFactory = GsonConverterFactory.create()

fun provideRetrofit(
    okHttpClient: OkHttpClient,
    gsonConverterFactory: GsonConverterFactory,
): Retrofit {
    val baseUrl = SdkConfig.baseUrl
    return Retrofit
        .Builder()
        .baseUrl(baseUrl)
        .client(okHttpClient)
        .addConverterFactory(gsonConverterFactory)
        .build()
}

fun provideService(retrofit: Retrofit): ApiService = retrofit.create(ApiService::class.java)

class RetryInterceptor(
    private val maxRetries: Int = 3,
) : Interceptor {
    override fun intercept(chain: Interceptor.Chain): Response {
        var attempt = 0
        var response: Response? = null
        var exception: Exception?

        do {
            try {
                response?.close()
                response = chain.proceed(chain.request())
                if (response.isSuccessful) {
                    return response
                }
                val errorMessage = response.extractErrorMessage()
                throw IOException(errorMessage)
            } catch (e: Exception) {
                exception = e
            } finally {
                attempt++
            }
        } while (attempt < maxRetries)
        response?.close()
        throw IOException(exception?.message ?: "We’re experiencing an issue on our end.")
    }

    private fun Response.extractErrorMessage(): String {
        val responseBodyString = this.body?.string() ?: return "Unknown error occurred"

        return try {
            val json = JSONObject(responseBodyString)
            json.optString("message", "Unknown error occurred")
        } catch (e: Exception) {
            "Unable to parse server error message"
        } finally {
            val newResponseBody = responseBodyString.toResponseBody(this.body?.contentType())
            this.newBuilder().body(newResponseBody).build()
        }
    }
}
