package nashid.verify.sdk.ui

import android.content.Intent
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Matrix
import android.os.Bundle
import android.util.TypedValue
import android.view.View
import android.view.animation.AnimationUtils
import android.widget.LinearLayout
import androidx.activity.addCallback
import androidx.activity.result.contract.ActivityResultContracts
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.async
import kotlinx.coroutines.delay
import kotlinx.coroutines.joinAll
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import nashid.verify.sdk.VerifySDKManager
import nashid.verify.sdk.model.VerifySDKExitStep
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.PermissionAndLocationHelper
import nashid.verify.sdk.utils.RetryHelper
import nashid.verify.sdk.utils.RetryHelper.RetryOperationType
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.ErrorUtility
import nashid.verify.sdk.utils.helpers.TextSizeConverter
import nashid.verify.sdk.viewmodel.SkipNfcLiveNessViewModel
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.ActivitySkipNfcBinding
import org.koin.android.ext.android.inject

class SkipNfcLiveNessActivity : BaseActivity() {
    private lateinit var binding: ActivitySkipNfcBinding
    private val textSizeConverter: TextSizeConverter by inject()
    private val viewModel: SkipNfcLiveNessViewModel by inject()
    private val locationRetryConfig = RetryHelper.getRetryConfig(RetryOperationType.LOCATION_REQUEST)
    private var currentRetryAttempt = 0
    private var hasSubmittedFailure = false
    private var lastRetryTimestamp = 0L

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivitySkipNfcBinding.inflate(layoutInflater)
        setContentView(binding.root)
        initView()
        backPress()
    }

    private fun handleIntentExtras() {
        if (intent.hasExtra(getString(R.string.doc_key))) {
            val selectedDoc = intent.getStringExtra(getString(R.string.doc_key))
            if (selectedDoc != null) {
                viewModel.getLiveNessData().setSelectedDoc(selectedDoc)
            }
        }
    }

    private fun observeViewModel() {
        viewModel.getLiveNessData().getFinishActivity().observe(this) { handleFinishActivity(it) }
        viewModel.getLiveNessData().getIsLiveness().observe(this) { handleLiveNess(it) }
        viewModel.getLiveNessData().getInternetConnection().observe(this) { handleInternetConnection(it) }
        viewModel.getLiveNessData().getHandleLocationData().observe(this) { handleLocationData(it) }
        viewModel.getLiveNessData().getLivenessBitmap().observe(this) {
            it?.let { it1 ->
                setLiveNessBitmap(
                    it1,
                )
            }
        }
        viewModel.getLiveNessData().getScanBitmap().observe(this) { it?.let { it1 -> setScanBitmap(it1) } }
        viewModel.getLiveNessData().getFaceMatchApiFail().observe(this) {}
        viewModel.getLiveNessData().getRetryUploadArtifacts().observe(this) {
            if (it) {
                handleRetryUploadArtifacts()
            }
        }
    }

    private fun handleInternetConnection(isAvailable: Boolean) {
        if (!isAvailable) {
            ErrorUtility.getInstance().showNoInternetDialog(this, !this.isFinishing && !this.isDestroyed)
        }
    }

    private fun setLiveNessBitmap(bitmap: Bitmap) {
        binding.imgTest123.setImageBitmap(bitmap)
    }

    private fun setScanBitmap(bitmap: Bitmap) {
        binding.imgTest12.setImageBitmap(bitmap)
    }

    private fun handleLiveNess(isLiveNess: Boolean) {
        if (isLiveNess) {
            // Check for missing liveness images and set failure flags before opening liveness screen
            if (SdkConfig.isActiveLiveNessEnabled && Utility.getInstance().getActiveLivenessImage() == null) {
                Utility.getInstance().setActiveLivenessFailed(true)
            }
            if (SdkConfig.isPassiveLiveNessEnabled && Utility.getInstance().getPassiveLiveNessImage() == null) {
                Utility.getInstance().setPassiveLivenessFailed(true)
            }
            openLiveNessScreen()
        } else {
            // Check for OCR face detection failure
            if (Utility.getInstance().hasFaceDetectionFailed()) {
                handleOcrFaceDetectionFailure()
            }
        }
    }

    private fun openLiveNessScreen() {
        val intent = Intent(this, LiveNessActivity::class.java)
        someActivityResultLauncher.launch(intent)
    }

    private val someActivityResultLauncher =
        registerForActivityResult(
            ActivityResultContracts.StartActivityForResult(),
        ) { result ->
            if (result.resultCode == RESULT_OK) {
                if (Utility.getInstance().liveImage != null) {
                    handleLiveNessImage()
                } else {
                    // If both enabled, set both failed; otherwise, set the relevant one
                    if (SdkConfig.isActiveLiveNessEnabled && SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().setActiveLivenessFailed(true)
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    } else if (SdkConfig.isActiveLiveNessEnabled) {
                        Utility.getInstance().setActiveLivenessFailed(true)
                    } else if (SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    }
                    viewModel.updateArtifacts()
                    viewModel.validateJustCompletion()
                }
            } else if (result.resultCode == RESULT_CANCELED) {
                if (Utility.getInstance().hasPassiveLivenessMaxRetries() || Utility.getInstance().hasActiveLivenessMaxRetries()) {
                    if (SdkConfig.isActiveLiveNessEnabled && SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().setActiveLivenessFailed(true)
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    } else if (SdkConfig.isActiveLiveNessEnabled) {
                        Utility.getInstance().setActiveLivenessFailed(true)
                    } else if (SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    }
                    viewModel.updateArtifacts()
                    viewModel.validateJustCompletion()
                } else {
                    VerifySDKManager.getInstance().getCallback()?.onSDKExit(VerifySDKExitStep.LivenessCancelled)
                    Utility.getInstance().restartApp(this)
                }
            }
        }

    private fun handleLiveNessImage() {
        CoroutineScope(Dispatchers.IO).launch {
            try {
                logger.log("Starting parallel processing of liveness images")

                // Get image bytes before parallel processing to avoid race conditions
                val activeImageBytes =
                    if (SdkConfig.isActiveLiveNessEnabled) {
                        Utility.getInstance().getActiveLivenessImage()
                    } else {
                        null
                    }

                val passiveImageBytes =
                    if (SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().getPassiveLiveNessImage()
                    } else {
                        null
                    }

                // Process both liveness images in parallel using async
                val activeJob =
                    if (SdkConfig.isActiveLiveNessEnabled && activeImageBytes != null) {
                        async {
                            logger.log("Processing active liveNess image with size: ${activeImageBytes.size}")
                            try {
                                val activeLiveNessImage = processActiveLiveNessImage(activeImageBytes).compressImage()
                                logger.log("Active liveNess image processed successfully")
                                Pair(activeLiveNessImage, ArtifactType.ACTIVE_LIVENESS_IMAGE)
                            } catch (e: Exception) {
                                logger.log("Failed to process active liveness image: ${e.message}")
                                Utility.getInstance().setActiveLivenessFailed(true)
                                null
                            }
                        }
                    } else {
                        if (SdkConfig.isActiveLiveNessEnabled) {
                            logger.log("Active liveNess image not found in Utility")
                            Utility.getInstance().setActiveLivenessFailed(true)
                        }
                        null
                    }

                val passiveJob =
                    if (SdkConfig.isPassiveLiveNessEnabled && passiveImageBytes != null) {
                        async {
                            logger.log("Processing passive liveNess image with size: ${passiveImageBytes.size}")
                            try {
                                val passiveLiveNessImage = processPassiveLiveNessImage(passiveImageBytes).compressImage()
                                logger.log("Passive liveNess image processed successfully")
                                Pair(passiveLiveNessImage, ArtifactType.PASSIVE_LIVENESS_IMAGE)
                            } catch (e: Exception) {
                                logger.log("Failed to process passive liveness image: ${e.message}")
                                Utility.getInstance().setPassiveLivenessFailed(true)
                                null
                            }
                        }
                    } else {
                        if (SdkConfig.isPassiveLiveNessEnabled) {
                            logger.log("Passive liveNess image not found in Utility")
                            Utility.getInstance().setPassiveLivenessFailed(true)
                        }
                        null
                    }

                // Update UI to show processing started
                withContext(Dispatchers.Main) {
                    binding.txtValidating.text = getString(R.string.validating)
                    binding.imgLoader.visibility = View.VISIBLE
                }

                // Wait for both processing jobs to complete
                val activeResult = activeJob?.await()
                val passiveResult = passiveJob?.await()

                logger.log("Both liveness images processed, starting parallel uploads")

                // Upload both images in parallel with delays to prevent race conditions
                val uploadJobs = mutableListOf<Job>()

                activeResult?.let { (bitmap, artifactType) ->
                    val uploadJob =
                        launch {
                            logger.log("Uploading active liveness image")
                            viewModel.uploadArtifact(bitmap, artifactType)
                            logger.log("Active liveness image upload initiated")
                        }
                    uploadJobs.add(uploadJob)
                }

                passiveResult?.let { (bitmap, artifactType) ->
                    val uploadJob =
                        launch {
                            // Small delay to ensure different timestamps
                            delay(100)
                            logger.log("Uploading passive liveness image")
                            viewModel.uploadArtifact(bitmap, artifactType)
                            logger.log("Passive liveness image upload initiated")
                        }
                    uploadJobs.add(uploadJob)
                }

                // Wait for all uploads to be initiated (not necessarily completed)
                uploadJobs.joinAll()
                logger.log("All liveness image uploads initiated successfully")

                withContext(Dispatchers.Main) {
                    viewModel.validateJustCompletion()
                    binding.imgLoader.visibility = View.GONE
                }
            } catch (e: Exception) {
                logger.log("Error in handleLivenNessImage : $e")
                withContext(Dispatchers.Main) {
                    binding.imgLoader.clearAnimation()
                    binding.imgLoader.visibility = View.GONE
                    binding.txtValidating.text = getString(R.string.some_thing_went_wrong)
                    binding.retryButton.visibility = View.VISIBLE

                    // Mark the correct liveness scan as failed
                    if (SdkConfig.isActiveLiveNessEnabled && SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().setActiveLivenessFailed(true)
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    } else if (SdkConfig.isActiveLiveNessEnabled) {
                        Utility.getInstance().setActiveLivenessFailed(true)
                    } else if (SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    }
                    viewModel.updateArtifacts()
                }
            }
        }
    }

    private suspend fun processActiveLiveNessImage(imageBytes: ByteArray): Bitmap =
        withContext(Dispatchers.IO) {
            try {
                logger.log("Processing active liveness image with ${imageBytes.size} bytes")
                val liveImage =
                    BitmapFactory.decodeByteArray(
                        imageBytes,
                        0,
                        imageBytes.size,
                    ) ?: throw IllegalStateException("Failed to decode active liveNess image")

                val matrix = Matrix()
                val rotation =
                    when (liveImage.width > liveImage.height) {
                        true -> 270f
                        false -> 0f
                    }
                matrix.postRotate(rotation)

                if (rotation != 0f) {
                    Bitmap.createBitmap(
                        liveImage,
                        0,
                        0,
                        liveImage.width,
                        liveImage.height,
                        matrix,
                        true,
                    ).also {
                        if (it != liveImage) {
                            liveImage.recycle()
                        }
                    }
                } else {
                    liveImage
                }
            } catch (e: Exception) {
                logger.log("Error processing active liveNess image : $e")
                throw e
            }
        }

    private suspend fun processPassiveLiveNessImage(imageBytes: ByteArray): Bitmap =
        withContext(Dispatchers.IO) {
            try {
                logger.log("Processing passive liveness image with ${imageBytes.size} bytes")
                val liveImage =
                    BitmapFactory.decodeByteArray(
                        imageBytes,
                        0,
                        imageBytes.size,
                    ) ?: throw IllegalStateException("Failed to decode passive liveNess image")

                val matrix = Matrix()
                val rotation =
                    when (liveImage.width > liveImage.height) {
                        true -> 270f
                        false -> 0f
                    }
                matrix.postRotate(rotation)

                if (rotation != 0f) {
                    Bitmap.createBitmap(
                        liveImage,
                        0,
                        0,
                        liveImage.width,
                        liveImage.height,
                        matrix,
                        true,
                    ).also {
                        if (it != liveImage) {
                            liveImage.recycle()
                        }
                    }
                } else {
                    liveImage
                }
            } catch (e: Exception) {
                logger.log("Error processing passive liveNess image : $e")
                throw e
            }
        }

    private fun handleFinishActivity(shouldFinish: Boolean) {
        if (shouldFinish) {
            binding.imgLoader.clearAnimation()
            binding.imgLoader.visibility = View.GONE
            binding.txtValidating.visibility = View.GONE
            Utility.getInstance().restartApp(this)
        }
    }

    private fun backPress() {
        onBackPressedDispatcher.addCallback(this) {
            if (viewModel.getLiveNessData().getIsApiCalled().value == false) onBackPressedDispatcher.onBackPressed()
        }
    }

    private fun setLayoutAndTextSize() {
        textSizeConverter.changeStatusBarColor(this)
        val layoutParams2 = binding.imgLoader.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(24)
        layoutParams2.height = textSizeConverter.getHeight(24)
        binding.imgLoader.layoutParams = layoutParams2

        binding.imgLoader.setColorFilter(SdkConfig.sdkAppTheme.getPrimaryColorInt())

        val marginLayoutParam = binding.txtValidating.layoutParams as LinearLayout.LayoutParams
        marginLayoutParam.setMargins(0, textSizeConverter.getPaddingOrMarginValue(32), 0, 0)
        binding.txtValidating.layoutParams = marginLayoutParam
        binding.txtValidating.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
        binding.txtValidating.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(20).toFloat(),
        )

        val rotation = AnimationUtils.loadAnimation(this, R.anim.rotate)
        rotation.fillAfter = true
        binding.imgLoader.startAnimation(rotation)

        binding.lytMainSkipNfc.setBackgroundColor(SdkConfig.sdkAppTheme.getBackgroundColorInt())
    }

    private fun initView() {
        setLayoutAndTextSize()
        handleIntentExtras()
        observeViewModel()
        viewModel.handleInternetConnectionData(isInternetAvailable)
    }

    override fun onAvailable() {
        if (viewModel.getLiveNessData().getIsApiCalled().value == false) {
            viewModel.handleInternetConnectionData(
                isInternetAvailable,
            )
        }
    }

    private fun handleLocationData(callLocationData: Boolean) {
        if (callLocationData) {
            CoroutineScope(Dispatchers.IO).launch {
                try {
                    // For regular location data requests (not missing artifacts)
                    withContext(Dispatchers.Main) {
                        PermissionAndLocationHelper.requestPermissionAndLocation(
                            this@SkipNfcLiveNessActivity,
                            object : PermissionAndLocationHelper.ResultCallback {
                                override fun onResult(
                                    latitude: Double,
                                    longitude: Double,
                                    address: String,
                                ) {
                                    viewModel.submitVerification(
                                        latitude,
                                        longitude,
                                        false,
                                    )
                                }
                            },
                        )
                    }
                } catch (e: Exception) {
                    logger.log("Error in handleLocationData: ${e.message}")
                    withContext(Dispatchers.Main) {
                        binding.imgLoader.visibility = View.GONE
                        binding.txtValidating.text = getString(R.string.some_thing_went_wrong)
                    }
                }
            }
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        ErrorUtility.getInstance().unregisterConnectivityManager(this)
        viewModel.getLiveNessData().clear()

        // Reset retry state on destroy
        currentRetryAttempt = 0
        hasSubmittedFailure = false
        lastRetryTimestamp = 0L
    }

    private fun Bitmap.compressImage(
        maxWidth: Int = 1024,
        maxHeight: Int = 1024,
    ): Bitmap {
        if (width <= maxWidth && height <= maxHeight) {
            return this
        }

        val ratioX = maxWidth.toFloat() / width
        val ratioY = maxHeight.toFloat() / height
        val ratio = minOf(ratioX, ratioY)

        val newWidth = (width * ratio).toInt()
        val newHeight = (height * ratio).toInt()

        return Bitmap.createScaledBitmap(this, newWidth, newHeight, true).also {
            if (this != it) {
                this.recycle()
            }
        }
    }

    private fun handleOcrFaceDetectionFailure() {
        if (!hasSubmittedFailure) {
            hasSubmittedFailure = true
            binding.txtValidating.text = getString(R.string.validating)
            binding.imgLoader.visibility = View.VISIBLE
            setLayoutAndTextSize()

            CoroutineScope(Dispatchers.IO).launch {
                try {
                    requestLocationWithRetry()
                } catch (e: Exception) {
                    logger.log("Error in handleOcrFaceDetectionFailure: ${e.message}")
                    withContext(Dispatchers.Main) {
                        binding.imgLoader.visibility = View.GONE
                        binding.txtValidating.text = getString(R.string.some_thing_went_wrong)
                    }
                }
            }
        }
    }

    private fun handleRetryUploadArtifacts() {
        CoroutineScope(Dispatchers.IO).launch {
            try {
                if (Utility.getInstance().hasFaceDetectionFailed()) {
                    handleFaceDetectionFailure()
                } else if (RetryHelper.shouldRetry(
                        currentRetryAttempt,
                        locationRetryConfig.maxRetries,
                        null,
                    )
                ) {
                    val currentTime = System.currentTimeMillis()
                    val timeSinceLastRetry = currentTime - lastRetryTimestamp
                    val minRetryInterval = 2000L
                    if (timeSinceLastRetry < minRetryInterval) {
                        delay(minRetryInterval - timeSinceLastRetry)
                    }
                    currentRetryAttempt++
                    lastRetryTimestamp = System.currentTimeMillis()
                    logger.log("Retrying artifact upload. Attempt $currentRetryAttempt of ${locationRetryConfig.maxRetries}")

                    withContext(Dispatchers.Main) {
                        binding.txtValidating.text = getString(R.string.validating)
                        binding.imgLoader.visibility = View.VISIBLE
                        setLayoutAndTextSize()
                    }

                    RetryHelper.delayWithBackoff(
                        attempt = currentRetryAttempt - 1,
                        customBaseDelay = locationRetryConfig.baseDelayMs,
                    )

                    withContext(Dispatchers.Main) {
                        handleLocationData(true)
                    }
                } else {
                    logger.log("Max retries reached or retry conditions not met. Submitting failure.")
                    handleFaceDetectionFailure()
                }
            } catch (e: Exception) {
                logger.log("Error in handleRetryUploadArtifacts: ${e.message}")
                withContext(Dispatchers.Main) {
                    handleFaceDetectionFailure()
                }
            }
        }
    }

    /**
     * Handle face detection failure with optimized retry
     */
    private suspend fun handleFaceDetectionFailure() {
        if (!hasSubmittedFailure) {
            hasSubmittedFailure = true

            withContext(Dispatchers.Main) {
                binding.txtValidating.text = getString(R.string.validating)
                binding.imgLoader.visibility = View.VISIBLE
                setLayoutAndTextSize()
            }

            requestLocationWithRetry()
        }
    }

    /**
     * Request location with retry mechanism
     */
    private suspend fun requestLocationWithRetry() {
        withContext(Dispatchers.Main) {
            PermissionAndLocationHelper.requestPermissionAndLocation(
                this@SkipNfcLiveNessActivity,
                object : PermissionAndLocationHelper.ResultCallback {
                    override fun onResult(
                        latitude: Double,
                        longitude: Double,
                        address: String,
                    ) {
                        CoroutineScope(Dispatchers.IO).launch {
                            try {
                                viewModel.missingArtifactsVerification(
                                    latitude,
                                    longitude,
                                    false,
                                )
                                logger.log("Location request successful after $currentRetryAttempt attempts")
                            } catch (e: Exception) {
                                logger.log("Error in location callback: ${e.message}")
                            }
                        }
                    }
                },
            )
        }
    }
}
