package nashid.verify.sdk.utils.helpers

import android.util.Log
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.GeneralHelper.Companion.fuzzyMapping

class CardReaderHelper {
    companion object {
        fun isOmanCard(
            a: String,
            b: String,
            threshold: Double,
        ): Boolean {
            val maxLength = maxOf(a.length, b.length)
            val distance = fuzzyMapping(a, b)
            val similarity = 1 - (distance.toDouble() / maxLength)
            Log.d("CardReaderHelper", "maxLength:$maxLength, distance:$distance, similarity:$similarity")
            return similarity >= threshold
        }

        fun normalizeOcrCharForBahrainType(ch: Char): Char {
            return when (ch) {
                'K' -> 'C'
                '0' -> 'O'
                else -> ch
            }
        }

        fun isOmanCard(input: String): Boolean = isOmanCard(input.uppercase(), b = "ROYAL OMAN POLICE D.G. OF CIVIL STATUS", threshold = 0.29)

        fun isKuwCard(input: String): Boolean {
            val kuwIdentifiers =
                listOf(
                    "مؤسسة ندى الخليج",
                    "الرقم المدني",
                    "فصيلة الدم",
                    "القطعة",
                    "الوحدة",
                    "شارع",
                    "منزل",
                    "Serial No:",
                )

            return kuwIdentifiers.any { identifier ->
                input.contains(identifier)
            }
        }

        fun isBahCard(input: String): Boolean {
            val bahIdentifiers =
                listOf(
                    "DATE OF BIRTH",
                    "GENDER",
                    "DISABILITY",
                    "EXPIRY DATE",
                    "SIGNATURE OF CARDHOLDER",
                    "VISION & DISABILITY",
                    "DRIVING LICENSE TYPE",
                    "FIRST ISSUE DATE",
                    "PLACE & DATE OF BIRTH",
                    "BLOOD GROUP",
                    "MANAMA",
                    "ICBHR",
                    "مكان وتاريخ الميلاد",
                    "الجنس",
                    "فصيلة الدم",
                    "توقيع حامل البطاقة",
                    "النظر و الإعاقة",
                    "نوع رخصة القيادة",
                    "تاريخ أول اصدار",
                )

            return bahIdentifiers.any { identifier ->
                input.uppercase().contains(identifier.uppercase())
            }
        }

        fun isUaeCard(input: String): Boolean {
            val uaeIdentifiers =
                listOf(
                    "CARD NUMBER",
                    "DATE OF BIRTH",
                    "SIGNATURE",
                    "UAE",
                    "IF YOU FIND THIS CARD",
                )

            return uaeIdentifiers.any { identifier ->
                input.uppercase().contains(identifier)
            }
        }

        fun cleanString(input: String): String {
            if (input.length < 18) return ""
            Log.d("CardReaderHelper", "length: " + input.length + ", input:$input")

            if (Utility.getInstance().isKuwCard == true) {
                val nationalityMatch = "(\\w{3})(\\d+)".toRegex().find(input)
                if (nationalityMatch != null) {
                    return input
                }
            }
            val groups =
                listOf(
                    input.substring(0, 7),
                    input.substring(7, 8),
                    input.substring(8, 15),
                    input.substring(15, 18),
                )

            val mappedGroups =
                groups.mapIndexed { index, group ->
                    when (index) {
                        0, 2 -> replaceAlfaWithNumber(group)
                        1, 3 -> replaceNumberWithAlfa(group)
                        else -> group
                    }
                }

            var result = mappedGroups.joinToString("")
            result += "<".repeat(29 - result.length)
            result += input.last()
            Log.d("CardReaderHelper", "length: " + result.length + ", result:$result")
            return result
        }

        fun replaceNumberWithAlfa(str: String): String {
            val replacementDigitMap =
                mapOf(
                    '0' to 'O',
                    '1' to 'I',
                    '2' to 'Z',
                    '5' to 'S',
                    'r' to 'F',
                )
            return str.map { replacementDigitMap.getOrElse(it) { it } }.joinToString("")
        }

        fun replaceAlfaWithNumber(str: String): String {
            val replacementCharMap =
                mapOf(
                    'Z' to '2',
                    'S' to '5',
                    'I' to '1',
                    'O' to '0',
                    'B' to '8',
                    'T' to '1',
                )
            return str.map { replacementCharMap.getOrElse(it) { it } }.joinToString("")
        }

        internal fun correctOcrName(name: String): String {
            return name
                .replace('1', 'I')
                .replace('0', 'O')
                .replace('5', 'S')
                .replace('8', 'B')
                .replace('6', 'G')
                .replace('2', 'Z')
                .replace('9', 'G')
                .trim()
        }
    }
}
