package nashid.verify.sdk.utils

import android.app.Activity
import android.content.Context
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.os.Build
import android.view.View
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.MultipartBody
import okhttp3.RequestBody.Companion.asRequestBody
import java.io.File
import java.net.NetworkInterface
import java.text.SimpleDateFormat
import java.util.Calendar
import java.util.Collections
import java.util.Date
import java.util.Locale
import java.util.TimeZone

internal class Utility private constructor() {
    var name: String? = null
    var passportNumber: String? = null
    var dateOfBirth: String? = null
    var expiryDate: String? = null
    var fullName: String? = null
    var gender: String = ""
    var nationality: String? = null
    var country: String? = null
    var documentType: String? = null
    var addressNumber: String? = null
    var scannedImage: ByteArray? = null
    var croppedFrontScanImage: Bitmap? = null
    var scannedIdFrontView: ByteArray? = null
    var cardNumber: String? = null

    var mrzLine1: String? = null
    var mrzLine2: String? = null
    var mrzLine3: String? = null

    var dgFile1: ByteArray? = null
    var dgFile2: ByteArray? = null
    var dgFile4: ByteArray? = null
    var dgFile6: ByteArray? = null
    var dgFile10: ByteArray? = null
    var dgFile11: ByteArray? = null
    var dgFile13: ByteArray? = null
    var livenessScannedImage: ByteArray? = null
    var nfcScannedImage: ByteArray? = null
    var matchPercentage: Int = 0
    var liveNessScore: Int = 0
    var dateOfExpiration: String? = null
    var selectedDoc: String? = null
    var liveImage: ByteArray? = null
    var isUAECard: Boolean? = false
    var isKuwCard: Boolean? = false
    var isBahCard: Boolean? = false
    var isOmanCard: Boolean? = false
    var isKSACard: Boolean? = false
    var isQatarCard: Boolean? = false
    var isBarCodeCard: Boolean = false
    var barcodeIdNumber: String = ""
    var barcodeSerialNumber: String = ""
    var isNationalIdCard: Boolean = false
    var isResidentIdCard: Boolean = false

    private val artifactFileKeys:
        MutableMap<ArtifactType, String> = mutableMapOf()
    private val artifacts =
        listOf(
            ArtifactType.FRONTSIDE_IMAGE,
            ArtifactType.BACKSIDE_IMAGE,
            ArtifactType.OCR_FACE_IMAGE,
            ArtifactType.NFC_IMAGE,
            ArtifactType.ACTIVE_LIVENESS_IMAGE,
            ArtifactType.PASSIVE_LIVENESS_IMAGE,
        )
    private var nfcScanFailed: Boolean = false
    private var ocrScanFailed: Boolean = false
    private var liveNessScanFailed: Boolean = false

    private var activeLivenessFailed: Boolean = false
    private var passiveLivenessFailed: Boolean = false

    private var activeLivenessImage: ByteArray? = null
    private var passiveLiveNessImage: ByteArray? = null

    private var ocrMaxRetries: Boolean = false
    private var nfcMaxRetries: Boolean = false
    private var activeLivenessMaxRetries: Boolean = false
    private var passiveLivenessMaxRetries: Boolean = false

    private var faceDetectionFailed = false

    fun cleanup() {
        name = null
        passportNumber = null
        dateOfBirth = null
        expiryDate = null
        fullName = null
        gender = ""
        nationality = null
        country = null
        scannedImage = null
        scannedIdFrontView = null
        documentType = null
        addressNumber = null
        dgFile1 = null
        dgFile2 = null
        dgFile4 = null
        dgFile6 = null
        dgFile10 = null
        dgFile11 = null
        dgFile13 = null
        mrzLine1 = null
        mrzLine2 = null
        mrzLine3 = null
        livenessScannedImage = null
        nfcScannedImage = null
        matchPercentage = 0
        liveNessScore = 0
        dateOfExpiration = null
        selectedDoc = null
        liveImage = null
        isOmanCard = null
        isUAECard = null
        isKuwCard = null
        isBahCard = null
        isKSACard = null
        isQatarCard = null
        isBarCodeCard = false
        artifactFileKeys.clear()
        ocrScanFailed = false
        nfcScanFailed = false
        liveNessScanFailed = false
        activeLivenessImage = null
        passiveLiveNessImage = null
        activeLivenessFailed = false
        passiveLivenessFailed = false
        ocrMaxRetries = false
        nfcMaxRetries = false
        activeLivenessMaxRetries = false
        passiveLivenessMaxRetries = false
        faceDetectionFailed = false

        cardNumber = null
        barcodeIdNumber = ""
        barcodeSerialNumber = ""
        isNationalIdCard = false
        isResidentIdCard = false
    }

    fun setVisibility(
        view1: View?,
        view2: View?,
        isVisible: Boolean,
    ) {
        val visibility = if (isVisible) View.VISIBLE else View.GONE
        view1?.visibility = visibility
        view2?.visibility = visibility
    }

    fun bitmapToFile(
        bitmap: Bitmap,
        context: Context,
    ): File? {
        val maxSize = 30 * 1024 * 1024
        val file = File(context.cacheDir, "upload_image_${System.currentTimeMillis()}.jpg")

        val pixels = bitmap.width * bitmap.height
        val estimatedBytes = pixels * 3
        var quality =
            when {
                estimatedBytes <= 2 * 1024 * 1024 -> 85
                estimatedBytes <= 5 * 1024 * 1024 -> 70
                estimatedBytes <= 10 * 1024 * 1024 -> 55
                else -> 40
            }

        while (quality >= 30) {
            try {
                file.outputStream().use { outputStream ->
                    bitmap.compress(Bitmap.CompressFormat.JPEG, quality, outputStream)
                    outputStream.flush()
                }

                if (file.length() <= maxSize) {
                    return file
                }

                quality -= if (file.length() > maxSize * 2) 20 else 10
            } catch (e: Exception) {
                e.printStackTrace()
                file.delete()
                return null
            }
        }
        file.delete()
        return null
    }

    fun createMultipartBody(file: File): MultipartBody.Part {
        val requestFile = file.asRequestBody("multipart/form-data".toMediaTypeOrNull())
        return MultipartBody.Part.createFormData("file", file.name, requestFile)
    }

    fun getAppKey(): String {
        return SdkConfig.appKey
    }

    fun getCurrentFormattedDateTime(): String? {
        return try {
            val calendar = Calendar.getInstance(TimeZone.getTimeZone("GMT"))
            val date = calendar.time
            val sdf = SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.ENGLISH)
            sdf.timeZone = TimeZone.getTimeZone("GMT")
            sdf.format(date)
        } catch (e: Exception) {
            e.printStackTrace()
            null
        }
    }

    fun getDeviceIdentifier(): String = "${Build.BRAND}_${Build.MODEL}"

    fun getDeviceType(): String = Build.MODEL

    fun getSystemVersion(): String = Build.VERSION.RELEASE

    @Suppress("DEPRECATION")
    fun getAppVersion(context: Context): String? {
        try {
            val pInfo: PackageInfo = context.packageManager.getPackageInfo(context.packageName, 0)
            return "${pInfo.versionName}(${if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) pInfo.longVersionCode else pInfo.versionCode})"
        } catch (e: PackageManager.NameNotFoundException) {
            e.printStackTrace()
        }
        return null
    }

    fun getSystemName(): String = "Android"

    fun fetchPublicIPAddresses(onResult: (String?, String?) -> Unit) {
        CoroutineScope(Dispatchers.IO).launch {
            val (ipv4, ipv6) = getLocalIPAddresses()
            withContext(Dispatchers.Main) {
                onResult(ipv4, ipv6)
            }
        }
    }

    private fun getLocalIPAddresses(): Pair<String?, String?> {
        var ipv4Address: String? = null
        var ipv6Address: String? = null

        try {
            val networkInterfaces = Collections.list(NetworkInterface.getNetworkInterfaces())
            for (networkInterface in networkInterfaces) {
                if (!networkInterface.isUp || networkInterface.isLoopback) continue

                val addresses = Collections.list(networkInterface.inetAddresses)
                for (address in addresses) {
                    if (address.isLoopbackAddress) continue

                    val hostAddress = address.hostAddress
                    if (hostAddress?.contains(':') == true) {
                        // IPv6 address
                        if (ipv6Address == null) {
                            // Remove scope id if present
                            ipv6Address = hostAddress.split('%')[0]
                        }
                    } else {
                        // IPv4 address
                        if (ipv4Address == null) {
                            ipv4Address = hostAddress
                        }
                    }
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }

        return Pair(ipv4Address, ipv6Address)
    }

    fun getFormattedDate(dateStr: String): String? {
        return try {
            val inputFormat = SimpleDateFormat("yyMMdd", Locale.ENGLISH)
            val date: Date? = inputFormat.parse(dateStr)
            val outputFormat = SimpleDateFormat("yyyy-MM-dd", Locale.ENGLISH)
            date?.let { outputFormat.format(it) }
        } catch (e: Exception) {
            e.printStackTrace()
            null
        }
    }

    fun getNfcFormattedDate(dateStr: String): String? {
        return try {
            val inputFormat = SimpleDateFormat("yyyyMMdd", Locale.ENGLISH)
            val date: Date? = inputFormat.parse(dateStr)
            val outputFormat = SimpleDateFormat("yyyy-MM-dd", Locale.ENGLISH)
            date?.let { outputFormat.format(it) }
        } catch (e: Exception) {
            e.printStackTrace()
            null
        }
    }

    fun saveFileKeyAndContinue(
        fileKey: String,
        artifactType: String,
    ) {
        val mappedArtifactType = artifacts.find { it.type == artifactType }
        if (mappedArtifactType != null) {
            artifactFileKeys[mappedArtifactType] = fileKey
            println("artifact saved.  type: $artifactType  :: file key ::$fileKey")
        } else {
            println("Invalid artifact type: $artifactType")
        }
    }

    fun containsArabic(text: String): Boolean {
        return text.any { char -> char.code in 0x0600..0x06FF || char.code in 0xFE70..0xFEFF }
    }

    fun getFileKeyForArtifact(artifactType: ArtifactType): String? = artifactFileKeys[artifactType]

    fun getLivenessScore(): Int = liveNessScore

    fun restartApp(activity: Activity) {
        activity.finish()
    }

    fun clearArtifactKeys() {
        artifactFileKeys.clear()
    }

    fun setNfcScanFailed(failed: Boolean) {
        nfcScanFailed = failed
    }

    fun hasNfcScanFailed(): Boolean = nfcScanFailed

    fun setOcrScanFailed(failed: Boolean) {
        ocrScanFailed = failed
    }

    fun hasOcrScanFailed(): Boolean = ocrScanFailed

    fun setLiveNessScanFailed(failed: Boolean) {
        liveNessScanFailed = failed
    }

    fun hasLiveNessScanFailed(): Boolean = liveNessScanFailed

    fun ocrScanningFailed(): Boolean = ocrScanFailed || nfcScanFailed || liveNessScanFailed

    fun setActiveLivenessImage(image: ByteArray?) {
        activeLivenessImage = image
    }

    fun getActiveLivenessImage(): ByteArray? = activeLivenessImage

    fun setActiveLivenessFailed(failed: Boolean) {
        activeLivenessFailed = failed
    }

    fun hasActiveLivenessFailed(): Boolean = activeLivenessFailed

    fun setPassiveLivenessFailed(failed: Boolean) {
        passiveLivenessFailed = failed
    }

    fun hasPassiveLivenessFailed(): Boolean = passiveLivenessFailed

    fun setPassiveLiveNessImage(image: ByteArray?) {
        passiveLiveNessImage = image
    }

    fun getPassiveLiveNessImage(): ByteArray? = passiveLiveNessImage

    fun setOcrMaxRetries(failed: Boolean) {
        ocrMaxRetries = failed
    }

    fun hasOcrMaxRetries(): Boolean = ocrMaxRetries

    fun setNfcMaxRetries(failed: Boolean) {
        nfcMaxRetries = failed
    }

    fun hasNfcMaxRetries(): Boolean = nfcMaxRetries

    fun setActiveLivenessMaxRetries(failed: Boolean) {
        activeLivenessMaxRetries = failed
    }

    fun hasActiveLivenessMaxRetries(): Boolean = activeLivenessMaxRetries

    fun setPassiveLivenessMaxRetries(failed: Boolean) {
        passiveLivenessMaxRetries = failed
    }

    fun hasPassiveLivenessMaxRetries(): Boolean = passiveLivenessMaxRetries

    fun setFaceDetectionFailed(failed: Boolean) {
        faceDetectionFailed = failed
    }

    fun hasFaceDetectionFailed(): Boolean = faceDetectionFailed

    companion object {
        @Volatile
        private var ourInstance: Utility? = null

        fun getInstance(): Utility {
            return ourInstance ?: synchronized(this) {
                ourInstance ?: Utility().also { ourInstance = it }
            }
        }
    }
}
