package nashid.verify.sdk.ui

import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.annotation.SuppressLint
import android.app.Activity
import android.app.Dialog
import android.app.PendingIntent
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.res.ColorStateList
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Color
import android.graphics.Matrix
import android.graphics.drawable.ColorDrawable
import android.nfc.NfcAdapter
import android.nfc.Tag
import android.nfc.tech.IsoDep
import android.os.Build
import android.os.Bundle
import android.provider.Settings
import android.util.Log
import android.util.TypedValue
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.view.Window
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.ProgressBar
import android.widget.TextView
import androidx.activity.addCallback
import androidx.activity.result.contract.ActivityResultContracts
import com.airbnb.lottie.LottieDrawable
import nashid.verify.sdk.VerifySDKManager
import nashid.verify.sdk.model.ScanDocumentResult
import nashid.verify.sdk.model.TagProvider
import nashid.verify.sdk.model.VerifySDKExitStep
import nashid.verify.sdk.ui.theme.FontManager
import nashid.verify.sdk.ui.theme.FontWeight
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.OtherConstant
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.DocumentImages
import nashid.verify.sdk.utils.helpers.ErrorUtility
import nashid.verify.sdk.utils.helpers.NfcLayoutAndTextSize
import nashid.verify.sdk.utils.helpers.TextSizeConverter
import nashid.verify.sdk.utils.id_card.jmrtd.BACKey
import nashid.verify.sdk.utils.id_card.jmrtd.BACKeySpec
import nashid.verify.sdk.viewmodel.NFCViewModel
import nashid.verify.sdk.viewmodel.ScanDocumentViewModel
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.ActivityWaitingForNfcBinding
import nashid.verify.sdkNew.databinding.BottomSheetDialogBinding
import nashid.verify.sdkNew.databinding.DialogEmailPhoneConfirmationBinding
import nashid.verify.sdkNew.databinding.DialogFailureNfcBinding
import nashid.verify.sdkNew.databinding.DialogSuccessBinding
import org.koin.android.ext.android.inject
import java.util.Objects

class NfcActivity : BaseActivity() {
    private var binding: ActivityWaitingForNfcBinding? = null
    private val viewModel: NFCViewModel by inject()
    private val textSizeConverter: TextSizeConverter by inject()
    private val scanDocumentViewModel: ScanDocumentViewModel by inject()

    private var mNfcAdapter: NfcAdapter? = null
    private var pendingIntent: PendingIntent? = null
    private lateinit var nfcLayoutAndTextSize: NfcLayoutAndTextSize

    private var errorDialog: Dialog? = null
    private var progressDialog: Dialog? = null
    private var isActivityRunning = false
    private var progressBar: ProgressBar? = null
    private var progressText: TextView? = null
    private var maxRetryCount = 0
    private var isStartedReadNFC = false
    private var isNFCPermissionDialogOpen = false
    private lateinit var nfcNotsupportDialog: Dialog

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityWaitingForNfcBinding.inflate(layoutInflater)
        setContentView(binding!!.root)
        initView()
        backPress()
    }

    private fun initView() {
        isActivityRunning = true
        handleIntentData()
        nfcLayoutAndTextSize =
            NfcLayoutAndTextSize(
                textSizeConverter,
                binding!!,
                this,
                storage,
                viewModel,
                scanDocumentViewModel,
            )
        nfcLayoutAndTextSize.setLayoutAndTextSize()
        setOtherViews()
        handleInternetConnection()
        setObservable()
    }

    private fun setObservable() {
        viewModel.getNfcLiveData().skipNFC.observe(this) { skipNFC: Boolean -> nfcLayoutAndTextSize.handleSkipNFC(skipNFC) }
        viewModel.getNfcLiveData().isInternetAvailable.observe(this) { isAvailable: Boolean ->
            handleInternetConnectionData(
                isAvailable,
            )
        }
        viewModel.getNfcLiveData().mrtdPercentage.observe(this) { progress: Int ->
            setMRTDProgressBarPercentage(
                progress,
            )
        }
        viewModel.getNfcLiveData().showError.observe(this) { message: String? -> showError(message) }
        viewModel.getNfcLiveData().showResult.observe(this) { isDisplaying: Boolean -> displayResult(isDisplaying) }
        scanDocumentViewModel.result.observeForever { result ->
            if (result is ScanDocumentResult.FileKeyResult) {
                Utility.getInstance().saveFileKeyAndContinue(result.fileKey, result.artifactType)
                viewModel.validateLivensUser(this)
                finish()
            }
        }
    }

    private fun handleInternetConnectionData(isAvailable: Boolean) {
        logger.log("handleInternetConnectionData: $isAvailable")
        if (!isAvailable) {
            ErrorUtility
                .getInstance()
                .showNoInternetDialog(this, !this.isFinishing && !this.isDestroyed)
        }
    }

    private fun handleInternetConnection() {
        logger.log("handleInternetConnection: ")
        viewModel.setInternetConnection(isInternetAvailable)
        if (java.lang.Boolean.TRUE == viewModel.getNfcLiveData().isInternetAvailable.value) {
            viewModel.setIsApiCalled(true)
            initAdapter()
            if (errorDialog != null) {
                errorDialog!!.dismiss()
            }
            if (progressDialog != null) {
                progressDialog!!.dismiss()
            }
            TagProvider.closeTag()
        }
    }

    public override fun onNewIntent(intent: Intent) {
        super.onNewIntent(intent)
        logger.log("onNewIntent: ")
        if (!isStartedReadNFC) return
        errorDialog?.let {
            if (it.isShowing) {
                return
            }
        }

        val skipNfc = viewModel.getNfcLiveData().skipNFC.value == true
        val retryLimit = SdkConfig.nfcMaximumRetries ?: 3

        if (skipNfc && maxRetryCount >= retryLimit && retryLimit != 0) {
            return
        }
        TagProvider.closeTag()
        val tagFromIntent: Tag? =
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                intent.getParcelableExtra(NfcAdapter.EXTRA_TAG, Tag::class.java)
            } else {
                @Suppress("DEPRECATION")
                intent.getParcelableExtra(NfcAdapter.EXTRA_TAG)
            }
        if (tagFromIntent != null) {
            TagProvider.setTag(IsoDep.get(tagFromIntent))
        }
        println("Got new intent!")
        if (errorDialog != null) {
            errorDialog!!.dismiss()
        }
        readPassport(tagFromIntent)
    }

    private fun readPassport(tagFromIntent: Tag?) {
        if (java.lang.Boolean.TRUE == viewModel.getNfcLiveData().isApiCalled.value) {
            viewModel.getNfcLiveData().updateData()
            showDialog()
            if (Objects
                    .requireNonNull(viewModel.getNfcLiveData().selectedDoc.value)
                    .equals(getString(R.string.e_passport), ignoreCase = true)
            ) {
                viewModel.readPassportNfc(tagFromIntent)
            } else {
                if (viewModel
                        .getNfcLiveData()
                        .passportNumber.value!!
                        .isNotEmpty() &&
                    viewModel
                        .getNfcLiveData()
                        .dateOfExpiration.value!!
                        .isNotEmpty() &&
                    viewModel
                        .getNfcLiveData()
                        .dateOfBirth.value!!
                        .isNotEmpty()
                ) {
                    if (TagProvider.getTag() != null) {
                        val bacKey: BACKeySpec =
                            BACKey(
                                viewModel.getNfcLiveData().passportNumber.value,
                                viewModel.getNfcLiveData().dateOfBirth.value,
                                viewModel.getNfcLiveData().dateOfExpiration.value,
                            )
                        logger.log("readPassport: $bacKey")
                        viewModel.cancelNfcRead()
                        viewModel.readData(TagProvider.getTag(), bacKey)
                    }
                }
            }
        }
    }

    private fun showDialog() {
        if (isActivityRunning) {
            isStartedReadNFC = true
            if (progressDialog == null) {
                progressDialog = Dialog(this)
                progressDialog!!.requestWindowFeature(Window.FEATURE_NO_TITLE)
                val binding =
                    BottomSheetDialogBinding.inflate(
                        progressDialog!!.layoutInflater,
                    )
                progressDialog!!.setContentView(binding.root)
                if (storage.getPreferredLocale().equals("ar", ignoreCase = true)) {
                    binding.progressBar.rotation = 180f
                }
                progressBar = binding.progressBar
                progressText = binding.txtDialogPercentage
                var padding = textSizeConverter.getPaddingOrMarginValue(16)
                binding.mainDialogLyt.setPadding(padding, padding, padding, padding)
                val layoutParams2 = binding.progressBar.layoutParams
                layoutParams2.width = textSizeConverter.getWidth(200)
                layoutParams2.height = ViewGroup.LayoutParams.WRAP_CONTENT
                binding.progressBar.layoutParams = layoutParams2
                var layoutParams = binding.progressBar.layoutParams as LinearLayout.LayoutParams
                layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(8), 0, 0)
                binding.progressBar.layoutParams = layoutParams

                binding.progressBar.progressTintList = ColorStateList.valueOf(SdkConfig.sdkAppTheme.getPrimaryColorInt())

                binding.txtDialogPercentage.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
                binding.txtScanning.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
                binding.txtDialogDesc.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
                binding.txtDialogPercentage.setTextSize(
                    TypedValue.COMPLEX_UNIT_PX,
                    textSizeConverter.getTextSize(16).toFloat(),
                )
                binding.txtScanning.setTextSize(
                    TypedValue.COMPLEX_UNIT_PX,
                    textSizeConverter.getTextSize(20).toFloat(),
                )
                binding.txtDialogDesc.setTextSize(
                    TypedValue.COMPLEX_UNIT_PX,
                    textSizeConverter.getTextSize(16).toFloat(),
                )
                layoutParams = binding.txtScanning.layoutParams as LinearLayout.LayoutParams
                layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(24), 0, 0)
                binding.txtScanning.layoutParams = layoutParams
                layoutParams = binding.txtDialogDesc.layoutParams as LinearLayout.LayoutParams
                layoutParams.setMargins(
                    0,
                    textSizeConverter.getPaddingOrMarginValue(16),
                    0,
                    textSizeConverter.getPaddingOrMarginValue(28),
                )
                binding.txtDialogDesc.layoutParams = layoutParams
                if (Objects
                        .requireNonNull<String?>(viewModel.getNfcLiveData().selectedDoc.value)
                        .equals(getString(R.string.e_passport), ignoreCase = true)
                ) {
                    binding.txtDialogDesc.text = getString(R.string.dialog_dontmove_pass)
                } else {
                    binding.txtDialogDesc.text = getString(R.string.dialog_dontmove)
                }

                layoutParams = binding.btnCancel.layoutParams as LinearLayout.LayoutParams
                layoutParams.setMargins(
                    0,
                    0,
                    0,
                    textSizeConverter.getPaddingOrMarginValue(12),
                )
                binding.btnCancel.layoutParams = layoutParams
                binding.btnCancel.setCardBackgroundColor(SdkConfig.sdkAppTheme.getPrimaryColorInt())
                binding.btnCancel.radius = textSizeConverter.calculateRadius(8).toFloat()
                binding.txtBtnCancel.setTextColor(SdkConfig.sdkAppTheme.getBackgroundColorInt())
                padding = textSizeConverter.getPaddingOrMarginValue(12)
                binding.txtBtnCancel.setPadding(0, padding, 0, padding)

                progressDialog!!.setCancelable(false)
                progressDialog!!.window!!.setLayout(
                    ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.WRAP_CONTENT,
                )
                progressDialog!!.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
                progressDialog!!.window!!.attributes.windowAnimations = R.style.DialogAnimation
                progressDialog!!.window!!.setGravity(Gravity.BOTTOM)

                binding.btnCancel.setOnClickListener {
                    if (java.lang.Boolean.TRUE == viewModel.getNfcLiveData().skipNFC.value) {
                        showFailureDialog()
                    } else {
                        if (maxRetryCount >= (SdkConfig.nfcMaximumRetries ?: 3)) {
                            OtherConstant.getInstance().skipNFC = true
                            Utility.getInstance().setNfcScanFailed(true)
                            Utility.getInstance().setLiveNessScanFailed(true)
                            dismissDialog()
                            viewModel.validateLivensUser(this)
                            finish()
                        } else {
                            showFailureDialog()
                        }
                    }
                    progressDialog!!.dismiss()
                }
            }
            progressDialog!!.show()
        }
    }

    private fun showError(message: String?) {
        Log.d("NfcActivity", "showError: $message")
        if (message != null) {
            runOnUiThread {
                Log.d("TAG", "showError: $message")
                showFailureDialog()
            }
        }
    }

    private fun displayResult(isDisplaying: Boolean) {
        if (isDisplaying && !isFinishing && isActivityRunning) {
            showResult()
        }
    }

    private fun showResult() {
        if (progressDialog?.isShowing == true) {
            progressDialog?.dismiss()
        }
        setMRTDProgressBarPercentage(100)
        showSuccessDialog()
    }

    @SuppressLint("SetTextI18n")
    fun setMRTDProgressBarPercentage(progress: Int) {
        Log.d("NfcActivity", "setMRTDProgressBarPercentage: $progress")
        if (progress != 0) {
            runOnUiThread {
                if (progressBar != null && progressText != null) {
                    progressBar!!.progress = progress
                    progressText!!.text = "$progress%"
                }
            }
        }
    }

    private fun showSuccessDialog() {
        if (isActivityRunning) {
            val dialog = Dialog(this)
            dialog.requestWindowFeature(Window.FEATURE_NO_TITLE)
            val binding = DialogSuccessBinding.inflate(dialog.layoutInflater)
            dialog.setContentView(binding.root)
            val padding = textSizeConverter.getPaddingOrMarginValue(16)
            binding.mainDialogLyt.setPadding(padding, padding, padding, padding)
            val layoutParams2 = binding.imgScanComplete.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(56)
            layoutParams2.height = textSizeConverter.getHeight(56)
            binding.imgScanComplete.layoutParams = layoutParams2
            var layoutParams = binding.imgScanComplete.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(8), 0, 0)
            binding.imgScanComplete.layoutParams = layoutParams
            binding.txtScanCompleted.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(24).toFloat(),
            )
            binding.txtScanCompleted.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
            layoutParams = binding.txtScanCompleted.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(
                0,
                textSizeConverter.getPaddingOrMarginValue(16),
                0,
                textSizeConverter.getPaddingOrMarginValue(28),
            )
            binding.txtScanCompleted.layoutParams = layoutParams
            binding.imgScanComplete.playAnimation()
            binding.imgScanComplete.setAnimation(R.raw.scan_completed)
            binding.imgScanComplete.addAnimatorListener(
                object : AnimatorListenerAdapter() {
                    override fun onAnimationEnd(animation: Animator) {
                        super.onAnimationEnd(animation)
                        dialog.dismiss()
                        TagProvider.closeTag()
                        dismissDialog()
                        if (Objects
                                .requireNonNull<String>(viewModel.getNfcLiveData().selectedDoc.value)
                                .equals(
                                    getString(R.string.e_passport),
                                    ignoreCase = true,
                                )
                        ) {
                            uploadArtifact(viewModel.getEDocument().personDetails.faceImage)
                        } else {
                            uploadArtifact(DocumentImages.instance.nfcPhoto)
                        }
                    }
                },
            )
            dialog.show()
            dialog.setCancelable(false)
            dialog.window?.setLayout(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
            )
            dialog.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
            dialog.window!!.attributes.windowAnimations = R.style.DialogAnimation
            dialog.window!!.setGravity(Gravity.BOTTOM)
        }
    }

    private fun uploadArtifact(bitmap: Bitmap?) {
        bitmap?.let {
            uploadArtifacts(it)
        }
    }

    private fun uploadArtifacts(bitmap: Bitmap?) {
        val file = Utility.getInstance().bitmapToFile(bitmap!!, applicationContext)
        val filePart = file?.let { Utility.getInstance().createMultipartBody(it) }
        if (filePart != null) {
            scanDocumentViewModel.uploadArtifact(
                SdkConfig.registerToken!!,
                Utility.getInstance().getAppKey(),
                filePart,
                ArtifactType.NFC_IMAGE.type,
            )
        }
    }

    private fun showFailureDialog() {
        if (isActivityRunning) {
            if (errorDialog != null) {
                errorDialog!!.dismiss()
            }
            maxRetryCount += 1
            errorDialog = Dialog(this)
            errorDialog!!.requestWindowFeature(Window.FEATURE_NO_TITLE)
            val binding =
                DialogFailureNfcBinding.inflate(
                    errorDialog!!.layoutInflater,
                )
            errorDialog!!.setContentView(binding.root)
            var padding = textSizeConverter.getPaddingOrMarginValue(16)
            binding.mainDialogLyt.setPadding(padding, padding, padding, padding)
            var layoutParams2 = binding.imgScanFailure.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(56)
            layoutParams2.height = textSizeConverter.getHeight(56)
            binding.imgScanFailure.layoutParams = layoutParams2
            layoutParams2 = binding.imgInfo1.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(20)
            layoutParams2.height = textSizeConverter.getHeight(20)
            binding.imgInfo1.layoutParams = layoutParams2
            layoutParams2 = binding.imgNfcFailed.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(72)
            layoutParams2.height = textSizeConverter.getHeight(72)
            binding.imgNfcFailed.layoutParams = layoutParams2
            val layoutParams1 = binding.imgNfcFailed.layoutParams as LinearLayout.LayoutParams
            layoutParams1.setMargins(
                0,
                textSizeConverter.getPaddingOrMarginValue(42),
                0,
                textSizeConverter.getPaddingOrMarginValue(22),
            )
            binding.imgNfcFailed.layoutParams = layoutParams1
            layoutParams2 = binding.imgInfo2.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(20)
            layoutParams2.height = textSizeConverter.getHeight(20)
            binding.imgInfo2.layoutParams = layoutParams2
            var layoutParams = binding.imgScanFailure.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(8), 0, 0)
            binding.imgScanFailure.layoutParams = layoutParams
            binding.txtScanCompleted.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(20).toFloat(),
            )
            layoutParams = binding.txtScanCompleted.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(24), 0, 0)
            binding.txtScanCompleted.layoutParams = layoutParams
            binding.txtScanCompleted.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
            binding.txtScanNotCompleteDesc1.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
            binding.txtScanNotCompleteDesc2.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
            binding.txtScanNotCompleteDesc1.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16).toFloat(),
            )
            layoutParams = binding.txtScanNotCompleteDesc1.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(
                textSizeConverter.getPaddingOrMarginValue(6),
                textSizeConverter.getPaddingOrMarginValue(16),
                0,
                0,
            )
            binding.txtScanNotCompleteDesc1.layoutParams = layoutParams
            binding.txtScanNotCompleteDesc2.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16).toFloat(),
            )
            binding.imgNfcFailed.playAnimation()
            binding.imgNfcFailed.repeatCount = LottieDrawable.INFINITE
            binding.imgNfcFailed.setAnimation(R.raw.nfc_read_failed)
            if (Objects
                    .requireNonNull<String?>(viewModel.getNfcLiveData().selectedDoc.value)
                    .equals(getString(R.string.e_passport), ignoreCase = true)
            ) {
                binding.txtScanNotCompleteDesc2.text =
                    resources.getString(R.string.passport_scan_not_complete_desc)
            } else {
                binding.txtScanNotCompleteDesc2.text =
                    resources.getString(R.string.scan_not_complete_desc2)
            }
            layoutParams = binding.txtScanNotCompleteDesc2.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(
                textSizeConverter.getPaddingOrMarginValue(6),
                textSizeConverter.getPaddingOrMarginValue(16),
                0,
                0,
            )
            binding.txtScanNotCompleteDesc2.layoutParams = layoutParams
            layoutParams = binding.imgInfo1.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(16), 0, 0)
            binding.imgInfo1.layoutParams = layoutParams
            layoutParams = binding.imgInfo2.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(16), 0, 0)
            binding.imgInfo2.layoutParams = layoutParams
            layoutParams = binding.btnRetry.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(
                0,
                textSizeConverter.getPaddingOrMarginValue(16),
                0,
                textSizeConverter.getPaddingOrMarginValue(12),
            )
            binding.btnRetry.layoutParams = layoutParams
            layoutParams = binding.btnSkip.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(
                0,
                textSizeConverter.getPaddingOrMarginValue(16),
                0,
                textSizeConverter.getPaddingOrMarginValue(12),
            )
            binding.btnSkip.layoutParams = layoutParams

            binding.imgInfo1.setColorFilter(SdkConfig.sdkAppTheme.getPrimaryColorInt())
            binding.imgInfo2.setColorFilter(SdkConfig.sdkAppTheme.getPrimaryColorInt())

            binding.btnRetry.setCardBackgroundColor(SdkConfig.sdkAppTheme.getPrimaryColorInt())
            binding.btnSkip.setCardBackgroundColor(SdkConfig.sdkAppTheme.getPrimaryColorInt())

            binding.txtBtnRetry.typeface = FontManager.getFont(context = applicationContext, FontWeight.MEDIUM)
            binding.txtBtnSkip.typeface = FontManager.getFont(context = applicationContext, FontWeight.MEDIUM)

            binding.txtBtnRetry.setTextColor(SdkConfig.sdkAppTheme.getBackgroundColorInt())
            padding = textSizeConverter.getPaddingOrMarginValue(12)
            binding.txtBtnRetry.setPadding(0, padding, 0, padding)
            binding.txtBtnSkip.setPadding(0, padding, 0, padding)

            binding.btnRetry.radius = textSizeConverter.calculateRadius(8).toFloat()
            binding.btnSkip.radius = textSizeConverter.calculateRadius(8).toFloat()

            binding.txtBtnRetry.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16).toFloat(),
            )
            binding.txtBtnSkip.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16).toFloat(),
            )
            binding.btnRetry.setOnClickListener {
                errorDialog?.dismiss()
                if (progressBar != null && progressText != null) {
                    progressBar!!.progress = 0
                    progressText!!.text = "0%"
                }
                viewModel.cancelNfcRead()
                showDialog()
            }
            errorDialog!!.show()
            errorDialog!!.setCancelable(false)
            errorDialog!!.window!!.setLayout(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
            )
            errorDialog!!.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
            errorDialog!!.window!!.attributes.windowAnimations = R.style.DialogAnimation
            errorDialog!!.window!!.setGravity(Gravity.BOTTOM)
            if (progressDialog != null) progressDialog!!.dismiss()

            val remainingRetry = (SdkConfig.nfcMaximumRetries ?: 3) - maxRetryCount
            binding.txtBtnRetry.text = getString(R.string.btn_retry) + " (" + remainingRetry + " " + getString(R.string.nfc_retry_remaining) + ")"

            binding.btnSkip.setOnClickListener {
                OtherConstant.getInstance().skipNFC = true
                dismissDialog()
                viewModel.validateLivensUser(this)
            }

            if (java.lang.Boolean.TRUE == viewModel.getNfcLiveData().skipNFC.value) {
                if (maxRetryCount >= (SdkConfig.nfcMaximumRetries ?: 3) && SdkConfig.nfcMaximumRetries != 0) {
                    Utility.getInstance().setNfcMaxRetries(true)
                    OtherConstant.getInstance().skipNFC = true
                    binding.btnSkip.visibility = View.VISIBLE
                    binding.btnRetry.visibility = View.GONE
                }
            } else {
                if (maxRetryCount >= (SdkConfig.nfcMaximumRetries ?: 3)) {
                    OtherConstant.getInstance().skipNFC = true
                    Utility.getInstance().setNfcScanFailed(true)
                    Utility.getInstance().setLiveNessScanFailed(true)
                    dismissDialog()
                    viewModel.validateLivensUser(this)
                    finish()
                }
            }
        }
    }

    private fun initAdapter() {
        mNfcAdapter = NfcAdapter.getDefaultAdapter(this)
        if (mNfcAdapter != null && mNfcAdapter!!.isEnabled) {
            // Yes NFC available
            pendingIntent =
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                    val intent =
                        Intent(
                            this@NfcActivity,
                            this@NfcActivity.javaClass,
                        ).addFlags(
                            Intent.FLAG_ACTIVITY_SINGLE_TOP,
                        )
                    PendingIntent.getActivity(
                        this@NfcActivity,
                        0,
                        intent,
                        PendingIntent.FLAG_MUTABLE,
                    )
                } else {
                    PendingIntent.getActivity(
                        this,
                        0,
                        Intent(this, javaClass).addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP),
                        0,
                    )
                }
        } else if (mNfcAdapter != null && !mNfcAdapter!!.isEnabled) {
            if (!isNFCPermissionDialogOpen) {
                isNFCPermissionDialogOpen = true
                showNfcNotSupportedDialog(true)
            } else {
                isActivityRunning = true
                showFailureDialog()
            }
        } else {
            showNfcNotSupportedDialog(false)
        }
    }

    private fun showNfcNotSupportedDialog(bool: Boolean) {
        nfcNotsupportDialog = Dialog(this)
        nfcNotsupportDialog.requestWindowFeature(Window.FEATURE_NO_TITLE)
        val binding =
            DialogFailureNfcBinding.inflate(
                nfcNotsupportDialog.layoutInflater,
            )
        nfcNotsupportDialog.setContentView(binding.root)
        var padding = textSizeConverter.getPaddingOrMarginValue(16)
        binding.mainDialogLyt.setPadding(padding, padding, padding, padding)
        var layoutParams2 = binding.imgScanFailure.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(56)
        layoutParams2.height = textSizeConverter.getHeight(56)
        binding.imgScanFailure.layoutParams = layoutParams2
        layoutParams2 = binding.imgInfo1.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(20)
        layoutParams2.height = textSizeConverter.getHeight(20)
        binding.imgInfo1.layoutParams = layoutParams2
        layoutParams2 = binding.imgNfcFailed.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(72)
        layoutParams2.height = textSizeConverter.getHeight(72)
        binding.imgNfcFailed.layoutParams = layoutParams2
        val layoutParams1 = binding.imgNfcFailed.layoutParams as LinearLayout.LayoutParams
        layoutParams1.setMargins(
            0,
            textSizeConverter.getPaddingOrMarginValue(42),
            0,
            textSizeConverter.getPaddingOrMarginValue(22),
        )
        binding.imgNfcFailed.layoutParams = layoutParams1
        layoutParams2 = binding.imgInfo2.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(20)
        layoutParams2.height = textSizeConverter.getHeight(20)
        binding.imgInfo2.layoutParams = layoutParams2
        var layoutParams = binding.imgScanFailure.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(8), 0, 0)
        binding.imgScanFailure.layoutParams = layoutParams
        binding.txtScanCompleted.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(20).toFloat(),
        )
        layoutParams = binding.txtScanCompleted.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(24), 0, 0)
        binding.txtScanCompleted.layoutParams = layoutParams
        binding.txtScanCompleted.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
        binding.txtScanNotCompleteDesc1.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
        binding.txtScanNotCompleteDesc2.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
        binding.txtScanNotCompleteDesc1.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(16).toFloat(),
        )
        layoutParams = binding.txtScanNotCompleteDesc1.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            textSizeConverter.getPaddingOrMarginValue(6),
            textSizeConverter.getPaddingOrMarginValue(16),
            0,
            0,
        )
        binding.txtScanNotCompleteDesc1.layoutParams = layoutParams
        binding.txtScanNotCompleteDesc2.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(16).toFloat(),
        )
        binding.imgNfcFailed.playAnimation()
        binding.imgNfcFailed.repeatCount = LottieDrawable.INFINITE
        binding.imgNfcFailed.setAnimation(R.raw.nfc_read_failed)
        if (Objects
                .requireNonNull<String?>(viewModel.getNfcLiveData().selectedDoc.value)
                .equals(getString(R.string.e_passport), ignoreCase = true)
        ) {
            binding.txtScanNotCompleteDesc2.text =
                resources.getString(R.string.passport_scan_not_complete_desc)
        } else {
            binding.txtScanNotCompleteDesc2.text =
                resources.getString(R.string.scan_not_complete_desc2)
        }
        layoutParams = binding.txtScanNotCompleteDesc2.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            textSizeConverter.getPaddingOrMarginValue(6),
            textSizeConverter.getPaddingOrMarginValue(16),
            0,
            0,
        )
        binding.lyoutError2.visibility = View.GONE
        if (!bool) {
            binding.txtScanNotCompleteDesc1.setText(getString(R.string.nfc_not_supported_desc))
        } else {
            binding.txtScanNotCompleteDesc1.setText(getString(R.string.error_nfc_is_disabled))
        }
        binding.txtScanNotCompleteDesc2.layoutParams = layoutParams
        layoutParams = binding.imgInfo1.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(16), 0, textSizeConverter.getPaddingOrMarginValue(12))
        binding.imgInfo1.layoutParams = layoutParams
        layoutParams = binding.imgInfo2.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(16), 0, 0)
        binding.imgInfo2.layoutParams = layoutParams
        layoutParams = binding.btnRetry.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            0,
            textSizeConverter.getPaddingOrMarginValue(16),
            0,
            textSizeConverter.getPaddingOrMarginValue(12),
        )
        binding.btnRetry.layoutParams = layoutParams
        layoutParams = binding.btnSkip.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            0,
            textSizeConverter.getPaddingOrMarginValue(16),
            0,
            textSizeConverter.getPaddingOrMarginValue(12),
        )
        binding.btnSkip.layoutParams = layoutParams

        binding.imgInfo1.setColorFilter(SdkConfig.sdkAppTheme.getPrimaryColorInt())
        binding.imgInfo2.setColorFilter(SdkConfig.sdkAppTheme.getPrimaryColorInt())

        binding.btnRetry.setCardBackgroundColor(SdkConfig.sdkAppTheme.getPrimaryColorInt())
        binding.btnSkip.setCardBackgroundColor(SdkConfig.sdkAppTheme.getPrimaryColorInt())

        binding.txtBtnRetry.typeface = FontManager.getFont(context = applicationContext, FontWeight.MEDIUM)
        binding.txtBtnSkip.typeface = FontManager.getFont(context = applicationContext, FontWeight.MEDIUM)

        binding.txtBtnRetry.setTextColor(SdkConfig.sdkAppTheme.getBackgroundColorInt())
        padding = textSizeConverter.getPaddingOrMarginValue(12)
        binding.txtBtnRetry.setPadding(0, padding, 0, padding)
        binding.txtBtnSkip.setPadding(0, padding, 0, padding)

        binding.btnRetry.radius = textSizeConverter.calculateRadius(8).toFloat()
        binding.btnSkip.radius = textSizeConverter.calculateRadius(8).toFloat()
        binding.txtBtnRetry.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(16).toFloat(),
        )
        binding.txtBtnSkip.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(16).toFloat(),
        )
        binding.btnRetry.setOnClickListener {
            if (!bool) {
                nfcNotsupportDialog.dismiss()
                OtherConstant.getInstance().skipNFC = false
                Utility.getInstance().setNfcScanFailed(true)
                Utility.getInstance().setLiveNessScanFailed(true)
                dismissDialog()
                viewModel.validateLivensUser(this)
                finish()
            } else {
                nfcNotsupportDialog.dismiss()
                val nfcIntent = Intent(Settings.ACTION_NFC_SETTINGS)
                someActivityResultLauncher.launch(nfcIntent)
            }
        }
        nfcNotsupportDialog.show()
        nfcNotsupportDialog.setCancelable(false)
        nfcNotsupportDialog.window!!.setLayout(
            ViewGroup.LayoutParams.MATCH_PARENT,
            ViewGroup.LayoutParams.WRAP_CONTENT,
        )
        nfcNotsupportDialog.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        nfcNotsupportDialog.window!!.attributes.windowAnimations = R.style.DialogAnimation
        nfcNotsupportDialog.window!!.setGravity(Gravity.BOTTOM)
        if (progressDialog != null) progressDialog!!.dismiss()

        val skipNfc = viewModel.getNfcLiveData().skipNFC.value == true
        if (!bool) {
            if (skipNfc) {
                binding.btnSkip.visibility = View.VISIBLE
                binding.btnRetry.visibility = View.GONE
            } else {
                binding.btnSkip.visibility = View.GONE
                binding.btnRetry.visibility = View.VISIBLE
                binding.txtBtnRetry.text = getString(R.string.nfc_not_enabled_btn_ok)
            }
        } else {
            binding.btnSkip.visibility = View.GONE
            binding.btnRetry.visibility = View.VISIBLE
            binding.txtBtnRetry.text = getString(R.string.turn_on_nfc)
        }

        binding.btnSkip.setOnClickListener {
            OtherConstant.getInstance().skipNFC = true
            dismissDialog()
            viewModel.validateLivensUser(this)
        }
    }

    private val someActivityResultLauncher =
        registerForActivityResult(
            ActivityResultContracts.StartActivityForResult(),
        ) { result ->
            isActivityRunning = true
            if (result.resultCode == Activity.RESULT_CANCELED) {
                mNfcAdapter = NfcAdapter.getDefaultAdapter(this)
                if (mNfcAdapter != null && mNfcAdapter!!.isEnabled) {

                    // Yes NFC available
                    pendingIntent =
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                            val intent =
                                Intent(
                                    this@NfcActivity,
                                    this@NfcActivity.javaClass,
                                ).addFlags(
                                    Intent.FLAG_ACTIVITY_SINGLE_TOP,
                                )
                            PendingIntent.getActivity(
                                this@NfcActivity,
                                0,
                                intent,
                                PendingIntent.FLAG_MUTABLE,
                            )
                        } else {
                            PendingIntent.getActivity(
                                this,
                                0,
                                Intent(this, javaClass).addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP),
                                0,
                            )
                        }
                } else {
                    isActivityRunning = true
                    showFailureDialog()
                }
            }
        }

    private fun setOtherViews() {
        binding!!.layoutHeader.imgBack.setOnClickListener { onBackPressedDispatcher.onBackPressed() }
        try {
            val matrix = Matrix()
            matrix.postRotate(90f)
            val bmp =
                BitmapFactory.decodeByteArray(
                    Utility.getInstance().scannedImage,
                    0,
                    Utility.getInstance().scannedImage!!.size,
                )
            val scaledBitmap = Bitmap.createScaledBitmap(bmp, 2048, 2048, true)
            val rotatedBitmap =
                Bitmap.createBitmap(
                    scaledBitmap,
                    0,
                    0,
                    scaledBitmap.width,
                    scaledBitmap.height,
                    matrix,
                    true,
                )
            val image = findViewById<ImageView>(R.id.imageview)
            image.setImageBitmap(rotatedBitmap)
        } catch (e: Exception) {
            e.printStackTrace()
        }
        binding!!.btnStartReadNfc.setOnClickListener {
            isStartedReadNFC = true
            showDialog()
        }
    }

    private fun dismissDialog() {
        if (errorDialog != null) {
            errorDialog!!.dismiss()
        }
        if (progressDialog != null) {
            progressDialog!!.dismiss()
        }
    }

    private fun handleIntentData() {
        if (intent.hasExtra(getString(R.string.doc_key))) {
            val selectedDoc = intent.getStringExtra(getString(R.string.doc_key))
            if (selectedDoc != null) {
                viewModel.setSelectedDoc(selectedDoc)
            }
        }
    }

    public override fun onPause() {
        super.onPause()
        unregisterReceiver(nfcStateReceiver)
        if (mNfcAdapter != null) {
            mNfcAdapter!!.disableForegroundDispatch(this)
        }
        if (this.isFinishing) {
            isActivityRunning = false
            viewModel.cancelNfcRead()
            if (progressDialog != null && progressDialog!!.isShowing) {
                progressDialog!!.dismiss()
            }
        }
    }

    private val nfcStateReceiver =
        object : BroadcastReceiver() {
            override fun onReceive(
                context: Context,
                intent: Intent,
            ) {
                Log.d("TAG", "onReceive: " + intent.action)
                if (NfcAdapter.ACTION_ADAPTER_STATE_CHANGED == intent.action) {
                    val state = intent.getIntExtra(NfcAdapter.EXTRA_ADAPTER_STATE, NfcAdapter.STATE_OFF)
                    when (state) {
                        NfcAdapter.STATE_OFF -> {
                            if (::nfcNotsupportDialog.isInitialized && nfcNotsupportDialog.isShowing) {
                                nfcNotsupportDialog.dismiss()
                            }
                            initAdapter()
                        }

                        NfcAdapter.STATE_ON -> {
                            if (mNfcAdapter != null && pendingIntent != null) {
                                mNfcAdapter!!.enableForegroundDispatch(this@NfcActivity, pendingIntent, null, null)
                            }
                            if (::nfcNotsupportDialog.isInitialized && nfcNotsupportDialog.isShowing) {
                                nfcNotsupportDialog.dismiss()
                            }
                            initAdapter()
                        }
                    }
                }
            }
        }

    public override fun onResume() {
        super.onResume()
        Log.d("TAG", "onResume--: ")
        try {
            val filter = IntentFilter(NfcAdapter.ACTION_ADAPTER_STATE_CHANGED)
            registerReceiver(nfcStateReceiver, filter)
            mNfcAdapter = NfcAdapter.getDefaultAdapter(this)
            if (mNfcAdapter != null && !mNfcAdapter!!.isEnabled) {
                if (!isNFCPermissionDialogOpen) {
                    isNFCPermissionDialogOpen = true
                    showNfcNotSupportedDialog(true)
                }
            }
            isActivityRunning = true
            if (mNfcAdapter != null && pendingIntent != null) {
                mNfcAdapter!!.enableForegroundDispatch(this, pendingIntent, null, null)
            }
        } catch (e: Exception) {
            println("onResume error")
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        ErrorUtility.getInstance().unregisterConnectivityManager(this)
        val disposable = viewModel.getDisposable()
        if (!disposable.isDisposed) {
            disposable.dispose()
            disposable.clear()
        }
        isActivityRunning = false
        viewModel.cancelNfcRead()
    }

    override fun onStop() {
        super.onStop()
        isActivityRunning = false
        viewModel.cancelNfcRead()
    }

    private fun backPress() {
        onBackPressedDispatcher.addCallback(this) {
            if (binding!!.customLoader.root.visibility != View.VISIBLE) openBackPressDialog()
        }
    }

    private fun openBackPressDialog() {
        val dialog = Dialog(this)
        dialog.requestWindowFeature(Window.FEATURE_NO_TITLE)
        val binding = DialogEmailPhoneConfirmationBinding.inflate(dialog.layoutInflater)
        dialog.setContentView(binding.root)
        var layoutParams = binding.dialogHeader.layoutParams as LinearLayout.LayoutParams
        var padding = textSizeConverter.getPaddingOrMarginValue(16)
        layoutParams.setMargins(padding, padding, padding, 0)
        binding.dialogHeader.layoutParams = layoutParams
        val layoutParams2 = binding.cardConfirmation.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(280)
        binding.cardConfirmation.layoutParams = layoutParams2
        binding.cardConfirmation.radius = textSizeConverter.calculateRadius(14).toFloat()
        binding.dialogHeader.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(18).toFloat(),
        )
        binding.dialogSubtitle.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(14).toFloat(),
        )
        binding.btnNo.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeConverter.getTextSize(16).toFloat())
        binding.btnYes.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeConverter.getTextSize(16).toFloat())
        layoutParams = binding.dialogSubtitle.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(padding, padding, padding, padding)
        binding.dialogSubtitle.layoutParams = layoutParams
        padding = textSizeConverter.getPaddingOrMarginValue(12)
        layoutParams = binding.btnNo.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(0, padding, 0, padding)
        binding.btnNo.layoutParams = layoutParams
        layoutParams = binding.btnYes.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            textSizeConverter.getPaddingOrMarginValue(24),
            padding,
            textSizeConverter.getPaddingOrMarginValue(16),
            padding,
        )
        binding.btnYes.layoutParams = layoutParams
        binding.dialogHeader.typeface = FontManager.getFont(context = applicationContext, FontWeight.MEDIUM)
        binding.dialogSubtitle.typeface = FontManager.getFont(context = applicationContext, FontWeight.REGULAR)
        binding.dialogHeader.text = getString(R.string.back_dialog_header)
        binding.dialogSubtitle.text = getString(R.string.back_dialog_title)
        binding.btnYes.setOnClickListener {
            dialog.dismiss()
            VerifySDKManager.getInstance().getCallback()?.onSDKExit(VerifySDKExitStep.NfcCancelled)
            viewModel.resetData()
            Utility.getInstance().restartApp(this)
        }
        binding.btnNo.setOnClickListener { dialog.dismiss() }
        dialog.show()
        dialog.setCancelable(false)
        dialog.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        dialog.window!!.attributes.windowAnimations = R.style.DialogAnimation
    }
}
