package nashid.verify.sdk.ui

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.os.Build
import android.util.Log
import android.view.View
import android.view.WindowInsetsController
import android.widget.Toast
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.SideEffect
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.luminance
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.navigation.NavController
import com.airbnb.lottie.compose.LottieAnimation
import com.airbnb.lottie.compose.LottieCompositionSpec
import com.airbnb.lottie.compose.rememberLottieComposition
import nashid.verify.sdk.model.DocumentType
import nashid.verify.sdk.ui.theme.FontManager
import nashid.verify.sdk.utils.ScanDocument
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.DocumentImages
import nashid.verify.sdkNew.R

@Composable
fun ScanningPreview(onBackClick: () -> Unit) {
    val context = LocalContext.current
    val activity = context as? Activity ?: return
    val statusBarColor = SdkConfig.sdkAppTheme.getBackgroundColorInt()
    SideEffect {
        val window = activity.window
        window.statusBarColor = statusBarColor

        // Adjust status bar icons based on background color (light/dark)
        val isLightBackground = Color(statusBarColor).luminance() > 0.5f
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            // API 30+: Use WindowInsetsController
            window.insetsController?.setSystemBarsAppearance(
                if (isLightBackground) WindowInsetsController.APPEARANCE_LIGHT_STATUS_BARS else 0,
                WindowInsetsController.APPEARANCE_LIGHT_STATUS_BARS,
            )
        } else {
            // API 23-29: Use deprecated flags with suppression
            @Suppress("DEPRECATION")
            window.decorView.systemUiVisibility =
                if (isLightBackground) {
                    window.decorView.systemUiVisibility or View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR
                } else {
                    window.decorView.systemUiVisibility and View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR.inv()
                }
        }
    }
    Box(
        modifier =
            Modifier
                .fillMaxSize()
                .background(SdkConfig.sdkAppTheme.getBackgroundColor()),
    ) {
        Column(
            modifier =
                Modifier
                    .fillMaxSize(),
        ) {
            Row(
                modifier =
                    Modifier
                        .fillMaxWidth()
                        .padding(top = 10.dp, end = 25.dp, start = 5.dp),
                verticalAlignment = Alignment.CenterVertically,
            ) {
                IconButton(onClick = onBackClick) {
                    Icon(
                        painter = painterResource(R.drawable.back),
                        contentDescription = "Back Arrow",
                        tint = SdkConfig.sdkAppTheme.getPrimaryColor(),
                    )
                }
                Text(
                    text = stringResource(id = R.string.header_doc_scan),
                    fontFamily = FontManager.getFontFamily(),
                    fontWeight = FontWeight.Medium,
                    color = SdkConfig.sdkAppTheme.getHeaderColor(),
                    fontSize = 18.sp,
                    textAlign = TextAlign.Center,
                    modifier = Modifier.weight(4f),
                )
            }

            Box(
                modifier =
                    Modifier
                        .weight(1f)
                        .padding(20.dp),
            ) {
                LottieAnimationScreen(
                    animationRes =
                        when (SdkConfig.viewType) {
                            DocumentType.OMAN_ID,
                            DocumentType.EMIRATI_ID,
                            DocumentType.BAHRAINI_ID,
                            DocumentType.SAUDI_ID,
                            DocumentType.QATARI_ID,
                            DocumentType.KUWAITI_ID,
                            -> R.raw.idscan

                            DocumentType.INTERNATIONAL_PASSPORT -> R.raw.passportscan
                            else -> R.raw.failure
                        },
                )
            }
            val descriptionRes =
                when (SdkConfig.viewType) {
                    DocumentType.OMAN_ID,
                    DocumentType.EMIRATI_ID,
                    DocumentType.BAHRAINI_ID,
                    DocumentType.SAUDI_ID,
                    DocumentType.QATARI_ID,
                    DocumentType.KUWAITI_ID,
                    -> R.string.doc_scan_desc

                    DocumentType.INTERNATIONAL_PASSPORT -> R.string.pass_scan_desc
                    else -> R.string.error_unknown
                }
            ScanDescription(stringResource(id = descriptionRes))
            StartVerification {
                Utility.getInstance().cleanup()
                ScanDocument.getInstance().apply {
                    documentNo = null
                    dateOfBirth = null
                    expiryDate = null
                    verificationResult = null
                }
                Log.d("TAG", "ScanningPreview: " + SdkConfig.sdkAppTheme + "   " + SdkConfig.fontName)
                if (SdkConfig.isOcrEnabled || SdkConfig.isMrzEnabled) {
                    openOcrView(context)
                } else if (SdkConfig.isNfcScanningEnabled) {
                    openNFC(context)
                } else if (SdkConfig.isActiveLiveNessEnabled || SdkConfig.isPassiveLiveNessEnabled) {
                    openLiveNess(context)
                } else {
                    Toast.makeText(context, context.getString(R.string.all_verification_processes_are_disabled), Toast.LENGTH_SHORT).show()
                }
            }
        }
    }
}

fun openOcrView(context: Context) {
    DocumentImages.instance.cleanup()
    val intent = Intent(context, CameraXLiveActivity::class.java)
    when (SdkConfig.viewType) {
        DocumentType.INTERNATIONAL_PASSPORT -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.e_passport))
        }

        DocumentType.OMAN_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.id_card))
        }

        DocumentType.EMIRATI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.uae_id))
        }

        DocumentType.BAHRAINI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.bahrain_id))
        }

        DocumentType.QATARI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.qatar_id))
        }

        DocumentType.SAUDI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.ksa_id))
        }

        DocumentType.KUWAITI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.kuwait_id))
        }

        else -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.error_unknown))
        }
    }

    intent.flags = Intent.FLAG_ACTIVITY_NEW_TASK
    (context as? Activity)?.let {
        it.finish()
        it.startActivity(intent)
    }
}

private fun openNFC(context: Context) {
    val intent = Intent(context, NfcActivity::class.java)
    when (SdkConfig.viewType) {
        DocumentType.INTERNATIONAL_PASSPORT -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.e_passport))
        }

        DocumentType.OMAN_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.id_card))
        }

        DocumentType.EMIRATI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.uae_id))
        }

        DocumentType.BAHRAINI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.bahrain_id))
        }

        DocumentType.QATARI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.qatar_id))
        }

        DocumentType.SAUDI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.ksa_id))
        }

        DocumentType.KUWAITI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.kuwait_id))
        }

        else -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.error_unknown))
        }
    }
    intent.setFlags(
        Intent.FLAG_ACTIVITY_NEW_TASK,
    )
    context.startActivity(intent)
}

private fun openLiveNess(context: Context) {
    val showInstruction =
        SdkConfig.showInstruction &&
            (SdkConfig.isActiveLiveNessEnabled || SdkConfig.isPassiveLiveNessEnabled) &&
            !Utility.getInstance().ocrScanningFailed()
    val targetActivity = if (showInstruction) LivenessInstructionActivity::class.java else SkipNfcLiveNessActivity::class.java
    val intent = Intent(context, targetActivity)
    when (SdkConfig.viewType) {
        DocumentType.INTERNATIONAL_PASSPORT -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.e_passport))
            Utility.getInstance().selectedDoc = context.getString(R.string.e_passport)
        }

        DocumentType.OMAN_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.id_card))
            Utility.getInstance().selectedDoc = context.getString(R.string.id_card)
        }

        DocumentType.EMIRATI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.uae_id))
            Utility.getInstance().selectedDoc = context.getString(R.string.uae_id)
        }

        DocumentType.BAHRAINI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.bahrain_id))
            Utility.getInstance().selectedDoc = context.getString(R.string.bahrain_id)
        }

        DocumentType.QATARI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.qatar_id))
            Utility.getInstance().selectedDoc = context.getString(R.string.qatar_id)
        }

        DocumentType.SAUDI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.ksa_id))
            Utility.getInstance().selectedDoc = context.getString(R.string.ksa_id)
        }

        DocumentType.KUWAITI_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.kuwait_id))
            Utility.getInstance().selectedDoc = context.getString(R.string.kuwait_id)
        }

        else -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.error_unknown))
        }
    }
    intent.setFlags(
        Intent.FLAG_ACTIVITY_CLEAR_TASK or
            Intent.FLAG_ACTIVITY_NEW_TASK,
    )
    context.startActivity(intent)
}

@Composable
fun ScanDescription(description: String) {
    Text(
        modifier = Modifier.padding(10.dp),
        text = description,
        fontFamily = FontManager.getFontFamily(),
        fontWeight = FontWeight.Normal,
        color = SdkConfig.sdkAppTheme.getHeaderColor(),
        fontSize = 18.sp,
        textAlign = TextAlign.Center,
    )
}

@Composable
fun StartVerification(startVerificationClick: () -> Unit) {
    Card(
        shape = RoundedCornerShape(6.dp),
        onClick = startVerificationClick,
        colors = CardDefaults.cardColors(SdkConfig.sdkAppTheme.getPrimaryColor()),
        modifier =
            Modifier
                .fillMaxWidth()
                .padding(16.dp),
    ) {
        Text(
            text = stringResource(id = R.string.btn_start_verificaion),
            fontFamily = FontManager.getFontFamily(),
            fontWeight = FontWeight.Medium,
            color = SdkConfig.sdkAppTheme.getBackgroundColor(),
            modifier =
                Modifier
                    .padding(vertical = 16.dp)
                    .fillMaxWidth(),
            fontSize = 16.sp,
            textAlign = TextAlign.Center,
        )
    }
}

@Composable
fun LottieAnimationScreen(animationRes: Int) {
    val composition by rememberLottieComposition(LottieCompositionSpec.RawRes(animationRes))
    Box(
        modifier =
            Modifier
                .fillMaxSize(),
    ) {
        LottieAnimation(
            composition = composition,
            modifier = Modifier.fillMaxSize(),
            iterations = Int.MAX_VALUE,
        )
    }
}

@Composable
fun StartScanning(navController: NavController) {
    val context = LocalContext.current
    ScanningPreview {
        (context as? Activity)?.finish()
    }
}

@Preview(showBackground = true)
@Composable
fun StartScanning() {
    ScanningPreview(
        onBackClick = {},
    )
}
