package nashid.verify.sdk.utils.helpers

import android.Manifest
import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.annotation.SuppressLint
import android.app.Activity
import android.content.Intent
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.Color
import android.graphics.PixelFormat
import android.graphics.drawable.Drawable
import android.nfc.NfcAdapter
import android.os.Build
import android.os.CountDownTimer
import android.provider.Settings
import android.util.Log
import android.util.Size
import android.view.Surface
import android.view.SurfaceHolder
import android.view.View
import android.view.View.GONE
import android.view.View.VISIBLE
import android.widget.CompoundButton
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.camera.core.CameraInfoUnavailableException
import androidx.camera.core.CameraSelector
import androidx.camera.core.ImageAnalysis
import androidx.camera.core.ImageCapture
import androidx.camera.core.ImageCapture.OnImageCapturedCallback
import androidx.camera.core.ImageProxy
import androidx.camera.core.Preview
import androidx.camera.core.resolutionselector.AspectRatioStrategy
import androidx.camera.core.resolutionselector.ResolutionSelector
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.core.content.ContextCompat
import androidx.core.graphics.createBitmap
import androidx.vectordrawable.graphics.drawable.Animatable2Compat
import com.airbnb.lottie.LottieDrawable
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.load.resource.gif.GifDrawable
import com.bumptech.glide.request.RequestListener
import com.bumptech.glide.request.target.Target
import com.google.mlkit.vision.barcode.BarcodeScanner
import com.google.mlkit.vision.barcode.BarcodeScannerOptions
import com.google.mlkit.vision.barcode.BarcodeScanning
import com.google.mlkit.vision.barcode.common.Barcode
import com.google.mlkit.vision.common.InputImage
import com.google.mlkit.vision.common.internal.ImageConvertUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import nashid.verify.sdk.Generated
import nashid.verify.sdk.ui.CameraXLiveActivity
import nashid.verify.sdk.utils.Constants
import nashid.verify.sdk.utils.GlareDetector
import nashid.verify.sdk.utils.Loggers
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.viewmodel.CameraXLiveViewModel
import nashid.verify.sdkNew.BuildConfig
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.ActivityCameraXliveAcitivityBinding
import java.io.ByteArrayOutputStream
import java.util.Objects
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import kotlin.math.min

class CameraOverlayManager(
    private val activity: CameraXLiveActivity,
    private val binding: ActivityCameraXliveAcitivityBinding,
    private val textSizeConverter: TextSizeConverter,
    private val storage: Storage,
    private val viewModel: CameraXLiveViewModel,
    private val isInternetAvailable: Boolean,
    private val logger: Loggers.LoggerInstance,
    private val activityResultLauncher: ActivityResultLauncher<Intent>? = null,
) : SurfaceHolder.Callback,
    CompoundButton.OnCheckedChangeListener {
    private lateinit var holder: SurfaceHolder
    private var cameraProvider: ProcessCameraProvider? = null
    private var analysisUseCase: ImageAnalysis? = null
    private val executor: Executor = Executors.newSingleThreadExecutor()
    private var imageCapture: ImageCapture? = null
    private lateinit var bitmapImage: Bitmap
    private var originalBitmapImage: Bitmap? = null
    private lateinit var croppedBitmap: Bitmap
    private var mPImage: ImageProxy? = null
    private var cTimer: CountDownTimer? = null
    private var previewUseCase: Preview? = null
    private val delayMillisecond: Long = 15000
    private val screenDivideValue = 1.8
    private var lensFacing = CameraSelector.LENS_FACING_BACK
    private var cameraSelector: CameraSelector? = null
    private var scanner: BarcodeScanner? = null
    private var boxWidth = 0
    private var boxHeight = 0
    private var left = 0
    private var right = 0
    private var top = 0
    private var bottom = 0
    private var diameter = 0
    private var scanFailCounter = 1
    private val maxAttempts = 3
    private var needToCloseTimer: Boolean = false

    @Generated
    fun setupOverlayAndAnimation() {
        // Reset scanning state
        scanFailCounter = 1

        // Cancel any existing timer
        cTimer?.cancel()

        scanner = BarcodeScanning.getClient()
        setupPaddingAndOverlay()
        setupHeaderAndAnimation()
        requestPermissions()
    }

    @Generated
    private fun setupPaddingAndOverlay() {
        val color = SdkConfig.sdkAppTheme.getBackgroundColorInt()
        val colorWithAlpha = androidx.core.graphics.ColorUtils.setAlphaComponent(color, (0.7 * 255).toInt())
        binding.lyoutScanFail.setBackgroundColor(colorWithAlpha)
        binding.txtBottom.setPadding(textSizeConverter.getPaddingOrMarginValue(6), 0, 0, 0)
        binding.overlay.setZOrderMediaOverlay(true)
        holder = binding.overlay.holder
        holder.setFormat(PixelFormat.TRANSPARENT)
        holder.addCallback(this)
        cameraSelector = CameraSelector.Builder().requireLensFacing(lensFacing).build()
        val remainingRetry = maxAttempts - scanFailCounter
        binding.txtRescanningDocuments.text = activity.getString(R.string.btn_retry) + " (" + remainingRetry + " " + activity.getString(R.string.nfc_retry_remaining) + ")"
    }

    @Generated
    private fun setupHeaderAndAnimation() {
        binding.layoutHeader.imgBack.setColorFilter(SdkConfig.sdkAppTheme.getPrimaryColorInt())
        binding.layoutHeader.imgBack.setImageResource(R.drawable.back_white)
        binding.layoutHeader.lytHeaderMain.setBackgroundColor(Color.TRANSPARENT)

        // Rotate the back button if the locale is Arabic
        if (storage.getPreferredLocale().equals("ar", ignoreCase = true)) {
            binding.layoutHeader.imgBack.rotation = 180f
        }

        // Set up the Lottie animation
        binding.lottieAnimationView.setAnimation(R.raw.scan)
        binding.lottieAnimationView.playAnimation()
        binding.lottieAnimationView.repeatCount = LottieDrawable.INFINITE
    }

    @Generated
    private fun requestPermissions() {
        PermissionHelper.requestsPermissions(
            activity,
            object : PermissionHelper.PermissionCallback {
                override fun onResult(
                    allGranted: Boolean,
                    grantedList: List<String>,
                    deniedList: List<String>,
                ) {
                    if (allGranted) {
                        initAdapter()
                    }
                }
            },
        )
    }

    @Generated
    private fun initAdapter() {
        var mNfcAdapter = NfcAdapter.getDefaultAdapter(activity)
        if (mNfcAdapter != null && mNfcAdapter.isEnabled) {
            startCameraInit()
        } else if (mNfcAdapter != null && !mNfcAdapter.isEnabled) {
            val nfcIntent = Intent(Settings.ACTION_NFC_SETTINGS)
            someActivityResultLauncher.launch(nfcIntent)
        } else {
            startCameraInit()
        }
    }

    private val someActivityResultLauncher =
        activityResultLauncher ?: activity.registerForActivityResult(
            ActivityResultContracts.StartActivityForResult(),
        ) { result ->
            Log.d("TAG", "resultcode: " + result.resultCode)
            if (result.resultCode == Activity.RESULT_OK) {
                startCameraInit()
            } else if (result.resultCode == Activity.RESULT_CANCELED) {
                initAdapter()
            }
        }

    // Implement required SurfaceHolder.Callback methods
    @Generated
    override fun surfaceCreated(holder: SurfaceHolder) {
        FocusRectDrawer(activity, holder, binding, screenDivideValue).drawFocusRect((ContextCompat.getColor(activity, R.color.overlay_color)))
    }

    @Generated
    override fun surfaceChanged(
        holder: SurfaceHolder,
        format: Int,
        width: Int,
        height: Int,
    ) {
    }

    @Generated
    override fun surfaceDestroyed(holder: SurfaceHolder) {
    }

    @Generated
    private fun startCameraInit() {
        logger.log("start camera..")
        initCameraViewModelProvider()
        initClick()
        startTimer()
    }

    @Generated
    private fun initCameraViewModelProvider() {
        viewModel.processCameraProvider.observe(activity) { provider: ProcessCameraProvider? ->
            cameraProvider = provider
            bindAllCameraUseCases()
            bindAnalysisUseCase()
        }
    }

    @Generated
    private fun initClick() {
        binding.layoutHeader.imgBack.setOnClickListener(::handleBackClick)
        binding.btnRescanningDocuments.setOnClickListener(::handleRescanClick)
    }

    private fun setOcrScanFailed() {
        scanFailCounter++
    }

    @Generated
    private fun handleBackClick(view: View) {
        activity.onBackPressedDispatcher.onBackPressed()
    }

    @Generated
    private fun handleRescanClick(view: View) {
        setOcrScanFailed()
        viewModel.cameraXLiveData.setIsOverlayVisible(true)
        viewModel.cameraXLiveData.setIsScanVisible(true)
        viewModel.cameraXLiveData.setIsScanFailVisible(false)
        viewModel.cameraXLiveData.setIsTransparentVisible(false)
        viewModel.cameraXLiveData.setIsScanCompleteVisible(false)
        viewModel.cameraXLiveData.setIsBackCardScanVisible(false)
        startTimer()
        mPImage?.close()
    }

    @Generated
    private fun bindAllCameraUseCases() {
        if (cameraProvider != null) {
            cameraProvider!!.unbindAll()
            bindPreviewUseCase()
        }
    }

    @Generated
    @SuppressLint("UnsafeOptInUsageError")
    fun bindAnalysisUseCase() {
        if (ContextCompat.checkSelfPermission(
                activity,
                Manifest.permission.CAMERA,
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            setUpAnalysisUseCase()
        }
    }

    @Generated
    @SuppressLint("RestrictedApi")
    private fun setUpAnalysisUseCase() {
        activity.runOnUiThread(::updateScanVisibility)
        if (cameraProvider == null) {
            return
        }
        if (analysisUseCase != null) {
            cameraProvider!!.unbind(analysisUseCase)
        }

        // Configure barcode scanner options with optimized settings for 1D barcodes
        scanner =
            BarcodeScanning.getClient(
                BarcodeScannerOptions
                    .Builder()
                    .setBarcodeFormats(Barcode.FORMAT_ALL_FORMATS).build(),
            )
        val resolutionSelector =
            ResolutionSelector.Builder()
                .setAspectRatioStrategy(AspectRatioStrategy.RATIO_16_9_FALLBACK_AUTO_STRATEGY)
                .build()
        analysisUseCase =
            ImageAnalysis.Builder()
                .setResolutionSelector(resolutionSelector)
                .setDefaultResolution(Size(1920, 1080))
                .setBackpressureStrategy(ImageAnalysis.STRATEGY_KEEP_ONLY_LATEST)
                .build()
                .also { analysis ->
                    analysis.setAnalyzer(executor) { pImage: ImageProxy ->
                        if (isInternetAvailable) {
                            analyzeImage(pImage)
                        } else {
                            pImage.close()
                        }
                    }
                }

        cameraProvider!!.bindToLifecycle(activity, cameraSelector!!, imageCapture, analysisUseCase)
    }

    private fun updateScanVisibility() {
        viewModel.cameraXLiveData.setIsScanVisible(true)
    }

    var isValueFound = false

    @Generated
    private fun analyzeImage(pImage: ImageProxy) {
        logger.log("analyzeImage: called")
        @SuppressLint("UnsafeOptInUsageError")
        val mediaImage = pImage.image
        if (mediaImage == null) {
            logger.log("analyzeImage: mediaImage is null, closing pImage and returning")
            pImage.close()
            return
        }

        logger.log("analyze:image (mediaImage not null)")
        val image = InputImage.fromMediaImage(mediaImage, pImage.imageInfo.rotationDegrees)

        logger.log("analyzeImage: isBarCodeCard=${viewModel.cameraXLiveData.getFlag().value}")
        if (Utility.getInstance().isBarCodeCard) {
            val selectedDoc = Objects.requireNonNull(viewModel.cameraXLiveData.getSelectedDoc().value)
            if (selectedDoc.equals(activity.getString(R.string.ksa_id), ignoreCase = true)) {
                if (viewModel.cameraXLiveData.getIsFrontView().value == false) {
                    readKSABarcodeData(image, pImage)
                } else {
                    if (DocumentImages.instance.scanPhoto == null && (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value)) {
                        bitmapImage = ImageConvertUtils.getInstance().convertToUpRightBitmap(image)
                        mPImage = pImage
                        detectFaceimage()
                    } else if (Utility.getInstance().isResidentIdCard) {
                        readKSABarcodeData(image, pImage)
                    } else if (Utility.getInstance().isNationalIdCard && Utility.getInstance().mrzLine1 != null && Utility.getInstance().mrzLine2 != null && Utility.getInstance().mrzLine3 != null) {
                        readKSABarcodeData(image, pImage)
                    } else {
                        try {
                            bitmapImage = ImageConvertUtils.getInstance().convertToUpRightBitmap(image)
                            originalBitmapImage = bitmapImage
                        } catch (e: Exception) {
                            e.printStackTrace()
                        }
                        cropBitmapByPercentage()
                        passImageToRead(pImage)
                    }
                }
            } else {
                readQatarBarcodeData(image, pImage)
            }
        } else {
            try {
                bitmapImage = ImageConvertUtils.getInstance().convertToUpRightBitmap(image)
                originalBitmapImage = bitmapImage
            } catch (e: Exception) {
                e.printStackTrace()
            }
            cropBitmapByPercentage()
            if (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getFlag().value) {
                passImageToRead(pImage)
            } else {
                assert(analysisUseCase != null)
                analysisUseCase!!.clearAnalyzer()
            }
        }
    }

    private fun readQatarBarcodeData(
        image: InputImage,
        pImage: ImageProxy,
    ) {
        try {
            // Crop the bottom 30% of the image for barcode scanning
            bitmapImage = ImageConvertUtils.getInstance().convertToUpRightBitmap(image)
            cropBitmapByPercentage()
            var croppedPercentage = 0.30
            if (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value) {
                croppedPercentage = 0.20
            }
            val cropHeight = (croppedBitmap.height * croppedPercentage).toInt()
            val cropY = croppedBitmap.height - cropHeight
            val bottomHalf = Bitmap.createBitmap(croppedBitmap, 0, cropY, croppedBitmap.width, cropHeight)
            var topHalfOfBottom: Bitmap
            val width = bottomHalf.width
            val height = bottomHalf.height

            val leftOffset = (width * 0.0f).toInt()

            val croppedWidth = (width * 0.70f).toInt()

            topHalfOfBottom =
                Bitmap.createBitmap(
                    bottomHalf,
                    leftOffset,
                    0,
                    croppedWidth,
                    height,
                )
            // Convert to grayscale for better barcode detection
            val grayscaleBitmap = createBitmap(topHalfOfBottom.width, topHalfOfBottom.height)
            val canvas = android.graphics.Canvas(grayscaleBitmap)
            val paint =
                android.graphics.Paint().apply {
                    isAntiAlias = true
                    isDither = true
                    isFilterBitmap = true
                }
            val colorMatrix = android.graphics.ColorMatrix()
            colorMatrix.setSaturation(0f)
            val filter = android.graphics.ColorMatrixColorFilter(colorMatrix)
            paint.colorFilter = filter
            canvas.drawBitmap(topHalfOfBottom, 0f, 0f, paint)
            // ZXing barcode detection
            val zxingResult = ZXingBarcodeUtil.decodeBarcodeFromBitmap(grayscaleBitmap)
            if (zxingResult != null && isValidBarcode(zxingResult)) {
                logger.log("ZXing detected barcode: $zxingResult")
                Utility.getInstance().apply {
                    barcodeIdNumber = zxingResult
                }
                processQRCode(zxingResult, pImage)
                return
            } else {
                logger.log("ZXing did not detect barcode, falling back to ML Kit")
            }

            val croppedInputImage = InputImage.fromBitmap(topHalfOfBottom, 0)

            scanner =
                BarcodeScanning.getClient(
                    BarcodeScannerOptions.Builder().setBarcodeFormats(Barcode.FORMAT_ALL_FORMATS).build(),
                )

            scanner?.process(croppedInputImage)?.addOnSuccessListener { barcodes: List<Barcode> ->
                logger.log("barcode value:$barcodes")
                if (barcodes.isEmpty()) {
                    logger.log("No barcodes detected in image.")
                    pImage.close()
                } else {
                    for (barcode in barcodes) {
                        val scannedData = barcode.rawValue
                        logger.log("Detected barcode rawValue: $scannedData")
                        if (scannedData != null) {
                            if (isValidBarcode(scannedData)) {
                                Utility.getInstance().apply {
                                    barcodeIdNumber = scannedData
                                }
                                processQRCode(scannedData, pImage)
                                return@addOnSuccessListener
                            } else {
                                pImage.close()
                            }
                        }
                    }
                }
            }?.addOnFailureListener { e: Exception? ->
                logger.log("Barcode scan failed: ${e?.message}")
                pImage.close()
            }
        } catch (e: Exception) {
            logger.log("Error processing image: ${e.message}")
            pImage.close()
        }
    }

    private fun readKSABarcodeData(
        image: InputImage,
        pImage: ImageProxy,
    ) {
        try {
            // Crop the bottom 30% of the image for barcode scanning
            bitmapImage = ImageConvertUtils.getInstance().convertToUpRightBitmap(image)
            cropBitmapByPercentage()
            var croppedPercentage = 0.30
            if (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value) {
                croppedPercentage = 0.20
            }
            val cropHeight = (croppedBitmap.height * croppedPercentage).toInt()
            val cropY = croppedBitmap.height - cropHeight
            val bottomHalf = Bitmap.createBitmap(croppedBitmap, 0, cropY, croppedBitmap.width, cropHeight)
            var topHalfOfBottom: Bitmap
            if (Utility.getInstance().isNationalIdCard) {
                val width = croppedBitmap.width
                val height = croppedBitmap.height

                val topHeight = (height * 0.30f).toInt()
                val centerHeight = (height * 0.40f).toInt()

                val centerBitmap = Bitmap.createBitmap(croppedBitmap, 0, topHeight, width, centerHeight)
                val centerWidth = centerBitmap.width
                val centerPartX = (centerWidth * 0.40f).toInt()
                val rightPartWidth = centerWidth - centerPartX

                topHalfOfBottom = Bitmap.createBitmap(centerBitmap, centerPartX, 0, rightPartWidth, centerBitmap.height)
            } else if (Utility.getInstance().isResidentIdCard) {
                val width = bottomHalf.width
                val height = bottomHalf.height

                val leftOffset = (width * 0.0f).toInt()

                val croppedWidth = (width * 0.70f).toInt()

                topHalfOfBottom =
                    Bitmap.createBitmap(
                        bottomHalf,
                        leftOffset,
                        0,
                        croppedWidth,
                        height,
                    )
            } else {
                val topOfBottomHalfHeight = bottomHalf.width / 2
                topHalfOfBottom =
                    Bitmap
                        .createBitmap(bottomHalf, 0, 0, topOfBottomHalfHeight, bottomHalf.height)
            }
            // Convert to grayscale for better barcode detection
            val grayscaleBitmap = createBitmap(topHalfOfBottom.width, topHalfOfBottom.height)
            val canvas = android.graphics.Canvas(grayscaleBitmap)
            val paint =
                android.graphics.Paint().apply {
                    isAntiAlias = true
                    isDither = true
                    isFilterBitmap = true
                }
            val colorMatrix = android.graphics.ColorMatrix()
            colorMatrix.setSaturation(0f)
            val filter = android.graphics.ColorMatrixColorFilter(colorMatrix)
            paint.colorFilter = filter
            canvas.drawBitmap(topHalfOfBottom, 0f, 0f, paint)
            // ZXing barcode detection
            val zxingResult = ZXingBarcodeUtil.decodeBarcodeFromBitmap(grayscaleBitmap)
            if (zxingResult != null && isValidBarcode(zxingResult) && !isValueFound) {
                logger.log("ZXing detected barcode: $zxingResult")
                if (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value) {
                    isValueFound = true
                    Utility.getInstance().apply {
                        barcodeSerialNumber = zxingResult
                    }
                    processQRCode(zxingResult, pImage)
                } else {
                    Utility.getInstance().apply {
                        barcodeIdNumber = zxingResult
                    }
                    activity.runOnUiThread {
                        viewModel.cameraXLiveData.apply {
                            setIdLine1(false)
                            setIdLine2(false)
                            setIdLine3(false)
                            setIsFrontView(true)
                            if (DocumentImages.instance.scanPhoto != null) {
                                setCaptureAnImage(true)
                                setIsSuccess(true)
                            } else {
                                pImage.close()
                            }
                            setCTimer(false)
                        }
                    }
                }
                return
            } else {
                logger.log("ZXing did not detect barcode, falling back to ML Kit")
            }

            val croppedInputImage = InputImage.fromBitmap(topHalfOfBottom, 0)

            scanner =
                BarcodeScanning.getClient(
                    BarcodeScannerOptions.Builder().setBarcodeFormats(Barcode.FORMAT_ALL_FORMATS).build(),
                )

            scanner?.process(croppedInputImage)?.addOnSuccessListener { barcodes: List<Barcode> ->
                logger.log("barcode value:$barcodes")
                if (barcodes.isEmpty()) {
                    logger.log("No barcodes detected in image.")
                    pImage.close()
                } else {
                    for (barcode in barcodes) {
                        val scannedData = barcode.rawValue
                        logger.log("Detected barcode rawValue: $scannedData")
                        if (scannedData != null) {
                            if (isValidBarcode(scannedData) && !isValueFound) {
                                logger.log("New barcode detected: $scannedData")
                                if (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value) {
                                    isValueFound = true
                                    Utility.getInstance().apply {
                                        barcodeSerialNumber = scannedData
                                    }
                                    processQRCode(scannedData, pImage)
                                } else {
                                    Utility.getInstance().apply {
                                        barcodeIdNumber = scannedData
                                    }
                                    activity.runOnUiThread {
                                        viewModel.cameraXLiveData.apply {
                                            setIdLine1(false)
                                            setIdLine2(false)
                                            setIdLine3(false)
                                            setIsFrontView(true)
                                            if (DocumentImages.instance.scanPhoto != null) {
                                                setCaptureAnImage(true)
                                                setIsSuccess(true)
                                            } else {
                                                pImage.close()
                                            }
                                            setCTimer(false)
                                        }
                                    }
                                }
                                return@addOnSuccessListener
                            } else {
                                pImage.close()
                                logger.log("Barcode failed validation: $scannedData")
                            }
                        }
                    }
                }
            }?.addOnFailureListener { e: Exception? ->
                logger.log("Barcode scan failed: ${e?.message}")
                pImage.close()
            }
        } catch (e: Exception) {
            logger.log("Error processing image: ${e.message}")
            pImage.close()
        }
    }

    private fun isValidBarcode(scannedData: String): Boolean {
        val data = scannedData.trim()
        val docType = SdkConfig.viewType
        var readLength = 14
        // Qatar ID: 11 digits
        if (docType.name.equals("QATARI_ID", ignoreCase = true) && data.length == 11 && data.all { it.isDigit() }) {
            logger.log("Barcode validation: Qatar ID format matched")
            return true
        }
        // KSA ID: 14 digits
        if (docType.name.equals("KSA_ID", ignoreCase = true) || docType.name.equals("SAUDI_ID", ignoreCase = true)) {
            if (java.lang.Boolean.FALSE == viewModel.cameraXLiveData.getIsFrontView().value) {
                readLength = 10
            }
            if (data.length == readLength && data.all { it.isDigit() }) {
                logger.log("Barcode validation: KSA ID format matched")
                return true
            }
        }
        logger.log("Barcode validation: failed for $data (docType=${docType.name})")
        return false
    }

    private fun processQRCode(
        qrData: String,
        pImage: ImageProxy,
    ) {
        try {
            analysisUseCase?.clearAnalyzer()
            activity.runOnUiThread {
                viewModel.cameraXLiveData.apply {
                    setIdNo(qrData)
                    setCTimer(true)
                    if (DocumentImages.instance.scanPhoto != null) {
                        setFlag(false)
                        setCloseAnalysisUseCase(true)
                        setIsSuccess(true)
                        setCaptureAnImage(true)
                    }
                }
            }
        } catch (e: Exception) {
            logger.log("Failed to process Qatar ID QR code: ${e.message}")
            activity.runOnUiThread {
                viewModel.cameraXLiveData.setIsSuccess(false)
            }
            pImage.close()
        }
    }

    private fun cropBitmapByPercentage() {
        val height = bitmapImage.height
        val width = bitmapImage.width
        diameter = min(height, width)

        val offsetPercent = if (Utility.getInstance().isBarCodeCard) 0.04 else Constants.VISION_API_OFFSET
        val offset = (offsetPercent * diameter).toInt()
        logger.log("analyzeImage: $width   $diameter")
        diameter -= offset
        val value = 4
        logger.log("analyzeImage:after removing offset $width   $diameter")
        left = width - diameter
        top = (height / screenDivideValue - diameter / value).toInt()
        right = diameter
        bottom = (height / screenDivideValue + diameter / value).toInt()
        boxHeight = bottom - top
        boxWidth = right - left
        if (Utility.getInstance().isBarCodeCard) {
            val cutPercentage = 5
            val cutAmount = (width * (cutPercentage / 100.0)).toInt()

            left = (left + cutAmount).coerceAtLeast(0)
            right = (right - cutAmount).coerceAtLeast(left)

            val newBoxWidth = right - left
            val newBoxHeight = boxHeight

            // ✅ Bound checking
            val finalWidth = if (left + newBoxWidth > width) width - left else newBoxWidth
            val finalHeight = if (top + newBoxHeight > height) height - top else newBoxHeight

            croppedBitmap = Bitmap.createBitmap(bitmapImage, left, top, finalWidth, finalHeight)
        } else {
            val cutPercentage = 5
            val cutAmount = (width * (cutPercentage / 100.0)).toInt()
            left += cutAmount
            right -= cutAmount
            val newBoxWidth = right - left
            croppedBitmap = Bitmap.createBitmap(bitmapImage, left, top, newBoxWidth, boxHeight)
            val stream = ByteArrayOutputStream()
            croppedBitmap.compress(Bitmap.CompressFormat.PNG, 100, stream)
        }
    }

    @Generated
    @SuppressLint("SetTextI18n")
    private fun passImageToRead(pImage: ImageProxy) {
        mPImage = pImage

        if (BuildConfig.BUILD_TYPE.equals("debug", ignoreCase = true)) {
            activity.runOnUiThread(::updateDebugUI)
        }

        if (binding.lyoutScanFail.visibility == GONE) {
            val imageLight = activity.getString(R.string.image_light_normal)
            if (imageLight.equals(activity.getString(R.string.image_light_normal), ignoreCase = true)) {
                val hasGlare = GlareDetector.isGlareDetected(croppedBitmap)
                if (!hasGlare) {
                    activity.runOnUiThread(::hideGlareText)
                    if (DocumentImages.instance.scanPhoto == null && (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value)) {
                        detectFaceimage()
                    } else {
                        filterScannedText()
                    }
                } else {
                    activity.runOnUiThread { showGlareTextAndCloseImage(pImage) }
                }
            } else {
                activity.runOnUiThread { showLightConditionTextAndCloseImage(imageLight, pImage) }
            }
        }
    }

    private fun detectFaceimage() {
//        This will crop a portion from the top and bottom sides.
        val cropBitmap = cropTopAndBottom(bitmapImage, 0.4f, 0.2f)
        val selectedDoc = Objects.requireNonNull(viewModel.cameraXLiveData.getSelectedDoc().value)
        if (selectedDoc.equals(activity.getString(R.string.e_passport), ignoreCase = true) || selectedDoc.equals(activity.getString(R.string.id_card), ignoreCase = true) ||
            selectedDoc.equals(activity.getString(R.string.ksa_id), ignoreCase = true)
        ) {
            val (leftHalf, rightHalf) = splitVertically(cropBitmap)
            croppedBitmap = leftHalf
        } else {
            croppedBitmap = cropBitmap
        }
        activity.runOnUiThread {
            binding.testimg2.visibility = VISIBLE
            binding.testimg2.setImageBitmap(croppedBitmap)
            binding.testimg1.visibility = VISIBLE
            binding.testimg1.setImageBitmap(cropBitmap)
        }
        FaceDetectionUtil.cropFaceFromIDCard(croppedBitmap) { faceBitmap ->
            if (faceBitmap != null) {
                DocumentImages.instance.scanPhoto = faceBitmap
                Utility.getInstance().setFaceDetectionFailed(false)
                viewModel.cameraXLiveData.apply {
                    if (selectedDoc.equals(activity.getString(R.string.e_passport), ignoreCase = true)) {
                        setFlag(false)
                    }
                    setCaptureAnImage(true)
                    setIsSuccess(true)
                }
            } else {
                Utility.getInstance().setFaceDetectionFailed(true)
            }
            mPImage?.close()
        }
    }

    @Generated
    private fun updateDebugUI() {
        binding.testimg1.visibility = VISIBLE
        binding.testimg1.setImageBitmap(bitmapImage)
    }

    @Generated
    private fun hideGlareText() {
        viewModel.cameraXLiveData.setGlareTextVisibility(false)
    }

    @Generated
    private fun showGlareTextAndCloseImage(pImage: ImageProxy) {
        viewModel.cameraXLiveData.setGlareTextVisibility(true)
        viewModel.cameraXLiveData.setGlareText(activity.getString(R.string.glare))
        activity.runOnUiThread { pImage.close() }
    }

    @Generated
    private fun showLightConditionTextAndCloseImage(
        imageLight: String,
        pImage: ImageProxy,
    ) {
        viewModel.cameraXLiveData.setGlareTextVisibility(true)
        viewModel.cameraXLiveData.setGlareText(activity.getString(R.string.image_light) + imageLight)
        activity.runOnUiThread { pImage.close() }
    }

    @Generated
    private fun filterScannedText() {
        try {
            val selectedDoc = Objects.requireNonNull(viewModel.cameraXLiveData.getSelectedDoc().value)
            if (selectedDoc.equals(activity.getString(R.string.ksa_id), ignoreCase = true) && viewModel.cameraXLiveData.getIsFrontView().value == true) {
                if (!Utility.getInstance().isNationalIdCard) {
                    croppedBitmap =
                        Bitmap.createBitmap(
                            croppedBitmap,
                            0,
                            0,
                            croppedBitmap.width,
                            croppedBitmap.height / 2,
                        )
                } else {
                    croppedBitmap =
                        Bitmap.createBitmap(
                            croppedBitmap,
                            0,
                            croppedBitmap.height / 2,
                            croppedBitmap.width,
                            croppedBitmap.height / 2,
                        )
                }
            } else if (selectedDoc.equals(activity.getString(R.string.id_card), ignoreCase = true) ||
                selectedDoc.equals(activity.getString(R.string.bahrain_id), ignoreCase = true)
            ) {
                //            this only happen id selected document is idCard
//            The following lines are used to divide the photo into 4 parts. The top left side of the photo is used to read the text from "ROYAL OMAN POLICE D.G.OF CIVIL STATUS." After successfully reading the data, the bitmap is divided horizontally, and the MRZ code is read from the bottom part of the image. This is done to increase the speed and accuracy of reading the data.
                if (java.lang.Boolean.FALSE == viewModel.cameraXLiveData.getAlgoHeaderDetect().value && java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value) {
                    croppedBitmap =
                        Bitmap.createBitmap(
                            croppedBitmap,
                            0,
                            0,
                            croppedBitmap.width / 2,
                            croppedBitmap.height / 2,
                        )
                } else {
                    if (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value) {
                        croppedBitmap =
                            Bitmap.createBitmap(
                                croppedBitmap,
                                0,
                                croppedBitmap.height / 2,
                                croppedBitmap.width,
                                croppedBitmap.height / 2,
                            )
                    }
                }
            } else if (selectedDoc.equals(activity.getString(R.string.e_passport), ignoreCase = true)) {
                croppedBitmap = bitmapImage
            } else if (selectedDoc.equals(activity.getString(R.string.uae_id), ignoreCase = true)) {
                if (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value) {
                    croppedBitmap =
                        Bitmap.createBitmap(
                            croppedBitmap,
                            0,
                            croppedBitmap.height / 2,
                            croppedBitmap.width,
                            croppedBitmap.height / 2,
                        )
                }
            }

            if (BuildConfig.BUILD_TYPE.equals("debug", ignoreCase = true)) {
                activity.runOnUiThread(::visibleImageInDebugMode)
            }
            viewModel.processTextFromBitmap(croppedBitmap)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    @Generated
    private fun visibleImageInDebugMode() {
        binding.testimg2.visibility = VISIBLE
        binding.testimg2.setImageBitmap(croppedBitmap)
    }

    @Generated
    private fun startTimer() {
        cTimer?.cancel()
        if (!needToCloseTimer) {
            cTimer =
                object : CountDownTimer(delayMillisecond, 1000) {
                    override fun onTick(millisUntilFinished: Long) {
                    }

                    override fun onFinish() {
                        val remainingRetry = maxAttempts - scanFailCounter
                        binding.txtRescanningDocuments.text = activity.getString(R.string.btn_retry) + " (" + remainingRetry + " " + activity.getString(R.string.nfc_retry_remaining) + ")"

                        val isFront = viewModel.cameraXLiveData.getIsFrontView().value == true

                        if (!isFront) {
                            setRescanningDocumentVisibility()
                            if (scanFailCounter >= maxAttempts) {
                                Log.d("TAG", "onFinish:inside " + scanFailCounter)
                                cTimer?.cancel()
                                Utility.getInstance().setOcrMaxRetries(true)
                                scanFailCounter = 1
                                binding.lyoutScanFail.visibility = View.INVISIBLE
                                Utility.getInstance().setOcrScanFailed(true)
                                Utility.getInstance().setNfcScanFailed(true)
                                Utility.getInstance().setLiveNessScanFailed(true)
                                viewModel.scanningFailed(activity)
                            }
                        } else {
                            setRescanningDocumentVisibility()
                            if (scanFailCounter >= maxAttempts) {
                                Log.d("TAG", "onFinish:inside " + scanFailCounter)
                                cTimer?.cancel()
                                Utility.getInstance().setOcrMaxRetries(true)
                                scanFailCounter = 1
                                binding.lyoutScanFail.visibility = View.INVISIBLE
                                Utility.getInstance().setOcrScanFailed(true)
                                Utility.getInstance().setNfcScanFailed(true)
                                Utility.getInstance().setLiveNessScanFailed(true)
                                viewModel.scanningFailed(activity)
                            }
                        }
                    }
                }.start()
        }
    }

    @Generated
    @SuppressLint("UnsafeOptInUsageError")
    private fun bindPreviewUseCase() {
        if (ContextCompat.checkSelfPermission(
                activity,
                Manifest.permission.CAMERA,
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            if (cameraProvider == null) {
                return
            }
            if (previewUseCase != null) {
                cameraProvider!!.unbind(previewUseCase)
            }
            val builder = Preview.Builder()
            previewUseCase = builder.build()
            previewUseCase!!.setSurfaceProvider(binding.previewView.surfaceProvider)
            val rotation =
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                    activity.display?.rotation ?: Surface.ROTATION_0
                } else {
                    @Suppress("DEPRECATION")
                    activity.windowManager.defaultDisplay.rotation
                }
            imageCapture = ImageCapture.Builder().setTargetRotation(rotation).build()
            cameraProvider!!.bindToLifecycle(activity, cameraSelector!!, imageCapture, previewUseCase)
        }
    }

    @Generated
    public fun setRescanningDocumentVisibility() {
        val isFront = viewModel.cameraXLiveData.getIsFrontView().value == true
        if (isFront) {
            if (scanFailCounter != 0) {
                viewModel.cameraXLiveData.setIsScanFailVisible(true)
                viewModel.cameraXLiveData.setIsOverlayVisible(false)
                viewModel.cameraXLiveData.setIsScanVisible(false)
                viewModel.cameraXLiveData.setIsTransparentVisible(true)
                viewModel.cameraXLiveData.setIsScanCompleteVisible(false)
                startTimer()
            } else {
                scanFailCounter++
                startTimer()
            }
        } else {
            viewModel.cameraXLiveData.setIsScanFailVisible(true)
        }
    }

    @Generated
    fun handleCloseImageProxy(needToClose: Boolean) {
        if (needToClose) {
            if (mPImage != null) {
                mPImage?.close()
            }
        }
    }

    fun handleCTimer(needToCloseTimer: Boolean) {
        this.needToCloseTimer = needToCloseTimer
        if (needToCloseTimer) {
            cTimer?.cancel()
        }
    }

    @Generated
    fun handleCaptureAnImage(needToCallCapture: Boolean) {
        if (needToCallCapture) {
            captureAnImage()
        }
    }

    @Generated
    fun handleAnalysisUseCase(needToClearAnalyzer: Boolean) {
        if (needToClearAnalyzer) {
            analysisUseCase?.clearAnalyzer()
        }
    }

    @Generated
    fun unBindCameraProvider(isUnbind: Boolean) {
        if (isUnbind) {
            assert(cameraProvider != null)
            cameraProvider?.unbindAll()
        }
    }

    @Generated
    private fun captureAnImage() {
        logger.log("capturing photo")
        imageCapture?.takePicture(
            executor,
            object : OnImageCapturedCallback() {
                override fun onCaptureSuccess(image: ImageProxy) {
                    binding.imgScanComplete.setMaxFrame(80)
                    super.onCaptureSuccess(image)
                    if (Objects.requireNonNull(viewModel.cameraXLiveData.getSelectedDoc().value).equals(activity.getString(R.string.e_passport), ignoreCase = true)) {
                        @SuppressLint("UnsafeOptInUsageError")
                        val mediaImage = image.image
                        if (mediaImage == null) {
                            image.close()
                            return
                        }
                        val image = InputImage.fromMediaImage(mediaImage, image.imageInfo.rotationDegrees)
                        try {
                            val capturedBitmap = ImageConvertUtils.getInstance().convertToUpRightBitmap(image)
                            croppedBitmap = capturedBitmap

                            capturePassportImage()
                        } catch (e: Exception) {
                            e.printStackTrace()
                        }
                    } else {
                        @SuppressLint("UnsafeOptInUsageError")
                        val mediaImage = image.image
                        if (mediaImage == null) {
                            image.close()
                            return
                        }
                        val image = InputImage.fromMediaImage(mediaImage, image.imageInfo.rotationDegrees)
                        try {
                            val capturedBitmap = ImageConvertUtils.getInstance().convertToUpRightBitmap(image)

                            val tempBitmap = capturedBitmap
                            val tempStream = ByteArrayOutputStream()
                            capturedBitmap.run {
                                if (tempBitmap != null) {
                                    tempBitmap.compress(Bitmap.CompressFormat.JPEG, 100, tempStream)
                                    tempBitmap.recycle()
                                }
                            }
                            if (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value && java.lang.Boolean.FALSE == viewModel.cameraXLiveData.getIsFrontViewScanned().value) {
                                captureFrontViewIdCard(tempStream)
                            } else {
                                captureBackViewIdCard(tempStream)
                            }
                        } catch (e: Exception) {
                            e.printStackTrace()
                        }
                    }
                }
            },
        )
    }

    @Generated
    private fun capturePassportImage() {
//        Reduce the image size by cropping 10% from the top and bottom to decrease the upload time.
        val reCroppedBitmap = croppedBitmap.let { Bitmap.createBitmap(it, 0, (it.height * 0.1).toInt(), it.width, it.height - (2 * (it.height * 0.1).toInt())) }
        if (reCroppedBitmap != null) {
            croppedBitmap = reCroppedBitmap
        }
        val stream = ByteArrayOutputStream()
        croppedBitmap.compress(Bitmap.CompressFormat.PNG, 100, stream)
        val byteArray = stream.toByteArray()
        Utility.getInstance().scannedImage = byteArray
        Utility.getInstance().croppedFrontScanImage = croppedBitmap
        activity.runOnUiThread(::updatePassportUI)
    }

    @Generated
    private fun updatePassportUI() {
        viewModel.cameraXLiveData.setStatusText("")
        viewModel.cameraXLiveData.setIsOverlayVisible(false)
        viewModel.cameraXLiveData.setIsScanVisible(false)
        viewModel.cameraXLiveData.setIsTransparentVisible(true)
        viewModel.cameraXLiveData.setIsScanCompleteVisible(true)
        viewModel.cameraXLiveData.setIsLayoutBottomVisible(false)
        binding.imgScanComplete.playAnimation()
        binding.imgScanComplete.addAnimatorListener(
            object : AnimatorListenerAdapter() {
                override fun onAnimationEnd(animation: Animator) {
                    super.onAnimationEnd(animation)
                    logger.log("passport scan completed")
                    viewModel.handleAnimationCompleted(activity)
                }
            },
        )
    }

    @Generated
    private fun captureFrontViewIdCard(stream: ByteArrayOutputStream) {
        val byteArray = stream.toByteArray()
        Utility.getInstance().croppedFrontScanImage = croppedBitmap

        Utility.getInstance().scannedImage = byteArray

        activity.runOnUiThread(::updateUIOnFrontViewScanned)

        activity.runOnUiThread { processFrontViewScanCompletion(byteArray) }
        scanFailCounter = 0
        activity.runOnUiThread(::updateLiveDataForFrontViewScanned)
        viewModel.cameraXLiveData.setIsScanFailVisible(false)
    }

    @Generated
    private fun updateUIOnFrontViewScanned() {
        viewModel.cameraXLiveData.setStatusText("")
        viewModel.cameraXLiveData.setIsOverlayVisible(false)
        viewModel.cameraXLiveData.setIsScanVisible(false)
        viewModel.cameraXLiveData.setIsTransparentVisible(true)
        viewModel.cameraXLiveData.setIsScanCompleteVisible(true)
        binding.imgScanComplete.playAnimation()
    }

    @Generated
    private fun processFrontViewScanCompletion(byteArray: ByteArray) {
        binding.imgScanComplete.addAnimatorListener(
            object : AnimatorListenerAdapter() {
                override fun onAnimationEnd(animation: Animator) {
                    super.onAnimationEnd(animation)
                    if (viewModel.cameraXLiveData.getIdNo().value == null && viewModel.cameraXLiveData.dOB.value == null && viewModel.cameraXLiveData.getExpiryDate().value == null) {
                        handleMissingIdDetails(byteArray)
                    }
                }
            },
        )
    }

    @Generated
    private fun handleMissingIdDetails(byteArray: ByteArray) {
        viewModel.cameraXLiveData.setIsScanCompleteVisible(false)
        viewModel.cameraXLiveData.setIsBackCardScanVisible(true)
        viewModel.cameraXLiveData.setStatusText("")

        CoroutineScope(Dispatchers.IO).launch {
            viewModel.scannedIdFrontViewCompleted(byteArray)
        }

        logger.log("loading card flip animation")

        val requestBuilder =
            Glide.with(activity).asGif().load(R.drawable.card_flip).listener(
                object : RequestListener<GifDrawable> {
                    override fun onLoadFailed(
                        e: GlideException?,
                        model: Any?,
                        target: Target<GifDrawable>,
                        isFirstResource: Boolean,
                    ): Boolean = false

                    override fun onResourceReady(
                        resource: GifDrawable,
                        model: Any,
                        target: Target<GifDrawable>,
                        dataSource: DataSource,
                        isFirstResource: Boolean,
                    ): Boolean {
                        handleAnimationResourceReady(resource)
                        return false
                    }
                },
            )

        requestBuilder.into(binding.imgBackcrdScan)
    }

    @Generated
    private fun handleAnimationResourceReady(resource: GifDrawable) {
        binding.txtTurnDoc.visibility = VISIBLE
        binding.txtTurnDoc.text = activity.getString(R.string.turn_document)
        logger.log("resource ready to load")

        resource.setLoopCount(1)
        resource.registerAnimationCallback(
            object : Animatable2Compat.AnimationCallback() {
                override fun onAnimationEnd(drawable: Drawable) {
                    handleAnimationEnd()
                }
            },
        )
    }

    @Generated
    private fun handleAnimationEnd() {
        logger.log("animation end")
        binding.txtTurnDoc.visibility = GONE
        viewModel.cameraXLiveData.setStatusText(activity.getString(R.string.scan_id_top_back_text))
        viewModel.cameraXLiveData.setIsOverlayVisible(true)
        viewModel.cameraXLiveData.setIsScanVisible(true)
        viewModel.cameraXLiveData.setIsTransparentVisible(false)
        viewModel.cameraXLiveData.setIsScanCompleteVisible(false)
        viewModel.cameraXLiveData.setIsBackCardScanVisible(false)
        logger.log("onAnimationEnd: start time " + viewModel.printCurrentTimeInLogs())

        mPImage?.close()
    }

    private fun updateLiveDataForFrontViewScanned() {
        viewModel.cameraXLiveData.setIdLine1(true)
        viewModel.cameraXLiveData.setIdLine2(true)
        viewModel.cameraXLiveData.setIdLine3(true)
        viewModel.cameraXLiveData.setIsFrontViewScanned(true)
    }

    @Generated
    private fun captureBackViewIdCard(stream: ByteArrayOutputStream) {
        activity.runOnUiThread(::clearStatusText)

        val byteArray = stream.toByteArray()
        Utility.getInstance().scannedIdFrontView = byteArray

        activity.runOnUiThread(::updateOverlayUI)
    }

    private fun clearStatusText() {
        viewModel.cameraXLiveData.setStatusText("")
    }

    @Generated
    private fun updateOverlayUI() {
        viewModel.cameraXLiveData.setIsOverlayVisible(false)
        viewModel.cameraXLiveData.setIsScanVisible(false)
        viewModel.cameraXLiveData.setIsTransparentVisible(true)
        viewModel.cameraXLiveData.setIsBackCardScanVisible(false)
        viewModel.cameraXLiveData.setIsScanCompleteVisible(true)

        binding.imgScanComplete.playAnimation()
        logger.log("onAnimationEnd: end time " + viewModel.printCurrentTimeInLogs())

        binding.imgScanComplete.addAnimatorListener(
            object : AnimatorListenerAdapter() {
                override fun onAnimationEnd(animation: Animator) {
                    super.onAnimationEnd(animation)
                    viewModel.handleAnimationCompleted(activity)
                }
            },
        )
    }

    @Generated
    override fun onCheckedChanged(
        buttonView: CompoundButton,
        isChecked: Boolean,
    ) {
        if (ContextCompat.checkSelfPermission(
                activity,
                Manifest.permission.CAMERA,
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            if (cameraProvider == null) {
                return
            }
            val newLensFacing = if (lensFacing == CameraSelector.LENS_FACING_FRONT) CameraSelector.LENS_FACING_BACK else CameraSelector.LENS_FACING_FRONT
            val newCameraSelector = CameraSelector.Builder().requireLensFacing(newLensFacing).build()
            try {
                if (cameraProvider!!.hasCamera(newCameraSelector)) {
                    lensFacing = newLensFacing
                    cameraSelector = newCameraSelector
                    bindAllCameraUseCases()
                }
            } catch (e: CameraInfoUnavailableException) {
                e.printStackTrace()
            }
        }
    }

    fun cropTopAndBottom(
        bitmap: Bitmap,
        topPercent: Float,
        bottomPercent: Float,
    ): Bitmap {
        val width = bitmap.width
        val height = bitmap.height

        val topCrop = (height * topPercent).toInt()
        val bottomCrop = (height * bottomPercent).toInt()

        val croppedHeight = height - topCrop - bottomCrop
        return Bitmap.createBitmap(bitmap, 0, topCrop, width, croppedHeight)
    }

    fun splitVertically(bitmap: Bitmap): Pair<Bitmap, Bitmap> {
        val width = bitmap.width
        val height = bitmap.height

        val mid = width / 2

        val leftHalf = Bitmap.createBitmap(bitmap, 0, 0, mid, height)
        val rightHalf = Bitmap.createBitmap(bitmap, mid, 0, width - mid, height)

        return Pair(leftHalf, rightHalf)
    }
}
