package nashid.verify.sdk.utils.helpers

import android.graphics.Bitmap
import android.util.Log
import com.google.mlkit.vision.common.InputImage
import com.google.mlkit.vision.face.FaceDetection
import com.google.mlkit.vision.face.FaceDetectorOptions

class FaceDetectionUtil {
    companion object {
        private const val TAG = "FaceDetectionUtil"

        fun cropFaceFromIDCard(
            idCardBitmap: Bitmap,
            onComplete: (Bitmap?) -> Unit,
        ) {
            try {
                val options =
                    FaceDetectorOptions
                        .Builder()
                        .setPerformanceMode(FaceDetectorOptions.PERFORMANCE_MODE_ACCURATE)
                        .setLandmarkMode(FaceDetectorOptions.LANDMARK_MODE_ALL)
                        .setContourMode(FaceDetectorOptions.CONTOUR_MODE_ALL)
                        .build()

                val detector = FaceDetection.getClient(options)
                val image = InputImage.fromBitmap(idCardBitmap, 0)
                detector
                    .process(image)
                    .addOnSuccessListener { faces ->
                        if (faces.isNotEmpty()) {
                            try {
                                Log.d(TAG, "Detected ${faces.size} faces in ${idCardBitmap.width}x${idCardBitmap.height} image")

                                var largeFace = faces[0]
                                if (faces.size == 2) {
                                    Log.d(TAG, "Found 2 faces, selecting the largest one")
                                    // Find the largest face by bounding box area
                                    largeFace =
                                        faces.maxByOrNull { face ->
                                            val bounds = face.boundingBox
                                            val area = bounds.width() * bounds.height()
                                            Log.d(TAG, "Face area: $area, bounds: $bounds")
                                            area
                                        } ?: run {
                                            Log.w(TAG, "No valid face found, using fallback")
                                            faces[0]
                                        }
                                }

                                val bounds = largeFace.boundingBox
                                Log.d(TAG, "Selected face bounds: $bounds")

                                // Calculate face size relative to image
                                val faceArea = bounds.width() * bounds.height()
                                val imageArea = idCardBitmap.width * idCardBitmap.height
                                val faceRatio = faceArea.toFloat() / imageArea.toFloat()
                                Log.d(TAG, "Face size ratio: ${String.format("%.4f", faceRatio)} (${bounds.width()}x${bounds.height()})")

                                val padding = 0.2f
                                val minPaddingPixels = 15 // Minimum 15 pixels padding for safety

                                // Calculate padding with minimum pixel guarantee
                                val paddingX = kotlin.math.max((bounds.width() * padding).toInt(), minPaddingPixels)
                                val paddingY = kotlin.math.max((bounds.height() * padding).toInt(), minPaddingPixels)

                                Log.d(TAG, "Padding: ${paddingX}x$paddingY pixels (${String.format("%.1f", paddingX.toFloat() / bounds.width() * 100)}% x ${String.format("%.1f", paddingY.toFloat() / bounds.height() * 100)}%)")

                                if (bounds.width() <= 0 || bounds.height() <= 0) {
                                    Log.w(TAG, "Invalid face bounds detected")
                                    onComplete(null)
                                    return@addOnSuccessListener
                                }

                                // Calculate crop boundaries with logging
                                val requestedLeft = bounds.left - paddingX
                                val requestedTop = bounds.top - paddingY
                                val requestedRight = bounds.right + paddingX
                                val requestedBottom = bounds.bottom + paddingY

                                val newLeft = requestedLeft.coerceAtLeast(0)
                                val newTop = requestedTop.coerceAtLeast(0)
                                val newRight = requestedRight.coerceAtMost(idCardBitmap.width)
                                val newBottom = requestedBottom.coerceAtMost(idCardBitmap.height)

                                // Log edge proximity and potential issues
                                if (requestedLeft < 0) {
                                    val lostPadding = kotlin.math.abs(requestedLeft)
                                    Log.w(TAG, "Face near LEFT edge! Lost $lostPadding pixels of left padding")
                                }
                                if (requestedTop < 0) {
                                    val lostPadding = kotlin.math.abs(requestedTop)
                                    Log.w(TAG, "Face near TOP edge! Lost $lostPadding pixels of top padding")
                                }
                                if (requestedRight > idCardBitmap.width) {
                                    val lostPadding = requestedRight - idCardBitmap.width
                                    Log.w(TAG, "Face near RIGHT edge! Lost $lostPadding pixels of right padding")
                                }
                                if (requestedBottom > idCardBitmap.height) {
                                    val lostPadding = requestedBottom - idCardBitmap.height
                                    Log.w(TAG, "Face near BOTTOM edge! Lost $lostPadding pixels of bottom padding")
                                }

                                // Calculate actual margins achieved
                                val actualLeftMargin = bounds.left - newLeft
                                val actualTopMargin = bounds.top - newTop
                                val actualRightMargin = newRight - bounds.right
                                val actualBottomMargin = newBottom - bounds.bottom

                                Log.d(TAG, "Actual margins: L:$actualLeftMargin T:$actualTopMargin R:$actualRightMargin B:$actualBottomMargin")

                                // Check for potential half-face scenarios
                                val minSafeMargin = 5 // 5 pixels minimum
                                if (actualLeftMargin < minSafeMargin || actualTopMargin < minSafeMargin ||
                                    actualRightMargin < minSafeMargin || actualBottomMargin < minSafeMargin
                                ) {
                                    Log.w(TAG, "POTENTIAL HALF-FACE RISK: Insufficient margins detected!")
                                }

                                if (newRight <= newLeft || newBottom <= newTop) {
                                    Log.e(TAG, "Invalid crop dimensions after boundary adjustment")
                                    onComplete(null)
                                    return@addOnSuccessListener
                                }

                                val cropWidth = newRight - newLeft
                                val cropHeight = newBottom - newTop

                                val croppedBitmap =
                                    Bitmap.createBitmap(
                                        idCardBitmap,
                                        newLeft,
                                        newTop,
                                        cropWidth,
                                        cropHeight,
                                    )

                                Log.d(TAG, "Successfully cropped face: ${cropWidth}x$cropHeight from ${idCardBitmap.width}x${idCardBitmap.height}")
                                onComplete(croppedBitmap)
                            } catch (e: Exception) {
                                Log.e(TAG, "Error processing face detection results", e)
                                onComplete(null)
                            }
                        } else {
                            Log.w(TAG, "No faces detected in the image")
                            onComplete(null)
                        }
                    }.addOnFailureListener { e ->
                        Log.e(TAG, "Face detection failed", e)
                        onComplete(null)
                    }.addOnCompleteListener {
                        detector.close()
                    }
            } catch (e: Exception) {
                Log.e(TAG, "Error initializing face detector", e)
                onComplete(null)
            }
        }
    }
}
