package nashid.mv.liveness

import android.content.Context
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.DashPathEffect
import android.graphics.Paint
import android.graphics.PorterDuff
import android.graphics.PorterDuffXfermode
import android.graphics.RectF
import android.util.AttributeSet
import android.view.View
import androidx.core.content.ContextCompat
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdkNew.R

internal class CoverViews : View {
    private var paint: Paint
    private var backColor: Int

    private var horizontalPadding: Int
    private var verticalPadding: Int
    private var borderColor: Int = ContextCompat.getColor(context, R.color.white)
    private var porterDuffXfermode: PorterDuffXfermode
    private val rectF = RectF()
    private val borderPaint = Paint()
    constructor(context: Context) : this(context, null)

    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)

    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr,
    ) {
        val typedArray = context.obtainStyledAttributes(attrs, R.styleable.CoverView)

        horizontalPadding = typedArray.getDimension(R.styleable.CoverView_horizontalPadding, 80F).toInt()
        verticalPadding = typedArray.getDimension(R.styleable.CoverView_verticalPadding, 60F).toInt()
        backColor =
            typedArray.getColor(
                R.styleable.CoverView_backgroundColor,
                ContextCompat.getColor(context, R.color.transparent),
            )

        typedArray.recycle()

        porterDuffXfermode = PorterDuffXfermode(PorterDuff.Mode.SRC_OUT)

        paint = Paint()
        paint.color = backColor
        paint.xfermode = porterDuffXfermode

        setLayerType(LAYER_TYPE_SOFTWARE, null)
    }

    override fun onDraw(canvas: Canvas) {
        val layerId =
            canvas.saveLayer(0F, 0F, width.toFloat(), height.toFloat(), null)

        canvas.drawRect(0F, 0F, width.toFloat(), height.toFloat(), paint)
        val color = SdkConfig.sdkAppTheme.getBackgroundColorInt()
        val colorWithAlpha = androidx.core.graphics.ColorUtils.setAlphaComponent(color, (0.3 * 255).toInt())
        canvas.drawColor(colorWithAlpha)

        rectF.apply {
            left = horizontalPadding.toFloat()
            top = verticalPadding.toFloat()
            right = width.minus(horizontalPadding).toFloat()
            bottom = height.minus(verticalPadding).toFloat()

            paint.color = Color.TRANSPARENT
            paint.strokeWidth = 30F
        }

        borderPaint.color = borderColor

        val scale = context.resources.displayMetrics.scaledDensity
        val desiredTextSizeInPixels: Int = (2 * scale + 0.5f).toInt()
        borderPaint.strokeWidth = desiredTextSizeInPixels.toFloat()
        borderPaint.style = Paint.Style.STROKE
        borderPaint.pathEffect = DashPathEffect(floatArrayOf(20f, 10f), 0f)
        canvas.drawOval(rectF, paint)
        canvas.drawOval(rectF, borderPaint)
        canvas.restoreToCount(layerId)
    }

    fun getRect(): RectF {
        return rectF
    }

    fun changeColor(color: Int) {
        borderColor = color
        invalidate()
    }
}
