package nashid.verify.sdk.ui

import android.content.Context
import android.content.Intent
import android.nfc.NfcAdapter
import android.os.Bundle
import android.provider.Settings
import android.widget.Toast
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.compose.setContent
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.res.stringResource
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.rememberNavController
import nashid.verify.sdk.utils.Screens
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.helpers.PermissionHelper
import nashid.verify.sdkNew.R

class MainScreen : BaseActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        requestAppPermissions()
    }

    private fun requestAppPermissions() {
        PermissionHelper.requestsPermissions(
            this,
            object : PermissionHelper.PermissionCallback {
                override fun onResult(
                    allGranted: Boolean,
                    grantedList: List<String>,
                    deniedList: List<String>,
                ) {
                    setContent {
                        if (allGranted) {
                            CheckNfcAndNavigate(context = this@MainScreen) {
                                AppNavigation()
                            }
                        } else {
                            // Permissions denied, show an informative message
                            PermissionDeniedScreen()
                            Toast.makeText(
                                this@MainScreen,
                                "These permissions are denied: $deniedList",
                                Toast.LENGTH_LONG,
                            ).show()
                        }
                    }
                }
            },
        )
    }

    @Composable
    fun CheckNfcAndNavigate(
        context: Context,
        onContinue: @Composable () -> Unit,
    ) {
        val hasLaunchedNfcIntent = remember { mutableStateOf(false) }

        val launcher =
            rememberLauncherForActivityResult(
                contract = ActivityResultContracts.StartActivityForResult(),
            ) { result ->
                hasLaunchedNfcIntent.value = false
            }

        val nfcAdapter = NfcAdapter.getDefaultAdapter(context)

        when {
            nfcAdapter == null || nfcAdapter.isEnabled -> {
                onContinue() // ✅ safe now — this is a Composable call
            }

            !hasLaunchedNfcIntent.value -> {
                LaunchedEffect(Unit) {
                    launcher.launch(Intent(Settings.ACTION_NFC_SETTINGS))
                    hasLaunchedNfcIntent.value = true
                }
            }
        }
    }

    @Composable
    fun AppNavigation() {
        val navController = rememberNavController()
        val startDestination =
            if (SdkConfig.showInstruction) {
                Screens.InstructionScreen.route
            } else {
                Screens.StartScanning.route
            }

        NavHost(navController = navController, startDestination = startDestination) {
            composable(Screens.InstructionScreen.route) {
                InstructionScreen(navController)
            }
            composable(Screens.StartScanning.route) {
                StartScanning(navController)
            }
        }
    }

    @Composable
    fun PermissionDeniedScreen() {
        Text(stringResource(R.string.the_following_permissions_are_required_for_this_feature_please_enable_them_in_settings))
    }
}
