package nashid.verify.sdk.utils

import kotlinx.coroutines.delay
import kotlin.math.min
import kotlin.math.pow
import kotlin.random.Random

class RetryHelper {
    companion object {
        private const val BASE_DELAY_MS = 500L
        private const val MAX_DELAY_MS = 30000L
        private const val MAX_JITTER_MS = 200L

        private const val FAST_NETWORK_MULTIPLIER = 0.5
        private const val SLOW_NETWORK_MULTIPLIER = 1.5
        private const val NO_NETWORK_MULTIPLIER = 3.0

        fun calculateBackoffDelay(
            attempt: Int,
            baseDelayMs: Long = BASE_DELAY_MS,
            maxDelayMs: Long = MAX_DELAY_MS,
            jitterMs: Long = MAX_JITTER_MS,
        ): Long {
            val exponentialDelay = (baseDelayMs * 2.0.pow(attempt.toDouble())).toLong()
            val cappedDelay = min(exponentialDelay, maxDelayMs)
            val jitter = Random.nextLong(-jitterMs, jitterMs)
            return maxOf(0, cappedDelay + jitter)
        }

        private fun calculateNetworkAwareDelay(
            attempt: Int,
            networkQuality: NetworkQuality = NetworkQuality.UNKNOWN,
        ): Long {
            val baseDelay = calculateBackoffDelay(attempt)

            return when (networkQuality) {
                NetworkQuality.FAST -> (baseDelay * FAST_NETWORK_MULTIPLIER).toLong()
                NetworkQuality.SLOW -> (baseDelay * SLOW_NETWORK_MULTIPLIER).toLong()
                NetworkQuality.NO_NETWORK -> (baseDelay * NO_NETWORK_MULTIPLIER).toLong()
                NetworkQuality.UNKNOWN -> baseDelay
            }
        }

        suspend fun delayWithBackoff(
            attempt: Int,
            networkQuality: NetworkQuality = NetworkQuality.UNKNOWN,
            customBaseDelay: Long? = null,
        ) {
            val delayMs =
                if (customBaseDelay != null) {
                    calculateBackoffDelay(attempt, customBaseDelay)
                } else {
                    calculateNetworkAwareDelay(attempt, networkQuality)
                }

            delay(delayMs)
        }

        fun shouldRetry(
            attempt: Int,
            maxRetries: Int,
            error: Throwable?,
            isCriticalError: Boolean = false,
        ): Boolean {
            if (isCriticalError) return false
            if (attempt >= maxRetries) return false
            error?.let {
                when {
                    it.message?.contains("401", ignoreCase = true) == true -> return false
                    it.message?.contains("403", ignoreCase = true) == true -> return false
                    it.message?.contains("400", ignoreCase = true) == true -> return false
                    it.message?.contains("file too large", ignoreCase = true) == true -> return false
                    else -> {}
                }
            }

            return true
        }

        fun getRetryConfig(operationType: RetryOperationType): RetryConfig {
            return when (operationType) {
                RetryOperationType.ARTIFACT_UPLOAD ->
                    RetryConfig(
                        maxRetries = 5,
                        baseDelayMs = 1000L,
                        maxDelayMs = 45000L,
                    )
                RetryOperationType.VERIFICATION_SUBMIT ->
                    RetryConfig(
                        maxRetries = 3,
                        baseDelayMs = 2000L,
                        maxDelayMs = 30000L,
                    )
                RetryOperationType.LOCATION_REQUEST ->
                    RetryConfig(
                        maxRetries = 3,
                        baseDelayMs = 500L,
                        maxDelayMs = 10000L,
                    )
                RetryOperationType.NETWORK_CHECK ->
                    RetryConfig(
                        maxRetries = 2,
                        baseDelayMs = 200L,
                        maxDelayMs = 5000L,
                    )
            }
        }
    }

    enum class NetworkQuality {
        FAST,
        SLOW,
        NO_NETWORK,
        UNKNOWN,
    }

    enum class RetryOperationType {
        ARTIFACT_UPLOAD,
        VERIFICATION_SUBMIT,
        LOCATION_REQUEST,
        NETWORK_CHECK,
    }

    data class RetryConfig(
        val maxRetries: Int,
        val baseDelayMs: Long,
        val maxDelayMs: Long,
    )
}
