package nashid.verify.sdk.viewmodel

import android.app.Application
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import nashid.verify.sdk.VerifySDKManager
import nashid.verify.sdk.data.repository.ScanDocumentRepository
import nashid.verify.sdk.model.Artifacts
import nashid.verify.sdk.model.DocumentType
import nashid.verify.sdk.model.DocumentVerification
import nashid.verify.sdk.model.FailureReason
import nashid.verify.sdk.model.ScanDocumentResult
import nashid.verify.sdk.model.VerifySDKExitStep
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.Loggers
import nashid.verify.sdk.utils.NetWorkResult
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdkNew.R
import okhttp3.MultipartBody
import java.io.File

class ScanDocumentViewModel(
    private val scanDocumentRepository: ScanDocumentRepository,
    private val context: Application,
    private val dispatcher: CoroutineDispatcher = Dispatchers.IO,
) : ViewModel() {
    private var frontSideImagePart: MultipartBody.Part? = null
    private var backSideImagePart: MultipartBody.Part? = null
    private var ocrFaceImagePart: MultipartBody.Part? = null
    private var passiveLiveImagePart: MultipartBody.Part? = null
    private var activeLiveImagePart: MultipartBody.Part? = null

    private val logger = Loggers.withTag("ScanDocumentViewModel")
    private val _result = MutableLiveData<ScanDocumentResult?>()
    val result: LiveData<ScanDocumentResult?> get() = _result
    private val temporaryFiles = mutableSetOf<File>()
    private val uploadedArtifacts = MutableStateFlow<Set<String>>(emptySet())
    private val uploadsInProgress = MutableStateFlow<Set<String>>(emptySet())
    private val requiredArtifacts = MutableStateFlow<Set<String>>(emptySet())

    private var isVerificationSubmissionInProgress = false
    private val maxRetries = 10
    private val retryDelayMs = 1500L
    private val uploadCompletionDelayMs = 2000L
    private val fileKeyProcessingDelayMs = 1000L

    init {
        updateRequiredArtifacts()
    }

    fun updateRequiredArtifacts() {
        val newRequiredArtifacts = mutableSetOf<String>()

        if (Utility.getInstance().hasOcrScanFailed()) {
            logger.log("OCR scan failed, no artifacts required")
            requiredArtifacts.value = emptySet()
            return
        }

        if (SdkConfig.isOcrEnabled || SdkConfig.isMrzEnabled) {
            newRequiredArtifacts.add(ArtifactType.FRONTSIDE_IMAGE.type)
            if (SdkConfig.viewType != DocumentType.INTERNATIONAL_PASSPORT) {
                newRequiredArtifacts.add(ArtifactType.BACKSIDE_IMAGE.type)
            }
            newRequiredArtifacts.add(ArtifactType.OCR_FACE_IMAGE.type)
        }

        if (SdkConfig.isNfcScanningEnabled &&
            SdkConfig.isNfcSkipable != true &&
            !Utility.getInstance().hasNfcScanFailed()
        ) {
            newRequiredArtifacts.add(ArtifactType.NFC_IMAGE.type)
        }

        if (!Utility.getInstance().hasLiveNessScanFailed()) {
            if (SdkConfig.isActiveLiveNessEnabled) {
                newRequiredArtifacts.add(ArtifactType.ACTIVE_LIVENESS_IMAGE.type)
            }
            if (SdkConfig.isPassiveLiveNessEnabled) {
                newRequiredArtifacts.add(ArtifactType.PASSIVE_LIVENESS_IMAGE.type)
            }
        }

        requiredArtifacts.value = newRequiredArtifacts
        logger.log("Required artifacts updated: $newRequiredArtifacts")
    }

    fun uploadArtifact(
        token: String,
        appKey: String,
        filePart: MultipartBody.Part,
        artifactType: String,
    ) {
        logger.log("uploadArtifact called for $artifactType with filePart: ${filePart.body != null}")

        uploadsInProgress.update { currentSet ->
            currentSet + artifactType
        }
        logger.log("Upload started for $artifactType. Uploads in progress: ${uploadsInProgress.value}")

        when (artifactType) {
            ArtifactType.FRONTSIDE_IMAGE.type -> {
                frontSideImagePart = filePart
                logger.log("Front side image part stored: ${frontSideImagePart != null}")
            }
            ArtifactType.BACKSIDE_IMAGE.type -> {
                backSideImagePart = filePart
                logger.log("Back side image part stored: ${backSideImagePart != null}")
            }
            ArtifactType.OCR_FACE_IMAGE.type -> {
                ocrFaceImagePart = filePart
                logger.log("OCR face image part stored: ${ocrFaceImagePart != null}")
            }
            ArtifactType.PASSIVE_LIVENESS_IMAGE.type -> {
                passiveLiveImagePart = filePart
                logger.log("Passive liveness image part stored: ${passiveLiveImagePart != null}")
            }
            ArtifactType.ACTIVE_LIVENESS_IMAGE.type -> {
                activeLiveImagePart = filePart
                logger.log("Active liveness image part stored: ${activeLiveImagePart != null}")
                logger.log("All file parts after setting active liveness - Front: ${frontSideImagePart != null}, Back: ${backSideImagePart != null}, OCR: ${ocrFaceImagePart != null}, Passive: ${passiveLiveImagePart != null}, Active: ${activeLiveImagePart != null}")
            }
        }

        viewModelScope.launch(dispatcher) {
            try {
                when (val response = scanDocumentRepository.uploadFile(token, appKey, filePart)) {
                    is NetWorkResult.Success -> {
                        if (response.data?.data?.fileKey != null) {
                            val fileKey = response.data.data.fileKey
                            val previousUploads = uploadedArtifacts.value
                            uploadedArtifacts.update { currentSet ->
                                currentSet + artifactType
                            }
                            logger.log("Upload successful for $artifactType. FileKey: $fileKey, Previous uploads: $previousUploads, Current uploads: ${uploadedArtifacts.value}")
                            logger.log("Posting FileKeyResult - fileKey: $fileKey, artifactType: $artifactType")
                            _result.postValue(ScanDocumentResult.FileKeyResult(fileKey, artifactType))
                            logger.log("FileKeyResult posted for $artifactType at ${System.currentTimeMillis()}")
                            logCurrentState()
                        } else {
                            logger.log("Upload failed for $artifactType: No fileKey returned")
                            _result.postValue(ScanDocumentResult.Error("Upload failed: No fileKey for $artifactType"))
                        }
                    }

                    is NetWorkResult.Error -> {
                        logger.log("Upload failed for $artifactType: ${response.message}")
                        _result.postValue(ScanDocumentResult.Error(response.message ?: context.getString(R.string.upload_failed)))
                    }

                    else -> {}
                }
            } catch (e: Exception) {
                logger.log("Upload failed for $artifactType: ${e.message}")
                _result.postValue(ScanDocumentResult.Error(e.message ?: context.getString(R.string.unexpected_error_occurred)))
            } finally {
                uploadsInProgress.update { currentSet ->
                    currentSet - artifactType
                }
                logger.log("Upload completed for $artifactType. Remaining uploads in progress: ${uploadsInProgress.value}")
            }
        }
    }

    private fun logCurrentState() {
        logger.log("Current state - Required: ${requiredArtifacts.value}, Uploaded: ${uploadedArtifacts.value}")
    }

    fun submitVerificationWithRetry(
        token: String,
        appKey: String,
        verification: DocumentVerification,
    ) {
        if (isVerificationSubmissionInProgress) {
            logger.log("Verification submission already in progress")
            _result.postValue(ScanDocumentResult.Error(context.getString(R.string.verification_in_progress)))
            return
        }

        updateRequiredArtifacts()
        viewModelScope.launch(dispatcher) {
            isVerificationSubmissionInProgress = true
            try {
                var retryCount = 0
                var missingFileKeys = emptyList<String>()

                while (retryCount < maxRetries && isVerificationSubmissionInProgress) {
                    delay(200)
                    logger.log("Verification attempt ${retryCount + 1} of $maxRetries")

                    val currentUploadsInProgress = uploadsInProgress.value
                    if (currentUploadsInProgress.isNotEmpty()) {
                        logger.log("Waiting for uploads to complete: $currentUploadsInProgress")
                        delay(uploadCompletionDelayMs)
                        continue
                    }

                    logger.log("All uploads completed, checking file keys...")

                    logger.log("Waiting ${fileKeyProcessingDelayMs}ms for FileKeyResult processing to complete...")
                    delay(fileKeyProcessingDelayMs)
                    logger.log("FileKeyResult processing delay completed, checking file keys...")

                    val artifacts =
                        Artifacts(
                            backSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.BACKSIDE_IMAGE) ?: "",
                            frontSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.FRONTSIDE_IMAGE) ?: "",
                            ocrFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.OCR_FACE_IMAGE) ?: "",
                            nfcFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.NFC_IMAGE) ?: "",
                            activeLivenessImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.ACTIVE_LIVENESS_IMAGE) ?: "",
                            passiveLiveNessImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.PASSIVE_LIVENESS_IMAGE) ?: "",
                        )
                    logger.log("File key states: frontSideImage=${artifacts.frontSideImage}, ocrFaceImage=${artifacts.ocrFaceImage}, backSideImage=${artifacts.backSideImage}, nfcFaceImage=${artifacts.nfcFaceImage}, activeLiveness=${artifacts.activeLivenessImage}, passiveLiveness=${artifacts.passiveLiveNessImage}")

                    val currentRequired = requiredArtifacts.value
                    logger.log("states: currentRequired=$currentRequired")

                    missingFileKeys =
                        currentRequired.filter { artifact ->
                            val key =
                                when (artifact) {
                                    ArtifactType.BACKSIDE_IMAGE.type -> artifacts.backSideImage
                                    ArtifactType.FRONTSIDE_IMAGE.type -> artifacts.frontSideImage
                                    ArtifactType.OCR_FACE_IMAGE.type -> artifacts.ocrFaceImage
                                    ArtifactType.NFC_IMAGE.type -> artifacts.nfcFaceImage
                                    ArtifactType.ACTIVE_LIVENESS_IMAGE.type -> artifacts.activeLivenessImage
                                    ArtifactType.PASSIVE_LIVENESS_IMAGE.type -> artifacts.passiveLiveNessImage
                                    else -> ""
                                }
                            key.isEmpty()
                        }

                    logger.log("states: missingFileKeys=$missingFileKeys")

                    if (missingFileKeys.isEmpty()) {
                        logger.log("All required artifacts have file keys present. Proceeding with verification submission.")
                        submitVerification(token, appKey, verification.copy(artifacts = artifacts))
                        break
                    } else {
                        logger.log("Missing file keys for artifacts: $missingFileKeys. Retrying uploads.")

                        val shouldProceedWithMissingArtifacts = shouldProceedWithMissingArtifacts(missingFileKeys)

                        if (shouldProceedWithMissingArtifacts) {
                            logger.log("Proceeding with verification despite missing artifacts: $missingFileKeys")
                            submitVerification(token, appKey, verification.copy(artifacts = artifacts))
                            break
                        } else {
                            _result.postValue(ScanDocumentResult.Error("Waiting for file keys: $missingFileKeys"))

                            var uploadsInProgress = false

                            logger.log("File parts status - Front: ${frontSideImagePart != null}, Back: ${backSideImagePart != null}, OCR: ${ocrFaceImagePart != null}, Passive: ${passiveLiveImagePart != null}, Active: ${activeLiveImagePart != null}")

                            if (missingFileKeys.contains(ArtifactType.FRONTSIDE_IMAGE.type)) {
                                logger.log("Retrying upload for missing front side image")
                                frontSideImagePart?.let {
                                    uploadArtifact(token, appKey, it, ArtifactType.FRONTSIDE_IMAGE.type)
                                    uploadsInProgress = true
                                } ?: logger.log("Front side image part is null - cannot retry")
                            }
                            if (missingFileKeys.contains(ArtifactType.BACKSIDE_IMAGE.type)) {
                                logger.log("Retrying upload for missing back side image")
                                backSideImagePart?.let {
                                    uploadArtifact(token, appKey, it, ArtifactType.BACKSIDE_IMAGE.type)
                                    uploadsInProgress = true
                                } ?: logger.log("Back side image part is null - cannot retry")
                            }
                            if (missingFileKeys.contains(ArtifactType.OCR_FACE_IMAGE.type)) {
                                logger.log("Retrying upload for missing ocr face image")
                                if (Utility.getInstance().hasFaceDetectionFailed()) {
                                    logger.log("OCR face image part is null - cannot retry due to face detection failure")
                                } else {
                                    ocrFaceImagePart?.let {
                                        uploadArtifact(token, appKey, it, ArtifactType.OCR_FACE_IMAGE.type)
                                        uploadsInProgress = true
                                    } ?: logger.log("OCR face image part is null - cannot retry")
                                }
                            }
                            if (missingFileKeys.contains(ArtifactType.PASSIVE_LIVENESS_IMAGE.type)) {
                                logger.log("Retrying upload for missing passive image")
                                passiveLiveImagePart?.let {
                                    uploadArtifact(token, appKey, it, ArtifactType.PASSIVE_LIVENESS_IMAGE.type)
                                    uploadsInProgress = true
                                } ?: logger.log("Passive liveness image part is null - cannot retry")
                            }
                            if (missingFileKeys.contains(ArtifactType.ACTIVE_LIVENESS_IMAGE.type)) {
                                logger.log("Retrying upload for missing active image")
                                activeLiveImagePart?.let {
                                    uploadArtifact(token, appKey, it, ArtifactType.ACTIVE_LIVENESS_IMAGE.type)
                                    uploadsInProgress = true
                                } ?: logger.log("Active liveness image part is null - cannot retry")
                            }

                            if (!uploadsInProgress) {
                                logger.log("No file parts available for retry. Missing artifacts: $missingFileKeys")
                                if (retryCount >= maxRetries - 1) {
                                    logger.log("Max retries reached with missing artifacts: $missingFileKeys. Calling missingArtifactsVerification.")
                                    missingArtifactsVerification(token, appKey, verification.copy(artifacts = artifacts))
                                    break
                                }
                            }

                            delay(retryDelayMs)
                            retryCount++
                        }
                    }
                }

                if (retryCount >= maxRetries) {
                    logger.log("Max retries reached. Missing file keys: $missingFileKeys")

                    if (missingFileKeys.isNotEmpty()) {
                        logger.log("Max retries reached with missing artifacts: $missingFileKeys. Calling missingArtifactsVerification.")
                        val finalArtifacts =
                            Artifacts(
                                backSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.BACKSIDE_IMAGE) ?: "",
                                frontSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.FRONTSIDE_IMAGE) ?: "",
                                ocrFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.OCR_FACE_IMAGE) ?: "",
                                nfcFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.NFC_IMAGE) ?: "",
                                activeLivenessImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.ACTIVE_LIVENESS_IMAGE) ?: "",
                                passiveLiveNessImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.PASSIVE_LIVENESS_IMAGE) ?: "",
                            )
                        missingArtifactsVerification(token, appKey, verification.copy(artifacts = finalArtifacts))
                    } else {
                        val errorMessage =
                            if (Utility.getInstance().hasFaceDetectionFailed() &&
                                missingFileKeys.contains(ArtifactType.OCR_FACE_IMAGE.type)
                            ) {
                                "Face detection failed - cannot upload OCR face image"
                            } else {
                                "Max retries reached - missing artifacts: $missingFileKeys"
                            }
                        logger.log("Max retries reached, posting error: $errorMessage")
                        _result.postValue(ScanDocumentResult.ScanDocumentResponse(false, errorMessage, ""))
                    }
                } else {
                    logger.log("Verification retry logic completed successfully")
                }
            } catch (e: Exception) {
                logger.log("Verification submission failed with exception: ${e.message}")
                _result.postValue(ScanDocumentResult.Error(context.getString(R.string.verification_failed) + ": ${e.message}"))
            } finally {
                isVerificationSubmissionInProgress = false
            }
        }
    }

    fun resetArtifacts() {
        logger.log("resetArtifacts called - Clearing uploaded artifacts from ${uploadedArtifacts.value}")
        logger.log("resetArtifacts called from:")
        uploadedArtifacts.value = emptySet()
        logger.log("Cleared uploaded artifacts")
        uploadsInProgress.value = emptySet()
        logger.log("Cleared uploads in progress")
        logger.log("Clearing file parts - Front: ${frontSideImagePart != null}, Back: ${backSideImagePart != null}, OCR: ${ocrFaceImagePart != null}, Passive: ${passiveLiveImagePart != null}, Active: ${activeLiveImagePart != null}")
        frontSideImagePart = null
        backSideImagePart = null
        ocrFaceImagePart = null
        passiveLiveImagePart = null
        activeLiveImagePart = null
        logger.log("File parts cleared")

        cleanupTemporaryFiles()

        Utility.getInstance().clearArtifactKeys()
        updateRequiredArtifacts()
        viewModelScope.launch(Dispatchers.Main) {
            _result.value = null
            isVerificationSubmissionInProgress = false
        }
        logger.log("Artifacts reset. Required: ${requiredArtifacts.value}, Uploaded: ${uploadedArtifacts.value}")
    }

    private fun cleanupTemporaryFiles() {
        temporaryFiles.forEach { file ->
            try {
                if (file.exists()) {
                    val deleted = file.delete()
                    logger.log("Cleaned up temporary file: ${file.name}, deleted: $deleted")
                }
            } catch (e: Exception) {
                logger.log("Failed to clean up temporary file: ${file.name}, error: ${e.message}")
            }
        }
        temporaryFiles.clear()
    }

    private suspend fun submitVerification(
        token: String,
        appKey: String,
        verification: DocumentVerification,
    ) {
        logger.log("Submitting verification with artifacts: ${verification.artifacts}")

        preserveFileParts()

        try {
            when (val response = scanDocumentRepository.submitVerification(token, appKey, verification)) {
                is NetWorkResult.Success -> {
                    if (response.data?.data?.verificationId != null) {
                        VerifySDKManager.getInstance().getCallback()?.onSDKExit(VerifySDKExitStep.Completed)
                        val verificationId = response.data.data.verificationId
                        logger.log("SubmitVerification successful for $verificationId: ${response.data.message}")
                        _result.postValue(ScanDocumentResult.ScanDocumentResponse(true, response.data.message ?: "", response.data.data.verificationId ?: ""))
                        clearFileParts()
                    }
                }

                is NetWorkResult.Error -> {
                    logger.log("Verification submission failed: ${response.message}")
                    _result.postValue(ScanDocumentResult.Error(response.message ?: context.getString(R.string.verification_failed)))

                    clearFileParts()
                }

                else -> {
                    logger.log("Verification submission returned unknown result")
                    _result.postValue(ScanDocumentResult.Error(context.getString(R.string.unexpected_error_occurred)))

                    clearFileParts()
                }
            }
        } catch (e: Exception) {
            logger.log("Verification submission exception: ${e.message}")
            _result.postValue(ScanDocumentResult.Error(e.message ?: context.getString(R.string.unexpected_error_occurred)))

            clearFileParts()
        }
    }

    fun getVerification(
        token: String,
        appKey: String,
        verificationId: String,
    ) {
        viewModelScope.launch(dispatcher) {
            when (val response = scanDocumentRepository.getVerification(token, appKey, verificationId)) {
                is NetWorkResult.Success -> {
                    logger.log("GetVerification successful for $verificationId: ${response.data?.message}")
                    _result.postValue(response.data?.let { ScanDocumentResult.GetScanResultResponse(true, it.message, response.data) })
                }

                is NetWorkResult.Error -> {
                    logger.log("GetVerification failed for $verificationId: ${response.message}")
                    _result.postValue(
                        ScanDocumentResult.GetScanResultResponse(
                            false,
                            response.message ?: context.getString(R.string.unexpected_error_occurred),
                            null,
                        ),
                    )
                }

                else -> {}
            }
        }
    }

    fun clearScanDocumentData() {
        _result.value = null
    }

    fun missingArtifactsVerification(
        token: String,
        appKey: String,
        verification: DocumentVerification,
    ) {
        viewModelScope.launch(dispatcher) {
            val artifacts = verification.artifacts
            val missingNotes = mutableListOf<String>()
            if (artifacts.frontSideImage.isEmpty()) missingNotes.add(FailureReason.FRONTSIDE_IMAGE_ARTIFACTS_ERROR.name)
            if (artifacts.backSideImage.isEmpty()) missingNotes.add(FailureReason.BACKSIDE_IMAGE_ARTIFACTS_ERROR.name)
            if (artifacts.ocrFaceImage.isEmpty()) {
                if (Utility.getInstance().hasFaceDetectionFailed()) {
                    missingNotes.add(FailureReason.OCR_FACE_IMAGE.name)
                    logger.log("OCR face image missing due to face detection failure")
                } else {
                    missingNotes.add(FailureReason.FACE_IMAGE_ARTIFACTS_ERROR.name)
                    logger.log("OCR face image missing due to upload failure")
                }
            }
            if (artifacts.nfcFaceImage.isEmpty() &&
                SdkConfig.isNfcScanningEnabled &&
                SdkConfig.isNfcSkipable != true &&
                !Utility.getInstance().hasNfcScanFailed()
            ) {
                missingNotes.add(FailureReason.NFC_ARTIFACTS_ERROR.name)
            }
            if (artifacts.activeLivenessImage.isEmpty() &&
                !Utility.getInstance().hasLiveNessScanFailed() &&
                SdkConfig.isActiveLiveNessEnabled
            ) {
                missingNotes.add(FailureReason.ACTIVE_LIVENESS_ARTIFACTS_ERROR.name)
            }
            if (artifacts.passiveLiveNessImage.isEmpty() &&
                !Utility.getInstance().hasLiveNessScanFailed() &&
                SdkConfig.isPassiveLiveNessEnabled
            ) {
                missingNotes.add(FailureReason.PASSIVE_LIVENESS_ARTIFACTS_ERROR.name)
            }
            val statusNote = missingNotes.joinToString(",")
            submitVerification(
                token,
                appKey,
                verification.copy(
                    status = 2,
                    statusNote = statusNote,
                ),
            )
        }
    }

    private fun preserveFileParts() {
        logger.log("Preserving file parts for verification - Front: ${frontSideImagePart != null}, Back: ${backSideImagePart != null}, OCR: ${ocrFaceImagePart != null}, Passive: ${passiveLiveImagePart != null}, Active: ${activeLiveImagePart != null}")
    }

    private fun clearFileParts() {
        logger.log("Clearing file parts after verification completion")
        frontSideImagePart = null
        backSideImagePart = null
        ocrFaceImagePart = null
        passiveLiveImagePart = null
        activeLiveImagePart = null
    }

    private fun shouldProceedWithMissingArtifacts(missingFileKeys: List<String>): Boolean {
        val utility = Utility.getInstance()

        if (missingFileKeys.contains(ArtifactType.OCR_FACE_IMAGE.type) && utility.hasFaceDetectionFailed()) {
            logger.log("OCR face image missing due to face detection failure - cannot proceed with verification")
            return false
        }

        if (missingFileKeys.contains(ArtifactType.NFC_IMAGE.type) && utility.hasNfcScanFailed()) {
            logger.log("NFC image missing due to NFC scan failure - proceeding with verification")
            return true
        }

        if ((
                missingFileKeys.contains(ArtifactType.ACTIVE_LIVENESS_IMAGE.type) ||
                    missingFileKeys.contains(ArtifactType.PASSIVE_LIVENESS_IMAGE.type)
            ) &&
            utility.hasLiveNessScanFailed()
        ) {
            logger.log("Liveness images missing due to liveness scan failure - proceeding with verification")
            return true
        }

        // Check for essential artifacts - frontside is always essential
        val missingFrontside = missingFileKeys.contains(ArtifactType.FRONTSIDE_IMAGE.type)

        // Backside is essential for non-passport documents
        val missingBackside =
            missingFileKeys.contains(ArtifactType.BACKSIDE_IMAGE.type) &&
                SdkConfig.viewType != DocumentType.INTERNATIONAL_PASSPORT

        val hasEssentialArtifacts = !missingFrontside && !missingBackside

        if (hasEssentialArtifacts) {
            val missingOptionalArtifacts =
                missingFileKeys.filter {
                    it == ArtifactType.ACTIVE_LIVENESS_IMAGE.type ||
                        it == ArtifactType.PASSIVE_LIVENESS_IMAGE.type ||
                        it == ArtifactType.NFC_IMAGE.type ||
                        it == ArtifactType.OCR_FACE_IMAGE.type ||
                        (it == ArtifactType.BACKSIDE_IMAGE.type && SdkConfig.viewType == DocumentType.INTERNATIONAL_PASSPORT)
                }

            if (missingFileKeys.size == missingOptionalArtifacts.size) {
                logger.log("Only optional artifacts are missing - proceeding with verification")
                return true
            }
        }

        logger.log("Cannot proceed with missing artifacts: $missingFileKeys")
        return false
    }

    override fun onCleared() {
        super.onCleared()
        cleanupTemporaryFiles()
        logger.log("ScanDocumentViewModel cleared - temporary files cleaned up")
    }
}
