package nashid.verify.sdk.ui

import android.content.Intent
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Matrix
import android.os.Bundle
import android.util.TypedValue
import android.view.View
import android.view.animation.AnimationUtils
import android.widget.LinearLayout
import androidx.activity.addCallback
import androidx.activity.result.contract.ActivityResultContracts
import androidx.lifecycle.Observer
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.joinAll
import kotlinx.coroutines.launch
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withContext
import nashid.verify.sdk.VerifySDKManager
import nashid.verify.sdk.model.ScanDocumentResult
import nashid.verify.sdk.model.VerifySDKExitStep
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.PermissionAndLocationHelper
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.ErrorUtility
import nashid.verify.sdk.utils.helpers.TextSizeConverter
import nashid.verify.sdk.viewmodel.SkipNfcLiveNessViewModel
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.ActivitySkipNfcBinding
import org.koin.android.ext.android.inject
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

class SkipNfcLiveNessActivity : BaseActivity() {
    private lateinit var binding: ActivitySkipNfcBinding
    private val textSizeConverter: TextSizeConverter by inject()
    private val viewModel: SkipNfcLiveNessViewModel by inject()
    private var maxRetries: Int = 3
    private var attempt = 0
    private var hasSubmittedFailure = false
    private var cachedLocationData: Triple<Double, Double, String>? = null
    private var isLocationRequestInProgress = false

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivitySkipNfcBinding.inflate(layoutInflater)
        setContentView(binding.root)
        initView()
        backPress()
    }

    private fun handleIntentExtras() {
        if (intent.hasExtra(getString(R.string.doc_key))) {
            val selectedDoc = intent.getStringExtra(getString(R.string.doc_key))
            if (selectedDoc != null) {
                viewModel.getLiveNessData().setSelectedDoc(selectedDoc)
            }
        }
    }

    private fun observeViewModel() {
        viewModel.getLiveNessData().getFinishActivity().observe(this) { handleFinishActivity(it) }
        viewModel.getLiveNessData().getIsLiveness().observe(this) { handleLiveNess(it) }
        viewModel.getLiveNessData().getInternetConnection().observe(this) { handleInternetConnection(it) }
        viewModel.getLiveNessData().getHandleLocationData().observe(this) {
            handleLocationData(it)
        }
        viewModel.getLiveNessData().getLivenessBitmap().observe(this) {
            it?.let { it1 ->
                setLiveNessBitmap(
                    it1,
                )
            }
        }
        viewModel.getLiveNessData().getScanBitmap().observe(this) { it?.let { it1 -> setScanBitmap(it1) } }
        viewModel.getLiveNessData().getFaceMatchApiFail().observe(this) {}
        viewModel.getLiveNessData().getRetryUploadArtifacts().observe(this) {
            if (it) {
                if (attempt < maxRetries) {
                    attempt++
                    binding.txtValidating.text = getString(R.string.validating)
                    binding.imgLoader.visibility = View.VISIBLE
                    setLayoutAndTextSize()
                    handleLocationData(true)
                } else {
                    if (!hasSubmittedFailure) {
                        hasSubmittedFailure = true
                        PermissionAndLocationHelper.requestPermissionAndLocation(
                            this,
                            object : PermissionAndLocationHelper.ResultCallback {
                                override fun onResult(
                                    latitude: Double,
                                    longitude: Double,
                                    address: String,
                                ) {
                                    viewModel.missingArtifactsVerification(
                                        latitude,
                                        longitude,
                                        false,
                                    )
                                }
                            },
                        )
                    }
                }
            }
        }
    }

    private fun handleInternetConnection(isAvailable: Boolean) {
        if (!isAvailable) {
            ErrorUtility.getInstance().showNoInternetDialog(this, !this.isFinishing && !this.isDestroyed)
        }
    }

    private fun setLiveNessBitmap(bitmap: Bitmap) {
        binding.imgTest123.setImageBitmap(bitmap)
    }

    private fun setScanBitmap(bitmap: Bitmap) {
        binding.imgTest12.setImageBitmap(bitmap)
    }

    private fun handleLiveNess(isLiveNess: Boolean) {
        if (isLiveNess) {
            // Check for missing liveness images and set failure flags before opening liveness screen
            if (SdkConfig.isActiveLiveNessEnabled && Utility.getInstance().getActiveLivenessImage() == null) {
                Utility.getInstance().setActiveLivenessFailed(true)
            }
            if (SdkConfig.isPassiveLiveNessEnabled && Utility.getInstance().getPassiveLiveNessImage() == null) {
                Utility.getInstance().setPassiveLivenessFailed(true)
            }
            openLiveNessScreen()
        }
    }

    private fun openLiveNessScreen() {
        val intent = Intent(this, LiveNessActivity::class.java)
        someActivityResultLauncher.launch(intent)
    }

    private val someActivityResultLauncher =
        registerForActivityResult(
            ActivityResultContracts.StartActivityForResult(),
        ) { result ->
            if (result.resultCode == RESULT_OK) {
                logger.log("LiveNess activity completed successfully")

                // Check if we have the required liveness images based on configuration
                var hasRequiredImages = false
                var missingImageType = ""

                if (SdkConfig.isActiveLiveNessEnabled && SdkConfig.isPassiveLiveNessEnabled) {
                    // Both enabled - check both images
                    val hasActive = Utility.getInstance().getActiveLivenessImage() != null
                    val hasPassive = Utility.getInstance().getPassiveLiveNessImage() != null
                    hasRequiredImages = hasActive && hasPassive
                    if (!hasActive) missingImageType += "active "
                    if (!hasPassive) missingImageType += "passive "
                    logger.log("Both liveness enabled - Active: $hasActive, Passive: $hasPassive")
                } else if (SdkConfig.isActiveLiveNessEnabled) {
                    // Only active enabled
                    hasRequiredImages = Utility.getInstance().getActiveLivenessImage() != null || Utility.getInstance().liveImage != null
                    if (!hasRequiredImages) missingImageType = "active"
                    logger.log("Active-only liveness - Available: $hasRequiredImages")
                } else if (SdkConfig.isPassiveLiveNessEnabled) {
                    // Only passive enabled
                    hasRequiredImages = Utility.getInstance().getPassiveLiveNessImage() != null || Utility.getInstance().liveImage != null
                    if (!hasRequiredImages) missingImageType = "passive"
                    logger.log("Passive-only liveness - Available: $hasRequiredImages")
                }

                if (hasRequiredImages) {
                    logger.log("All required liveness images captured successfully")
                    handleLiveNessImage()
                } else {
                    logger.log("Missing liveness images: $missingImageType")
                    logger.log("image is not captured!")
                    // Set failure flags based on what's missing
                    if (SdkConfig.isActiveLiveNessEnabled && SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().setActiveLivenessFailed(true)
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    } else if (SdkConfig.isActiveLiveNessEnabled) {
                        Utility.getInstance().setActiveLivenessFailed(true)
                    } else if (SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    }
                    if (Utility.getInstance().hasActiveLivenessFailed() || Utility.getInstance().hasPassiveLivenessFailed()) {
                        binding.txtValidating.text = getString(R.string.scan_max_retry_reach)
                    }
                    viewModel.updateArtifacts()
                    viewModel.validateJustCompletion()
                }
            } else if (result.resultCode == RESULT_CANCELED) {
                if (Utility.getInstance().hasPassiveLivenessMaxRetries() || Utility.getInstance().hasActiveLivenessMaxRetries()) {
                    if (SdkConfig.isActiveLiveNessEnabled && SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().setActiveLivenessFailed(true)
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    } else if (SdkConfig.isActiveLiveNessEnabled) {
                        Utility.getInstance().setActiveLivenessFailed(true)
                    } else if (SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    }
                    if (Utility.getInstance().hasActiveLivenessFailed() || Utility.getInstance().hasPassiveLivenessFailed()) {
                        binding.txtValidating.text = getString(R.string.scan_max_retry_reach)
                    }
                    viewModel.updateArtifacts()
                    viewModel.validateJustCompletion()
                } else {
                    VerifySDKManager.getInstance().getCallback()?.onSDKExit(VerifySDKExitStep.LivenessCancelled)
                    Utility.getInstance().restartApp(this)
                }
            }
        }

    private fun handleLiveNessImage() {
        CoroutineScope(Dispatchers.IO).launch {
            try {
                val uploadJobs = mutableListOf<Job>()

                // Process and upload active liveNess image if enabled
                if (SdkConfig.isActiveLiveNessEnabled) {
                    logger.log("Processing active liveNess image")
                    val activeImageBytes = Utility.getInstance().getActiveLivenessImage()
                    if (activeImageBytes == null) {
                        logger.log("Active liveNess image not found in Utility")
                        Utility.getInstance().setActiveLivenessFailed(true)
                    } else {
                        logger.log("Active liveNess image size: ${activeImageBytes.size}")
                        val activeLiveNessImage = processActiveLiveNessImage().compressImage()
                        withContext(Dispatchers.Main) {
                            binding.txtValidating.text = getString(R.string.validating)
                            binding.imgLoader.visibility = View.VISIBLE
                        }
                        val activeUploadJob =
                            launch {
                                uploadArtifactAndWait(activeLiveNessImage, ArtifactType.ACTIVE_LIVENESS_IMAGE)
                                logger.log("Active liveNess image processed and uploaded")
                            }
                        uploadJobs.add(activeUploadJob)
                    }
                }

                // Process and upload passive liveNess image if enabled
                if (SdkConfig.isPassiveLiveNessEnabled) {
                    logger.log("Processing passive liveNess image")
                    val passiveImageBytes = Utility.getInstance().getPassiveLiveNessImage()
                    if (passiveImageBytes == null) {
                        logger.log("Passive liveNess image not found in Utility")
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    } else {
                        val passiveLiveNessImage = processPassiveLiveNessImage().compressImage()
                        withContext(Dispatchers.Main) {
                            binding.txtValidating.text = getString(R.string.validating)
                            binding.imgLoader.visibility = View.VISIBLE
                        }
                        val passiveUploadJob =
                            launch {
                                uploadArtifactAndWait(passiveLiveNessImage, ArtifactType.PASSIVE_LIVENESS_IMAGE)
                                logger.log("Passive liveNess image processed and uploaded")
                            }
                        uploadJobs.add(passiveUploadJob)
                    }
                }

                logger.log("Waiting for ${uploadJobs.size} liveness upload jobs to complete...")
                uploadJobs.joinAll()
                logger.log("All liveness uploads completed")

                logger.log("Checking for cached location data...")
                var locationToUse = cachedLocationData

                if (locationToUse == null && isLocationRequestInProgress) {
                    logger.log("Location request still in progress, waiting for completion...")
                    var waitTime = 0
                    while (locationToUse == null && waitTime < 2000 && isLocationRequestInProgress) {
                        delay(100)
                        waitTime += 100
                        locationToUse = cachedLocationData
                    }
                }

                // Use cached location or default
                val finalLocation = locationToUse ?: Triple(0.0, 0.0, "")
                logger.log("Using location data: lat=${finalLocation.first}, lng=${finalLocation.second}")
                delay(500)
                logger.log("File key processing delay completed, proceeding with verification")

                withContext(Dispatchers.Main) {
                    logger.log("Triggering direct verification submission with location data")
                    viewModel.submitVerification(finalLocation.first, finalLocation.second, false)
                    binding.imgLoader.visibility = View.GONE
                }
            } catch (e: Exception) {
                logger.log("Error in handleLivenNessImage : $e")
                withContext(Dispatchers.Main) {
                    binding.imgLoader.clearAnimation()
                    binding.imgLoader.visibility = View.GONE
                    binding.txtValidating.text = getString(R.string.some_thing_went_wrong)
                    binding.retryButton.visibility = View.VISIBLE

                    // Mark the correct liveness scan as failed
                    if (SdkConfig.isActiveLiveNessEnabled && SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().setActiveLivenessFailed(true)
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    } else if (SdkConfig.isActiveLiveNessEnabled) {
                        Utility.getInstance().setActiveLivenessFailed(true)
                    } else if (SdkConfig.isPassiveLiveNessEnabled) {
                        Utility.getInstance().setPassiveLivenessFailed(true)
                    }
                    viewModel.updateArtifacts()
                }
            }
        }
    }

    private suspend fun uploadArtifactAndWait(
        bitmap: Bitmap,
        artifactType: ArtifactType,
    ) {
        return suspendCancellableCoroutine { continuation ->
            var isResumed = false
            val observer =
                object : Observer<ScanDocumentResult?> {
                    override fun onChanged(result: ScanDocumentResult?) {
                        if (result is ScanDocumentResult.FileKeyResult && result.artifactType == artifactType.type) {
                            logger.log("Upload completed for ${artifactType.type} with fileKey: ${result.fileKey}")
                            viewModel.getScanDocumentViewModel().result.removeObserver(this)
                            if (!isResumed) {
                                isResumed = true
                                continuation.resume(Unit)
                            }
                        } else if (result is ScanDocumentResult.Error) {
                            logger.log("Upload failed for ${artifactType.type}: ${result.message}")
                            viewModel.getScanDocumentViewModel().result.removeObserver(this)
                            if (!isResumed) {
                                isResumed = true
                                continuation.resumeWithException(Exception("Upload failed: ${result.message}"))
                            }
                        }
                    }
                }
            CoroutineScope(Dispatchers.Main).launch {
                viewModel.getScanDocumentViewModel().result.observeForever(observer)
                viewModel.uploadArtifact(bitmap, artifactType)
            }
            continuation.invokeOnCancellation {
                CoroutineScope(Dispatchers.Main).launch {
                    viewModel.getScanDocumentViewModel().result.removeObserver(observer)
                }
            }
        }
    }

    private suspend fun processActiveLiveNessImage(): Bitmap =
        withContext(Dispatchers.IO) {
            val activeImageBytes = Utility.getInstance().getActiveLivenessImage()
            if (activeImageBytes == null) {
                logger.log("Active liveNess image bytes are null")
                throw IllegalStateException("Active liveNess image not found")
            }

            try {
                val liveImage =
                    BitmapFactory.decodeByteArray(
                        activeImageBytes,
                        0,
                        activeImageBytes.size,
                    ) ?: throw IllegalStateException("Failed to decode active liveNess image")

                val matrix = Matrix()
                val rotation =
                    when (liveImage.width > liveImage.height) {
                        true -> 270f
                        false -> 0f
                    }
                matrix.postRotate(rotation)

                if (rotation != 0f) {
                    Bitmap.createBitmap(
                        liveImage,
                        0,
                        0,
                        liveImage.width,
                        liveImage.height,
                        matrix,
                        true,
                    ).also {
                        if (it != liveImage) {
                            liveImage.recycle()
                        }
                    }
                } else {
                    liveImage
                }
            } catch (e: Exception) {
                logger.log("Error processing active liveNess image : $e")
                throw e
            }
        }

    private suspend fun processPassiveLiveNessImage(): Bitmap =
        withContext(Dispatchers.IO) {
            val passiveImageBytes = Utility.getInstance().getPassiveLiveNessImage()
            if (passiveImageBytes == null) {
                logger.log("Passive liveNess image bytes are null")
                throw IllegalStateException("Passive liveNess image not found")
            }

            try {
                val liveImage =
                    BitmapFactory.decodeByteArray(
                        passiveImageBytes,
                        0,
                        passiveImageBytes.size,
                    ) ?: throw IllegalStateException("Failed to decode passive liveNess image")

                val matrix = Matrix()
                val rotation =
                    when (liveImage.width > liveImage.height) {
                        true -> 270f
                        false -> 0f
                    }
                matrix.postRotate(rotation)

                if (rotation != 0f) {
                    Bitmap.createBitmap(
                        liveImage,
                        0,
                        0,
                        liveImage.width,
                        liveImage.height,
                        matrix,
                        true,
                    ).also {
                        if (it != liveImage) {
                            liveImage.recycle()
                        }
                    }
                } else {
                    liveImage
                }
            } catch (e: Exception) {
                logger.log("Error processing passive liveNess image : $e")
                throw e
            }
        }

    private fun handleFinishActivity(shouldFinish: Boolean) {
        if (shouldFinish) {
            binding.imgLoader.clearAnimation()
            binding.imgLoader.visibility = View.GONE
            binding.txtValidating.visibility = View.GONE
            Utility.getInstance().restartApp(this)
        }
    }

    private fun backPress() {
        onBackPressedDispatcher.addCallback(this) {
            if (viewModel.getLiveNessData().getIsApiCalled().value == false) onBackPressedDispatcher.onBackPressed()
        }
    }

    private fun setLayoutAndTextSize() {
        textSizeConverter.changeStatusBarColor(this)
        val layoutParams2 = binding.imgLoader.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(24)
        layoutParams2.height = textSizeConverter.getHeight(24)
        binding.imgLoader.layoutParams = layoutParams2

        binding.imgLoader.setColorFilter(SdkConfig.sdkAppTheme.getPrimaryColorInt())

        val marginLayoutParam = binding.txtValidating.layoutParams as LinearLayout.LayoutParams
        marginLayoutParam.setMargins(textSizeConverter.getPaddingOrMarginValue(12), textSizeConverter.getPaddingOrMarginValue(32), textSizeConverter.getPaddingOrMarginValue(12), 0)
        binding.txtValidating.layoutParams = marginLayoutParam
        binding.txtValidating.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
        binding.txtValidating.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(20).toFloat(),
        )

        val rotation = AnimationUtils.loadAnimation(this, R.anim.rotate)
        rotation.fillAfter = true
        binding.imgLoader.startAnimation(rotation)

        binding.lytMainSkipNfc.setBackgroundColor(SdkConfig.sdkAppTheme.getBackgroundColorInt())
    }

    private fun initView() {
        setLayoutAndTextSize()
        handleIntentExtras()
        observeViewModel()
        // Start location request immediately for better performance
        startEarlyLocationRequest()
        viewModel.handleInternetConnectionData(isInternetAvailable)
    }

    private fun startEarlyLocationRequest() {
        if (!isLocationRequestInProgress && cachedLocationData == null) {
            logger.log("Starting early location request at activity initialization")
            isLocationRequestInProgress = true

            PermissionAndLocationHelper.requestPermissionAndLocation(
                this,
                object : PermissionAndLocationHelper.ResultCallback {
                    override fun onResult(
                        latitude: Double,
                        longitude: Double,
                        address: String,
                    ) {
                        if (latitude == 0.0 && longitude == 0.0) {
                            logger.log("Early location request returned default values (0.0, 0.0) - likely permission denied or location unavailable")
                        } else {
                            logger.log("Early location data cached successfully: lat=$latitude, lng=$longitude, address=$address")
                        }
                        cachedLocationData = Triple(latitude, longitude, address)
                        isLocationRequestInProgress = false
                    }
                },
            )
        } else {
            logger.log("Skipping early location request - already in progress: $isLocationRequestInProgress, cached data: ${cachedLocationData != null}")
        }
    }

    override fun onAvailable() {
        runOnUiThread {
            if (viewModel.getLiveNessData().getIsApiCalled().value == false) {
                viewModel.handleInternetConnectionData(
                    isInternetAvailable,
                )
            }
        }
    }

    private fun handleLocationData(callLocationData: Boolean) {
        if (callLocationData) {
            logger.log("Using fallback location request method")
            if (Utility.getInstance().hasOcrMaxRetries() || Utility.getInstance().hasNfcMaxRetries()) {
                binding.txtValidating.text = getString(R.string.scan_max_retry_reach)
            }
            PermissionAndLocationHelper.requestPermissionAndLocation(
                this,
                object : PermissionAndLocationHelper.ResultCallback {
                    override fun onResult(
                        latitude: Double,
                        longitude: Double,
                        address: String,
                    ) {
                        logger.log("Triggering direct verification submission with location data")
                        viewModel.submitVerification(
                            latitude,
                            longitude,
                            false,
                        )
                    }
                },
            )
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        ErrorUtility.getInstance().unregisterConnectivityManager(this)
        viewModel.getLiveNessData().clear()
    }

    private fun Bitmap.compressImage(
        maxWidth: Int = 1024,
        maxHeight: Int = 1024,
    ): Bitmap {
        if (width <= maxWidth && height <= maxHeight) {
            return this
        }

        val ratioX = maxWidth.toFloat() / width
        val ratioY = maxHeight.toFloat() / height
        val ratio = minOf(ratioX, ratioY)

        val newWidth = (width * ratio).toInt()
        val newHeight = (height * ratio).toInt()

        return Bitmap.createScaledBitmap(this, newWidth, newHeight, true).also {
            if (this != it) {
                this.recycle()
            }
        }
    }
}
