package nashid.verify.sdk.utils.helpers

import android.app.Activity
import android.content.Context
import android.os.Build
import android.view.View
import android.view.WindowInsets
import android.view.WindowInsetsController
import android.view.WindowManager
import android.widget.LinearLayout
import androidx.compose.ui.graphics.luminance
import nashid.verify.sdk.utils.SdkConfig

class TextSizeConverter(
    private val context: Context,
) {
    private val scale: Float
        get() {
            val density = context.resources.displayMetrics.density
            val fontScale = context.resources.configuration.fontScale
            return density * fontScale
        }
    private val densityScale: Float
        get() = context.resources.displayMetrics.density // Used for dp-to-pixel conversions

    fun changeStatusBarColor(activity: Activity) {
        // Status bar color from branding (null means transparent)
        val statusBarColor = SdkConfig.sdkAppTheme.getBackgroundColorInt()

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            activity.window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)

            if (statusBarColor != null) {
                // Color provided: set the status bar color
                activity.window.statusBarColor = statusBarColor

                // Adjust status bar icons based on background color (light/dark)
                val isLightBackground = androidx.compose.ui.graphics.Color(statusBarColor).luminance() > 0.5f
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                    // API 30+: Use WindowInsetsController
                    activity.window.insetsController?.setSystemBarsAppearance(
                        if (isLightBackground) WindowInsetsController.APPEARANCE_LIGHT_STATUS_BARS else 0,
                        WindowInsetsController.APPEARANCE_LIGHT_STATUS_BARS,
                    )
                } else {
                    // API 23-29: Use deprecated flags with suppression
                    @Suppress("DEPRECATION")
                    activity.window.decorView.systemUiVisibility =
                        if (isLightBackground) {
                            activity.window.decorView.systemUiVisibility or View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR
                        } else {
                            activity.window.decorView.systemUiVisibility and View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR.inv()
                        }
                }
            } else {
                // No color provided: make status bar transparent and fullscreen
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                    activity.window.setDecorFitsSystemWindows(false)
                    activity.window.insetsController?.apply {
                        hide(WindowInsets.Type.statusBars())
                        systemBarsBehavior = WindowInsetsController.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
                    }
                } else {
                    @Suppress("DEPRECATION")
                    activity.window.decorView.systemUiVisibility =
                        View.SYSTEM_UI_FLAG_LAYOUT_STABLE or
                        View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                }
                activity.window.statusBarColor = android.graphics.Color.TRANSPARENT
            }
        }
    }

    // Convert DP to pixels for width
    fun getWidth(dp: Int): Int = (dp * densityScale + 0.5f).toInt()

    // Convert DP to pixels for height
    fun getHeight(dp: Int): Int = (dp * densityScale + 0.5f).toInt()

    // Convert DP to pixels for padding or margin
    fun getPaddingOrMarginValue(dp: Int): Int = (dp * densityScale + 0.5f).toInt()

    // Convert DP to pixels for corner radius
    fun calculateRadius(dp: Int): Int = (dp * densityScale + 0.5f).toInt()

    fun getLinearLayoutParam(): LinearLayout.LayoutParams =
        LinearLayout.LayoutParams(
            LinearLayout.LayoutParams.MATCH_PARENT,
            LinearLayout.LayoutParams.WRAP_CONTENT,
        )

    fun getMatchParentLinearLayoutParam(): LinearLayout.LayoutParams =
        LinearLayout.LayoutParams(
            LinearLayout.LayoutParams.MATCH_PARENT,
            LinearLayout.LayoutParams.MATCH_PARENT,
        )

    // Get scaled text size in pixels
    fun getTextSize(textSize: Int): Int = (textSize * scale + 0.5f).toInt()
}
