package nashid.verify.sdk.data.repository

import nashid.verify.sdk.utils.Constants.KSA_ID_CARD_REGEX1
import nashid.verify.sdk.utils.Constants.KSA_ID_CARD_REGEX2
import nashid.verify.sdk.utils.Constants.KSA_ID_CARD_REGEX3
import nashid.verify.sdk.utils.ScanDocument
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.CardReaderHelper
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.isBahCard
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.isOmanCard
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.replaceAlfaWithNumber
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.replaceNumberWithAlfa
import nashid.verify.sdk.viewmodel.CameraXLiveData
import java.util.Locale
import java.util.Objects
import java.util.regex.Pattern

class KSAIdCardRepository(
    private val cameraXLiveData: CameraXLiveData,
    private val scanDocument: ScanDocument,
) {
    private var retryCount = 0

    fun readKSAFrontViewData(originalText: String) {
        var text = originalText
        text = text.replace(" ", "").uppercase(Locale.getDefault())
        text = text.replace("\n", " ").uppercase(Locale.getDefault())
        val isKsaIdDetected = (
            (text.contains("KINGDOMOFSAUDIARABIA") || text.contains("MINISTRYOFINTERIOR")) &&
                (text.contains("NATIONALIDCARD") || text.contains("RESIDENTIDENTITY"))
        )

        if (retryCount == 3 || isKsaIdDetected) {
            retryCount = 0
            handleKSAView()
        } else {
            retryCount += 1
            cameraXLiveData.setIsSuccess(false)
        }
    }

    private fun handleKSAView() {
        cameraXLiveData.apply {
            setIdNo(null)
            setDOB(null)
            setExpiryDate(null)
        }
        Utility.getInstance().apply {
            isBarCodeCard = true
            isKSACard = true
            isUAECard = false
            isBahCard = false
            isOmanCard = false
        }
    }

    fun readKSABackViewData(originalText: String) {
        var text = originalText
        text = text.replace(" ", "").uppercase(Locale.getDefault())
        text = text.replace("\n", " ").uppercase(Locale.getDefault())
        val textAll = originalText.split("\n".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()
        if (Utility.getInstance().isNationalIdCard) {
            val pattern1 = Pattern.compile(KSA_ID_CARD_REGEX1)
            val pattern2 = Pattern.compile(KSA_ID_CARD_REGEX2)
            val pattern3 = Pattern.compile(KSA_ID_CARD_REGEX3)
            for (line in textAll) {
                var matcherText = line.replace(" ", "").replace('«', '<')
                if (matcherText.length == 30) {
                    val matcher1 = pattern1.matcher(matcherText)
                    val matcher2 = pattern2.matcher(matcherText)
                    val matcher3 = pattern3.matcher(matcherText)
                    if (matcher1.matches()) {
                        Utility.getInstance().mrzLine1 = removeKValueFromString(matcherText)
                        cameraXLiveData.setIdNo(
                            replaceAlfaWithNumber(
                                Objects.requireNonNull(
                                    matcher1.group(3),
                                ),
                            ),
                        )
                        val country = replaceNumberWithAlfa(Objects.requireNonNull(matcher1.group(2)))
                        Utility.getInstance().country = country
                        Utility.getInstance().documentType = matcher1.group(1)
                        Utility.getInstance().passportNumber = matcher1.group(3)
                        scanDocument.documentNo = matcher1.group(3)
                        cameraXLiveData.setIdLine1(false)
                    }
                    if (matcher2.matches()) {
                        Utility.getInstance().mrzLine2 = removeKValueFromString(matcherText.toString())
                        cameraXLiveData.setDOB(matcher2.group(1))
                        cameraXLiveData.setExpiryDate(matcher2.group(4))
                        val rawGender = matcher2.group(3)?.toString()?.trim()?.uppercase()
                        val gender =
                            when (rawGender) {
                                "F", "E" -> "F" // Sometimes OCR might read F as E
                                "M" -> "M"
                                "X" -> "X"
                                else -> {
                                    ""
                                }
                            }
                        Utility.getInstance().dateOfBirth = matcher2.group(1)
                        Utility.getInstance().expiryDate = matcher2.group(4)
                        Utility.getInstance().gender = gender.uppercase()

                        scanDocument.dateOfBirth = matcher2.group(1)
                        scanDocument.expiryDate = matcher2.group(4)
                        val nationality = matcher2.group(6)
                        Utility.getInstance().nationality = nationality
                        cameraXLiveData.setIdLine2(false)
                    }
                    if (matcher3.matches()) {
                        matcherText = removeKValueFromString(matcherText)
                        val rawMrz =
                            matcherText
                                .replace(Regex("K$"), "<") // fix the wrong K pattern
                                .replace(Regex("[<«]"), " ") // replace all < and « with space
                                .replace(Regex("\\s+"), " ") // clean multiple spaces
                                .trim()

                        Utility.getInstance().name = CardReaderHelper.correctOcrName(rawMrz)
                        val mrzLine = matcherText.replace(Regex("K$"), "<")
                        Utility.getInstance().mrzLine3 = CardReaderHelper.correctOcrName(mrzLine)
                        cameraXLiveData.setIdLine3(false)
                    }
                    if (cameraXLiveData.getIdLine1().value == false && cameraXLiveData.getIdLine2().value == false && cameraXLiveData.getIdLine3().value == false) {
                        break
                    }
                } else {
                    cameraXLiveData.setIsSuccess(false)
                }
            }
        } else {
            if (text.contains("KINGDOMOFSAUDIARABIA") || text.contains("MINISTRYOFINTERIOR") &&
                text.contains("NATIONALIDCARD")
            ) {
                Utility.getInstance().isNationalIdCard = true
                cameraXLiveData.setIsSuccess(false)
            } else if (text.contains("KINGDOMOFSAUDIARABIA") || text.contains("MINISTRYOFINTERIOR") &&
                text.contains("RESIDENTIDENTITY")
            ) {
                Utility.getInstance().isResidentIdCard = true
                cameraXLiveData.setIsSuccess(false)
            } else {
                cameraXLiveData.setIsSuccess(false)
            }
        }
    }

    private fun removeKValueFromString(modifiedMRZText: String): String {
        val latest = StringBuilder()
        for (i in modifiedMRZText.indices) {
            if (modifiedMRZText[i] == 'K' || modifiedMRZText[i] == 'C') {
                val prevChar = if (i > 0) modifiedMRZText[i - 1] else null
                val nextChar = if (i < modifiedMRZText.length - 1) modifiedMRZText[i + 1] else null
                if ((prevChar == '<' || prevChar == 'K' || prevChar == 'C') && (nextChar == '<' || nextChar == 'K' || nextChar == 'C')) {
                    latest.append('<')
                } else {
                    latest.append('K')
                }
            } else {
                latest.append(modifiedMRZText[i])
            }
        }
        return latest.toString()
    }
}
