package nashid.verify.sdk.ui

import android.Manifest
import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.annotation.SuppressLint
import android.annotation.TargetApi
import android.app.Dialog
import android.content.Context
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Color
import android.graphics.PointF
import android.graphics.Rect
import android.graphics.RectF
import android.graphics.SurfaceTexture
import android.graphics.drawable.ColorDrawable
import android.hardware.camera2.CameraCharacteristics
import android.hardware.camera2.CameraManager
import android.os.Build
import android.os.Bundle
import android.os.CountDownTimer
import android.util.Size
import android.util.TypedValue
import android.view.Surface
import android.view.View.GONE
import android.view.View.INVISIBLE
import android.view.View.VISIBLE
import android.view.ViewGroup
import android.view.Window
import android.widget.LinearLayout
import android.widget.Toast
import androidx.activity.addCallback
import androidx.camera.core.CameraSelector
import androidx.camera.core.ExperimentalGetImage
import androidx.camera.core.ImageAnalysis
import androidx.camera.core.ImageCapture
import androidx.camera.core.ImageCaptureException
import androidx.camera.core.ImageProxy
import androidx.camera.core.Preview
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.core.content.ContextCompat
import androidx.core.view.doOnLayout
import com.google.mlkit.vision.common.InputImage
import com.google.mlkit.vision.face.Face
import com.google.mlkit.vision.face.FaceDetection
import com.google.mlkit.vision.face.FaceDetector
import com.google.mlkit.vision.face.FaceDetectorOptions
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import nashid.mv.liveness.DetectionResults
import nashid.mv.liveness.EngineWrappers
import nashid.verify.sdk.ui.theme.FontManager
import nashid.verify.sdk.ui.theme.FontWeight
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.EdgeToEdgeHelper
import nashid.verify.sdk.utils.helpers.LiveNessCameraLayout
import nashid.verify.sdk.utils.helpers.TextSizeConverter
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.DialogBackBinding
import nashid.verify.sdkNew.databinding.LivenessActivityMainBinding
import org.koin.android.ext.android.inject
import java.io.ByteArrayOutputStream
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors
import kotlin.math.abs

@OptIn(ExperimentalCoroutinesApi::class)
internal class LiveNessActivity : BaseActivity() {
    private lateinit var liveNessLayout: LiveNessCameraLayout
    private lateinit var binding: LivenessActivityMainBinding
    private lateinit var engineWrappers: EngineWrappers
    private var enginePrepared: Boolean = false
    private var cameraProvider: ProcessCameraProvider? = null
    private val textSizeConverter: TextSizeConverter by inject()
    private var startToCaptureImage = false
    private var cameraSelector: CameraSelector? = null
    private var imageCount = 0
    private var scanFailCounter = 1
    private val maxAttempts = 3

    @OptIn(DelicateCoroutinesApi::class)
    private val detectionContext = newSingleThreadContext("detection")

    @TargetApi(Build.VERSION_CODES.M)
    private fun requestPermission() = requestPermissions(permissions, PERMISSION_REQUEST_CODE)

    private var working: Boolean = false
    private lateinit var result: DetectionResults
    private lateinit var detectionResults: DetectionResults
    private var delayMillis: Long = 15000
    private var cTimer: CountDownTimer? = null
    private lateinit var imageProxy: ImageProxy
    private val previewWidth: Int = 640
    private val previewHeight: Int = 480
    private lateinit var cameraExecutor: ExecutorService
    private val frameOrientation: Int = 7
    private var threshold: Float = DEFAULT_THRESHOLD
    private var startRecording = false
    private var isStartTimer: Boolean = false
    private var factorX: Float = 0F
    private var factorY: Float = 0F
    private var screenWidth: Int = 0
    private var screenHeight: Int = 0
    private var currentAction: LiveNessAction = LiveNessAction.NONE
    private var actionCompleted =
        mutableMapOf(
            LiveNessAction.FACE_CHECK to false,
            LiveNessAction.TURN_HEAD_LEFT to false,
            LiveNessAction.TURN_HEAD_RIGHT to false,
            LiveNessAction.BLINK to false,
            LiveNessAction.SMILE to false,
        )

    // Add new property to track passive liveNess completion
    private var isPassiveLiveNessComplete = false
    private var stableFrameCount = 0
    private val requiredStableFrames = 5

    enum class LiveNessAction {
        NONE,
        FACE_CHECK,
        TURN_HEAD_LEFT,
        TURN_HEAD_RIGHT,
        BLINK,
        SMILE,
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = LivenessActivityMainBinding.inflate(layoutInflater)
        setContentView(binding.root)
        if (hasPermissions()) {
            init()
        } else {
            requestPermission()
        }
    }

    private fun init() {
        try {
            logger.log("Initializing LiveNessActivity")
            cameraExecutor = Executors.newSingleThreadExecutor()
            try {
                engineWrappers = EngineWrappers(assets)
                enginePrepared = engineWrappers.init()
                if (!enginePrepared) {
                    logger.log("Engine initialization failed")
                    if (!SdkConfig.isPassiveLiveNessEnabled) {
                        Toast.makeText(this, "Engine init failed.", Toast.LENGTH_LONG).show()
                        setResult(RESULT_CANCELED)
                        finish()
                        return
                    }
                } else {
                    logger.log("Engine initialized successfully")
                }
            } catch (e: Exception) {
                logger.log("Exception during engine initialization: $e")
                enginePrepared = false
                // Continue for passive liveness only
                if (!SdkConfig.isPassiveLiveNessEnabled) {
                    setResult(RESULT_CANCELED)
                    finish()
                    return
                }
            }

            backPress()
            initFaceDetector()
            detectionResults = DetectionResults()
            liveNessLayout = LiveNessCameraLayout(this, binding, storage, textSizeConverter)
            liveNessLayout.changeTextSize()
            startCamera()
            binding.layoutHeader.imgBack.setColorFilter(SdkConfig.sdkAppTheme.getPrimaryColorInt())
            binding.layoutHeader.imgBack.setOnClickListener { backPress() }
            binding.layoutHeader.imgBack.setOnClickListener { showLeaveOnboardingDialog() }
            EdgeToEdgeHelper.applySystemInsets(binding.root, applyTopInset = true, applyBottomInset = true)
            binding.lytStartRecording.setOnClickListener {
                startRecording = true
                binding.cardPressDesc.visibility = GONE
                binding.cardStartDesc.visibility = GONE

                binding.lytStartRecording.visibility = INVISIBLE
                binding.preview.visibility = VISIBLE
                binding.overlayMainLayout.setBackgroundColor(
                    ContextCompat.getColor(
                        applicationContext,
                        R.color.transparent,
                    ),
                )
            }

            // Initialize based on liveness configuration
            when {
                SdkConfig.isPassiveLiveNessEnabled && SdkConfig.isActiveLiveNessEnabled -> {
                    // Both passive and active liveness enabled - start with passive
                    currentAction = LiveNessAction.NONE
                    binding.txtTakeSelfie.text = getString(R.string.face_scanning)
                    isPassiveLiveNessComplete = false
                    logger.log("Initialized for both passive and active liveness - starting with passive")
                }

                SdkConfig.isPassiveLiveNessEnabled -> {
                    // Only passive liveness
                    binding.txtTakeSelfie.text = getString(R.string.face_scanning)
                    isPassiveLiveNessComplete = false
                    currentAction = LiveNessAction.FACE_CHECK
                    showActionPrompt(currentAction)
                    logger.log("Initialized for passive-only liveness")
                }

                SdkConfig.isActiveLiveNessEnabled -> {
                    // Only active liveness
                    currentAction = LiveNessAction.FACE_CHECK
                    showActionPrompt(currentAction)
                    scanFailCounter = 1
                    isPassiveLiveNessComplete = true // No passive liveness needed
                    logger.log("Initialized for active-only liveness")
                }

                else -> {
                    logger.log("No liveness enabled - should not reach here")
                    setResult(RESULT_CANCELED)
                    finish()
                }
            }

            logger.log("LiveNessActivity initialization completed successfully")
        } catch (e: Exception) {
            logger.log("Error initializing LiveNessActivity: $e")
            setResult(RESULT_CANCELED)
            finish()
        }
    }

    private fun backPress() {
        onBackPressedDispatcher.addCallback(this) {
            showLeaveOnboardingDialog()
        }
    }

    private fun hasPermissions(): Boolean {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            for (permission in permissions) {
                if (checkSelfPermission(permission) != PackageManager.PERMISSION_GRANTED) {
                    return false
                }
            }
        }
        return true
    }

    private fun takeImage() {
        logger.log("takeImage: Starting active liveness image capture")
        runOnUiThread {
            binding.txtTakeSelfie.text = getString(R.string.face_smile)
            binding.coverView.changeColor(SdkConfig.sdkAppTheme.getPrimaryColorInt())
            val imageCapture =
                ImageCapture
                    .Builder()
                    .setTargetRotation(Surface.ROTATION_0)
                    .build()
            startToCaptureImage = true
            cameraProvider?.let {
                try {
                    it.unbindAll()
                    it.bindToLifecycle(this, cameraSelector!!, imageCapture)
                    imageCapture.takePicture(
                        ContextCompat.getMainExecutor(this),
                        object : ImageCapture.OnImageCapturedCallback() {
                            @SuppressLint("RestrictedApi", "UnsafeOptInUsageError")
                            override fun onCaptureSuccess(image: ImageProxy) {
                                try {
                                    logger.log("takeImage: Active liveness image captured successfully")
                                    val mediaImage = image.image
                                    if (mediaImage != null) {
                                        val buffer = mediaImage.planes[0].buffer
                                        val bytes = ByteArray(buffer.capacity())
                                        buffer.get(bytes)

                                        // Store active liveNess image if active liveNess was performed
                                        if (SdkConfig.isActiveLiveNessEnabled) {
                                            Utility.getInstance().setActiveLivenessImage(bytes)
                                            logger.log("takeImage: Active liveness image stored, size: ${bytes.size}")
                                        }

                                        // For backward compatibility - set liveImage
                                        Utility.getInstance().liveImage = bytes
                                        logger.log("takeImage: Set as main liveImage")

                                        cameraProvider?.unbindAll()
                                        binding.lytStartRecording.visibility = INVISIBLE
                                        binding.lyutSucess.visibility = VISIBLE
                                        binding.imgScanComplete.playAnimation()
                                        binding.imgScanComplete.setMaxFrame(80)
                                        binding.imgScanComplete.addAnimatorListener(
                                            object : AnimatorListenerAdapter() {
                                                override fun onAnimationEnd(animation: Animator) {
                                                    super.onAnimationEnd(animation)
                                                    logger.log("takeImage: Success animation complete - finishing activity")
                                                    binding.txtTakeSelfie.visibility = INVISIBLE
                                                    setResult(RESULT_OK)
                                                    finish()
                                                }
                                            },
                                        )
                                    } else {
                                        logger.log("takeImage: Media image is null")
                                        setResult(RESULT_CANCELED)
                                        finish()
                                    }
                                } catch (e: Exception) {
                                    logger.log("takeImage: Error saving active liveness image: $e")
                                    setResult(RESULT_CANCELED)
                                    finish()
                                } finally {
                                    image.close()
                                }
                            }

                            override fun onError(exc: ImageCaptureException) {
                                logger.log("takeImage: Error capturing active liveness image: $exc")
                                setResult(RESULT_CANCELED)
                                finish()
                            }
                        },
                    )
                } catch (ex: Exception) {
                    logger.log("takeImage: Exception in takeImage: $ex")
                    setResult(RESULT_CANCELED)
                    finish()
                }
            } ?: run {
                logger.log("takeImage: Camera provider is null")
                setResult(RESULT_CANCELED)
                finish()
            }
        }
    }

    private fun startTimer() {
        cTimer?.cancel()
        cTimer =
            object : CountDownTimer(delayMillis, 1000) {
                override fun onTick(millisUntilFinished: Long) {
                }

                override fun onFinish() {
                    try {
                        if (scanFailCounter >= maxAttempts) {
                            if (SdkConfig.isActiveLiveNessEnabled && !SdkConfig.isPassiveLiveNessEnabled) {
                                Utility.getInstance().setActiveLivenessMaxRetries(true)
                            } else if (!SdkConfig.isActiveLiveNessEnabled && SdkConfig.isPassiveLiveNessEnabled) {
                                Utility.getInstance().setPassiveLivenessMaxRetries(true)
                            } else if (SdkConfig.isActiveLiveNessEnabled && SdkConfig.isPassiveLiveNessEnabled) {
                                Utility.getInstance().setPassiveLivenessMaxRetries(true)
                                Utility.getInstance().setActiveLivenessMaxRetries(true)
                            }
                            scanFailCounter = 1
                            binding.lyoutScanFail.visibility = INVISIBLE
                            setResult(RESULT_CANCELED)
                            finish()
                        } else {
                            setRescanningDocumentVisibility()
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }
                }
            }.start()
    }

    private fun setRescanningDocumentVisibility() {
        binding.lyoutScanFail.visibility = VISIBLE
        hideAnimation()
        val remainingRetry = maxAttempts - scanFailCounter
        binding.txtRescanningDocuments.text = getString(R.string.btn_retry) + " (" + remainingRetry + " " + getString(R.string.nfc_retry_remaining) + ")"
        binding.btnRescanningDocuments.setOnClickListener {
            showAnimation()
            setLiveNessScanFailed()
            binding.cardStartDesc.visibility = GONE
            binding.lyoutScanFail.visibility = GONE
            imageProxy.close()
            startTimer()
        }
    }

    private fun setLiveNessScanFailed() {
        scanFailCounter++
        currentAction = LiveNessAction.FACE_CHECK
        showActionPrompt(currentAction)
        actionCompleted =
            mutableMapOf(
                LiveNessAction.FACE_CHECK to false,
                LiveNessAction.TURN_HEAD_LEFT to false,
                LiveNessAction.TURN_HEAD_RIGHT to false,
                LiveNessAction.BLINK to false,
                LiveNessAction.SMILE to false,
            )
    }

    @OptIn(DelicateCoroutinesApi::class)
    @SuppressLint("UnsafeOptInUsageError")
    private fun onPreviewFrame(imageProxy: ImageProxy) {
        val byteArray = liveNessLayout.getYuvDataFromImage(imageProxy)
        if (binding.lyoutScanFail.visibility == GONE) {
            if (!startToCaptureImage) {
                if (enginePrepared) {
                    if (!working) {
                        GlobalScope.launch(detectionContext) {
                            working = true

                            result =
                                engineWrappers.detect(
                                    byteArray,
                                    previewWidth,
                                    previewHeight,
                                    frameOrientation,
                                )
                            result.threshold = threshold
                            runOnUiThread {
                                if (!isStartTimer) {
                                    startTimer()
                                    isStartTimer = true
                                }
                            }

                            factorX = screenWidth / previewHeight.toFloat()
                            factorY = screenHeight / previewWidth.toFloat()
                            val rect =
                                calculateBoxLocationOnScreen(
                                    result.left,
                                    result.top,
                                    result.right,
                                    result.bottom,
                                )
                            detectionResults = result.updateLocation(rect)

                            val isOutside = isRectOutside(result)
                            if (isOutside) {
                                this@LiveNessActivity.runOnUiThread {
                                    if (startRecording) {
                                        binding.coverView.changeColor(SdkConfig.sdkAppTheme.getBackgroundColorInt())
                                        binding.txtTakeSelfie.text =
                                            getString(R.string.face_scanning)
                                        binding.txtStartRecording.text =
                                            getString(R.string.face_outside)
                                        binding.cardStartDesc.visibility = VISIBLE
                                        binding.txtStartRecording.visibility = VISIBLE
                                        binding.txtStartRecordingDesc2.visibility = GONE
                                        binding.livenessAnimation.pauseAnimation()
                                    }
                                }
                                if (binding.lyoutScanFail.visibility != VISIBLE) {
                                    imageProxy.close()
                                }
                            } else {
                                this@LiveNessActivity.runOnUiThread {
                                    if (startRecording) {
                                        binding.livenessAnimation.apply {
                                            if (currentAction == LiveNessAction.FACE_CHECK) {
                                                visibility = VISIBLE
                                            }
                                        }
                                        binding.cardStartDesc.visibility = GONE
                                        binding.txtStartRecording.visibility = GONE
                                        binding.txtStartRecordingDesc2.visibility = GONE
                                    }
                                }
                                if (result.confidence >= 0.985F) {
                                    if (startRecording) {
                                        logger.log("onPreviewFrame: Face detected with confidence ${result.confidence}")
                                        // Handle liveness based on SDK configuration and current state
                                        if (SdkConfig.isPassiveLiveNessEnabled && !isPassiveLiveNessComplete) {
                                            // Passive liveness first (for both passive-only and mixed configs)
                                            logger.log("onPreviewFrame: Handling passive liveness")
                                            handlePassiveLiveness(imageProxy)
                                        } else if (SdkConfig.isActiveLiveNessEnabled && isPassiveLiveNessComplete) {
                                            // Active liveness (for both active-only and mixed configs after passive)
                                            logger.log("onPreviewFrame: Handling active liveness")
                                            handleActiveLiveNess(imageProxy)
                                        } else {
                                            // Edge case - should not happen
                                            logger.log("onPreviewFrame: Unexpected state - closing imageProxy")
                                            if (binding.lyoutScanFail.visibility != VISIBLE) {
                                                imageProxy.close()
                                            }
                                        }
                                    } else {
                                        if (binding.lyoutScanFail.visibility != VISIBLE) {
                                            imageProxy.close()
                                        }
                                    }
                                } else {
                                    if (binding.lyoutScanFail.visibility != VISIBLE) {
                                        imageProxy.close()
                                    }
                                }
                            }
                            working = false
                        }
                    } else {
                        if (binding.lyoutScanFail.visibility != VISIBLE) {
                            imageProxy.close()
                        }
                    }
                } else {
                    if (binding.lyoutScanFail.visibility != VISIBLE) {
                        imageProxy.close()
                    }
                }
            }
        }
    }

    private fun hideAnimation() {
        binding.livenessAnimation.apply {
            visibility = GONE
        }
    }

    private fun showAnimation() {
        binding.livenessAnimation.apply {
            visibility = VISIBLE
        }
    }

    private fun handlePassiveLiveness(imageProxy: ImageProxy) {
        logger.log("handlePassiveLiveness: imageCount = $imageCount")
        if (imageCount == 10) {
            logger.log("Passive liveness scan complete")
            // Take and save passive liveness image
            takePassiveLivenessImage()

            // Check configuration to determine next action
            if (!SdkConfig.isActiveLiveNessEnabled) {
                // Passive-only liveness - take image first, then complete the process
                logger.log("Passive-only liveness - completing process")
                startToCaptureImage = true // Set this flag to prevent further processing
                setLiveNessScore()
                // Don't show success animation here - wait for image capture to complete
            } else if (SdkConfig.isActiveLiveNessEnabled) {
                // Both passive and active enabled - passive complete, start active
                logger.log("Passive liveness complete - transitioning to active liveness")
                isPassiveLiveNessComplete = true
                scanFailCounter = 1
                currentAction = LiveNessAction.FACE_CHECK
                imageCount = 0 // Reset counter for active liveness
                showActionPrompt(currentAction)
                runOnUiThread { startTimer() }
            }
        }

        if (binding.lyoutScanFail.visibility != VISIBLE) {
            imageProxy.close()
        }
        imageCount++
    }

    private fun takePassiveLivenessImage() {
        logger.log("takePassiveLivenessImage: Starting image capture")
        runOnUiThread {
            val imageCapture =
                ImageCapture
                    .Builder()
                    .setTargetRotation(Surface.ROTATION_0)
                    .build()

            cameraProvider?.let {
                try {
                    it.unbindAll()
                    it.bindToLifecycle(this, cameraSelector!!, imageCapture)

                    imageCapture.takePicture(
                        ContextCompat.getMainExecutor(this),
                        object : ImageCapture.OnImageCapturedCallback() {
                            @SuppressLint("RestrictedApi", "UnsafeOptInUsageError")
                            override fun onCaptureSuccess(image: ImageProxy) {
                                try {
                                    logger.log("takePassiveLivenessImage: Image captured successfully")
                                    val mediaImage = image.image
                                    if (mediaImage != null) {
                                        // Convert ImageProxy to bitmap first, then to byte array
                                        val bitmap = imageProxyToBitmap(image)
                                        val stream = ByteArrayOutputStream()
                                        bitmap.compress(Bitmap.CompressFormat.JPEG, 100, stream)
                                        val bytes = stream.toByteArray()

                                        // Always store passive liveness image when passive is enabled
                                        Utility.getInstance().setPassiveLiveNessImage(bytes)
                                        logger.log("takePassiveLivenessImage: Passive liveness image stored, size: ${bytes.size}")

                                        // Set liveImage for backward compatibility - use passive image if no active
                                        if (!SdkConfig.isActiveLiveNessEnabled) {
                                            Utility.getInstance().liveImage = bytes
                                            logger.log("takePassiveLivenessImage: Set as main liveImage for passive-only config")

                                            // Now show success animation and finish activity for passive-only
                                            runOnUiThread {
                                                binding.lytStartRecording.visibility = INVISIBLE
                                                binding.lyutSucess.visibility = VISIBLE
                                                binding.imgScanComplete.playAnimation()
                                                binding.imgScanComplete.setMaxFrame(80)
                                                binding.imgScanComplete.addAnimatorListener(
                                                    object : AnimatorListenerAdapter() {
                                                        override fun onAnimationEnd(animation: Animator) {
                                                            super.onAnimationEnd(animation)
                                                            logger.log("Success animation complete - finishing activity")
                                                            binding.txtTakeSelfie.visibility = INVISIBLE
                                                            setResult(RESULT_OK)
                                                            finish()
                                                        }
                                                    },
                                                )
                                            }
                                        }
                                    } else {
                                        logger.log("takePassiveLivenessImage: Media image is null")
                                    }
                                } catch (e: Exception) {
                                    logger.log("takePassiveLivenessImage: Error saving passive liveness image: $e")
                                } finally {
                                    image.close()
                                    if (!SdkConfig.isActiveLiveNessEnabled) {
                                        // For passive-only, we don't need to restart the camera
                                        logger.log("takePassiveLivenessImage: Passive-only - not restarting camera")
                                    } else {
                                        // For mixed config, restart camera for active liveness
                                        logger.log("takePassiveLivenessImage: Restarting camera for active liveness")
                                        runOnUiThread {
                                            startCamera()
                                        }
                                    }
                                }
                            }

                            override fun onError(exc: ImageCaptureException) {
                                logger.log("takePassiveLivenessImage: Error capturing passive liveness image: $exc")
                                runOnUiThread {
                                    startCamera()
                                }
                            }
                        },
                    )
                } catch (ex: Exception) {
                    logger.log("takePassiveLivenessImage: Error in takePassiveLivenessImage: $ex")
                    runOnUiThread {
                        startCamera()
                    }
                }
            } ?: run {
                logger.log("takePassiveLivenessImage: Camera provider is null")
            }
        }
    }

    @androidx.annotation.OptIn(ExperimentalGetImage::class)
    private fun handleActiveLiveNess(imageProxy: ImageProxy) {
        if (!isPassiveLiveNessComplete && SdkConfig.isPassiveLiveNessEnabled) {
            // Handle as passive liveNess first
            handlePassiveLiveness(imageProxy)
            return
        }

        // Existing active liveNess logic
        if (imageCount >= 10) {
            val mediaImage = imageProxy.image
            if (mediaImage == null) {
                imageProxy.close()
                return
            }
            val image =
                InputImage.fromMediaImage(
                    mediaImage,
                    imageProxy.imageInfo.rotationDegrees,
                )
            faceDetector.process(image)
                .addOnSuccessListener { faces ->
                    if (faces.isNotEmpty()) {
                        val face = faces[0]
                        when (currentAction) {
                            LiveNessAction.FACE_CHECK -> checkFace()
                            LiveNessAction.TURN_HEAD_LEFT -> checkHeadTurnLeft(face)
                            LiveNessAction.TURN_HEAD_RIGHT -> checkHeadTurnRight(face)
                            LiveNessAction.BLINK -> checkForBlink(face)
                            LiveNessAction.SMILE -> checkForSmile(face)
                            LiveNessAction.NONE -> {
                            }
                        }
                    }
                }
                .addOnFailureListener {
                }
                .addOnCompleteListener {
                    if (binding.lyoutScanFail.visibility != VISIBLE) {
                        imageProxy.close()
                    }
                }
        } else {
            if (binding.lyoutScanFail.visibility != VISIBLE) {
                imageProxy.close()
            }
        }
        imageCount++
    }

    // Add ML Kit face detector initialization
    private lateinit var faceDetector: FaceDetector

    private fun initFaceDetector() {
        val options =
            FaceDetectorOptions.Builder()
                .setPerformanceMode(FaceDetectorOptions.PERFORMANCE_MODE_ACCURATE)
                .setLandmarkMode(FaceDetectorOptions.LANDMARK_MODE_ALL)
                .setClassificationMode(FaceDetectorOptions.CLASSIFICATION_MODE_ALL)
                .build()

        faceDetector = FaceDetection.getClient(options)
    }

    @SuppressLint("UnsafeOptInUsageError")
    private fun imageProxyToBitmap(imageProxy: ImageProxy): Bitmap {
        val buffer = imageProxy.planes[0].buffer
        val bytes = ByteArray(buffer.remaining())
        buffer.get(bytes)
        return BitmapFactory.decodeByteArray(bytes, 0, bytes.size)
    }

    private fun setNextAction() {
        currentAction =
            when {
                !(actionCompleted[LiveNessAction.FACE_CHECK] ?: false) -> LiveNessAction.FACE_CHECK
                !(actionCompleted[LiveNessAction.TURN_HEAD_LEFT] ?: false) -> LiveNessAction.TURN_HEAD_LEFT
                !(actionCompleted[LiveNessAction.TURN_HEAD_RIGHT] ?: false) -> LiveNessAction.TURN_HEAD_RIGHT
                !(actionCompleted[LiveNessAction.BLINK] ?: false) -> LiveNessAction.BLINK
                !(actionCompleted[LiveNessAction.SMILE] ?: false) -> LiveNessAction.SMILE
                else -> {
                    // All actions completed - check if image hasn't been captured yet
                    if (!startToCaptureImage) {
                        logger.log("setNextAction: All actions completed, capturing final image")
                        cTimer?.cancel()
                        takeImage()
                        setLiveNessScore()
                    }
                    LiveNessAction.NONE
                }
            }

        showActionPrompt(currentAction)
    }

    private fun showActionPrompt(action: LiveNessAction) {
        runOnUiThread {
            binding.livenessAnimation.apply {
                when (action) {
                    LiveNessAction.FACE_CHECK -> {
                        if (!SdkConfig.isActiveLiveNessEnabled && SdkConfig.isPassiveLiveNessEnabled) {
                            visibility = GONE
                        }
                        setAnimation(R.raw.face_point)
                        binding.txtTakeSelfie.text = getString(R.string.face_scanning)
                        playAnimation()
                    }

                    LiveNessAction.BLINK -> {
                        visibility = GONE
                        binding.txtTakeSelfie.text = getString(R.string.liveness_blink)
                    }

                    LiveNessAction.TURN_HEAD_LEFT -> {
                        visibility = VISIBLE
                        setAnimation(R.raw.turn_head_left)
                        binding.txtTakeSelfie.text = getString(R.string.liveness_turn_left)
                        playAnimation()
                    }

                    LiveNessAction.TURN_HEAD_RIGHT -> {
                        visibility = VISIBLE
                        setAnimation(R.raw.turn_head_right)
                        binding.txtTakeSelfie.text = getString(R.string.liveness_turn_right)
                        playAnimation()
                    }

                    LiveNessAction.SMILE -> {
                        visibility = GONE
                        binding.txtTakeSelfie.text = getString(R.string.liveness_smile)
                    }

                    LiveNessAction.NONE -> {
                        visibility = GONE
                    }
                }
            }
        }
    }

    private fun checkForBlink(face: Face) {
        if (face.leftEyeOpenProbability != null && face.rightEyeOpenProbability != null) {
            val leftEyeClosed = face.leftEyeOpenProbability!! < 0.1
            val rightEyeClosed = face.rightEyeOpenProbability!! < 0.1

            if (leftEyeClosed && rightEyeClosed) {
                actionCompleted[LiveNessAction.BLINK] = true
                setNextAction()
            }
        }
    }

    private fun checkHeadTurnLeft(face: Face) {
        val headEulerY = face.headEulerAngleY
        if (headEulerY > 30) {
            actionCompleted[LiveNessAction.TURN_HEAD_LEFT] = true
            setNextAction()
        }
    }

    private fun checkHeadTurnRight(face: Face) {
        val headEulerY = face.headEulerAngleY
        if (headEulerY < -30) {
            actionCompleted[LiveNessAction.TURN_HEAD_RIGHT] = true
            setNextAction()
        }
    }

    private fun checkForSmile(face: Face) {
        if (face.smilingProbability != null && face.smilingProbability!! > 0.8) {
            // Check if we have a good quality image before capturing
            if (isImageQualityGood(face)) {
                stableFrameCount++
                logger.log("checkForSmile: Good quality frame $stableFrameCount/$requiredStableFrames")
                // Only capture when we have enough stable frames
                if (stableFrameCount >= requiredStableFrames) {
                    actionCompleted[LiveNessAction.SMILE] = true
                    cTimer?.cancel()
                    logger.log("checkForSmile: Smile detected with stable good quality, capturing image")
                    takeImage()
                    setLiveNessScore()
                }
            } else {
                stableFrameCount = 0
                logger.log("checkForSmile: Smile detected but image quality not good enough, continue smiling")
            }
        } else {
            stableFrameCount = 0
        }
    }

    private fun isImageQualityGood(face: Face): Boolean {
        if (isRectOutside(result)) {
            logger.log("isImageQualityGood: Face outside oval bounds")
            return false
        }
        val faceBounds = face.boundingBox
        val faceArea = faceBounds.width() * faceBounds.height()

        if (faceArea < 10000) {
            logger.log("isImageQualityGood: Face too small, area: $faceArea")
            return false
        }
        val headEulerY = face.headEulerAngleY
        val headEulerZ = face.headEulerAngleZ

        if (kotlin.math.abs(headEulerY) > 45 || kotlin.math.abs(headEulerZ) > 45) {
            logger.log("isImageQualityGood: Head pose too extreme, Y: $headEulerY, Z: $headEulerZ")
            return false
        }

        // Check if both eyes are visible - more lenient
        val leftEyeOpen = face.leftEyeOpenProbability ?: 0.5f
        val rightEyeOpen = face.rightEyeOpenProbability ?: 0.5f

        if (leftEyeOpen < 0.1f || rightEyeOpen < 0.1f) {
            logger.log("isImageQualityGood: Eyes not open enough, left: $leftEyeOpen, right: $rightEyeOpen")
            return false
        }

        // More lenient confidence check
        if (result.confidence < 0.95f) {
            logger.log("isImageQualityGood: Detection confidence too low: ${result.confidence}")
            return false
        }

        logger.log("isImageQualityGood: All quality checks passed")
        return true
    }

    private fun checkFace() {
        if (result.confidence >= 0.985F) {
            actionCompleted[LiveNessAction.FACE_CHECK] = true
            setNextAction()
        }
    }

    private fun setLiveNessScore() {
        val liveNessScore = result.confidence * 100
        logger.log("Setting liveness score: $liveNessScore")
        Utility.getInstance().liveNessScore = liveNessScore.toInt()
    }

    private fun isRectOutside(parentRect: DetectionResults): Boolean {
        val rect =
            RectF(
                parentRect.left.toFloat(),
                parentRect.top.toFloat(),
                parentRect.right.toFloat(),
                parentRect.bottom.toFloat(),
            )
        val ovalRect =
            RectF(
                binding.coverView.getRect().left,
                binding.coverView.getRect().top,
                binding.coverView.getRect().right,
                binding.coverView.getRect().bottom,
            )

        val ovalCenterX = ovalRect.centerX()
        val ovalCenterY = ovalRect.centerY()
        val ovalRadiusX = ovalRect.width() / 2
        val ovalRadiusY = ovalRect.height() / 2

        val rectCorners =
            arrayOf(
                PointF(rect.left, rect.top),
                PointF(rect.right, rect.top),
                PointF(rect.left, rect.bottom),
                PointF(rect.right, rect.bottom),
            )

        for (corner in rectCorners) {
            val dx = corner.x - ovalCenterX
            val dy = corner.y - ovalCenterY

            val distance = (dx * dx) / (ovalRadiusX * ovalRadiusX) + (dy * dy) / (ovalRadiusY * ovalRadiusY)
            if (distance > 1) {
                this@LiveNessActivity.runOnUiThread {
                    if (startRecording) {
                        binding.coverView.changeColor(SdkConfig.sdkAppTheme.getBackgroundColorInt())
                        binding.txtTakeSelfie.text = getString(R.string.face_scanning)
                        binding.livenessAnimation.pauseAnimation()
                    }
                }
                return true
            }
        }

        this@LiveNessActivity.runOnUiThread {
            if (startRecording) {
                if (SdkConfig.isActiveLiveNessEnabled) {
                    when (currentAction) {
                        LiveNessAction.FACE_CHECK -> binding.txtTakeSelfie.text = getString(R.string.face_scanning)
                        LiveNessAction.TURN_HEAD_LEFT -> binding.txtTakeSelfie.text = getString(R.string.liveness_turn_left)
                        LiveNessAction.TURN_HEAD_RIGHT -> binding.txtTakeSelfie.text = getString(R.string.liveness_turn_right)
                        LiveNessAction.BLINK -> binding.txtTakeSelfie.text = getString(R.string.liveness_blink)
                        LiveNessAction.SMILE -> binding.txtTakeSelfie.text = getString(R.string.liveness_smile)
                        LiveNessAction.NONE -> binding.txtTakeSelfie.text = getString(R.string.face_scanning)
                    }
                    binding.livenessAnimation.resumeAnimation()
                } else if (SdkConfig.isPassiveLiveNessEnabled) {
                    binding.txtTakeSelfie.text = getString(R.string.face_scanning)
                }
            }
        }
        return false
    }

    private fun calculateBoxLocationOnScreen(
        left: Int,
        top: Int,
        right: Int,
        bottom: Int,
    ): Rect =
        Rect(
            (left * factorX).toInt(),
            (top * factorY).toInt(),
            (right * factorX).toInt(),
            (bottom * factorY).toInt(),
        )

    private fun getBestPreviewSize(): Size? {
        val cameraManager = getSystemService(Context.CAMERA_SERVICE) as CameraManager
        val characteristics = cameraManager.getCameraCharacteristics(liveNessLayout.getFrontCameraId().toString())
        val streamConfigurationMap =
            characteristics.get(CameraCharacteristics.SCALER_STREAM_CONFIGURATION_MAP)

        val outputSizes = streamConfigurationMap?.getOutputSizes(SurfaceTexture::class.java)

        var bestPreviewSize: Size? = null
        var bestPreviewSizeArea = 0

        outputSizes?.forEach { size ->
            val area = size.width * size.height
            if (size.width >= previewWidth && size.height >= previewHeight && area > bestPreviewSizeArea) {
                bestPreviewSize = size
                bestPreviewSizeArea = area
            }
        }

        return bestPreviewSize
    }

    @SuppressLint("UnsafeOptInUsageError")
    private fun startCamera() {
        calculateSize()
        cameraExecutor = Executors.newSingleThreadExecutor()
        getBestPreviewSize()

        val preview =
            Preview
                .Builder()
                .build()
                .also {
                    it.surfaceProvider = binding.preview.surfaceProvider
                }

        val imageAnalysis =
            ImageAnalysis.Builder().build().also {
                it.setAnalyzer(
                    cameraExecutor,
                ) { imageProxy ->
                    if (binding.lyoutScanFail.visibility == GONE) {
                        if (startRecording) {
                            this.imageProxy = imageProxy
                            onPreviewFrame(imageProxy)
                        } else {
                            imageProxy.close()
                        }
                    }
                }
            }
        cameraSelector =
            CameraSelector
                .Builder()
                .requireLensFacing(CameraSelector.LENS_FACING_FRONT)
                .build()

        val cameraProviderFuture = ProcessCameraProvider.getInstance(this)

        cameraProviderFuture.addListener({
            cameraProvider = cameraProviderFuture.get()

            try {
                cameraProvider?.unbindAll()
                cameraProvider?.bindToLifecycle(this, cameraSelector!!, preview, imageAnalysis)
            } catch (_: Exception) {
            }
        }, ContextCompat.getMainExecutor(this))
    }

    private fun calculateSize() {
        binding.coverView.doOnLayout {
            screenWidth = binding.coverView.width
            screenHeight = binding.coverView.height
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        try {
            logger.log("Destroying LiveNessActivity")
            cTimer?.cancel()
            cameraProvider?.unbindAll()
            cameraExecutor.shutdown()
            if (::engineWrappers.isInitialized) {
                engineWrappers.destroy()
                logger.log("Engine destroyed successfully")
            }

            logger.log("LiveNessActivity destroyed successfully")
        } catch (e: Exception) {
            logger.log("Error in onDestroy: $e")
        }
    }

    override fun onPause() {
        super.onPause()
        try {
            cTimer?.cancel()
        } catch (e: Exception) {
            logger.log("Error in onPause: $e")
        }
    }

    override fun onStop() {
        super.onStop()
        try {
            if (!isFinishing) {
                logger.log("Activity stopped but not finishing - preserving state")
            }
        } catch (e: Exception) {
            logger.log("Error in onStop: $e")
        }
    }

    @SuppressLint("RestrictedApi", "VisibleForTests")
    private fun showLeaveOnboardingDialog() {
        val dialog = Dialog(this)
        dialog.requestWindowFeature(Window.FEATURE_NO_TITLE)
        val binding: DialogBackBinding =
            DialogBackBinding.inflate(dialog.layoutInflater)
        dialog.setContentView(binding.root)

        var layoutParams = binding.dialogHeader.layoutParams as LinearLayout.LayoutParams
        var padding: Int = textSizeConverter.getPaddingOrMarginValue(16)
        layoutParams.setMargins(textSizeConverter.getPaddingOrMarginValue(16), textSizeConverter.getPaddingOrMarginValue(16), padding, 0)
        binding.dialogHeader.layoutParams = layoutParams

        val layoutParams2: ViewGroup.LayoutParams = binding.cardConfirmation.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(280)
        binding.cardConfirmation.layoutParams = layoutParams2

        binding.cardConfirmation.radius = textSizeConverter.calculateRadius(14).toFloat()
        binding.dialogHeader.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(18).toFloat(),
        )
        binding.dialogSubtitle.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(14).toFloat(),
        )
        binding.btnNo.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeConverter.getTextSize(16).toFloat())
        binding.btnYes.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeConverter.getTextSize(16).toFloat())

        layoutParams = binding.dialogSubtitle.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(padding, padding, padding, padding)
        binding.dialogSubtitle.layoutParams = layoutParams

        padding = textSizeConverter.getPaddingOrMarginValue(12)
        layoutParams = binding.btnNo.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(0, padding, 0, padding)
        binding.btnNo.layoutParams = layoutParams

        layoutParams = binding.btnYes.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            textSizeConverter.getPaddingOrMarginValue(24),
            padding,
            textSizeConverter.getPaddingOrMarginValue(16),
            padding,
        )
        binding.btnYes.layoutParams = layoutParams
        binding.dialogHeader.typeface = FontManager.getFont(context = applicationContext, FontWeight.MEDIUM)
        binding.dialogSubtitle.typeface = FontManager.getFont(context = applicationContext, FontWeight.REGULAR)
        binding.dialogHeader.text = getString(R.string.liveness_back_dialog_header)
        binding.dialogSubtitle.text = getString(R.string.liveness_back_dialog_title)
        binding.btnNo.text = getString(R.string.liveness_back_dialog_cotionue)
        binding.btnYes.text = getString(R.string.liveness_back_dialog_leave)

        binding.btnYes.setOnClickListener {
            dialog.dismiss()
            cameraProvider?.unbindAll()
            cameraProvider?.shutdownAsync()
            setResult(RESULT_CANCELED)
            finish()
        }
        binding.btnNo.setOnClickListener { dialog.dismiss() }

        dialog.show()
        dialog.setCancelable(false)
        dialog.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        dialog.window!!.attributes.windowAnimations = R.style.DialogAnimation
    }

    companion object {
        const val TAG = "LiveNessActivity"
        const val DEFAULT_THRESHOLD = 0.915F
        val permissions: Array<String> = arrayOf(Manifest.permission.CAMERA)
        const val PERMISSION_REQUEST_CODE = 1
    }
}
