package nashid.verify.sdk.utils.helpers

import android.app.Activity
import android.os.Build
import android.view.View
import android.view.WindowInsets
import android.view.WindowInsetsController
import android.view.WindowManager
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.luminance
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.updatePadding
import nashid.verify.sdk.utils.SdkConfig

/**
 * Helper class for handling edge-to-edge mode on Android 15+
 *
 * IMPORTANT: When calling from onCreate(), either:
 * 1. Use safeEnableEdgeToEdge() which handles window initialization timing
 * 2. Call enableEdgeToEdge() after setContentView()
 */
object EdgeToEdgeHelper {
    /**
     * Enables edge-to-edge mode for Android 15+ with proper system bar handling
     */
    @Suppress("DEPRECATION")
    private fun enableEdgeToEdge(activity: Activity) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.VANILLA_ICE_CREAM) {
            // Check if window is available
            val window = activity.window ?: return

            // Android 15+ - Enable edge-to-edge by default
            window.setDecorFitsSystemWindows(false)

            // Set transparent system bars
            window.statusBarColor = android.graphics.Color.TRANSPARENT
            window.navigationBarColor = android.graphics.Color.TRANSPARENT

            // Handle system bar appearance
            val statusBarColor = SdkConfig.sdkAppTheme.getBackgroundColorInt()
            if (statusBarColor != null) {
                val isLightBackground = Color(statusBarColor).luminance() > 0.5f
                setSystemBarAppearance(activity, isLightBackground)
            }
        }
    }

    /**
     * Sets up edge-to-edge mode with custom status bar color
     */
    @Suppress("DEPRECATION")
    fun setupEdgeToEdgeWithStatusBar(
        activity: Activity,
        statusBarColor: Int?,
    ) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            val window = activity.window ?: return

            window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)

            when {
                Build.VERSION.SDK_INT >= Build.VERSION_CODES.VANILLA_ICE_CREAM -> {
                    // Android 15+ - Always use edge-to-edge
                    window.setDecorFitsSystemWindows(false)

                    if (statusBarColor != null) {
                        // Use colored status bar with edge-to-edge
                        window.statusBarColor = statusBarColor
                        val isLightBackground = Color(statusBarColor).luminance() > 0.5f
                        setSystemBarAppearance(activity, isLightBackground)
                    } else {
                        // Use transparent status bar
                        window.statusBarColor = android.graphics.Color.TRANSPARENT
                        hideSystemBars(activity)
                    }
                }
                statusBarColor != null -> {
                    // Pre-Android 15 with colored status bar
                    window.statusBarColor = statusBarColor
                    val isLightBackground = Color(statusBarColor).luminance() > 0.5f
                    setSystemBarAppearance(activity, isLightBackground)
                }
                else -> {
                    // Pre-Android 15 with transparent status bar
                    makeStatusBarTransparent(activity)
                }
            }
        }
    }

    /**
     * Applies system insets to a view for edge-to-edge compatibility
     */
    fun applySystemInsets(
        view: View,
        applyTopInset: Boolean = true,
        applyBottomInset: Boolean = true,
        applyLeftInset: Boolean = false,
        applyRightInset: Boolean = false,
    ) {
        ViewCompat.setOnApplyWindowInsetsListener(view) { v, windowInsets ->
            val insets = windowInsets.getInsets(WindowInsetsCompat.Type.systemBars())

            val topInset = if (applyTopInset) insets.top else 0
            val bottomInset = if (applyBottomInset) insets.bottom else 0
            val leftInset = if (applyLeftInset) insets.left else 0
            val rightInset = if (applyRightInset) insets.right else 0

            v.updatePadding(
                top = topInset,
                bottom = bottomInset,
                left = leftInset,
                right = rightInset,
            )

            windowInsets
        }
    }

    /**
     * Applies system insets as margins instead of padding
     */
    fun applySystemInsetsAsMargins(
        view: View,
        applyTopInset: Boolean = true,
        applyBottomInset: Boolean = true,
        applyLeftInset: Boolean = false,
        applyRightInset: Boolean = false,
    ) {
        ViewCompat.setOnApplyWindowInsetsListener(view) { v, windowInsets ->
            val insets = windowInsets.getInsets(WindowInsetsCompat.Type.systemBars())

            val layoutParams = v.layoutParams
            if (layoutParams is android.view.ViewGroup.MarginLayoutParams) {
                val topMargin = if (applyTopInset) insets.top else layoutParams.topMargin
                val bottomMargin = if (applyBottomInset) insets.bottom else layoutParams.bottomMargin
                val leftMargin = if (applyLeftInset) insets.left else layoutParams.leftMargin
                val rightMargin = if (applyRightInset) insets.right else layoutParams.rightMargin

                layoutParams.setMargins(leftMargin, topMargin, rightMargin, bottomMargin)
                v.layoutParams = layoutParams
            }

            windowInsets
        }
    }

    /**
     * Sets system bar appearance (light or dark icons)
     */
    private fun setSystemBarAppearance(
        activity: Activity,
        isLightBackground: Boolean,
    ) {
        val window = activity.window ?: return

        // Check if decorView is ready (required for both API levels)
        val decorView = window.decorView

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            // API 30+: Use WindowInsetsController
            window.insetsController?.setSystemBarsAppearance(
                if (isLightBackground) WindowInsetsController.APPEARANCE_LIGHT_STATUS_BARS else 0,
                WindowInsetsController.APPEARANCE_LIGHT_STATUS_BARS,
            )
        } else {
            // API 23-29: Use deprecated flags
            @Suppress("DEPRECATION")
            decorView.systemUiVisibility =
                if (isLightBackground) {
                    decorView.systemUiVisibility or View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR
                } else {
                    decorView.systemUiVisibility and View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR.inv()
                }
        }
    }

    /**
     * Hides system bars for fullscreen experience
     */
    private fun hideSystemBars(activity: Activity) {
        val window = activity.window ?: return

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            window.insetsController?.apply {
                hide(WindowInsets.Type.statusBars())
                systemBarsBehavior = WindowInsetsController.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
            }
        } else {
            val decorView = window.decorView
            if (decorView != null) {
                @Suppress("DEPRECATION")
                decorView.systemUiVisibility = (
                    View.SYSTEM_UI_FLAG_LAYOUT_STABLE or
                        View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                )
            }
        }
    }

    /**
     * Makes status bar transparent for pre-Android 15
     */
    private fun makeStatusBarTransparent(activity: Activity) {
        val window =
            activity.window
                ?: return

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            window.setDecorFitsSystemWindows(false)
            hideSystemBars(activity)
        } else {
            val decorView = window.decorView
            if (decorView != null) {
                @Suppress("DEPRECATION")
                decorView.systemUiVisibility = (
                    View.SYSTEM_UI_FLAG_LAYOUT_STABLE or
                        View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                )
            }
        }
        window.statusBarColor = android.graphics.Color.TRANSPARENT
    }

    /**
     * Safely enables edge-to-edge mode - can be called at any point in activity lifecycle
     * This method will retry if the window is not ready yet
     */
    fun safeEnableEdgeToEdge(activity: Activity) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.VANILLA_ICE_CREAM) {
            val window = activity.window
            if (window == null) {
                // Window not ready yet, schedule for later
                activity.runOnUiThread {
                    // Try again on next UI thread cycle
                    if (activity.window != null) {
                        enableEdgeToEdge(activity)
                    }
                }
                return
            }
            enableEdgeToEdge(activity)
        }
    }
}
