package nashid.verify.sdk.viewmodel

import android.app.Application
import android.content.Intent
import android.content.res.Resources
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.core.content.ContextCompat
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import com.google.mlkit.vision.common.InputImage
import com.google.mlkit.vision.text.Text
import com.google.mlkit.vision.text.TextRecognizer
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import nashid.verify.sdk.Generated
import nashid.verify.sdk.data.repository.IdCardRepository
import nashid.verify.sdk.data.repository.KSAIdCardRepository
import nashid.verify.sdk.data.repository.PassportRepository
import nashid.verify.sdk.data.repository.QatarIdCardRepository
import nashid.verify.sdk.data.repository.UAEIdCardRepository
import nashid.verify.sdk.model.ScanDocumentResult
import nashid.verify.sdk.ui.CameraXLiveActivity
import nashid.verify.sdk.ui.NfcActivity
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdkNew.R
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale
import java.util.Objects

class CameraXLiveViewModel(
    val cameraXLiveData: CameraXLiveData,
    private val textRecognizer: TextRecognizer,
    private val app: Application,
    private val idCardRepository: IdCardRepository,
    private val uaeIdCardRepository: UAEIdCardRepository,
    private val ksaIdCardRepository: KSAIdCardRepository,
    private val qatarIdCardRepository: QatarIdCardRepository,
    private val passportRepository: PassportRepository,
    private val scanDocumentViewModel: ScanDocumentViewModel,
) : ViewModel() {
    private var cameraProviderLiveData: MutableLiveData<ProcessCameraProvider>? = null

    init {
        observeScanDocumentViewModel()
    }

    private fun observeScanDocumentViewModel() {
        scanDocumentViewModel.result.observeForever { result ->
            if (result is ScanDocumentResult.FileKeyResult) {
                Utility.getInstance().saveFileKeyAndContinue(result.fileKey, result.artifactType)
            }
        }
    }

    fun setSelectedDoc(
        selectedDoc: String,
        resources: Resources,
    ) {
        resetScanningState()

        cameraXLiveData.setSelectedDoc(selectedDoc)
        if (selectedDoc.equals(resources.getString(R.string.e_passport), ignoreCase = true)) {
            cameraXLiveData.setStatusText(resources.getString(R.string.scan_passport_top_text))
            cameraXLiveData.setBottomText(resources.getString(R.string.scan_passport_bottom))
        } else {
            cameraXLiveData.setStatusText(resources.getString(R.string.scan_id_top_text))
            cameraXLiveData.setBottomText(resources.getString(R.string.scan_id_bottom))
        }
    }

    private fun resetScanningState() {
        cameraXLiveData.apply {
            setIsSuccess(false)
            setIsFrontView(false)
            setIsFrontViewScanned(false)
            setIsOverlayVisible(true)
            setIsScanVisible(true)
            setIsScanFailVisible(false)
            setIsTransparentVisible(false)
            setIsScanCompleteVisible(false)
            setIsBackCardScanVisible(false)
            setGlareTextVisibility(false)
            setCloseImageProxy(false)
            setCTimer(false)
            setCloseAnalysisUseCase(false)
            setUnbindCameraProvider(false)
        }

        Utility.getInstance().apply {
            scannedImage = null
            scannedIdFrontView = null
            passportNumber = null
            dateOfBirth = null
            expiryDate = null
            setOcrScanFailed(false)
            setNfcScanFailed(false)
            setLiveNessScanFailed(false)
        }
    }

    fun processTextFromBitmap(bitmap: Bitmap?) {
        textRecognizer
            .process(InputImage.fromBitmap(bitmap!!, 0))
            .addOnSuccessListener { textResult: Text ->
                val blocks = textResult.textBlocks
                passportRepository.readPassportData(blocks)
                val text = textResult.text
                readTextFromString(text)
            }.addOnFailureListener {
                cameraXLiveData.setIsSuccess(false)
            }.addOnCompleteListener {
                if (cameraXLiveData.getIsSuccess().value == false) {
                    cameraXLiveData.setCloseImageProxy(true)
                }
            }
    }

    private fun readTextFromString(text: String) {
        try {
            val textAll = text.split("\n".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()
            if (Objects.requireNonNull(cameraXLiveData.getSelectedDoc().value).equals(
                    app.getString(R.string.e_passport),
                    ignoreCase = true,
                )
            ) {
                passportRepository.readAllDetailOfPassport()
            } else if (Objects.requireNonNull(cameraXLiveData.getSelectedDoc().value).equals(
                    app.getString(R.string.uae_id),
                    ignoreCase = true,
                )
            ) {
                if (cameraXLiveData.getIsFrontView().value == false) {
                    uaeIdCardRepository.readIdCardFrontView(text)
                } else {
                    uaeIdCardRepository.readIdCardBackView(textAll)
                }
            } else if (Objects.requireNonNull(cameraXLiveData.getSelectedDoc().value).equals(
                    app.getString(R.string.qatar_id),
                    ignoreCase = true,
                )
            ) {
                if (cameraXLiveData.getIsFrontView().value == false) {
                    qatarIdCardRepository.readQatarFrontViewData(text)
                }
            } else if (Objects.requireNonNull(cameraXLiveData.getSelectedDoc().value).equals(
                    app.getString(R.string.ksa_id),
                    ignoreCase = true,
                )
            ) {
                if (cameraXLiveData.getIsFrontView().value == false) {
                    ksaIdCardRepository.readKSAFrontViewData(text)
                } else {
                    ksaIdCardRepository.readKSABackViewData(text)
                }
            } else {
                if (cameraXLiveData.getIsFrontView().value == false) {
                    idCardRepository.readIdCardFrontView(text)
                } else {
                    idCardRepository.readIdCardBackView(textAll)
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
            cameraXLiveData.setIsSuccess(false)
        }
    }

    @Generated
    fun printCurrentTimeInLogs(): String {
        val sdf = SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.getDefault())
        val currentTime = sdf.format(Date())
        return currentTime
    }

    @Generated
    @Override
    fun scanningFailed(activity: CameraXLiveActivity) {
        val showInstruction = SdkConfig.showInstruction && (SdkConfig.isActiveLiveNessEnabled || SdkConfig.isPassiveLiveNessEnabled) && !Utility.getInstance().ocrScanningFailed()
        val targetActivity = if (showInstruction) nashid.verify.sdk.ui.LivenessInstructionActivity::class.java else nashid.verify.sdk.ui.SkipNfcLiveNessActivity::class.java
        val intent =
            Intent(
                app,
                targetActivity,
            )
        intent.putExtra(
            app.resources.getString(R.string.doc_key),
            cameraXLiveData.getSelectedDoc().value,
        )
        intent.setFlags(
            Intent.FLAG_ACTIVITY_NEW_TASK,
        )
        app.startActivity(intent)
        activity.finish()
    }

    fun handleAnimationCompleted(activity: CameraXLiveActivity) {
        cameraXLiveData.setCTimer(true)
        Utility.getInstance().selectedDoc = cameraXLiveData.getSelectedDoc().value
        val capturedImage =
            BitmapFactory.decodeByteArray(
                Utility.getInstance().scannedImage,
                0,
                Utility.getInstance().scannedImage?.size ?: 0,
            )
        if (Utility.getInstance().selectedDoc.equals(app.resources.getString(R.string.e_passport), ignoreCase = true)) {
            CoroutineScope(Dispatchers.IO).launch {
                callUploadArtifact(capturedImage, ArtifactType.FRONTSIDE_IMAGE)
            }
        } else {
            val bitmap =
                BitmapFactory.decodeByteArray(
                    Utility.getInstance().scannedIdFrontView,
                    0,
                    Utility.getInstance().scannedIdFrontView?.size ?: 0,
                )
            CoroutineScope(Dispatchers.IO).launch {
                callUploadArtifact(bitmap, ArtifactType.BACKSIDE_IMAGE)
            }
            if (SdkConfig.isNfcScanningEnabled) {
                val intent = Intent(app, NfcActivity::class.java)
                intent.putExtra(
                    app.resources.getString(R.string.doc_key),
                    cameraXLiveData.getSelectedDoc().value,
                )
                intent.setFlags(
                    Intent.FLAG_ACTIVITY_NEW_TASK,
                )
                app.startActivity(intent)
                activity.finish()
            } else {
                openLiveNessScreen(activity)
            }
        }
    }

    @Generated
    private fun openLiveNessScreen(activity: CameraXLiveActivity) {
        cameraXLiveData.setUnbindCameraProvider(true)
        Utility.getInstance().selectedDoc = cameraXLiveData.getSelectedDoc().value
        val showInstruction = SdkConfig.showInstruction && (SdkConfig.isActiveLiveNessEnabled || SdkConfig.isPassiveLiveNessEnabled) && !Utility.getInstance().ocrScanningFailed()
        val targetActivity = if (showInstruction) nashid.verify.sdk.ui.LivenessInstructionActivity::class.java else nashid.verify.sdk.ui.SkipNfcLiveNessActivity::class.java
        val intent = Intent(app, targetActivity)
        intent.putExtra(
            app.resources.getString(R.string.doc_key),
            cameraXLiveData.getSelectedDoc().value,
        )
        intent.setFlags(
            Intent.FLAG_ACTIVITY_NEW_TASK,
        )
        app.startActivity(intent)
        activity.finish()
    }

    @get:Generated
    val processCameraProvider: LiveData<ProcessCameraProvider>
        get() {
            if (cameraProviderLiveData == null) {
                cameraProviderLiveData = MutableLiveData()
                val cameraProviderFuture = ProcessCameraProvider.getInstance(app)
                cameraProviderFuture.addListener(
                    {
                        try {
                            cameraProviderLiveData!!.setValue(cameraProviderFuture.get())
                        } catch (e: InterruptedException) {
                            e.printStackTrace()
                        }
                    },
                    ContextCompat.getMainExecutor(app),
                )
            }
            return cameraProviderLiveData!!
        }

    internal interface OnImageProcessedListener {
        fun onImageProcessed(croppedBitmap: Bitmap?)
    }

    private fun uploadArtifact(
        bitmap: Bitmap?,
        artifactType: ArtifactType,
    ) {
        val file = Utility.getInstance().bitmapToFile(bitmap!!, app)
        val filePart = file?.let { Utility.getInstance().createMultipartBody(it) }
        if (filePart != null) {
            scanDocumentViewModel.uploadArtifact(
                SdkConfig.registerToken!!,
                Utility.getInstance().getAppKey(),
                filePart,
                artifactType.type,
            )
        }
    }

    internal fun callUploadArtifact(
        bitmap: Bitmap?,
        artifactType: ArtifactType,
    ) {
        bitmap?.let {
            uploadArtifact(it, artifactType)
        }
    }

    fun scannedIdFrontViewCompleted(byteArray: ByteArray) {
        val fullBitmap =
            BitmapFactory.decodeByteArray(
                byteArray,
                0,
                byteArray.size,
            )
        try {
            uploadArtifact(fullBitmap, ArtifactType.FRONTSIDE_IMAGE)
        } catch (e: Exception) {
        }
    }

    override fun onCleared() {
        super.onCleared()
        resetScanningState()
        cameraXLiveData.resetState()
    }
}
