package nashid.verify.sdk.data.repository

import android.content.Context
import android.widget.Toast
import nashid.verify.sdk.model.Artifacts
import nashid.verify.sdk.model.Liveness
import nashid.verify.sdk.model.Location
import nashid.verify.sdk.model.MainData
import nashid.verify.sdk.model.MetaDataModel
import nashid.verify.sdk.model.NFC
import nashid.verify.sdk.model.Scan
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.LivenessData
import nashid.verify.sdk.utils.OtherConstant
import nashid.verify.sdk.utils.ScanDocument
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG10File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG11File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG13File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG1File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG4File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG6File
import nashid.verify.sdk.viewmodel.SkipNfcLiveNessLiveData

class SkipLiveNessRepository(
    private val skipNfcLiveNessLiveData: SkipNfcLiveNessLiveData,
    private val context: Context,
) {
    fun callToComparison() {
        skipNfcLiveNessLiveData.setHandleLocationData(true)
    }

    fun createData(
        isNFCEnabled: Boolean,
        documentType: Int,
        isLiveNessEnabled: Boolean?,
    ): MainData {
        return MainData(
            scan = createScanData(documentType) ?: Scan(),
            nfc = createNfcData(documentType, isNFCEnabled) ?: NFC(),
            liveness = createLiveNess(isLiveNessEnabled ?: false),
        )
    }

    private fun createLiveNess(isLiveNessEnabled: Boolean): Liveness {
        if (isLiveNessEnabled) {
            return Liveness(
                activeLivenessConfirmed = false,
                passiveLivenessConfirmed = true,
                score = Utility.getInstance().getLivenessScore().toString(),
            )
        }
        return Liveness()
    }

    private fun createScanData(documentType: Int): Scan? {
        val mrzLines =
            when (documentType) {
                1 ->
                    listOfNotNull(
                        Utility.getInstance().mrzLine1,
                        Utility.getInstance().mrzLine2,
                        Utility.getInstance().mrzLine3,
                    )

                2 ->
                    listOfNotNull(
                        Utility.getInstance().mrzLine1,
                        Utility.getInstance().mrzLine2,
                    )

                else -> return null
            }
        return createScan(mrzLines)
    }

    private fun createScan(mrzLines: List<String>): Scan {
        val utility = Utility.getInstance()
        val scanDocument = ScanDocument.getInstance()

        return Scan(
            mrzText = mrzLines.joinToString("\n"),
            documentNo = utility.passportNumber?.replace("<", "") ?: "",
            documentType = utility.documentType ?: "",
            expiryDate = scanDocument.expiryDate?.let { utility.getFormattedDate(it) } ?: "",
            dateOfBirth = scanDocument.dateOfBirth?.let { utility.getFormattedDate(it) } ?: "",
            fullName = utility.name ?: "",
            nationality = utility.nationality ?: "",
            country = utility.country ?: "",
            gender = if (utility.gender.equals("f", ignoreCase = true)) "FEMALE" else "MALE",
        )
    }

    private fun createNfcData(
        documentType: Int,
        isNFCEnabled: Boolean,
    ): NFC? {
        return if (isNFCEnabled) {
            when (documentType) {
                1 -> createIdCardNfcData()
                2 -> createPassportNfcData()
                else -> null
            }
        } else {
            null
        }
    }

    private fun createPassportNfcData(): NFC? {
        return try {
            val eDocument = LivenessData.getInstance().geteDocument()
            val personDetails = eDocument.personDetails
            val additionalPersonDetails = eDocument.additionalPersonDetails
            val additionalDGFiles = eDocument.additionalDGFiles
            val placeOfBirth = additionalPersonDetails.placeOfBirth?.getOrNull(0) ?: ""

            NFC(
                fullNameEnglish = additionalPersonDetails.nameOfHolder!!.replace("null ", ""),
                nationalityEnglish = personDetails.personalNumber,
                genderEnglish = personDetails.gender,
                genderArabic = personDetails.gender,
                companyNameEnglish = additionalPersonDetails?.profession ?: "",
                permitType = additionalPersonDetails?.title ?: "",
                countryOfBirthArabic = personDetails.nationality,
                placeOfIssueArabic = personDetails.issuerAuthority,
                identityNumber = personDetails.serialNumber,
                placeOfIssueEnglish = personDetails.issuerAuthority,
                expiryDate = personDetails.expiryDate,
                visaNumber = additionalDGFiles?.personalizationDeviceSerialNumber ?: "",
                issueDate = additionalDGFiles?.dateOfIssue ?: "",
                fullNameArabic = additionalPersonDetails?.nameOfHolder ?: "",
                companyNameArabic = additionalPersonDetails?.profession ?: "",
                countryOfBirthEnglish = placeOfBirth,
                nationalityArabic = personDetails.nationality,
                companyAddressArabic = additionalPersonDetails?.permanentAddress?.get(0) ?: "",
                permitNumber = additionalDGFiles?.personalizationDeviceSerialNumber ?: "",
                dateOfBirth = personDetails.birthDate,
                useByDate = personDetails.expiryDate,
            )
        } catch (e: Exception) {
            e.printStackTrace()
            null
        }
    }

    private fun createIdCardNfcData(): NFC? {
        return try {
            val dg1Parser = Utility.getInstance().dgFile1?.let { OmanCardDG1File(it) }
            val dg4Parser = Utility.getInstance().dgFile4?.let { OmanCardDG4File(it) }
            val dg6Parser = Utility.getInstance().dgFile6?.let { OmanCardDG6File(it) }
            val dg10Parser = Utility.getInstance().dgFile10?.let { OmanCardDG10File(it) }
            val dg13Parser = Utility.getInstance().dgFile13?.let { OmanCardDG13File(it) }
            val dg11Parser = Utility.getInstance().dgFile11?.let { OmanCardDG11File(it) }

            val placeOfIssueArabic =
                dg1Parser?.placeOfIssueArabic?.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                } ?: ""

            val fullNameArabic =
                dg4Parser?.fullNameArabic?.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                } ?: ""

            val countryOfBirthArabic =
                dg6Parser?.countryOfBirthArabic?.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                } ?: ""

            val countryOfBirthEnglish =
                dg6Parser?.placeOfBirthEnglish?.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                } ?: ""

            val nationalityArabic =
                dg6Parser?.nationalityArabic?.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                } ?: ""

            val genderArabic =
                dg6Parser?.genderArabic?.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                } ?: ""

            NFC(
                fullNameEnglish = dg4Parser?.fullNameEnglish ?: "",
                nationalityEnglish = dg6Parser?.nationalityEnglish ?: "",
                genderEnglish = dg6Parser?.genderEnglish ?: "",
                genderArabic = genderArabic,
                companyNameEnglish = dg11Parser?.companyNameEnglish ?: "",
                countryOfBirthArabic = countryOfBirthArabic,
                placeOfIssueArabic = placeOfIssueArabic,
                identityNumber = dg1Parser?.idNumber ?: "",
                placeOfIssueEnglish = dg1Parser?.placeOfIssueEnglish ?: "",
                expiryDate = dg1Parser?.expiryDate ?.let { Utility.getInstance().getFormattedDate(it) } ?: "",
                issueDate = dg1Parser?.issueDate ?.let { Utility.getInstance().getFormattedDate(it) } ?: "",
                fullNameArabic = fullNameArabic,
                companyNameArabic = dg11Parser?.companyNameArabic ?: "",
                countryOfBirthEnglish = countryOfBirthEnglish,
                nationalityArabic = nationalityArabic,
                companyAddressArabic = dg11Parser?.companyAddressArabic ?: "",
                dateOfBirth = dg6Parser?.dateOfBirth?.let { Utility.getInstance().getFormattedDate(it) } ?: "",
                visaNumber = dg10Parser?.visaNumber ?: "",
                permitNumber = dg13Parser?.sponsorNumber ?: "",
                permitType = dg13Parser?.sponsorRelationship ?: "",
                useByDate = dg10Parser?.visaExpiryDate?.let { Utility.getInstance().getFormattedDate(it) } ?: "",
            )
        } catch (e: Exception) {
            e.printStackTrace()
            Toast.makeText(context, "Error creating NFC data: ${e.message}", Toast.LENGTH_LONG).show()
            null
        }
    }

    fun createMetaData(
        latitude: Double,
        longitude: Double,
        onMetaDataReady: (MetaDataModel) -> Unit,
    ) {
        Utility.getInstance().fetchPublicIPAddresses { ipv4, ipv6 ->
            val metadata =
                MetaDataModel(
                    timestamp = Utility.getInstance().getCurrentFormattedDateTime() ?: "",
                    location = Location(latitude, longitude),
                    deviceIp4 = ipv4 ?: "Unknown",
                    deviceIp6 = ipv6 ?: "Unknown",
                    systemName = Utility.getInstance().getSystemName(),
                    systemVersion = Utility.getInstance().getSystemVersion(),
                    deviceIdentifier = Utility.getInstance().getDeviceIdentifier(),
                    deviceType = Utility.getInstance().getDeviceType(),
                    gender = if (Utility.getInstance().gender.equals("f", ignoreCase = true)) "FEMALE" else "MALE",
                    extraData = SdkConfig.isExtraData,
                )
            onMetaDataReady(metadata)
        }
    }

    fun handleArtifacts(): Artifacts {
        return Artifacts(
            backSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.BACKSIDE_IMAGE) ?: "",
            frontSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.FRONTSIDE_IMAGE) ?: "",
            ocrFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.OCR_FACE_IMAGE) ?: "",
            nfcFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.NFC_IMAGE) ?: "",
            livenessImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.LIVENESS_IMAGE) ?: "",
            livenessImageWithoutBg = Utility.getInstance().getFileKeyForArtifact(ArtifactType.LIVENESS_IMAGE_WITHOUT_BG) ?: "",
        )
    }
}
