package nashid.verify.sdk.ui

import android.annotation.SuppressLint
import android.graphics.Bitmap
import android.os.Bundle
import android.util.TypedValue
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.TextView
import androidx.fragment.app.Fragment
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.bumptech.glide.Glide
import nashid.verify.sdk.model.AdditionalPersonDetails
import nashid.verify.sdk.ui.adapter.NFCDetailAdapter
import nashid.verify.sdk.utils.AdditionalDGFiles
import nashid.verify.sdk.utils.Constants
import nashid.verify.sdk.utils.EDocument
import nashid.verify.sdk.utils.LivenessData
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.TextSizeConverter
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG10File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG11File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG13File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG1File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG4File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG6File
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.FragmentNfcBinding
import org.json.JSONObject
import java.io.IOException
import java.text.DateFormat
import java.text.SimpleDateFormat
import java.util.Locale

class NFCFragment : Fragment() {
    private var binding: FragmentNfcBinding? = null
    private val personalHashmap = LinkedHashMap<String, String>()
    private val documentHashmap = LinkedHashMap<String, String>()
    private val visaHashmap = LinkedHashMap<String, String>()
    private val sponsorHashmap = LinkedHashMap<String, String>()
    private val companyHashmap = LinkedHashMap<String, String>()
    private val emptyString = ""
    private val textSize = 20

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?,
    ): View {
        binding = FragmentNfcBinding.inflate(inflater, container, false)
        initView()
        return binding!!.root
    }

    private fun initView() {
        val textSizeConverter = TextSizeConverter(requireContext())
        setupLayoutParameters(textSizeConverter)
        setupRecyclerViews()

        try {
            if (Utility.getInstance().selectedDoc == getString(R.string.e_passport)) {
                setPassportDetail()
                binding!!.imgSignature.visibility = View.VISIBLE
            } else {
                setIDCardDetail()
                binding!!.imgSignature.visibility = View.GONE
            }
        } catch (_: Exception) {
        }
    }

    private fun setupLayoutParameters(textSizeConverter: TextSizeConverter) {
        binding!!.mainNfcFragmnent.setPaddingAllSides(textSizeConverter.getPaddingOrMarginValue(16))
        binding!!.mainPhotoLayout.setMargins(0, textSizeConverter.getPaddingOrMarginValue(28), 0, 0)

        binding!!.apply {
            profileCard.resize(textSizeConverter.getWidth(106), textSizeConverter.getHeight(106))
            imgSignature.resize(textSizeConverter.getWidth(106), textSizeConverter.getHeight(52))
            setTextSize(txtUserName, textSizeConverter.getTextSize(14).toFloat())
            setTextSize(txtDocNumber, textSizeConverter.getTextSize(12).toFloat())
            setTextSize(txtPersonalInfoHeader, textSizeConverter.getTextSize(textSize).toFloat())
            setTextSize(txtPersonalImfo, textSizeConverter.getTextSize(textSize).toFloat())
            setTextSize(txtDocument, textSizeConverter.getTextSize(textSize).toFloat())
            setTextSize(txtVisa, textSizeConverter.getTextSize(textSize).toFloat())
            setTextSize(txtSponser, textSizeConverter.getTextSize(textSize).toFloat())
            setTextSize(txtCompany, textSizeConverter.getTextSize(textSize).toFloat())
        }
    }

    private fun setupRecyclerViews() {
        binding!!.apply {
            recyclerviewPersonalInfo.setLayoutManagerAndAdapter(personalHashmap)
            rcyclerviewDocument.setLayoutManagerAndAdapter(documentHashmap)
            recyclerviewVisa.setLayoutManagerAndAdapter(visaHashmap)
            recyclerviewSponser.setLayoutManagerAndAdapter(sponsorHashmap)
            recyclerviewCompany.setLayoutManagerAndAdapter(companyHashmap)
        }
    }

    @SuppressLint("SetTextI18n")
    private fun setPassportDetail() {
        val eDocument = LivenessData.getInstance().geteDocument()
        val personDetails = eDocument.personDetails
        val additionalPersonDetails = eDocument.additionalPersonDetails
        val additionalDGFiles = eDocument.additionalDGFiles

        personalHashmap.apply {
            put(getString(R.string.passdocumentType), eDocument.docType.name)
            put(getString(R.string.passdocumentNumber), personDetails.serialNumber)
            put(getString(R.string.passissuingAuthority), personDetails.issuerAuthority)
            put(getString(R.string.passdateOfBirth), personDetails.birthDate)
            put(getString(R.string.passgender), personDetails.gender)
            put(getString(R.string.passdocumentExpiryDate), personDetails.expiryDate)
        }

        setupPersonImages(personDetails.faceImage, personDetails.signature)
        setAdditionalPersonDetails(additionalPersonDetails)
        setAdditionalDocumentFiles(additionalDGFiles, eDocument)
        binding!!.txtDocNumber.text = "${personDetails.nationality} | ${personDetails.serialNumber}"
    }

    private fun setupPersonImages(
        faceImage: Bitmap?,
        signatureImage: Bitmap?,
    ) {
        binding!!.apply {
            imgProfile.setImageBitmap(faceImage)
            Glide.with(requireActivity()).load(signatureImage).into(imgSignature)
        }
    }

    private fun setAdditionalPersonDetails(details: AdditionalPersonDetails?) {
        details?.let {
            binding!!.txtUserName.text = it.nameOfHolder
            personalHashmap.apply {
                put(getString(R.string.passfullName), it.nameOfHolder.cleanNull())
                put(getString(R.string.passplaceOfBirth), it.placeOfBirth.getOrEmpty())
                put(getString(R.string.passaddress), it.permanentAddress.getOrEmpty())
                it.profession?.let { it1 -> put(getString(R.string.passprofession), it1) }
                it.title?.let { it1 -> put(getString(R.string.passtitle), it1) }
                it.personalSummary?.let { it1 -> put(getString(R.string.passpersonalSummary), it1) }
                put(getString(R.string.passtdNumbers), it.otherValidTDNumbers.getOrEmpty())
                it.custodyInformation?.let { it1 -> put(getString(R.string.passcustodyInfo), it1) }
                put(getString(R.string.passproofOfCitizenship), it.proofOfCitizenship.toString())
                it.telephone?.let { it1 -> put(getString(R.string.passtelephone), it1) }
            }
        }
    }

    private fun setAdditionalDocumentFiles(
        files: AdditionalDGFiles,
        eDocument: EDocument,
    ) {
        personalHashmap.apply {
            put(getString(R.string.passnfcissuingCountry), files.issueingAuthority)
            put(getString(R.string.passfrontImage), files.frontImage.toString())
            put(getString(R.string.passrearImage), files.backImage.toString())
            put(getString(R.string.passPersonalizationNumber), files.personalizationDeviceSerialNumber)
            put(getString(R.string.passPersonalizationTime), files.personalizationTime)
            put(getString(R.string.passsecurityInfos), files.securityInfos)
            put(getString(R.string.passrsaPublicKey), eDocument.rsaPublicKey)
            put(getString(R.string.passecdsaPublicKey), eDocument.ecdsaPublicKey)
            put(getString(R.string.passive_authentication), eDocument.personDetails.isDigitalSigVerification.toString())
        }
    }

    @Throws(IOException::class)
    private fun setIDCardDetail() {
        binding?.imgProfile?.setImageBitmap(LivenessData.getInstance().nfcImage)
        try {
            setDocumentDetails()
            setPersonalDetails()
            setVisaDetails()
            setCompanyDetails()
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    @SuppressLint("SetTextI18n")
    private fun setDocumentDetails() {
        val dg1Parser = OmanCardDG1File(Utility.getInstance().dgFile1)
        binding?.txtUserName?.text = nfcData?.getString(getString(R.string.idcardfullNameEnglish))
        binding?.txtDocNumber?.text = "${OmanCardDG6File(Utility.getInstance().dgFile6).nationalityEnglish} | ${dg1Parser.idNumber}"

        documentHashmap.apply {
            put(getString(R.string.idcardidNumber), dg1Parser.idNumber)
            put(getString(R.string.idcardissueDate), formatDate(dg1Parser.issueDate))
            put(getString(R.string.idcardexpiryDate), formatDate(dg1Parser.expiryDate))
            put(getString(R.string.idcardplaceOfIssueArabic), dg1Parser.placeOfIssueArabic)
            put(getString(R.string.idcardplaceOfIssueEnglish), dg1Parser.placeOfIssueEnglish)
        }
    }

    private fun setPersonalDetails() {
        val dg4Parser = OmanCardDG4File(Utility.getInstance().dgFile4)
        val dg6Parser = OmanCardDG6File(Utility.getInstance().dgFile6)

        dg4Parser.let {
            personalHashmap[getString(R.string.idcardfullNameArabic)] = it.fullNameArabic
            personalHashmap[getString(R.string.idcardfullNameEnglish)] = it.fullNameEnglish
        }

        dg6Parser.let {
            personalHashmap.apply {
                put(getString(R.string.idcarddateOfBirth), formatDate(it.dateOfBirth))
                put(getString(R.string.idcardcountryOfBirthArabic), it.countryOfBirthArabic)
                put(getString(R.string.idcardplaceOfBirthEnglish), it.placeOfBirthEnglish)
                put(getString(R.string.idcardnationalityArabic), it.nationalityArabic)
                put(getString(R.string.idcardgenderArabic), it.genderArabic)
                put(getString(R.string.idcardnationalityEnglish), it.nationalityEnglish)
                put(getString(R.string.idcardgenderEnglish), it.genderEnglish)
            }
        }
    }

    private fun setVisaDetails() {
        val dg13Parser = Utility.getInstance().dgFile13?.let { OmanCardDG13File(it) }
        val dg10Parser = Utility.getInstance().dgFile10?.let { OmanCardDG10File(it) }

        Utility.getInstance().setVisibility(binding?.txtSponser, binding?.recyclerviewSponser, dg13Parser != null)
        dg13Parser?.let {
            visaHashmap[getString(R.string.idcardvisaNumber)] = it.sponsorNumber
            sponsorHashmap[getString(R.string.idcardPermitType)] = it.sponsorRelationship
            visaHashmap[getString(R.string.idcardsponsorIssueDate)] = formatDate(it.sponsorIssueDate)
            visaHashmap[getString(R.string.idcardsponsorExpiryDate)] = formatDate(it.sponsorExpiryDate)
        }

        Utility.getInstance().setVisibility(binding?.txtVisa, binding?.recyclerviewVisa, dg10Parser != null)
        dg10Parser?.let {
            visaHashmap[getString(R.string.idcardvisaExpiryDate)] = formatDate(it.visaExpiryDate)
            visaHashmap[getString(R.string.idcardvisaPlaceOfIssueArabic)] = it.visaPlaceOfIssueArabic
            sponsorHashmap[getString(R.string.idcardPermitNumber)] = it.visaNumber
        }
    }

    private fun setCompanyDetails() {
        val dg11Parser = Utility.getInstance().dgFile11?.let { OmanCardDG11File(it) }
        Utility.getInstance().setVisibility(binding?.txtCompany, binding?.recyclerviewCompany, dg11Parser != null)
        dg11Parser?.let {
            companyHashmap.apply {
                put(getString(R.string.idcardcompanyNameArabic), it.companyNameArabic)
                put(getString(R.string.idcardcompanyNameEnglish), it.companyNameEnglish)
                put(getString(R.string.idcardcompanyAddressArabic), it.companyAddressArabic)
            }
        }
    }

    private fun formatDate(inputDate: String): String {
        try {
            val sdf: DateFormat = SimpleDateFormat("yyyyMMdd", Locale.ENGLISH)
            val date = sdf.parse(inputDate)
            return SimpleDateFormat(Constants.DATE_FORMAT, Locale.ENGLISH).format(date!!)
        } catch (_: Exception) {
        }
        return inputDate
    }

    private fun setTextSize(
        view: TextView,
        size: Float,
    ) {
        view.setTextSize(TypedValue.COMPLEX_UNIT_PX, size)
    }

    private fun View.setPaddingAllSides(value: Int) {
        setPadding(value, value, value, value)
    }

    private fun View.setMargins(
        left: Int,
        top: Int,
        right: Int,
        bottom: Int,
    ) {
        (layoutParams as? ViewGroup.MarginLayoutParams)?.setMargins(left, top, right, bottom)
    }

    private fun View.resize(
        width: Int,
        height: Int,
    ) {
        layoutParams.width = width
        layoutParams.height = height
    }

    private fun RecyclerView.setLayoutManagerAndAdapter(dataMap: HashMap<String, String>) {
        layoutManager = LinearLayoutManager(context, LinearLayoutManager.VERTICAL, false)
        adapter = NFCDetailAdapter(requireContext(), dataMap)
    }

    private fun String?.cleanNull(): String = this?.replace("null ", "") ?: emptyString

    private fun List<String>?.getOrEmpty(): String = this?.firstOrNull() ?: emptyString

    companion object {
        var nfcData: JSONObject? = null
    }
}
