package nashid.verify.sdk.utils

import android.app.Activity
import android.content.Context
import android.graphics.Bitmap
import android.os.Build
import android.view.View
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.launch
import nashid.verify.sdk.utils.Constants.APP_KEY_PRODUCTION
import nashid.verify.sdk.utils.Constants.APP_KEY_STAGING
import nashid.verify.sdkNew.BuildConfig
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.MultipartBody
import okhttp3.RequestBody.Companion.asRequestBody
import org.json.JSONObject
import java.io.File
import java.net.HttpURLConnection
import java.net.URL
import java.text.SimpleDateFormat
import java.util.Calendar
import java.util.Date
import java.util.Locale
import java.util.TimeZone

internal class Utility private constructor() {
    var name: String? = null
    var passportNumber: String? = null
    var dateOfBirth: String? = null
    var expiryDate: String? = null
    var fullName: String? = null
    var gender: String = ""
    var nationality: String? = null
    var country: String? = null
    var documentType: String? = null
    var scannedImage: ByteArray? = null
    var scannedIdFrontView: ByteArray? = null

    var mrzLine1: String? = null
    var mrzLine2: String? = null
    var mrzLine3: String? = null

    var dgFile1: ByteArray? = null
    var dgFile2: ByteArray? = null
    var dgFile4: ByteArray? = null
    var dgFile6: ByteArray? = null
    var dgFile10: ByteArray? = null
    var dgFile11: ByteArray? = null
    var dgFile13: ByteArray? = null
    var livenessScannedImage: ByteArray? = null
    var nfcScannedImage: ByteArray? = null
    var matchPercentage: Int = 0
    var liveNessScore: Int = 0
    var dateOfExpiration: String? = null
    var selectedDoc: String? = null
    var liveImage: ByteArray? = null

    private val artifactFileKeys:
        MutableMap<ArtifactType, String> = mutableMapOf()
    private val artifacts =
        listOf(
            ArtifactType.FRONTSIDE_IMAGE,
            ArtifactType.BACKSIDE_IMAGE,
            ArtifactType.OCR_FACE_IMAGE,
            ArtifactType.NFC_IMAGE,
            ArtifactType.LIVENESS_IMAGE,
            ArtifactType.LIVENESS_IMAGE_WITHOUT_BG,
        )

    fun cleanup() {
        name = null
        passportNumber = null
        dateOfBirth = null
        expiryDate = null
        fullName = null
        gender = ""
        nationality = null
        country = null
        scannedImage = null
        scannedIdFrontView = null
        documentType = null
        dgFile1 = null
        dgFile2 = null
        dgFile4 = null
        dgFile6 = null
        dgFile10 = null
        dgFile11 = null
        dgFile13 = null
        livenessScannedImage = null
        nfcScannedImage = null
        matchPercentage = 0
        liveNessScore = 0
        dateOfExpiration = null
        selectedDoc = null
        liveImage = null
    }

    fun setVisibility(
        view1: View?,
        view2: View?,
        isVisible: Boolean,
    ) {
        val visibility = if (isVisible) View.VISIBLE else View.GONE
        view1?.visibility = visibility
        view2?.visibility = visibility
    }

    fun bitmapToFile(
        bitmap: Bitmap,
        context: Context,
    ): File {
        val file = File(context.cacheDir, "upload_image_${System.currentTimeMillis()}.png")
        file.outputStream().use { outputStream ->
            bitmap.compress(Bitmap.CompressFormat.PNG, 100, outputStream)
            outputStream.flush()
        }
        return file
    }

    fun createMultipartBody(file: File): MultipartBody.Part {
        val requestFile = file.asRequestBody("multipart/form-data".toMediaTypeOrNull())
        return MultipartBody.Part.createFormData("file", file.name, requestFile)
    }

    fun getAppKey(): String = if (BuildConfig.BUILD_TYPE.equals("release", ignoreCase = true)) APP_KEY_PRODUCTION else APP_KEY_STAGING

    fun getCurrentFormattedDateTime(): String? {
        val calendar = Calendar.getInstance()
        val date = calendar.time
        val sdf = SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.ENGLISH)
        sdf.timeZone = TimeZone.getTimeZone("GMT")
        return sdf.format(date)
    }

    fun getDeviceIdentifier(): String = "${Build.BRAND}_${Build.MODEL}"

    fun getDeviceType(): String = Build.MODEL

    fun getSystemVersion(): String = Build.VERSION.RELEASE

    fun getSystemName(): String = "Android"

    fun fetchPublicIPAddresses(onResult: (String?, String?) -> Unit) {
        CoroutineScope(Dispatchers.Main).launch {
            val ipv4Service = "https://api64.ipify.org?format=json"
            val ipv6Service = "https://ipv6.icanhazip.com/"
            val ipv4Deferred = async(Dispatchers.IO) { fetchPublicIP(ipv4Service) }
            val ipv6Deferred = async(Dispatchers.IO) { fetchPublicIP(ipv6Service) }
            val ipv4 = ipv4Deferred.await()
            val ipv6 = ipv6Deferred.await()
            onResult(ipv4, ipv6)
        }
    }

    private fun fetchPublicIP(serviceUrl: String): String? =
        try {
            val url = URL(serviceUrl)
            val connection = url.openConnection() as HttpURLConnection
            connection.requestMethod = "GET"
            connection.setRequestProperty("User-Agent", "Mozilla/5.0")
            connection.inputStream.bufferedReader().use {
                val response = it.readLine()
                val json = JSONObject(response)
                json.getString("ip")
            }
        } catch (e: Exception) {
            e.printStackTrace()
            null
        }

    fun getFormattedDate(dateStr: String): String? {
        val inputFormat = SimpleDateFormat("yyMMdd", Locale.getDefault())
        val date: Date? = inputFormat.parse(dateStr)
        val outputFormat = SimpleDateFormat("yyyy-MM-dd", Locale.getDefault())
        return date?.let { outputFormat.format(it) }
    }

    fun getNfcFormattedDate(dateStr: String): String? {
        val inputFormat = SimpleDateFormat("yyyyMMdd", Locale.getDefault())
        val date: Date? = inputFormat.parse(dateStr)
        val outputFormat = SimpleDateFormat("yyyy-MM-dd", Locale.getDefault())
        return date?.let { outputFormat.format(it) }
    }

    fun saveFileKeyAndContinue(
        fileKey: String,
        artifactType: String,
    ) {
        val mappedArtifactType = artifacts.find { it.type == artifactType }
        if (mappedArtifactType != null) {
            artifactFileKeys[mappedArtifactType] = fileKey
            println("artifact saved.  type: $artifactType  :: file key ::$fileKey")
        } else {
            println("Invalid artifact type: $artifactType")
        }
    }

    fun getFileKeyForArtifact(artifactType: ArtifactType): String? = artifactFileKeys[artifactType]

    fun getLivenessScore(): Int = liveNessScore

    fun restartApp(activity: Activity) {
        activity.finish()
    }

    companion object {
        @Volatile
        private var ourInstance: Utility? = null

        fun getInstance(): Utility =
            ourInstance ?: synchronized(this) {
                ourInstance ?: Utility().also { ourInstance = it }
            }
    }
}
