package nashid.verify.sdk

import android.app.Activity
import android.app.Dialog
import android.content.Context
import android.content.Intent
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.util.TypedValue
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.view.Window
import android.widget.LinearLayout
import androidx.lifecycle.Observer
import nashid.verify.sdk.model.AuthCallback
import nashid.verify.sdk.model.AuthResponse
import nashid.verify.sdk.model.DocumentType
import nashid.verify.sdk.model.DocumentVerification
import nashid.verify.sdk.model.ExtraData
import nashid.verify.sdk.model.SDKStatus
import nashid.verify.sdk.model.ScanDocumentResult
import nashid.verify.sdk.model.VerifySDKCallback
import nashid.verify.sdk.model.VerifySdkEnvironment
import nashid.verify.sdk.request.AuthRequest
import nashid.verify.sdk.ui.MainScreen
import nashid.verify.sdk.ui.dialog.ActivityTracker
import nashid.verify.sdk.ui.theme.AppTheme
import nashid.verify.sdk.ui.theme.FontManager
import nashid.verify.sdk.ui.theme.FontWeight
import nashid.verify.sdk.utils.Loggers
import nashid.verify.sdk.utils.NetWorkResult
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.LocaleUtil.Companion.applyLocalizedContext
import nashid.verify.sdk.utils.helpers.Storage
import nashid.verify.sdk.utils.helpers.TextSizeConverter
import nashid.verify.sdk.viewmodel.AuthViewModel
import nashid.verify.sdk.viewmodel.ScanDocumentViewModel
import nashid.verify.sdk.viewmodel.SkipNfcLiveNessViewModel
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.DialogFailureNfcBinding
import org.koin.core.component.KoinComponent
import org.koin.core.component.inject

internal class VerifySDKManagerImpl :
    KoinComponent,
    VerifySDKManager {
    private var sdkCallback: VerifySDKCallback? = null
    private val viewModel: AuthViewModel by inject()
    private val scanDocumentViewModel: ScanDocumentViewModel by inject()
    private val skipNfcLiveNessViewModel: SkipNfcLiveNessViewModel by inject()
    private val context: Context by inject()
    private val logger = Loggers.withTag("VerifySDKManager")
    private var authResultObserver: Observer<NetWorkResult<AuthResponse>?>? = null
    private var scanDocumentObserver: Observer<ScanDocumentResult?>? = null

    override fun setEnvironment(env: VerifySdkEnvironment) {
        SdkConfig.selectedEnvironment = env
        SdkConfig.updateEnvironmentConfig(env)
        logger.log("Environment set to $env")
    }

    override fun setCallback(callback: VerifySDKCallback?) {
        this.sdkCallback = callback
        logger.log("SDK callback listener has been ${if (callback != null) "set" else "cleared"}")
    }

    /**
     * Get the current callback listener.
     */
    override fun getCallback(): VerifySDKCallback? {
        logger.log("getCallback() called, callback is: ${if (sdkCallback != null) "set" else "null"}")
        return sdkCallback
    }

    override fun initialize(
        sdkKey: String,
        sdkSecret: String,
        languageType: String,
        extraData: ExtraData?,
        callback: AuthCallback,
    ) {
        resetInstance()
        logger.log("Starting initialization with SDK Keys")
        authResultObserver =
            Observer { result ->
                when (result) {
                    is NetWorkResult.Success -> {
                        logger.log("SDK initialization successful: ${result.data?.toString()}")
                        callback.initializeResponse(true, result.data?.message)
                        result.data?.data?.sdkConfig?.let { sdkConfig ->
                            var appTheme =
                                AppTheme(
                                    primaryColor = sdkConfig.colorTheme.primaryColorCode,
                                    secondaryColor = sdkConfig.colorTheme.secondaryColorCode,
                                    backgroundColor = sdkConfig.colorTheme.backgroundColorCode,
                                    headerColor = "#000000",
                                )

                            SdkConfig.updateConfig(
                                sdkKey = sdkConfig.sandboxSdkKey,
                                sdkSecret = sdkConfig.sandboxSdkSecret,
                                languageType = languageType,
                                sdkName = sdkConfig.name,
                                sdkDescription = sdkConfig.description,
                                nfcScanningEnabled = sdkConfig.isNfcScanningEnabled,
                                faceMatchingEnabled = sdkConfig.isFaceMatchingEnabled,
                                activeLiveNessEnabled = sdkConfig.isActiveLivenessEnabled,
                                passiveLiveNessEnabled = sdkConfig.isPassiveLivenessEnabled,
                                ocrEnabled = sdkConfig.isOcrEnabled,
                                mrzEnabled = sdkConfig.isMrzEnabled,
                                protocol = sdkConfig.callbackProtocol,
                                host = sdkConfig.callbackHost,
                                instructionsVisible = sdkConfig.showInstructions,
                                nfcSkippable = sdkConfig.isNfcSkippable,
                                nfcRetries = sdkConfig.nfcMaximumRetries ?: 0,
                                nfcCountCancelAsRetry = sdkConfig.nfcCountCancelAsRetry,
                                extraData = extraData,
                                appTheme = appTheme,
                                fontName = sdkConfig.fontFamily.name,
                            )
                        }
                    }

                    is NetWorkResult.Error -> {
                        logger.log("Initialization failed: ${result.message}")
                        SdkConfig.sdkStatus = SDKStatus.FAILED
                        callback.initializeResponse(false, result.message)
                    }

                    else -> {}
                }
            }

        authResultObserver?.let { viewModel.authResult.observeForever(it) }

        // Trigger authentication
        viewModel.authenticateUser(AuthRequest(sdkKey, sdkSecret))

        // Configure the SDK appearance
        Storage(context).setPreferredLocale(languageType)
        applyLocalizedContext(context, languageType)
    }

    override fun verify(
        documentType: DocumentType,
        callback: (ScanDocumentResult.ScanDocumentResponse) -> Unit,
    ) {
        if (documentType == DocumentType.OTHER) {
            logger.log("Invalid document type: ${documentType.displayName}")
            callback(ScanDocumentResult.ScanDocumentResponse(false, context.getString(R.string.invalid_document_type), ""))
            return
        }

        if (SdkConfig.sdkStatus == SDKStatus.INITIALIZED) {
            logger.log("scanDocument called with DocumentType: ${documentType.displayName}")
            SdkConfig.viewType = documentType
            scanDocumentObserver =
                Observer {
                    if (it is ScanDocumentResult.ScanDocumentResponse) {
                        if (!it.result) {
                            skipNfcLiveNessViewModel.getLiveNessData().retryUploadArtifacts()
                        } else {
                            callback(it)
                            skipNfcLiveNessViewModel.getLiveNessData().requestFinishActivity()
                        }
                    }
                }
            scanDocumentObserver?.let { scanDocumentViewModel.result.observeForever(it) }
            val isAllFeaturesDisabled =
                !SdkConfig.isOcrEnabled &&
                    !SdkConfig.isNfcScanningEnabled &&
                    !(SdkConfig.isActiveLiveNessEnabled || SdkConfig.isPassiveLiveNessEnabled)
            if (isAllFeaturesDisabled) {
                ActivityTracker.getActivity()?.let { showFailureDialog(it) }
            } else {
                val intent =
                    Intent(context, MainScreen::class.java).apply {
                        flags = Intent.FLAG_ACTIVITY_NEW_TASK
                    }
                context.startActivity(intent)
            }
        } else {
            logger.log("Failed to initialize SDK")
            callback(ScanDocumentResult.ScanDocumentResponse(false, "Failed to initialize SDK", ""))
        }
    }

    @Generated
    private fun showFailureDialog(activity: Activity) {
        val textSizeConverter: TextSizeConverter by inject()
        var errorDialog = Dialog(activity)
        errorDialog.requestWindowFeature(Window.FEATURE_NO_TITLE)
        val binding =
            DialogFailureNfcBinding.inflate(
                errorDialog.layoutInflater,
            )
        errorDialog.setContentView(binding.root)

        binding.txtScanCompleted.visibility = View.GONE
        binding.imgNfcFailed.visibility = View.GONE
        binding.imgInfo1.visibility = View.GONE
        binding.lyoutError2.visibility = View.GONE

        binding.txtScanNotCompleteDesc1.setText(context.getString(R.string.no_access_feature))
        binding.txtScanNotCompleteDesc1.gravity = Gravity.CENTER
        binding.txtBtnRetry.setText(context.getString(R.string.btn_no_access_feature))

        var padding = textSizeConverter.getPaddingOrMarginValue(16)
        binding.mainDialogLyt.setPadding(padding, padding, padding, padding)
        var layoutParams2 = binding.imgScanFailure.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(56)
        layoutParams2.height = textSizeConverter.getHeight(56)
        binding.imgScanFailure.layoutParams = layoutParams2
        var layoutParams = binding.imgScanFailure.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(8), 0, 0)
        binding.imgScanFailure.layoutParams = layoutParams
        binding.txtScanNotCompleteDesc1.setTextColor(SdkConfig.sdkAppTheme.getHeaderColorInt())
        binding.txtScanNotCompleteDesc1.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(16).toFloat(),
        )
        layoutParams = binding.txtScanNotCompleteDesc1.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            textSizeConverter.getPaddingOrMarginValue(6),
            textSizeConverter.getPaddingOrMarginValue(16),
            0,
            0,
        )
        binding.txtScanNotCompleteDesc1.layoutParams = layoutParams
        layoutParams = binding.btnRetry.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            0,
            textSizeConverter.getPaddingOrMarginValue(16),
            0,
            0,
        )
        binding.btnRetry.layoutParams = layoutParams

        binding.btnRetry.setCardBackgroundColor(SdkConfig.sdkAppTheme.getPrimaryColorInt())

        binding.txtBtnRetry.typeface = FontManager.getFont(context = context, FontWeight.MEDIUM)

        binding.txtBtnRetry.setTextColor(SdkConfig.sdkAppTheme.getBackgroundColorInt())
        padding = textSizeConverter.getPaddingOrMarginValue(12)
        binding.txtBtnRetry.setPadding(0, padding, 0, padding)

        binding.btnRetry.radius = textSizeConverter.calculateRadius(8).toFloat()

        binding.txtBtnRetry.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(16).toFloat(),
        )
        binding.btnRetry.setOnClickListener {
            errorDialog.dismiss()
        }
        errorDialog.show()
        errorDialog.window!!.setLayout(
            ViewGroup.LayoutParams.MATCH_PARENT,
            ViewGroup.LayoutParams.WRAP_CONTENT,
        )
        errorDialog.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        errorDialog.window!!.attributes.windowAnimations = R.style.DialogAnimation
        errorDialog.window!!.setGravity(Gravity.BOTTOM)
    }

    override fun getVerificationResult(
        verificationId: String,
        callback: (ScanDocumentResult.GetScanResultResponse) -> Unit,
    ) {
        scanDocumentObserver?.let { observer ->
            scanDocumentViewModel.result.removeObserver(observer)
        }
        logger.log("getScanResult called with verificationId: $verificationId")
        if (SdkConfig.sdkStatus == SDKStatus.INITIALIZED) {
            scanDocumentViewModel.getVerification(
                SdkConfig.registerToken!!,
                Utility.getInstance().getAppKey(),
                verificationId,
            )
            scanDocumentObserver =
                Observer { result ->
                    if (result is ScanDocumentResult.GetScanResultResponse) {
                        scanDocumentObserver?.let { observer ->
                            scanDocumentViewModel.result.removeObserver(observer)
                            scanDocumentObserver = null
                        }
                        callback(result)
                    }
                }
            scanDocumentObserver?.let { scanDocumentViewModel.result.observeForever(it) }
        } else {
            logger.log("SDK not initialized")
        }
    }

    companion object {
        @Volatile
        private var instance: VerifySDKManagerImpl? = null

        @JvmStatic
        fun getInstance(): VerifySDKManagerImpl =
            instance ?: synchronized(this) {
                instance ?: VerifySDKManagerImpl().also { instance = it }
            }
    }

    override fun submitVerification(verification: DocumentVerification) {
        scanDocumentViewModel.submitVerificationWithRetry(
            SdkConfig.registerToken!!,
            Utility.getInstance().getAppKey(),
            verification,
        )
    }

    override fun failedArtifactsVerification(verification: DocumentVerification) {
        scanDocumentViewModel.missingArtifactsVerification(
            SdkConfig.registerToken!!,
            Utility.getInstance().getAppKey(),
            verification,
        )
    }

    private fun resetInstance() {
        instance = null
        SdkConfig.clear()
        authResultObserver?.let { viewModel.authResult.removeObserver(it) }
        scanDocumentObserver?.let { scanDocumentViewModel.result.removeObserver(it) }
        authResultObserver = null
        scanDocumentObserver = null
        viewModel.clearAuthData()
        scanDocumentViewModel.clearScanDocumentData()
        scanDocumentViewModel.resetArtifacts()
        skipNfcLiveNessViewModel.getLiveNessData().clear()
        logger.log("VerifySDKManager instance and observers have been reset.")
    }
}
