/*
 * Decompiled with CFR 0.152.
 */
package io.nats.bridge.support;

import io.nats.bridge.MessageBridge;
import io.nats.bridge.MessageBus;
import io.nats.bridge.jms.JMSMessageBus;
import io.nats.bridge.messages.Message;
import io.nats.bridge.messages.transform.TransformMessage;
import io.nats.bridge.messages.transform.TransformResult;
import io.nats.bridge.messages.transform.Transformers;
import io.nats.bridge.metrics.Metrics;
import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.LinkedTransferQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageBridgeImpl
implements MessageBridge {
    private final MessageBus sourceBus;
    private final MessageBus destinationBus;
    private final boolean requestReply;
    private final String name;
    private final boolean transformMessage;
    private Logger runtimeLogger = LoggerFactory.getLogger((String)"runtime");
    private Logger logger = LoggerFactory.getLogger(MessageBridgeImpl.class);
    private final Queue<MessageBridgeRequestReply> replyMessageQueue;
    private final List<String> transforms;
    private final Map<String, TransformMessage> transformers;
    private final List<String> outputTransforms;
    long lastRestart = System.currentTimeMillis();
    final Duration ignoreRestartBackoffAfter = Duration.ofMinutes(10L);
    final int backoffMax = 60;
    int backoffSeconds = 1;

    public MessageBridgeImpl(String name, MessageBus sourceBus, MessageBus destinationBus, boolean requestReply, Queue<MessageBridgeRequestReply> replyMessageQueue, List<String> transforms, List<String> outputTransforms) {
        this.sourceBus = sourceBus;
        this.destinationBus = destinationBus;
        this.requestReply = requestReply;
        this.replyMessageQueue = replyMessageQueue != null ? replyMessageQueue : new LinkedTransferQueue();
        this.name = "bridge-" + name.toLowerCase().replace(".", "-").replace(" ", "-");
        this.transforms = transforms;
        this.outputTransforms = outputTransforms;
        boolean inputTransformEnabled = transforms != null && transforms.size() > 0;
        boolean outputTransformEnabled = outputTransforms != null && outputTransforms.size() > 0;
        this.transformMessage = inputTransformEnabled || outputTransformEnabled;
        this.transformers = this.transformMessage ? Transformers.loadTransforms() : Collections.emptyMap();
    }

    @Override
    public String name() {
        return this.name;
    }

    @Override
    public int process() {
        Optional<Object> receiveMessageFromSourceOption;
        try {
            receiveMessageFromSourceOption = this.sourceBus.receive();
        }
        catch (Exception ex) {
            receiveMessageFromSourceOption = Optional.empty();
            this.restartSourceBus(ex);
        }
        return this.doProcess(receiveMessageFromSourceOption);
    }

    private Message transformMessageIfNeeded(Message receiveMessageFromSource, List<String> transforms) {
        if (transforms.isEmpty()) {
            return receiveMessageFromSource;
        }
        Message currentMessage = receiveMessageFromSource;
        if (this.transformMessage) {
            TransformResult result = Transformers.runTransforms(this.transformers, transforms, (Message)currentMessage);
            switch (result.getResult()) {
                case SKIP: {
                    if (this.runtimeLogger.isTraceEnabled()) {
                        this.runtimeLogger.trace("Message was skipped");
                    }
                    return null;
                }
                case SYSTEM_ERROR: 
                case ERROR: {
                    if (result.getStatusMessage().isPresent()) {
                        this.logger.error((String)result.getStatusMessage().get(), result.getError());
                    } else {
                        this.logger.error("Error handling transform ", result.getError());
                    }
                    return null;
                }
                case TRANSFORMED: {
                    if (this.runtimeLogger.isTraceEnabled()) {
                        if (!result.getStatusMessage().isPresent()) {
                            this.runtimeLogger.trace("Message was transformed");
                        } else {
                            this.runtimeLogger.trace("Message was transformed " + (String)result.getStatusMessage().get());
                        }
                    }
                    currentMessage = result.getTransformedMessage();
                }
            }
        }
        return currentMessage;
    }

    private int doProcess(Optional<Message> receiveMessageFromSourceOption) {
        if (receiveMessageFromSourceOption.isPresent() && this.runtimeLogger.isTraceEnabled()) {
            this.runtimeLogger.trace("The {} bridge received message with body {}", (Object)this.name(), (Object)receiveMessageFromSourceOption.get().bodyAsString());
        }
        int count = 0;
        if (receiveMessageFromSourceOption.isPresent()) {
            ++count;
        }
        if (this.requestReply) {
            receiveMessageFromSourceOption.ifPresent(receiveMessageFromSource -> {
                Message currentMessageFinal = this.transformMessageIfNeeded((Message)receiveMessageFromSource, this.transforms);
                if (currentMessageFinal == null) {
                    return;
                }
                try {
                    this.destinationBus.request(currentMessageFinal, replyMessage -> {
                        Message replyMessageFinal;
                        if (this.runtimeLogger.isTraceEnabled()) {
                            this.runtimeLogger.info("The bridge {} got reply message {} \n for request message {} ", new Object[]{this.name, replyMessage.bodyAsString(), currentMessageFinal});
                        }
                        this.replyMessageQueue.add(new MessageBridgeRequestReply(currentMessageFinal, (replyMessageFinal = this.transformMessageIfNeeded((Message)replyMessage, this.outputTransforms)) != null ? replyMessageFinal : replyMessage));
                    });
                }
                catch (Exception ex) {
                    this.restartDestinationBus(ex);
                    this.destinationBus.request(currentMessageFinal, replyMessage -> {
                        Message replyMessageFinal;
                        if (this.runtimeLogger.isTraceEnabled()) {
                            this.runtimeLogger.info("The bridge {} got reply message {} \n for request message {} ", new Object[]{this.name, replyMessage.bodyAsString(), currentMessageFinal});
                        }
                        this.replyMessageQueue.add(new MessageBridgeRequestReply(currentMessageFinal, (replyMessageFinal = this.transformMessageIfNeeded((Message)replyMessage, this.outputTransforms)) != null ? replyMessageFinal : replyMessage));
                    });
                }
            });
        } else {
            receiveMessageFromSourceOption.ifPresent(receiveMessageFromSource -> {
                Message currentMessageFinal = this.transformMessageIfNeeded((Message)receiveMessageFromSource, this.transforms);
                if (currentMessageFinal == null) {
                    return;
                }
                try {
                    this.destinationBus.publish(currentMessageFinal);
                }
                catch (Exception ex) {
                    this.restartDestinationBus(ex);
                    this.destinationBus.publish(currentMessageFinal);
                }
            });
        }
        try {
            count += this.sourceBus.process();
        }
        catch (Exception ex) {
            this.restartSourceBus(ex);
            count += this.sourceBus.process();
        }
        try {
            count += this.destinationBus.process();
        }
        catch (Exception ex) {
            this.restartDestinationBus(ex);
            count += this.destinationBus.process();
        }
        return count += this.processReplies();
    }

    private void rethrow(Exception ex) {
        if (ex instanceof RuntimeException) {
            throw (RuntimeException)ex;
        }
        throw new RuntimeException(ex);
    }

    private void restartMessageBus(Exception ex, MessageBus messageBus) {
        if (messageBus instanceof JMSMessageBus) {
            this.logger.info("Restarting Message Bus {} {}", (Object)this.name, (Object)this.backoffSeconds);
            long now = System.currentTimeMillis();
            if (this.lastRestart > now - this.ignoreRestartBackoffAfter.toMillis() && this.backoffSeconds < 60) {
                this.backoffSeconds *= 2;
            }
            this.logger.info("Restart reason for " + this.name, (Throwable)ex);
            try {
                messageBus.close();
                this.logger.info("Restart reason {} === CLOSED", (Object)this.name);
            }
            catch (Exception exClose) {
                this.logger.debug("Unable to close", (Throwable)exClose);
            }
            try {
                messageBus.init();
                this.logger.info("Restart reason {} === RESTARTED", (Object)this.name);
            }
            catch (Exception exClose) {
                this.logger.error("Unable to recreate", (Throwable)exClose);
                try {
                    Thread.sleep(this.backoffSeconds * 1000);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                return;
            }
            this.backoffSeconds = 1;
            this.lastRestart = System.currentTimeMillis();
            this.logger.info("Restarting Message Bus for {}, sleeping {}", (Object)this.name, (Object)this.backoffSeconds);
            this.logger.info("Restarted Message Bus for {}", (Object)this.name);
        } else {
            this.rethrow(ex);
        }
    }

    private void restartDestinationBus(Exception ex) {
        this.restartMessageBus(ex, this.destinationBus);
    }

    private void restartSourceBus(Exception ex) {
        this.restartMessageBus(ex, this.sourceBus);
    }

    @Override
    public int process(Duration duration) {
        Optional<Message> receiveMessageFromSourceOption;
        try {
            receiveMessageFromSourceOption = this.sourceBus.receive(duration);
        }
        catch (Exception ex) {
            receiveMessageFromSourceOption = Optional.empty();
            this.restartSourceBus(ex);
        }
        return this.doProcess(receiveMessageFromSourceOption);
    }

    private int processReplies() {
        int i = 0;
        MessageBridgeRequestReply requestReply = this.replyMessageQueue.poll();
        while (requestReply != null) {
            ++i;
            requestReply.respond();
            requestReply = this.replyMessageQueue.poll();
        }
        return i;
    }

    @Override
    public void close() {
        this.sourceBus.close();
        this.destinationBus.close();
    }

    @Override
    public Metrics sourceMetrics() {
        return this.sourceBus.metrics();
    }

    @Override
    public Metrics destinationMetrics() {
        return this.destinationBus.metrics();
    }

    public static class MessageBridgeRequestReply {
        final Message request;
        final Message reply;

        public MessageBridgeRequestReply(Message request, Message reply) {
            this.request = request;
            this.reply = reply;
        }

        public void respond() {
            this.request.reply(this.reply);
        }
    }
}

