/*
 * Decompiled with CFR 0.152.
 */
package io.nats.client.impl;

import io.nats.client.impl.NatsMessage;
import java.time.Duration;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.LockSupport;
import java.util.function.Predicate;

class MessageQueue {
    private final AtomicLong length;
    private final AtomicLong sizeInBytes;
    private final AtomicBoolean running;
    private final boolean singleThreadedReader;
    private final ConcurrentLinkedQueue<NatsMessage> queue = new ConcurrentLinkedQueue();
    private final ConcurrentLinkedQueue<Thread> waiters;
    public static final int MAX_SPINS = 200;
    public static final int SPIN_WAIT = 50;
    public static final int MAX_SPIN_TIME = 10000;

    MessageQueue(boolean singleReaderMode) {
        this.running = new AtomicBoolean(true);
        this.sizeInBytes = new AtomicLong(0L);
        this.length = new AtomicLong(0L);
        this.waiters = new ConcurrentLinkedQueue();
        this.singleThreadedReader = singleReaderMode;
    }

    boolean isSingleReaderMode() {
        return this.singleThreadedReader;
    }

    boolean isRunning() {
        return this.running.get();
    }

    void pause() {
        this.running.set(false);
        this.signalAll();
    }

    void resume() {
        this.running.set(true);
        this.signalAll();
    }

    void signalOne() {
        Thread t = this.waiters.poll();
        if (t != null) {
            LockSupport.unpark(t);
        }
    }

    void signalIfNotEmpty() {
        if (this.length.get() > 0L) {
            this.signalOne();
        }
    }

    void signalAll() {
        Thread t = this.waiters.poll();
        while (t != null) {
            LockSupport.unpark(t);
            t = this.waiters.poll();
        }
    }

    void push(NatsMessage msg) {
        this.queue.add(msg);
        this.length.incrementAndGet();
        this.sizeInBytes.getAndAdd(msg.getSizeInBytes());
        this.signalOne();
    }

    void waitForTimeout(Duration timeout) throws InterruptedException {
        long timeoutNanos;
        long l = timeoutNanos = timeout != null ? timeout.toNanos() : -1L;
        if (timeoutNanos >= 0L) {
            Thread t = Thread.currentThread();
            long start = System.nanoTime();
            if (timeoutNanos > 10000L) {
                for (int count = 0; this.length.get() == 0L && this.running.get() && count < 200; ++count) {
                    LockSupport.parkNanos(50L);
                }
            }
            long now = start;
            while (this.length.get() == 0L && this.running.get() && (timeoutNanos <= 0L || (timeoutNanos -= (now = System.nanoTime()) - (start = now)) > 0L)) {
                this.waiters.add(t);
                if (timeoutNanos == 0L) {
                    LockSupport.park();
                } else {
                    LockSupport.parkNanos(timeoutNanos);
                }
                this.waiters.remove(t);
                if (!Thread.interrupted()) continue;
                throw new InterruptedException("Interrupted during timeout");
            }
        }
    }

    NatsMessage pop(Duration timeout) throws InterruptedException {
        if (!this.running.get()) {
            return null;
        }
        NatsMessage retVal = this.queue.poll();
        if (retVal == null && timeout != null) {
            this.waitForTimeout(timeout);
            if (!this.running.get()) {
                return null;
            }
            retVal = this.queue.poll();
        }
        if (retVal != null) {
            this.sizeInBytes.getAndAdd(-retVal.getSizeInBytes());
            this.length.decrementAndGet();
            this.signalIfNotEmpty();
        }
        return retVal;
    }

    NatsMessage accumulate(long maxSize, long maxMessages, Duration timeout) throws InterruptedException {
        NatsMessage next;
        if (!this.singleThreadedReader) {
            throw new IllegalStateException("Accumulate is only supported in single reader mode.");
        }
        if (!this.running.get()) {
            return null;
        }
        NatsMessage msg = this.queue.poll();
        if (msg == null) {
            this.waitForTimeout(timeout);
            if (!this.running.get() || this.queue.peek() == null) {
                return null;
            }
            msg = this.queue.poll();
        }
        long size = msg.getSizeInBytes();
        if (maxMessages <= 1L || size >= maxSize) {
            this.sizeInBytes.addAndGet(-size);
            this.length.decrementAndGet();
            this.signalIfNotEmpty();
            return msg;
        }
        long count = 1L;
        NatsMessage cursor = msg;
        while (cursor != null && (next = this.queue.peek()) != null) {
            long s = next.getSizeInBytes();
            if (maxSize >= 0L && size + s >= maxSize) break;
            size += s;
            cursor = cursor.next = this.queue.poll();
            if (++count != maxMessages) continue;
            break;
        }
        this.sizeInBytes.addAndGet(-size);
        this.length.addAndGet(-count);
        this.signalIfNotEmpty();
        return msg;
    }

    NatsMessage popNow() throws InterruptedException {
        return this.pop(null);
    }

    long length() {
        return this.length.get();
    }

    long sizeInBytes() {
        return this.sizeInBytes.get();
    }

    void filter(Predicate<NatsMessage> p) {
        if (this.running.get()) {
            throw new IllegalStateException("Filter is only supported when the queue is paused");
        }
        ConcurrentLinkedQueue<NatsMessage> newQueue = new ConcurrentLinkedQueue<NatsMessage>();
        NatsMessage cursor = this.queue.poll();
        while (cursor != null) {
            if (!p.test(cursor)) {
                newQueue.add(cursor);
            } else {
                this.sizeInBytes.addAndGet(-cursor.getSizeInBytes());
                this.length.decrementAndGet();
            }
            cursor = this.queue.poll();
        }
        this.queue.addAll(newQueue);
    }
}

