/*
 * Decompiled with CFR 0.152.
 */
package io.nats.client;

import io.nats.client.DecodedSeed;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Random;
import net.i2p.crypto.eddsa.EdDSAEngine;
import net.i2p.crypto.eddsa.EdDSAPrivateKey;
import net.i2p.crypto.eddsa.EdDSAPublicKey;
import net.i2p.crypto.eddsa.spec.EdDSANamedCurveSpec;
import net.i2p.crypto.eddsa.spec.EdDSANamedCurveTable;
import net.i2p.crypto.eddsa.spec.EdDSAParameterSpec;
import net.i2p.crypto.eddsa.spec.EdDSAPrivateKeySpec;
import net.i2p.crypto.eddsa.spec.EdDSAPublicKeySpec;

public class NKey {
    private static int PREFIX_BYTE_SEED;
    private static int PREFIX_BYTE_PRIVATE;
    private static int PREFIX_BYTE_SERVER;
    private static int PREFIX_BYTE_CLUSTER;
    private static int PREFIX_BYTE_ACCOUNT;
    private static int PREFIX_BYTE_USER;
    private static int PREFIX_BYTE_OPERATOR;
    private static final int ED25519_PUBLIC_KEYSIZE = 32;
    private static final int ED25519_PRIVATE_KEYSIZE = 64;
    private static final int ED25519_SEED_SIZE = 32;
    private static final EdDSANamedCurveSpec ed25519;
    private static final int[] crc16table;
    private static final String BASE32_CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567";
    private static int[] BASE32_LOOKUP;
    private static int MASK;
    private static int SHIFT;
    private char[] privateKeyAsSeed;
    private char[] publicKey;
    private Type type;

    static int crc16(byte[] bytes) {
        int crc = 0;
        for (byte b : bytes) {
            crc = crc << 8 & 0xFFFF ^ crc16table[(crc >> 8 ^ b & 0xFF) & 0xFF];
        }
        return crc;
    }

    static char[] base32Encode(byte[] bytes) {
        int nonBlank;
        int last = bytes.length;
        char[] charBuff = new char[(last + 7) * 8 / SHIFT];
        int offset = 0;
        int buffer = bytes[offset++];
        int bitsLeft = 8;
        int i = 0;
        while (bitsLeft > 0 || offset < last) {
            if (bitsLeft < SHIFT) {
                if (offset < last) {
                    buffer <<= 8;
                    buffer |= bytes[offset++] & 0xFF;
                    bitsLeft += 8;
                } else {
                    int pad = SHIFT - bitsLeft;
                    buffer <<= pad;
                    bitsLeft += pad;
                }
            }
            int index = MASK & buffer >> bitsLeft - SHIFT;
            bitsLeft -= SHIFT;
            charBuff[i] = BASE32_CHARS.charAt(index);
            ++i;
        }
        for (nonBlank = charBuff.length - 1; nonBlank >= 0 && charBuff[nonBlank] == '\u0000'; --nonBlank) {
        }
        char[] retVal = new char[nonBlank + 1];
        System.arraycopy(charBuff, 0, retVal, 0, retVal.length);
        for (int j = 0; j < charBuff.length; ++j) {
            charBuff[j] = '\u0000';
        }
        return retVal;
    }

    static byte[] base32Decode(char[] input) {
        byte[] bytes = new byte[input.length * SHIFT / 8];
        int buffer = 0;
        int next = 0;
        int bitsLeft = 0;
        for (int i = 0; i < input.length; ++i) {
            int lookup = input[i] - 48;
            if (lookup < 0 || lookup >= BASE32_LOOKUP.length) continue;
            int c = BASE32_LOOKUP[lookup];
            buffer <<= SHIFT;
            buffer |= c & MASK;
            if ((bitsLeft += SHIFT) < 8) continue;
            bytes[next++] = (byte)(buffer >> bitsLeft - 8);
            bitsLeft -= 8;
        }
        return bytes;
    }

    private static boolean checkValidPublicPrefixByte(int prefix) {
        if (prefix == PREFIX_BYTE_SERVER) {
            return true;
        }
        if (prefix == PREFIX_BYTE_CLUSTER) {
            return true;
        }
        if (prefix == PREFIX_BYTE_OPERATOR) {
            return true;
        }
        if (prefix == PREFIX_BYTE_ACCOUNT) {
            return true;
        }
        return prefix == PREFIX_BYTE_USER;
    }

    static char[] removePaddingAndClear(char[] withPad) {
        int i;
        for (i = withPad.length - 1; i >= 0 && withPad[i] == '='; --i) {
        }
        char[] withoutPad = new char[i + 1];
        System.arraycopy(withPad, 0, withoutPad, 0, withoutPad.length);
        for (int j = 0; j < withPad.length; ++j) {
            withPad[j] = '\u0000';
        }
        return withoutPad;
    }

    static char[] encode(Type type, byte[] src) throws IOException {
        ByteArrayOutputStream bytes = new ByteArrayOutputStream();
        bytes.write(type.prefix);
        bytes.write(src);
        int crc = NKey.crc16(bytes.toByteArray());
        byte[] littleEndian = ByteBuffer.allocate(2).order(ByteOrder.LITTLE_ENDIAN).putShort((short)crc).array();
        bytes.write(littleEndian);
        char[] withPad = NKey.base32Encode(bytes.toByteArray());
        return NKey.removePaddingAndClear(withPad);
    }

    static char[] encodeSeed(Type type, byte[] src) throws IOException {
        if (src.length != 64 && src.length != 32) {
            throw new IllegalArgumentException("Source is not the correct size for an ED25519 seed");
        }
        int b1 = PREFIX_BYTE_SEED | type.prefix >> 5;
        int b2 = (type.prefix & 0x1F) << 3;
        ByteArrayOutputStream bytes = new ByteArrayOutputStream();
        bytes.write(b1);
        bytes.write(b2);
        bytes.write(src);
        int crc = NKey.crc16(bytes.toByteArray());
        byte[] littleEndian = ByteBuffer.allocate(2).order(ByteOrder.LITTLE_ENDIAN).putShort((short)crc).array();
        bytes.write(littleEndian);
        char[] withPad = NKey.base32Encode(bytes.toByteArray());
        return NKey.removePaddingAndClear(withPad);
    }

    static byte[] decode(char[] src) {
        byte[] raw = NKey.base32Decode(src);
        if (raw == null || raw.length < 4) {
            throw new IllegalArgumentException("Invalid encoding for source string");
        }
        byte[] crcBytes = Arrays.copyOfRange(raw, raw.length - 2, raw.length);
        byte[] dataBytes = Arrays.copyOfRange(raw, 0, raw.length - 2);
        int crc = ByteBuffer.wrap(crcBytes).order(ByteOrder.LITTLE_ENDIAN).getShort() & 0xFFFF;
        int actual = NKey.crc16(dataBytes);
        if (actual != crc) {
            throw new IllegalArgumentException("CRC is invalid");
        }
        return dataBytes;
    }

    static byte[] decode(Type expectedType, char[] src, boolean safe) {
        byte[] raw = NKey.decode(src);
        byte[] dataBytes = Arrays.copyOfRange(raw, 1, raw.length);
        Type type = Type.fromPrefix(raw[0] & 0xFF);
        if (type != expectedType) {
            if (safe) {
                return null;
            }
            throw new IllegalArgumentException("Unexpected type");
        }
        return dataBytes;
    }

    static DecodedSeed decodeSeed(char[] seed) {
        byte[] raw = NKey.decode(seed);
        int b1 = raw[0] & 0xF8;
        int b2 = (raw[0] & 7) << 5 | (raw[1] & 0xF8) >> 3;
        if (b1 != PREFIX_BYTE_SEED) {
            throw new IllegalArgumentException("Invalid encoding");
        }
        if (!NKey.checkValidPublicPrefixByte(b2)) {
            throw new IllegalArgumentException("Invalid encoded prefix byte");
        }
        byte[] dataBytes = Arrays.copyOfRange(raw, 2, raw.length);
        DecodedSeed retVal = new DecodedSeed();
        retVal.prefix = b2;
        retVal.bytes = dataBytes;
        return retVal;
    }

    private static NKey createPair(Type type, SecureRandom random) throws IOException, NoSuchProviderException, NoSuchAlgorithmException {
        if (random == null) {
            random = SecureRandom.getInstance("SHA1PRNG", "SUN");
        }
        byte[] seed = new byte[ed25519.getCurve().getField().getb() / 8];
        random.nextBytes(seed);
        return NKey.createPair(type, seed);
    }

    private static NKey createPair(Type type, byte[] seed) throws IOException, NoSuchProviderException, NoSuchAlgorithmException {
        EdDSAPrivateKeySpec privKeySpec = new EdDSAPrivateKeySpec(seed, (EdDSAParameterSpec)ed25519);
        EdDSAPrivateKey privKey = new EdDSAPrivateKey(privKeySpec);
        EdDSAPublicKeySpec pubKeySpec = new EdDSAPublicKeySpec(privKey.getA(), (EdDSAParameterSpec)ed25519);
        EdDSAPublicKey pubKey = new EdDSAPublicKey(pubKeySpec);
        byte[] pubBytes = pubKey.getAbyte();
        byte[] bytes = new byte[pubBytes.length + seed.length];
        System.arraycopy(seed, 0, bytes, 0, seed.length);
        System.arraycopy(pubBytes, 0, bytes, seed.length, pubBytes.length);
        char[] encoded = NKey.encodeSeed(type, bytes);
        return new NKey(type, null, encoded);
    }

    public static NKey createAccount(SecureRandom random) throws IOException, NoSuchProviderException, NoSuchAlgorithmException {
        return NKey.createPair(Type.ACCOUNT, random);
    }

    public static NKey createCluster(SecureRandom random) throws IOException, NoSuchProviderException, NoSuchAlgorithmException {
        return NKey.createPair(Type.CLUSTER, random);
    }

    public static NKey createOperator(SecureRandom random) throws IOException, NoSuchProviderException, NoSuchAlgorithmException {
        return NKey.createPair(Type.OPERATOR, random);
    }

    public static NKey createServer(SecureRandom random) throws IOException, NoSuchProviderException, NoSuchAlgorithmException {
        return NKey.createPair(Type.SERVER, random);
    }

    public static NKey createUser(SecureRandom random) throws IOException, NoSuchProviderException, NoSuchAlgorithmException {
        return NKey.createPair(Type.USER, random);
    }

    public static NKey fromPublicKey(char[] publicKey) {
        byte[] raw = NKey.decode(publicKey);
        int prefix = raw[0] & 0xFF;
        if (!NKey.checkValidPublicPrefixByte(prefix)) {
            throw new IllegalArgumentException("Not a valid public NKey");
        }
        Type type = Type.fromPrefix(prefix);
        return new NKey(type, publicKey, null);
    }

    public static NKey fromSeed(char[] seed) {
        DecodedSeed decoded = NKey.decodeSeed(seed);
        if (decoded.bytes.length == 64) {
            return new NKey(Type.fromPrefix(decoded.prefix), null, seed);
        }
        try {
            return NKey.createPair(Type.fromPrefix(decoded.prefix), decoded.bytes);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Bad seed value", e);
        }
    }

    public static boolean isValidPublicAccountKey(char[] src) {
        return NKey.decode(Type.ACCOUNT, src, true) != null;
    }

    public static boolean isValidPublicClusterKey(char[] src) {
        return NKey.decode(Type.CLUSTER, src, true) != null;
    }

    public static boolean isValidPublicOperatorKey(char[] src) {
        return NKey.decode(Type.OPERATOR, src, true) != null;
    }

    public static boolean isValidPublicServerKey(char[] src) {
        return NKey.decode(Type.SERVER, src, true) != null;
    }

    public static boolean isValidPublicUserKey(char[] src) {
        return NKey.decode(Type.USER, src, true) != null;
    }

    private NKey(Type t, char[] publicKey, char[] privateKey) {
        this.type = t;
        this.privateKeyAsSeed = privateKey;
        this.publicKey = publicKey;
    }

    public void clear() {
        int i;
        Random r = new Random();
        if (this.privateKeyAsSeed != null) {
            for (i = 0; i < this.privateKeyAsSeed.length; ++i) {
                this.privateKeyAsSeed[i] = (char)(r.nextInt(26) + 97);
            }
            Arrays.fill(this.privateKeyAsSeed, '\u0000');
        }
        if (this.publicKey != null) {
            for (i = 0; i < this.publicKey.length; ++i) {
                this.publicKey[i] = (char)(r.nextInt(26) + 97);
            }
            Arrays.fill(this.publicKey, '\u0000');
        }
    }

    public char[] getSeed() {
        if (this.privateKeyAsSeed == null) {
            throw new IllegalStateException("Public-only NKey");
        }
        DecodedSeed decoded = NKey.decodeSeed(this.privateKeyAsSeed);
        byte[] seedBytes = new byte[32];
        System.arraycopy(decoded.bytes, 0, seedBytes, 0, seedBytes.length);
        try {
            return NKey.encodeSeed(Type.fromPrefix(decoded.prefix), seedBytes);
        }
        catch (Exception e) {
            throw new IllegalStateException("Unable to create seed.", e);
        }
    }

    public char[] getPublicKey() throws GeneralSecurityException, IOException {
        if (this.publicKey != null) {
            return this.publicKey;
        }
        KeyPair keys = this.getKeyPair();
        EdDSAPublicKey pubKey = (EdDSAPublicKey)keys.getPublic();
        byte[] pubBytes = pubKey.getAbyte();
        return NKey.encode(this.type, pubBytes);
    }

    public char[] getPrivateKey() throws GeneralSecurityException, IOException {
        if (this.privateKeyAsSeed == null) {
            throw new IllegalStateException("Public-only NKey");
        }
        DecodedSeed decoded = NKey.decodeSeed(this.privateKeyAsSeed);
        return NKey.encode(Type.PRIVATE, decoded.bytes);
    }

    public KeyPair getKeyPair() throws GeneralSecurityException, IOException {
        if (this.privateKeyAsSeed == null) {
            throw new IllegalStateException("Public-only NKey");
        }
        DecodedSeed decoded = NKey.decodeSeed(this.privateKeyAsSeed);
        byte[] seedBytes = new byte[32];
        byte[] pubBytes = new byte[32];
        System.arraycopy(decoded.bytes, 0, seedBytes, 0, seedBytes.length);
        System.arraycopy(decoded.bytes, seedBytes.length, pubBytes, 0, pubBytes.length);
        EdDSAPrivateKeySpec privKeySpec = new EdDSAPrivateKeySpec(seedBytes, (EdDSAParameterSpec)ed25519);
        EdDSAPrivateKey privKey = new EdDSAPrivateKey(privKeySpec);
        EdDSAPublicKeySpec pubKeySpec = new EdDSAPublicKeySpec(pubBytes, (EdDSAParameterSpec)ed25519);
        EdDSAPublicKey pubKey = new EdDSAPublicKey(pubKeySpec);
        return new KeyPair((PublicKey)pubKey, (PrivateKey)privKey);
    }

    public Type getType() {
        return this.type;
    }

    public byte[] sign(byte[] input) throws GeneralSecurityException, IOException {
        EdDSAEngine sgr = new EdDSAEngine(MessageDigest.getInstance(ed25519.getHashAlgorithm()));
        PrivateKey sKey = this.getKeyPair().getPrivate();
        sgr.initSign(sKey);
        sgr.update(input);
        return sgr.sign();
    }

    public boolean verify(byte[] input, byte[] signature) throws GeneralSecurityException, IOException {
        EdDSAEngine sgr = new EdDSAEngine(MessageDigest.getInstance(ed25519.getHashAlgorithm()));
        PublicKey sKey = null;
        if (this.privateKeyAsSeed != null) {
            sKey = this.getKeyPair().getPublic();
        } else {
            char[] encodedPublicKey = this.getPublicKey();
            byte[] decodedPublicKey = NKey.decode(this.type, encodedPublicKey, false);
            EdDSAPublicKeySpec pubKeySpec = new EdDSAPublicKeySpec(decodedPublicKey, (EdDSAParameterSpec)ed25519);
            sKey = new EdDSAPublicKey(pubKeySpec);
        }
        sgr.initVerify(sKey);
        sgr.update(input);
        return sgr.verify(signature);
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (!(o instanceof NKey)) {
            return false;
        }
        NKey otherNKey = (NKey)o;
        if (this.type != otherNKey.type) {
            return false;
        }
        if (this.privateKeyAsSeed == null) {
            return Arrays.equals(this.publicKey, otherNKey.publicKey);
        }
        return Arrays.equals(this.privateKeyAsSeed, otherNKey.privateKeyAsSeed);
    }

    public int hashCode() {
        int result = 17;
        result = 31 * result + this.type.prefix;
        result = this.privateKeyAsSeed == null ? 31 * result + Arrays.hashCode(this.publicKey) : 31 * result + Arrays.hashCode(this.privateKeyAsSeed);
        return result;
    }

    static {
        int i;
        PREFIX_BYTE_SEED = 144;
        PREFIX_BYTE_PRIVATE = 120;
        PREFIX_BYTE_SERVER = 104;
        PREFIX_BYTE_CLUSTER = 16;
        PREFIX_BYTE_ACCOUNT = 0;
        PREFIX_BYTE_USER = 160;
        PREFIX_BYTE_OPERATOR = 112;
        ed25519 = EdDSANamedCurveTable.getByName((String)"Ed25519");
        crc16table = new int[]{0, 4129, 8258, 12387, 16516, 20645, 24774, 28903, 33032, 37161, 41290, 45419, 49548, 53677, 57806, 61935, 4657, 528, 12915, 8786, 21173, 17044, 29431, 25302, 37689, 33560, 45947, 41818, 54205, 50076, 62463, 58334, 9314, 13379, 1056, 5121, 25830, 29895, 17572, 21637, 42346, 46411, 34088, 38153, 58862, 62927, 50604, 54669, 13907, 9842, 5649, 1584, 30423, 26358, 22165, 18100, 46939, 42874, 38681, 34616, 63455, 59390, 55197, 51132, 18628, 22757, 26758, 30887, 2112, 6241, 10242, 14371, 51660, 55789, 59790, 63919, 35144, 39273, 43274, 47403, 23285, 19156, 31415, 27286, 6769, 2640, 14899, 10770, 56317, 52188, 64447, 60318, 39801, 35672, 47931, 43802, 27814, 31879, 19684, 23749, 11298, 15363, 3168, 7233, 60846, 64911, 52716, 56781, 44330, 48395, 36200, 40265, 32407, 28342, 24277, 20212, 15891, 11826, 7761, 3696, 65439, 61374, 57309, 53244, 48923, 44858, 40793, 36728, 37256, 33193, 45514, 41451, 53516, 49453, 61774, 57711, 4224, 161, 12482, 8419, 20484, 16421, 28742, 24679, 33721, 37784, 41979, 46042, 49981, 54044, 58239, 62302, 689, 4752, 8947, 13010, 16949, 21012, 25207, 29270, 46570, 42443, 38312, 34185, 62830, 58703, 54572, 50445, 13538, 9411, 5280, 1153, 29798, 25671, 21540, 17413, 42971, 47098, 34713, 38840, 59231, 63358, 50973, 55100, 9939, 14066, 1681, 5808, 26199, 30326, 17941, 22068, 55628, 51565, 63758, 59695, 39368, 35305, 47498, 43435, 22596, 18533, 30726, 26663, 6336, 2273, 14466, 10403, 52093, 56156, 60223, 64286, 35833, 39896, 43963, 48026, 19061, 23124, 27191, 31254, 2801, 6864, 10931, 14994, 64814, 60687, 56684, 52557, 48554, 44427, 40424, 36297, 31782, 27655, 23652, 19525, 15522, 11395, 7392, 3265, 61215, 65342, 53085, 57212, 44955, 49082, 36825, 40952, 28183, 32310, 20053, 24180, 11923, 16050, 3793, 7920};
        MASK = 31;
        SHIFT = 5;
        BASE32_LOOKUP = new int[256];
        for (i = 0; i < BASE32_LOOKUP.length; ++i) {
            NKey.BASE32_LOOKUP[i] = 255;
        }
        i = 0;
        while (i < BASE32_CHARS.length()) {
            int index = BASE32_CHARS.charAt(i) - 48;
            NKey.BASE32_LOOKUP[index] = i++;
        }
    }

    public static enum Type {
        USER(PREFIX_BYTE_USER),
        ACCOUNT(PREFIX_BYTE_ACCOUNT),
        SERVER(PREFIX_BYTE_SERVER),
        OPERATOR(PREFIX_BYTE_OPERATOR),
        CLUSTER(PREFIX_BYTE_CLUSTER),
        PRIVATE(PREFIX_BYTE_PRIVATE);

        private final int prefix;

        private Type(int prefix) {
            this.prefix = prefix;
        }

        public static Type fromPrefix(int prefix) {
            if (prefix == PREFIX_BYTE_ACCOUNT) {
                return ACCOUNT;
            }
            if (prefix == PREFIX_BYTE_SERVER) {
                return SERVER;
            }
            if (prefix == PREFIX_BYTE_USER) {
                return USER;
            }
            if (prefix == PREFIX_BYTE_CLUSTER) {
                return CLUSTER;
            }
            if (prefix == PREFIX_BYTE_PRIVATE) {
                return ACCOUNT;
            }
            if (prefix == PREFIX_BYTE_OPERATOR) {
                return OPERATOR;
            }
            throw new IllegalArgumentException("Unknown prefix");
        }
    }
}

