/*
 * Decompiled with CFR 0.152.
 */
package io.neow3j.crypto;

import io.neow3j.constants.NeoConstants;
import io.neow3j.crypto.ECDSASignature;
import io.neow3j.crypto.ECKeyPair;
import io.neow3j.crypto.Hash;
import io.neow3j.types.Hash160;
import io.neow3j.utils.ArrayUtils;
import io.neow3j.utils.Assertions;
import io.neow3j.utils.Numeric;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.SignatureException;
import java.util.Arrays;
import org.bouncycastle.asn1.x9.X9IntegerConverter;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.signers.ECDSASigner;
import org.bouncycastle.math.ec.ECAlgorithms;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.math.ec.FixedPointCombMultiplier;
import org.bouncycastle.math.ec.custom.sec.SecP256R1Curve;

public class Sign {
    private static final int LOWER_REAL_V = 27;

    public static SignatureData signHexMessage(String messageHex, ECKeyPair keyPair) {
        return Sign.signMessage(Numeric.hexStringToByteArray(messageHex), keyPair, true);
    }

    public static SignatureData signMessage(String message, ECKeyPair keyPair) {
        return Sign.signMessage(message.getBytes(StandardCharsets.UTF_8), keyPair, true);
    }

    public static SignatureData signMessage(byte[] message, ECKeyPair keyPair) {
        return Sign.signMessage(message, keyPair, true);
    }

    public static SignatureData signMessage(byte[] message, ECKeyPair keyPair, boolean needToHash) {
        byte[] messageHash = needToHash ? Hash.sha256(message) : message;
        ECDSASignature sig = keyPair.signAndGetECDSASignature(messageHash);
        int recId = -1;
        for (int i = 0; i < 4; ++i) {
            ECKeyPair.ECPublicKey k = Sign.recoverFromSignature(i, sig, messageHash);
            if (k == null || !k.equals(keyPair.getPublicKey())) continue;
            recId = i;
            break;
        }
        if (recId == -1) {
            throw new RuntimeException("Could not construct a recoverable key. This should never happen.");
        }
        int headerByte = recId + 27;
        byte v = (byte)headerByte;
        byte[] r = Numeric.toBytesPadded(sig.r, 32);
        byte[] s = Numeric.toBytesPadded(sig.s, 32);
        return new SignatureData(v, r, s);
    }

    public static ECKeyPair.ECPublicKey recoverFromSignature(int recId, ECDSASignature sig, byte[] message) {
        Assertions.verifyPrecondition(recId >= 0, "recId must be positive");
        Assertions.verifyPrecondition(sig.r.signum() >= 0, "r must be positive");
        Assertions.verifyPrecondition(sig.s.signum() >= 0, "s must be positive");
        Assertions.verifyPrecondition(message != null, "message cannot be null");
        BigInteger n = NeoConstants.secp256r1DomainParams().getN();
        BigInteger i = BigInteger.valueOf((long)recId / 2L);
        BigInteger x = sig.r.add(i.multiply(n));
        BigInteger prime = SecP256R1Curve.q;
        if (x.compareTo(prime) >= 0) {
            return null;
        }
        ECPoint R = Sign.decompressKey(x, (recId & 1) == 1);
        if (!R.multiply(n).isInfinity()) {
            return null;
        }
        BigInteger e = new BigInteger(1, message);
        BigInteger eInv = BigInteger.ZERO.subtract(e).mod(n);
        BigInteger rInv = sig.r.modInverse(n);
        BigInteger srInv = rInv.multiply(sig.s).mod(n);
        BigInteger eInvrInv = rInv.multiply(eInv).mod(n);
        ECPoint q = ECAlgorithms.sumOfTwoMultiplies((ECPoint)NeoConstants.secp256r1DomainParams().getG(), (BigInteger)eInvrInv, (ECPoint)R, (BigInteger)srInv);
        return new ECKeyPair.ECPublicKey(q);
    }

    private static ECPoint decompressKey(BigInteger xBN, boolean yBit) {
        X9IntegerConverter x9 = new X9IntegerConverter();
        byte[] compEnc = x9.integerToBytes(xBN, 1 + x9.getByteLength(NeoConstants.secp256r1DomainParams().getCurve()));
        compEnc[0] = (byte)(yBit ? 3 : 2);
        return NeoConstants.secp256r1DomainParams().getCurve().decodePoint(compEnc);
    }

    public static ECKeyPair.ECPublicKey signedMessageToKey(byte[] message, SignatureData signatureData) throws SignatureException {
        byte[] messageHash;
        byte[] r = signatureData.getR();
        byte[] s = signatureData.getS();
        Assertions.verifyPrecondition(r != null && r.length == 32, "r must be 32 bytes.");
        Assertions.verifyPrecondition(s != null && s.length == 32, "s must be 32 bytes.");
        int header = signatureData.getV() & 0xFF;
        if (header < 27 || header > 34) {
            throw new SignatureException("Header byte out of range: " + header);
        }
        int recId = header - 27;
        ECDSASignature sig = new ECDSASignature(new BigInteger(1, signatureData.getR()), new BigInteger(1, signatureData.getS()));
        ECKeyPair.ECPublicKey key = Sign.recoverFromSignature(recId, sig, messageHash = Hash.sha256(message));
        if (key == null) {
            throw new SignatureException("Could not recover public key from signature");
        }
        return key;
    }

    public static ECKeyPair.ECPublicKey publicKeyFromPrivate(ECKeyPair.ECPrivateKey privKey) {
        return new ECKeyPair.ECPublicKey(Sign.publicPointFromPrivateKey(privKey));
    }

    public static ECPoint publicPointFromPrivateKey(ECKeyPair.ECPrivateKey privKey) {
        BigInteger key = privKey.getInt();
        if (key.bitLength() > NeoConstants.secp256r1DomainParams().getN().bitLength()) {
            key = key.mod(NeoConstants.secp256r1DomainParams().getN());
        }
        return new FixedPointCombMultiplier().multiply(NeoConstants.secp256r1DomainParams().getG(), key).normalize();
    }

    public static Hash160 recoverSigningScriptHash(byte[] message, SignatureData signatureData) throws SignatureException {
        byte v = signatureData.getV();
        byte[] r = signatureData.getR();
        byte[] s = signatureData.getS();
        SignatureData signatureDataV = new SignatureData(Sign.getRealV(v), r, s);
        ECKeyPair.ECPublicKey key = Sign.signedMessageToKey(message, signatureDataV);
        return Hash160.fromPublicKey(key.getEncoded(true));
    }

    private static byte getRealV(byte v) {
        if (v == 27 || v == 28) {
            return v;
        }
        int realV = 27;
        int inc = 0;
        if (v % 2 == 0) {
            inc = 1;
        }
        return (byte)(realV + inc);
    }

    public static boolean verifySignature(byte[] message, SignatureData sig, ECKeyPair.ECPublicKey pubKey, boolean hashMessage) {
        byte[] messageHash = hashMessage ? Hash.sha256(message) : message;
        ECDSASigner verifier = new ECDSASigner();
        verifier.init(false, (CipherParameters)new ECPublicKeyParameters(pubKey.getECPoint(), NeoConstants.secp256r1DomainParams()));
        return verifier.verifySignature(messageHash, new BigInteger(1, sig.getR()), new BigInteger(1, sig.getS()));
    }

    public static boolean verifySignature(byte[] message, SignatureData sig, ECKeyPair.ECPublicKey pubKey) {
        return Sign.verifySignature(message, sig, pubKey, true);
    }

    public static class SignatureData {
        private final byte v;
        private final byte[] r;
        private final byte[] s;

        public SignatureData(byte v, byte[] r, byte[] s) {
            this.v = v;
            this.r = r;
            this.s = s;
        }

        public static SignatureData fromByteArray(byte[] signature) {
            return SignatureData.fromByteArray((byte)0, signature);
        }

        public static SignatureData fromByteArray(byte v, byte[] signature) {
            return new SignatureData(v, Arrays.copyOfRange(signature, 0, 32), Arrays.copyOfRange(signature, 32, 64));
        }

        public byte getV() {
            return this.v;
        }

        public byte[] getR() {
            return this.r;
        }

        public byte[] getS() {
            return this.s;
        }

        public byte[] getConcatenated() {
            return ArrayUtils.concatenate(this.r, this.s);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            SignatureData that = (SignatureData)o;
            if (this.v != that.v) {
                return false;
            }
            if (!Arrays.equals(this.r, that.r)) {
                return false;
            }
            return Arrays.equals(this.s, that.s);
        }

        public int hashCode() {
            int result = this.v;
            result = 31 * result + Arrays.hashCode(this.r);
            result = 31 * result + Arrays.hashCode(this.s);
            return result;
        }

        public String toString() {
            return "SignatureData{v=" + this.v + ", r=" + Arrays.toString(this.r) + ", s=" + Arrays.toString(this.s) + '}';
        }
    }
}

