/*
 * Decompiled with CFR 0.152.
 */
package io.netty.buffer;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.CompositeByteBuf;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.buffer.SwappedByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.buffer.WrappedByteBuf;
import io.netty.buffer.WrappedCompositeByteBuf;
import io.netty.util.AsciiString;
import io.netty.util.CharsetUtil;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.function.Executable;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;

public class ByteBufUtilTest {
    private static final String PARAMETERIZED_NAME = "bufferType = {0}";
    private static final int[][] INVALID_RANGES = new int[][]{{-1, 5}, {5, 30}, {10, 5}};

    private ByteBuf buffer(BufferType bufferType, int capacity) {
        switch (bufferType) {
            case DIRECT_UNPOOLED: {
                return Unpooled.directBuffer((int)capacity);
            }
            case HEAP_UNPOOLED: {
                return Unpooled.buffer((int)capacity);
            }
            case DIRECT_POOLED: {
                return PooledByteBufAllocator.DEFAULT.directBuffer(capacity);
            }
            case HEAP_POOLED: {
                return PooledByteBufAllocator.DEFAULT.buffer(capacity);
            }
        }
        throw new AssertionError((Object)("unexpected buffer type: " + (Object)((Object)bufferType)));
    }

    public static Collection<Object[]> noUnsafe() {
        return Arrays.asList({BufferType.DIRECT_POOLED}, {BufferType.DIRECT_UNPOOLED}, {BufferType.HEAP_POOLED}, {BufferType.HEAP_UNPOOLED});
    }

    @Test
    public void decodeRandomHexBytesWithEvenLength() {
        ByteBufUtilTest.decodeRandomHexBytes(256);
    }

    @Test
    public void decodeRandomHexBytesWithOddLength() {
        ByteBufUtilTest.decodeRandomHexBytes(257);
    }

    private static void decodeRandomHexBytes(int len) {
        byte[] b = new byte[len];
        Random rand = new Random();
        rand.nextBytes(b);
        String hexDump = ByteBufUtil.hexDump((byte[])b);
        for (int i = 0; i <= len; ++i) {
            byte[] b2 = Arrays.copyOfRange(b, i, b.length);
            byte[] decodedBytes = ByteBufUtil.decodeHexDump((CharSequence)hexDump, (int)(i * 2), (int)((len - i) * 2));
            Assertions.assertArrayEquals((byte[])b2, (byte[])decodedBytes);
        }
    }

    @Test
    public void decodeHexDumpWithOddLength() {
        Assertions.assertThrows(IllegalArgumentException.class, (Executable)new Executable(){

            public void execute() throws Throwable {
                ByteBufUtil.decodeHexDump((CharSequence)"abc");
            }
        });
    }

    @Test
    public void decodeHexDumpWithInvalidChar() {
        Assertions.assertThrows(IllegalArgumentException.class, (Executable)new Executable(){

            public void execute() throws Throwable {
                ByteBufUtil.decodeHexDump((CharSequence)"fg");
            }
        });
    }

    @Test
    public void testIndexOf() {
        ByteBuf haystack = Unpooled.copiedBuffer((CharSequence)"abc123", (Charset)CharsetUtil.UTF_8);
        Assertions.assertEquals((int)0, (int)ByteBufUtil.indexOf((ByteBuf)Unpooled.copiedBuffer((CharSequence)"a", (Charset)CharsetUtil.UTF_8), (ByteBuf)haystack));
        Assertions.assertEquals((int)1, (int)ByteBufUtil.indexOf((ByteBuf)Unpooled.copiedBuffer((byte[])"bc".getBytes(CharsetUtil.UTF_8)), (ByteBuf)haystack));
        Assertions.assertEquals((int)2, (int)ByteBufUtil.indexOf((ByteBuf)Unpooled.copiedBuffer((byte[])"c".getBytes(CharsetUtil.UTF_8)), (ByteBuf)haystack));
        Assertions.assertEquals((int)0, (int)ByteBufUtil.indexOf((ByteBuf)Unpooled.copiedBuffer((byte[])"abc12".getBytes(CharsetUtil.UTF_8)), (ByteBuf)haystack));
        Assertions.assertEquals((int)-1, (int)ByteBufUtil.indexOf((ByteBuf)Unpooled.copiedBuffer((byte[])"abcdef".getBytes(CharsetUtil.UTF_8)), (ByteBuf)haystack));
        Assertions.assertEquals((int)-1, (int)ByteBufUtil.indexOf((ByteBuf)Unpooled.copiedBuffer((byte[])"abc12x".getBytes(CharsetUtil.UTF_8)), (ByteBuf)haystack));
        Assertions.assertEquals((int)-1, (int)ByteBufUtil.indexOf((ByteBuf)Unpooled.copiedBuffer((byte[])"abc123def".getBytes(CharsetUtil.UTF_8)), (ByteBuf)haystack));
        ByteBuf needle = Unpooled.copiedBuffer((CharSequence)"abc12", (Charset)CharsetUtil.UTF_8);
        haystack.readerIndex(1);
        needle.readerIndex(1);
        Assertions.assertEquals((int)1, (int)ByteBufUtil.indexOf((ByteBuf)needle, (ByteBuf)haystack));
        haystack.readerIndex(2);
        needle.readerIndex(3);
        Assertions.assertEquals((int)3, (int)ByteBufUtil.indexOf((ByteBuf)needle, (ByteBuf)haystack));
        haystack.readerIndex(1);
        needle.readerIndex(2);
        Assertions.assertEquals((int)2, (int)ByteBufUtil.indexOf((ByteBuf)needle, (ByteBuf)haystack));
        haystack.release();
        haystack = new WrappedByteBuf(Unpooled.copiedBuffer((CharSequence)"abc123", (Charset)CharsetUtil.UTF_8));
        Assertions.assertEquals((int)0, (int)ByteBufUtil.indexOf((ByteBuf)Unpooled.copiedBuffer((CharSequence)"a", (Charset)CharsetUtil.UTF_8), (ByteBuf)haystack));
        Assertions.assertEquals((int)1, (int)ByteBufUtil.indexOf((ByteBuf)Unpooled.copiedBuffer((byte[])"bc".getBytes(CharsetUtil.UTF_8)), (ByteBuf)haystack));
        Assertions.assertEquals((int)-1, (int)ByteBufUtil.indexOf((ByteBuf)Unpooled.copiedBuffer((byte[])"abcdef".getBytes(CharsetUtil.UTF_8)), (ByteBuf)haystack));
        haystack.release();
        haystack = Unpooled.copiedBuffer((CharSequence)"123aab123", (Charset)CharsetUtil.UTF_8);
        Assertions.assertEquals((int)3, (int)ByteBufUtil.indexOf((ByteBuf)Unpooled.copiedBuffer((CharSequence)"aab", (Charset)CharsetUtil.UTF_8), (ByteBuf)haystack));
        haystack.release();
        needle.release();
    }

    @Test
    public void equalsBufferSubsections() {
        byte[] b1 = new byte[128];
        byte[] b2 = new byte[256];
        Random rand = new Random();
        rand.nextBytes(b1);
        rand.nextBytes(b2);
        int iB1 = b1.length / 2;
        int iB2 = iB1 + b1.length;
        int length = b1.length - iB1;
        System.arraycopy(b1, iB1, b2, iB2, length);
        Assertions.assertTrue((boolean)ByteBufUtil.equals((ByteBuf)Unpooled.wrappedBuffer((byte[])b1), (int)iB1, (ByteBuf)Unpooled.wrappedBuffer((byte[])b2), (int)iB2, (int)length));
    }

    private static int random(Random r, int min, int max) {
        return r.nextInt(max - min + 1) + min;
    }

    @Test
    public void notEqualsBufferSubsections() {
        int diffIndex;
        byte[] b1 = new byte[50];
        byte[] b2 = new byte[256];
        Random rand = new Random();
        rand.nextBytes(b1);
        rand.nextBytes(b2);
        int iB1 = b1.length / 2;
        int iB2 = iB1 + b1.length;
        int length = b1.length - iB1;
        System.arraycopy(b1, iB1, b2, iB2, length);
        int n = diffIndex = ByteBufUtilTest.random(rand, iB1, iB1 + length - 1);
        b1[n] = (byte)(b1[n] + 1);
        Assertions.assertFalse((boolean)ByteBufUtil.equals((ByteBuf)Unpooled.wrappedBuffer((byte[])b1), (int)iB1, (ByteBuf)Unpooled.wrappedBuffer((byte[])b2), (int)iB2, (int)length));
    }

    @Test
    public void notEqualsBufferOverflow() {
        byte[] b1 = new byte[8];
        byte[] b2 = new byte[16];
        Random rand = new Random();
        rand.nextBytes(b1);
        rand.nextBytes(b2);
        int iB1 = b1.length / 2;
        int iB2 = iB1 + b1.length;
        int length = b1.length - iB1;
        System.arraycopy(b1, iB1, b2, iB2, length - 1);
        Assertions.assertFalse((boolean)ByteBufUtil.equals((ByteBuf)Unpooled.wrappedBuffer((byte[])b1), (int)iB1, (ByteBuf)Unpooled.wrappedBuffer((byte[])b2), (int)iB2, (int)(Math.max(b1.length, b2.length) * 2)));
    }

    @Test
    public void notEqualsBufferUnderflow() {
        final byte[] b1 = new byte[8];
        final byte[] b2 = new byte[16];
        Random rand = new Random();
        rand.nextBytes(b1);
        rand.nextBytes(b2);
        final int iB1 = b1.length / 2;
        final int iB2 = iB1 + b1.length;
        int length = b1.length - iB1;
        System.arraycopy(b1, iB1, b2, iB2, length - 1);
        Assertions.assertThrows(IllegalArgumentException.class, (Executable)new Executable(){

            public void execute() {
                ByteBufUtil.equals((ByteBuf)Unpooled.wrappedBuffer((byte[])b1), (int)iB1, (ByteBuf)Unpooled.wrappedBuffer((byte[])b2), (int)iB2, (int)-1);
            }
        });
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void writeShortBE(BufferType bufferType) {
        int expected = 4660;
        ByteBuf buf = this.buffer(bufferType, 2).order(ByteOrder.BIG_ENDIAN);
        ByteBufUtil.writeShortBE((ByteBuf)buf, (int)expected);
        Assertions.assertEquals((int)expected, (int)buf.readShort());
        buf.resetReaderIndex();
        Assertions.assertEquals((short)ByteBufUtil.swapShort((short)((short)expected)), (short)buf.readShortLE());
        buf.release();
        buf = this.buffer(bufferType, 2).order(ByteOrder.LITTLE_ENDIAN);
        ByteBufUtil.writeShortBE((ByteBuf)buf, (int)expected);
        Assertions.assertEquals((short)ByteBufUtil.swapShort((short)((short)expected)), (short)buf.readShortLE());
        buf.resetReaderIndex();
        Assertions.assertEquals((short)ByteBufUtil.swapShort((short)((short)expected)), (short)buf.readShort());
        buf.release();
    }

    @Test
    public void setShortBE() {
        int shortValue = 4660;
        ByteBuf buf = Unpooled.wrappedBuffer((byte[])new byte[2]).order(ByteOrder.BIG_ENDIAN);
        ByteBufUtil.setShortBE((ByteBuf)buf, (int)0, (int)shortValue);
        Assertions.assertEquals((int)shortValue, (int)buf.readShort());
        buf.resetReaderIndex();
        Assertions.assertEquals((short)ByteBufUtil.swapShort((short)((short)shortValue)), (short)buf.readShortLE());
        buf.release();
        buf = Unpooled.wrappedBuffer((byte[])new byte[2]).order(ByteOrder.LITTLE_ENDIAN);
        ByteBufUtil.setShortBE((ByteBuf)buf, (int)0, (int)shortValue);
        Assertions.assertEquals((short)ByteBufUtil.swapShort((short)((short)shortValue)), (short)buf.readShortLE());
        buf.resetReaderIndex();
        Assertions.assertEquals((short)ByteBufUtil.swapShort((short)((short)shortValue)), (short)buf.readShort());
        buf.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void writeMediumBE(BufferType bufferType) {
        int mediumValue = 1193046;
        ByteBuf buf = this.buffer(bufferType, 4).order(ByteOrder.BIG_ENDIAN);
        ByteBufUtil.writeMediumBE((ByteBuf)buf, (int)mediumValue);
        Assertions.assertEquals((int)mediumValue, (int)buf.readMedium());
        buf.resetReaderIndex();
        Assertions.assertEquals((int)ByteBufUtil.swapMedium((int)mediumValue), (int)buf.readMediumLE());
        buf.release();
        buf = this.buffer(bufferType, 4).order(ByteOrder.LITTLE_ENDIAN);
        ByteBufUtil.writeMediumBE((ByteBuf)buf, (int)mediumValue);
        Assertions.assertEquals((int)ByteBufUtil.swapMedium((int)mediumValue), (int)buf.readMediumLE());
        buf.resetReaderIndex();
        Assertions.assertEquals((int)ByteBufUtil.swapMedium((int)mediumValue), (int)buf.readMedium());
        buf.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUsAscii(BufferType bufferType) {
        String usAscii = "NettyRocks";
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(usAscii.getBytes(CharsetUtil.US_ASCII));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        ByteBufUtil.writeAscii((ByteBuf)buf2, (CharSequence)usAscii);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUsAsciiSwapped(BufferType bufferType) {
        String usAscii = "NettyRocks";
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(usAscii.getBytes(CharsetUtil.US_ASCII));
        SwappedByteBuf buf2 = new SwappedByteBuf(this.buffer(bufferType, 16));
        ByteBufUtil.writeAscii((ByteBuf)buf2, (CharSequence)usAscii);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUsAsciiWrapped(BufferType bufferType) {
        String usAscii = "NettyRocks";
        ByteBuf buf = Unpooled.unreleasableBuffer((ByteBuf)this.buffer(bufferType, 16));
        ByteBufUtilTest.assertWrapped(buf);
        buf.writeBytes(usAscii.getBytes(CharsetUtil.US_ASCII));
        ByteBuf buf2 = Unpooled.unreleasableBuffer((ByteBuf)this.buffer(bufferType, 16));
        ByteBufUtilTest.assertWrapped(buf2);
        ByteBufUtil.writeAscii((ByteBuf)buf2, (CharSequence)usAscii);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        buf.unwrap().release();
        buf2.unwrap().release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUsAsciiComposite(BufferType bufferType) {
        String usAscii = "NettyRocks";
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(usAscii.getBytes(CharsetUtil.US_ASCII));
        CompositeByteBuf buf2 = Unpooled.compositeBuffer().addComponent(this.buffer(bufferType, 8)).addComponent(this.buffer(bufferType, 24));
        buf2.writeByte(1);
        ByteBufUtil.writeAscii((ByteBuf)buf2, (CharSequence)usAscii);
        Assertions.assertEquals((Object)buf, (Object)buf2.skipBytes(1));
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUsAsciiCompositeWrapped(BufferType bufferType) {
        String usAscii = "NettyRocks";
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(usAscii.getBytes(CharsetUtil.US_ASCII));
        WrappedCompositeByteBuf buf2 = new WrappedCompositeByteBuf(Unpooled.compositeBuffer().addComponent(this.buffer(bufferType, 8)).addComponent(this.buffer(bufferType, 24)));
        buf2.writeByte(1);
        ByteBufUtil.writeAscii((ByteBuf)buf2, (CharSequence)usAscii);
        Assertions.assertEquals((Object)buf, (Object)buf2.skipBytes(1));
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8(BufferType bufferType) {
        String usAscii = "Some UTF-8 like \u00e4\u00c4\u220f\u0152\u0152";
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(usAscii.getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)usAscii);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8Composite(BufferType bufferType) {
        String utf8 = "Some UTF-8 like \u00e4\u00c4\u220f\u0152\u0152";
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(utf8.getBytes(CharsetUtil.UTF_8));
        CompositeByteBuf buf2 = Unpooled.compositeBuffer().addComponent(this.buffer(bufferType, 8)).addComponent(this.buffer(bufferType, 24));
        buf2.writeByte(1);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)utf8);
        Assertions.assertEquals((Object)buf, (Object)buf2.skipBytes(1));
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8CompositeWrapped(BufferType bufferType) {
        String utf8 = "Some UTF-8 like \u00e4\u00c4\u220f\u0152\u0152";
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(utf8.getBytes(CharsetUtil.UTF_8));
        WrappedCompositeByteBuf buf2 = new WrappedCompositeByteBuf(Unpooled.compositeBuffer().addComponent(this.buffer(bufferType, 8)).addComponent(this.buffer(bufferType, 24)));
        buf2.writeByte(1);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)utf8);
        Assertions.assertEquals((Object)buf, (Object)buf2.skipBytes(1));
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8Surrogates(BufferType bufferType) {
        String surrogateString = new StringBuilder(2).append('a').append('\ud800').append('\udc00').append('b').toString();
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(surrogateString.getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)surrogateString);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        Assertions.assertEquals((int)buf.readableBytes(), (int)ByteBufUtil.utf8Bytes((CharSequence)surrogateString));
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8InvalidOnlyTrailingSurrogate(BufferType bufferType) {
        String surrogateString = new StringBuilder(2).append('a').append('\udc00').append('b').toString();
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(surrogateString.getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)surrogateString);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        Assertions.assertEquals((int)buf.readableBytes(), (int)ByteBufUtil.utf8Bytes((CharSequence)surrogateString));
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8InvalidOnlyLeadingSurrogate(BufferType bufferType) {
        String surrogateString = new StringBuilder(2).append('a').append('\ud800').append('b').toString();
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(surrogateString.getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)surrogateString);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        Assertions.assertEquals((int)buf.readableBytes(), (int)ByteBufUtil.utf8Bytes((CharSequence)surrogateString));
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8InvalidSurrogatesSwitched(BufferType bufferType) {
        String surrogateString = new StringBuilder(2).append('a').append('\udc00').append('\ud800').append('b').toString();
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(surrogateString.getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)surrogateString);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        Assertions.assertEquals((int)buf.readableBytes(), (int)ByteBufUtil.utf8Bytes((CharSequence)surrogateString));
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8InvalidTwoLeadingSurrogates(BufferType bufferType) {
        String surrogateString = new StringBuilder(2).append('a').append('\ud800').append('\ud800').append('b').toString();
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(surrogateString.getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)surrogateString);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        Assertions.assertEquals((int)buf.readableBytes(), (int)ByteBufUtil.utf8Bytes((CharSequence)surrogateString));
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8InvalidTwoTrailingSurrogates(BufferType bufferType) {
        String surrogateString = new StringBuilder(2).append('a').append('\udc00').append('\udc00').append('b').toString();
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(surrogateString.getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)surrogateString);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        Assertions.assertEquals((int)buf.readableBytes(), (int)ByteBufUtil.utf8Bytes((CharSequence)surrogateString));
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8InvalidEndOnLeadingSurrogate(BufferType bufferType) {
        String surrogateString = new StringBuilder(2).append('\ud800').toString();
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(surrogateString.getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)surrogateString);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        Assertions.assertEquals((int)buf.readableBytes(), (int)ByteBufUtil.utf8Bytes((CharSequence)surrogateString));
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8InvalidEndOnTrailingSurrogate(BufferType bufferType) {
        String surrogateString = new StringBuilder(2).append('\udc00').toString();
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(surrogateString.getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)surrogateString);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        Assertions.assertEquals((int)buf.readableBytes(), (int)ByteBufUtil.utf8Bytes((CharSequence)surrogateString));
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUsAsciiString(BufferType bufferType) {
        AsciiString usAscii = new AsciiString((CharSequence)"NettyRocks");
        int expectedCapacity = usAscii.length();
        ByteBuf buf = this.buffer(bufferType, expectedCapacity);
        buf.writeBytes(usAscii.toString().getBytes(CharsetUtil.US_ASCII));
        ByteBuf buf2 = this.buffer(bufferType, expectedCapacity);
        ByteBufUtil.writeAscii((ByteBuf)buf2, (CharSequence)usAscii);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8Wrapped(BufferType bufferType) {
        String usAscii = "Some UTF-8 like \u00e4\u00c4\u220f\u0152\u0152";
        ByteBuf buf = Unpooled.unreleasableBuffer((ByteBuf)this.buffer(bufferType, 16));
        ByteBufUtilTest.assertWrapped(buf);
        buf.writeBytes(usAscii.getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = Unpooled.unreleasableBuffer((ByteBuf)this.buffer(bufferType, 16));
        ByteBufUtilTest.assertWrapped(buf2);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)usAscii);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        buf.unwrap().release();
        buf2.unwrap().release();
    }

    private static void assertWrapped(ByteBuf buf) {
        MatcherAssert.assertThat((Object)buf, (Matcher)Matchers.instanceOf(WrappedByteBuf.class));
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8Subsequence(BufferType bufferType) {
        String usAscii = "Some UTF-8 like \u00e4\u00c4\u220f\u0152\u0152";
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(usAscii.substring(5, 18).getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)usAscii, (int)5, (int)18);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8SubsequenceSplitSurrogate(BufferType bufferType) {
        String usAscii = "\ud800\udc00";
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(usAscii.substring(0, 1).getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        ByteBufUtil.writeUtf8((ByteBuf)buf2, (CharSequence)usAscii, (int)0, (int)1);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        buf.release();
        buf2.release();
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testReserveAndWriteUtf8Subsequence(BufferType bufferType) {
        String usAscii = "Some UTF-8 like \u00e4\u00c4\u220f\u0152\u0152";
        ByteBuf buf = this.buffer(bufferType, 16);
        buf.writeBytes(usAscii.substring(5, 18).getBytes(CharsetUtil.UTF_8));
        ByteBuf buf2 = this.buffer(bufferType, 16);
        int count = ByteBufUtil.reserveAndWriteUtf8((ByteBuf)buf2, (CharSequence)usAscii, (int)5, (int)18, (int)16);
        Assertions.assertEquals((Object)buf, (Object)buf2);
        Assertions.assertEquals((int)buf.readableBytes(), (int)count);
        buf.release();
        buf2.release();
    }

    @Test
    public void testUtf8BytesSubsequence() {
        String usAscii = "Some UTF-8 like \u00e4\u00c4\u220f\u0152\u0152";
        Assertions.assertEquals((int)usAscii.substring(5, 18).getBytes(CharsetUtil.UTF_8).length, (int)ByteBufUtil.utf8Bytes((CharSequence)usAscii, (int)5, (int)18));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testInvalidSubsequences(BufferType bufferType, TestMethod method) {
        for (int[] range : INVALID_RANGES) {
            ByteBuf buf = this.buffer(bufferType, 16);
            try {
                method.invoke(buf, "Some UTF-8 like \u00e4\u00c4\u220f\u0152\u0152", range[0], range[1]);
                Assertions.fail((String)("Did not throw IndexOutOfBoundsException for range (" + range[0] + ", " + range[1] + ")"));
            }
            catch (IndexOutOfBoundsException indexOutOfBoundsException) {
            }
            finally {
                Assertions.assertFalse((boolean)buf.isReadable());
                buf.release();
            }
        }
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testWriteUtf8InvalidSubsequences(BufferType bufferType) {
        this.testInvalidSubsequences(bufferType, new TestMethod(){

            @Override
            public int invoke(Object ... args) {
                return ByteBufUtil.writeUtf8((ByteBuf)((ByteBuf)args[0]), (CharSequence)((String)args[1]), (int)((Integer)args[2]), (int)((Integer)args[3]));
            }
        });
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testReserveAndWriteUtf8InvalidSubsequences(BufferType bufferType) {
        this.testInvalidSubsequences(bufferType, new TestMethod(){

            @Override
            public int invoke(Object ... args) {
                return ByteBufUtil.reserveAndWriteUtf8((ByteBuf)((ByteBuf)args[0]), (CharSequence)((String)args[1]), (int)((Integer)args[2]), (int)((Integer)args[3]), (int)32);
            }
        });
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testUtf8BytesInvalidSubsequences(BufferType bufferType) {
        this.testInvalidSubsequences(bufferType, new TestMethod(){

            @Override
            public int invoke(Object ... args) {
                return ByteBufUtil.utf8Bytes((CharSequence)((String)args[1]), (int)((Integer)args[2]), (int)((Integer)args[3]));
            }
        });
    }

    @Test
    public void testDecodeUsAscii() {
        ByteBufUtilTest.testDecodeString("This is a test", CharsetUtil.US_ASCII);
    }

    @Test
    public void testDecodeUtf8() {
        ByteBufUtilTest.testDecodeString("Some UTF-8 like \u00e4\u00c4\u220f\u0152\u0152", CharsetUtil.UTF_8);
    }

    private static void testDecodeString(String text, Charset charset) {
        ByteBuf buffer = Unpooled.copiedBuffer((CharSequence)text, (Charset)charset);
        Assertions.assertEquals((Object)text, (Object)ByteBufUtil.decodeString((ByteBuf)buffer, (int)0, (int)buffer.readableBytes(), (Charset)charset));
        buffer.release();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testToStringDoesNotThrowIndexOutOfBounds(BufferType bufferType) {
        CompositeByteBuf buffer = Unpooled.compositeBuffer();
        try {
            byte[] bytes = "1234".getBytes(CharsetUtil.UTF_8);
            buffer.addComponent(this.buffer(bufferType, bytes.length).writeBytes(bytes));
            buffer.addComponent(this.buffer(bufferType, bytes.length).writeBytes(bytes));
            Assertions.assertEquals((Object)"1234", (Object)buffer.toString(bytes.length, bytes.length, CharsetUtil.UTF_8));
        }
        finally {
            buffer.release();
        }
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testIsTextWithUtf8(BufferType bufferType) {
        byte[][] invalidUtf8Bytes;
        byte[][] validUtf8Bytes;
        for (byte[] bytes : validUtf8Bytes = new byte[][]{"netty".getBytes(CharsetUtil.UTF_8), {36}, {-62, -94}, {-30, -126, -84}, {-16, -112, -115, -120}, {36, -62, -94, -30, -126, -84, -16, -112, -115, -120}}) {
            this.assertIsText(bufferType, bytes, true, CharsetUtil.UTF_8);
        }
        for (byte[] bytes : invalidUtf8Bytes = new byte[][]{{-128}, {-16, -126, -126, -84}, {-62}, {-30, -126}, {-16, -112, -115}, {-62, -64}, {-30, -126, -64}, {-16, -112, -115, -64}, {-63, -128}, {-32, -128, -128}, {-19, -81, -128}}) {
            this.assertIsText(bufferType, bytes, false, CharsetUtil.UTF_8);
        }
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testIsTextWithoutOptimization(BufferType bufferType) {
        byte[] validBytes = new byte[]{1, -40, 55, -36};
        byte[] invalidBytes = new byte[]{1, -40};
        this.assertIsText(bufferType, validBytes, true, CharsetUtil.UTF_16LE);
        this.assertIsText(bufferType, invalidBytes, false, CharsetUtil.UTF_16LE);
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testIsTextWithAscii(BufferType bufferType) {
        byte[] validBytes = new byte[]{0, 1, 55, 127};
        byte[] invalidBytes = new byte[]{-128, -1};
        this.assertIsText(bufferType, validBytes, true, CharsetUtil.US_ASCII);
        this.assertIsText(bufferType, invalidBytes, false, CharsetUtil.US_ASCII);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testIsTextWithInvalidIndexAndLength(BufferType bufferType) {
        ByteBuf buffer = this.buffer(bufferType, 4);
        try {
            int[][] invalidIndexLengthPairs;
            int[][] validIndexLengthPairs;
            buffer.writeBytes(new byte[4]);
            for (int[] pair : validIndexLengthPairs = new int[][]{{4, 0}, {0, 4}, {1, 3}}) {
                Assertions.assertTrue((boolean)ByteBufUtil.isText((ByteBuf)buffer, (int)pair[0], (int)pair[1], (Charset)CharsetUtil.US_ASCII));
            }
            for (int[] pair : invalidIndexLengthPairs = new int[][]{{4, 1}, {-1, 2}, {3, -1}, {3, -2}, {5, 0}, {1, 5}}) {
                try {
                    ByteBufUtil.isText((ByteBuf)buffer, (int)pair[0], (int)pair[1], (Charset)CharsetUtil.US_ASCII);
                    Assertions.fail((String)"Expected IndexOutOfBoundsException");
                }
                catch (IndexOutOfBoundsException indexOutOfBoundsException) {
                    // empty catch block
                }
            }
        }
        finally {
            buffer.release();
        }
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testUtf8Bytes(BufferType bufferType) {
        String s = "Some UTF-8 like \u00e4\u00c4\u220f\u0152\u0152";
        this.checkUtf8Bytes(bufferType, "Some UTF-8 like \u00e4\u00c4\u220f\u0152\u0152");
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testUtf8BytesWithSurrogates(BufferType bufferType) {
        String s = "a\ud800\udc00b";
        this.checkUtf8Bytes(bufferType, "a\ud800\udc00b");
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testUtf8BytesWithNonSurrogates3Bytes(BufferType bufferType) {
        String s = "a\ue000b";
        this.checkUtf8Bytes(bufferType, "a\ue000b");
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testUtf8BytesWithNonSurrogatesNonAscii(BufferType bufferType) {
        int nonAscii = 129;
        String s = "a\u0081b";
        this.checkUtf8Bytes(bufferType, "a\u0081b");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkUtf8Bytes(BufferType bufferType, CharSequence charSequence) {
        ByteBuf buf = this.buffer(bufferType, ByteBufUtil.utf8MaxBytes((CharSequence)charSequence));
        try {
            int writtenBytes = ByteBufUtil.writeUtf8((ByteBuf)buf, (CharSequence)charSequence);
            int utf8Bytes = ByteBufUtil.utf8Bytes((CharSequence)charSequence);
            Assertions.assertEquals((int)writtenBytes, (int)utf8Bytes);
        }
        finally {
            buf.release();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void assertIsText(BufferType bufferType, byte[] bytes, boolean expected, Charset charset) {
        ByteBuf buffer = this.buffer(bufferType, bytes.length);
        try {
            buffer.writeBytes(bytes);
            Assertions.assertEquals((Object)expected, (Object)ByteBufUtil.isText((ByteBuf)buffer, (Charset)charset));
        }
        finally {
            buffer.release();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testIsTextMultiThreaded(BufferType bufferType) throws Throwable {
        Assumptions.assumeTrue((bufferType == BufferType.HEAP_UNPOOLED ? 1 : 0) != 0);
        final ByteBuf buffer = Unpooled.copiedBuffer((CharSequence)"Hello, World!", (Charset)CharsetUtil.ISO_8859_1);
        try {
            final AtomicInteger counter = new AtomicInteger(60000);
            final AtomicReference errorRef = new AtomicReference();
            ArrayList<Thread> threads = new ArrayList<Thread>();
            for (int i = 0; i < 10; ++i) {
                Thread thread = new Thread(new Runnable(){

                    @Override
                    public void run() {
                        try {
                            while (errorRef.get() == null && counter.decrementAndGet() > 0) {
                                Assertions.assertTrue((boolean)ByteBufUtil.isText((ByteBuf)buffer, (Charset)CharsetUtil.ISO_8859_1));
                            }
                        }
                        catch (Throwable cause) {
                            errorRef.compareAndSet(null, cause);
                        }
                    }
                });
                threads.add(thread);
            }
            for (Thread thread : threads) {
                thread.start();
            }
            for (Thread thread : threads) {
                thread.join();
            }
            Throwable error = (Throwable)errorRef.get();
            if (error != null) {
                throw error;
            }
        }
        finally {
            buffer.release();
        }
    }

    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testGetBytes(BufferType bufferType) {
        ByteBuf buf = this.buffer(bufferType, 4);
        try {
            ByteBufUtilTest.checkGetBytes(buf);
        }
        finally {
            buf.release();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testGetBytesHeapWithNonZeroArrayOffset(BufferType bufferType) {
        Assumptions.assumeTrue((bufferType == BufferType.HEAP_UNPOOLED ? 1 : 0) != 0);
        ByteBuf buf = this.buffer(bufferType, 5);
        try {
            buf.setByte(0, 5);
            ByteBuf slice = buf.slice(1, 4);
            slice.writerIndex(0);
            Assertions.assertTrue((boolean)slice.hasArray());
            MatcherAssert.assertThat((Object)slice.arrayOffset(), (Matcher)Matchers.is((Object)1));
            MatcherAssert.assertThat((Object)slice.array().length, (Matcher)Matchers.is((Object)buf.capacity()));
            ByteBufUtilTest.checkGetBytes(slice);
        }
        finally {
            buf.release();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @ParameterizedTest(name="bufferType = {0}")
    @MethodSource(value={"noUnsafe"})
    public void testGetBytesHeapWithArrayLengthGreaterThanCapacity(BufferType bufferType) {
        Assumptions.assumeTrue((bufferType == BufferType.HEAP_UNPOOLED ? 1 : 0) != 0);
        ByteBuf buf = this.buffer(bufferType, 5);
        try {
            buf.setByte(4, 5);
            ByteBuf slice = buf.slice(0, 4);
            slice.writerIndex(0);
            Assertions.assertTrue((boolean)slice.hasArray());
            MatcherAssert.assertThat((Object)slice.arrayOffset(), (Matcher)Matchers.is((Object)0));
            MatcherAssert.assertThat((Object)slice.array().length, (Matcher)Matchers.greaterThan((Comparable)Integer.valueOf(slice.capacity())));
            ByteBufUtilTest.checkGetBytes(slice);
        }
        finally {
            buf.release();
        }
    }

    private static void checkGetBytes(ByteBuf buf) {
        buf.writeInt(16909060);
        byte[] expected = new byte[]{1, 2, 3, 4};
        Assertions.assertArrayEquals((byte[])expected, (byte[])ByteBufUtil.getBytes((ByteBuf)buf));
        Assertions.assertArrayEquals((byte[])expected, (byte[])ByteBufUtil.getBytes((ByteBuf)buf, (int)0, (int)buf.readableBytes(), (boolean)false));
        expected = new byte[]{1, 2, 3};
        Assertions.assertArrayEquals((byte[])expected, (byte[])ByteBufUtil.getBytes((ByteBuf)buf, (int)0, (int)3));
        Assertions.assertArrayEquals((byte[])expected, (byte[])ByteBufUtil.getBytes((ByteBuf)buf, (int)0, (int)3, (boolean)false));
        expected = new byte[]{2, 3, 4};
        Assertions.assertArrayEquals((byte[])expected, (byte[])ByteBufUtil.getBytes((ByteBuf)buf, (int)1, (int)3));
        Assertions.assertArrayEquals((byte[])expected, (byte[])ByteBufUtil.getBytes((ByteBuf)buf, (int)1, (int)3, (boolean)false));
        expected = new byte[]{2, 3};
        Assertions.assertArrayEquals((byte[])expected, (byte[])ByteBufUtil.getBytes((ByteBuf)buf, (int)1, (int)2));
        Assertions.assertArrayEquals((byte[])expected, (byte[])ByteBufUtil.getBytes((ByteBuf)buf, (int)1, (int)2, (boolean)false));
    }

    static interface TestMethod {
        public int invoke(Object ... var1);
    }

    private static enum BufferType {
        DIRECT_UNPOOLED,
        DIRECT_POOLED,
        HEAP_POOLED,
        HEAP_UNPOOLED;

    }
}

