/*
 * Copyright 2015 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/*
 * Copyright 2014 Twitter, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.netty5.handler.codec.http2;

import io.netty5.buffer.Buffer;
import io.netty5.util.AsciiString;
import io.netty5.util.ByteProcessor;

import static java.util.Objects.requireNonNull;

final class HpackHuffmanEncoder {

    private final int[] codes;
    private final byte[] lengths;
    private final EncodedLengthProcessor encodedLengthProcessor = new EncodedLengthProcessor();
    private final EncodeProcessor encodeProcessor = new EncodeProcessor();

    HpackHuffmanEncoder() {
        this(HpackUtil.HUFFMAN_CODES, HpackUtil.HUFFMAN_CODE_LENGTHS);
    }

    /**
     * Creates a new Huffman encoder with the specified Huffman coding.
     *
     * @param codes the Huffman codes indexed by symbol
     * @param lengths the length of each Huffman code
     */
    private HpackHuffmanEncoder(int[] codes, byte[] lengths) {
        this.codes = codes;
        this.lengths = lengths;
    }

    /**
     * Compresses the input string literal using the Huffman coding.
     *
     * @param out the output stream for the compressed data
     * @param data the string literal to be Huffman encoded
     */
    public void encode(Buffer out, CharSequence data) {
        requireNonNull(out, "out");
        if (data instanceof AsciiString) {
            AsciiString string = (AsciiString) data;
            encodeProcessor.out = out;
            try {
                string.forEachByte(encodeProcessor);
            } finally {
                encodeProcessor.end();
            }
        } else {
            encodeSlowPath(out, data);
        }
    }

    private void encodeSlowPath(Buffer out, CharSequence data) {
        long current = 0;
        int n = 0;

        for (int i = 0; i < data.length(); i++) {
            int b = data.charAt(i) & 0xFF;
            int code = codes[b];
            int nbits = lengths[b];

            current <<= nbits;
            current |= code;
            n += nbits;

            while (n >= 8) {
                n -= 8;
                out.writeByte((byte) (current >> n));
            }
        }

        if (n > 0) {
            current <<= 8 - n;
            current |= 0xFF >>> n; // this should be EOS symbol
            out.writeByte((byte) current);
        }
    }

    /**
     * Returns the number of bytes required to Huffman encode the input string literal.
     *
     * @param data the string literal to be Huffman encoded
     * @return the number of bytes required to Huffman encode {@code data}
     */
    int getEncodedLength(CharSequence data) {
        if (data instanceof AsciiString) {
            AsciiString string = (AsciiString) data;
            encodedLengthProcessor.reset();
            string.forEachByte(encodedLengthProcessor);
            return encodedLengthProcessor.length();
        } else {
            return getEncodedLengthSlowPath(data);
        }
    }

    private int getEncodedLengthSlowPath(CharSequence data) {
        long len = 0;
        for (int i = 0; i < data.length(); i++) {
            len += lengths[data.charAt(i) & 0xFF];
        }
        return (int) (len + 7 >> 3);
    }

    private final class EncodeProcessor implements ByteProcessor {
        Buffer out;
        private long current;
        private int n;

        @Override
        public boolean process(byte value) {
            int b = value & 0xFF;
            int nbits = lengths[b];

            current <<= nbits;
            current |= codes[b];
            n += nbits;

            while (n >= 8) {
                n -= 8;
                out.writeByte((byte) (current >> n));
            }
            return true;
        }

        void end() {
            try {
                if (n > 0) {
                    current <<= 8 - n;
                    current |= 0xFF >>> n; // this should be EOS symbol
                    out.writeByte((byte) current);
                }
            } finally {
                out = null;
                current = 0;
                n = 0;
            }
        }
    }

    private final class EncodedLengthProcessor implements ByteProcessor {
        private long len;

        @Override
        public boolean process(byte value) {
            len += lengths[value & 0xFF];
            return true;
        }

        void reset() {
            len = 0;
        }

        int length() {
            return (int) (len + 7 >> 3);
        }
    }
}
