/*
 * Decompiled with CFR 0.152.
 */
package com.upplication.s3fs;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AbortMultipartUploadRequest;
import com.amazonaws.services.s3.model.CompleteMultipartUploadRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadResult;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3ObjectId;
import com.amazonaws.services.s3.model.StorageClass;
import com.amazonaws.services.s3.model.UploadPartRequest;
import com.amazonaws.util.Base64;
import com.upplication.s3fs.util.ByteBufferInputStream;
import com.upplication.s3fs.util.S3UploadRequest;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.Phaser;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class S3OutputStream
extends OutputStream {
    private static final Logger log = LoggerFactory.getLogger(S3OutputStream.class);
    private final AmazonS3 s3;
    private final S3ObjectId objectId;
    private final StorageClass storageClass;
    private final ObjectMetadata metadata;
    private volatile boolean closed;
    private volatile boolean aborted;
    private volatile String uploadId;
    private Queue<PartETag> partETags;
    private final S3UploadRequest request;
    private final Queue<ByteBuffer> bufferPool = new ConcurrentLinkedQueue<ByteBuffer>();
    private ExecutorService executor;
    private ByteBuffer buf;
    private MessageDigest md5;
    private Phaser phaser;
    private int partsCount;
    private int chunkSize;
    private static volatile ExecutorService executorSingleton;

    public S3OutputStream(AmazonS3 s3, S3ObjectId objectId) {
        this(s3, new S3UploadRequest().setObjectId(objectId));
    }

    public S3OutputStream(AmazonS3 s3, S3UploadRequest request) {
        this.s3 = Objects.requireNonNull(s3);
        this.objectId = Objects.requireNonNull(request.getObjectId());
        this.metadata = request.getMetadata() != null ? request.getMetadata() : new ObjectMetadata();
        this.storageClass = request.getStorageClass();
        this.request = request;
        this.chunkSize = request.getChunkSize();
    }

    private ByteBuffer expandBuffer(ByteBuffer byteBuffer) {
        float expandFactor = 2.5f;
        int newCapacity = Math.min((int)((float)byteBuffer.capacity() * 2.5f), this.chunkSize);
        byteBuffer.flip();
        ByteBuffer expanded = ByteBuffer.allocate(newCapacity);
        expanded.order(byteBuffer.order());
        expanded.put(byteBuffer);
        return expanded;
    }

    private MessageDigest createMd5() {
        try {
            return MessageDigest.getInstance("MD5");
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Cannot find a MD5 algorithm provider", e);
        }
    }

    @Override
    public void write(int b) throws IOException {
        if (this.buf == null) {
            this.buf = this.allocate();
            this.md5 = this.createMd5();
        } else if (!this.buf.hasRemaining()) {
            if (this.buf.position() < this.chunkSize) {
                this.buf = this.expandBuffer(this.buf);
            } else {
                this.flush();
                this.buf = this.allocate();
                this.md5 = this.createMd5();
            }
        }
        this.buf.put((byte)b);
        this.md5.update((byte)b);
    }

    @Override
    public void flush() throws IOException {
        this.uploadBuffer(this.buf);
        this.buf = null;
        this.md5 = null;
    }

    private ByteBuffer allocate() {
        if (this.partsCount == 0) {
            return ByteBuffer.allocate(10240);
        }
        ByteBuffer result = this.bufferPool.poll();
        if (result != null) {
            result.clear();
        } else {
            result = ByteBuffer.allocateDirect(this.request.getChunkSize());
        }
        return result;
    }

    private void uploadBuffer(ByteBuffer buf) throws IOException {
        if (buf == null || buf.position() == 0) {
            return;
        }
        if (this.partsCount == 0) {
            this.init();
        }
        this.executor.submit(this.task(buf, this.md5.digest(), ++this.partsCount));
    }

    private void init() throws IOException {
        this.uploadId = this.initiateMultipartUpload().getUploadId();
        if (this.uploadId == null) {
            throw new IOException("Failed to get a valid multipart upload ID from Amazon S3");
        }
        this.executor = S3OutputStream.getOrCreateExecutor(this.request.getMaxThreads());
        this.partETags = new LinkedBlockingQueue<PartETag>();
        this.phaser = new Phaser();
        this.phaser.register();
        log.trace("Starting S3 upload: {}; chunk-size: {}; max-threads: {}", new Object[]{this.uploadId, this.request.getChunkSize(), this.request.getMaxThreads()});
    }

    private Runnable task(final ByteBuffer buffer, final byte[] checksum, final int partIndex) {
        this.phaser.register();
        return new Runnable(){

            @Override
            public void run() {
                try {
                    S3OutputStream.this.uploadPart(buffer, checksum, partIndex, false);
                }
                catch (IOException e) {
                    StringWriter writer = new StringWriter();
                    e.printStackTrace(new PrintWriter(writer));
                    log.error("Upload: {} > Error for part: {}\nCaused by: {}", new Object[]{S3OutputStream.this.uploadId, partIndex, writer.toString()});
                }
                finally {
                    S3OutputStream.this.phaser.arriveAndDeregister();
                }
            }
        };
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        if (this.uploadId == null) {
            if (this.buf != null) {
                this.putObject(this.buf, this.md5.digest());
            } else {
                this.putObject(new ByteArrayInputStream(new byte[0]), 0L, this.createMd5().digest());
            }
        } else {
            if (this.buf != null) {
                this.uploadBuffer(this.buf);
            }
            this.phaser.arriveAndAwaitAdvance();
            this.completeMultipartUpload();
        }
        this.closed = true;
    }

    private InitiateMultipartUploadResult initiateMultipartUpload() throws IOException {
        InitiateMultipartUploadRequest request = new InitiateMultipartUploadRequest(this.objectId.getBucket(), this.objectId.getKey(), this.metadata);
        if (this.storageClass != null) {
            request.setStorageClass(this.storageClass);
        }
        try {
            return this.s3.initiateMultipartUpload(request);
        }
        catch (AmazonClientException e) {
            throw new IOException("Failed to initiate Amazon S3 multipart upload", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void uploadPart(ByteBuffer buf, byte[] checksum, int partNumber, boolean lastPart) throws IOException {
        block9: {
            buf.flip();
            buf.mark();
            int attempt = 0;
            boolean success = false;
            block5: while (true) {
                while (!success) {
                    ++attempt;
                    int len = buf.limit();
                    try {
                        log.trace("Uploading part {} with length {} attempt {} for {} ", new Object[]{partNumber, len, attempt, this.objectId});
                        this.uploadPart(new ByteBufferInputStream(buf), len, checksum, partNumber, lastPart);
                        success = true;
                        continue block5;
                    }
                    catch (AmazonClientException | IOException e) {
                        if (attempt == this.request.getMaxAttempts()) {
                            throw new IOException("Failed to upload multipart data to Amazon S3", e);
                        }
                        log.debug("Failed to upload part {} attempt {} for {} -- Caused by: {}", new Object[]{partNumber, attempt, this.objectId, e.getMessage()});
                        this.sleep(this.request.getRetrySleep());
                        buf.reset();
                    }
                }
                break block9;
                {
                    continue block5;
                    break;
                }
                break;
            }
            finally {
                if (!success) {
                    this.closed = true;
                    this.abortMultipartUpload();
                }
                this.bufferPool.offer(buf);
            }
        }
    }

    private void uploadPart(InputStream content, long contentLength, byte[] checksum, int partNumber, boolean lastPart) throws IOException {
        if (this.aborted) {
            return;
        }
        UploadPartRequest request = new UploadPartRequest();
        request.setBucketName(this.objectId.getBucket());
        request.setKey(this.objectId.getKey());
        request.setUploadId(this.uploadId);
        request.setPartNumber(partNumber);
        request.setPartSize(contentLength);
        request.setInputStream(content);
        request.setLastPart(lastPart);
        request.setMd5Digest(Base64.encodeAsString((byte[])checksum));
        PartETag partETag = this.s3.uploadPart(request).getPartETag();
        log.trace("Uploaded part {} with length {} for {}: {}", new Object[]{partETag.getPartNumber(), contentLength, this.objectId, partETag.getETag()});
        this.partETags.add(partETag);
    }

    private void sleep(long millis) {
        try {
            Thread.sleep(millis);
        }
        catch (InterruptedException e) {
            log.trace("Sleep was interrupted -- Cause: {}", (Object)e.getMessage());
        }
    }

    private synchronized void abortMultipartUpload() {
        if (this.aborted) {
            return;
        }
        log.debug("Aborting multipart upload {} for {}", (Object)this.uploadId, (Object)this.objectId);
        try {
            this.s3.abortMultipartUpload(new AbortMultipartUploadRequest(this.objectId.getBucket(), this.objectId.getKey(), this.uploadId));
        }
        catch (AmazonClientException e) {
            log.warn("Failed to abort multipart upload {}: {}", (Object)this.uploadId, (Object)e.getMessage());
        }
        this.aborted = true;
        this.phaser.arriveAndDeregister();
    }

    private void completeMultipartUpload() throws IOException {
        if (this.aborted) {
            return;
        }
        int partCount = this.partETags.size();
        log.trace("Completing upload to {} consisting of {} parts", (Object)this.objectId, (Object)partCount);
        try {
            this.s3.completeMultipartUpload(new CompleteMultipartUploadRequest(this.objectId.getBucket(), this.objectId.getKey(), this.uploadId, new ArrayList<PartETag>(this.partETags)));
        }
        catch (AmazonClientException e) {
            throw new IOException("Failed to complete Amazon S3 multipart upload", e);
        }
        log.trace("Completed upload to {} consisting of {} parts", (Object)this.objectId, (Object)partCount);
        this.uploadId = null;
        this.partETags = null;
    }

    private void putObject(ByteBuffer buf, byte[] checksum) throws IOException {
        buf.flip();
        this.putObject(new ByteBufferInputStream(buf), buf.limit(), checksum);
    }

    private void putObject(InputStream content, long contentLength, byte[] checksum) throws IOException {
        ObjectMetadata meta = this.metadata.clone();
        meta.setContentLength(contentLength);
        meta.setContentMD5(Base64.encodeAsString((byte[])checksum));
        PutObjectRequest request = new PutObjectRequest(this.objectId.getBucket(), this.objectId.getKey(), content, meta);
        if (this.storageClass != null) {
            request.setStorageClass(this.storageClass);
        }
        try {
            this.s3.putObject(request);
        }
        catch (AmazonClientException e) {
            throw new IOException("Failed to put data into Amazon S3 object", e);
        }
    }

    int getPartsCount() {
        return this.partsCount;
    }

    static synchronized ExecutorService getOrCreateExecutor(int maxThreads) {
        if (executorSingleton == null) {
            ThreadPoolExecutor pool = new ThreadPoolExecutor(maxThreads, Integer.MAX_VALUE, 60L, TimeUnit.SECONDS, new LimitedQueue<Runnable>(maxThreads * 3), new ThreadPoolExecutor.CallerRunsPolicy());
            pool.allowCoreThreadTimeOut(true);
            executorSingleton = pool;
            log.trace("Created singleton upload executor -- max-treads: {}", (Object)maxThreads);
        }
        return executorSingleton;
    }

    public static synchronized void shutdownExecutor() {
        log.trace("Uploader shutdown -- Executor: {}", (Object)executorSingleton);
        if (executorSingleton != null) {
            executorSingleton.shutdown();
            log.trace("Uploader await completion");
            S3OutputStream.awaitExecutorCompletion();
            executorSingleton = null;
            log.trace("Uploader shutdown completed");
        }
    }

    private static void awaitExecutorCompletion() {
        try {
            executorSingleton.awaitTermination(Long.MAX_VALUE, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            log.trace("Executor await interrupted -- Cause: {}", (Object)e.getMessage());
        }
    }

    static class LimitedQueue<E>
    extends LinkedBlockingQueue<E> {
        public LimitedQueue(int maxSize) {
            super(maxSize);
        }

        @Override
        public boolean offer(E e) {
            try {
                this.put(e);
                return true;
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                return false;
            }
        }
    }
}

