/*
 * Decompiled with CFR 0.152.
 */
package io.nflow.engine.service;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.nflow.engine.internal.dao.MaintenanceDao;
import io.nflow.engine.internal.dao.NflowTable;
import io.nflow.engine.internal.dao.TableMetadataChecker;
import io.nflow.engine.internal.dao.TableType;
import io.nflow.engine.internal.util.PeriodicLogger;
import io.nflow.engine.service.MaintenanceConfiguration;
import io.nflow.engine.service.MaintenanceResults;
import io.nflow.engine.service.WorkflowDefinitionService;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Stream;
import javax.inject.Inject;
import javax.inject.Named;
import org.apache.commons.lang3.time.StopWatch;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Named
public class MaintenanceService {
    private static final Logger log = LoggerFactory.getLogger(MaintenanceService.class);
    private final MaintenanceDao maintenanceDao;
    private final TableMetadataChecker tableMetadataChecker;
    private final WorkflowDefinitionService workflowDefinitionService;

    @Inject
    public MaintenanceService(MaintenanceDao maintenanceDao, TableMetadataChecker tableMetadataChecker, WorkflowDefinitionService workflowDefinitionService) {
        this.maintenanceDao = maintenanceDao;
        this.tableMetadataChecker = tableMetadataChecker;
        this.workflowDefinitionService = workflowDefinitionService;
    }

    @SuppressFBWarnings(value={"BAS_BLOATED_ASSIGNMENT_SCOPE"}, justification="periodicLogger is defined in correct scope")
    public MaintenanceResults cleanupWorkflows(MaintenanceConfiguration configuration) {
        this.validateConfiguration(configuration);
        if (configuration.archiveWorkflows != null || configuration.deleteArchivedWorkflows != null) {
            Arrays.stream(NflowTable.values()).forEach(table -> this.tableMetadataChecker.ensureCopyingPossible(table.main, table.archive));
        }
        MaintenanceResults.Builder builder = new MaintenanceResults.Builder();
        if (configuration.deleteArchivedWorkflows != null) {
            builder.setDeletedArchivedWorkflows(this.doAction("Deleting archived workflows", configuration.deleteArchivedWorkflows, TableType.ARCHIVE, idList -> this.maintenanceDao.deleteWorkflows(TableType.ARCHIVE, (Collection<Long>)idList)));
        }
        if (configuration.archiveWorkflows != null) {
            builder.setArchivedWorkflows(this.doAction("Archiving workflows", configuration.archiveWorkflows, TableType.MAIN, this.maintenanceDao::archiveWorkflows));
        }
        if (configuration.deleteWorkflows != null) {
            builder.setDeletedWorkflows(this.doAction("Deleting workflows", configuration.deleteWorkflows, TableType.MAIN, idList -> this.maintenanceDao.deleteWorkflows(TableType.MAIN, (Collection<Long>)idList)));
        }
        return builder.build();
    }

    private void validateConfiguration(MaintenanceConfiguration configuration) {
        Stream.of(configuration.archiveWorkflows, configuration.deleteArchivedWorkflows, configuration.deleteWorkflows).filter(Objects::nonNull).flatMap(configItem -> configItem.workflowTypes.stream()).filter(workflowType -> this.workflowDefinitionService.getWorkflowDefinition((String)workflowType) == null).findAny().ifPresent(workflowType -> {
            throw new IllegalArgumentException("Workflow type " + workflowType + " is not valid");
        });
    }

    @SuppressFBWarnings(value={"BAS_BLOATED_ASSIGNMENT_SCOPE"}, justification="periodLogger scope is correct")
    private int doAction(String type, MaintenanceConfiguration.ConfigurationItem configuration, TableType tableType, Function<List<Long>, Integer> doAction) {
        List<Long> workflowIds;
        DateTime olderThan = DateTime.now().minus(configuration.olderThanPeriod);
        log.info("{} older than {}, in batches of {}.", new Object[]{type, olderThan, configuration.batchSize});
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        PeriodicLogger periodicLogger = new PeriodicLogger(log, 60);
        int totalWorkflows = 0;
        while (!(workflowIds = this.maintenanceDao.getOldWorkflowIds(tableType, olderThan, configuration.batchSize, configuration.workflowTypes)).isEmpty()) {
            int workflows = doAction.apply(workflowIds);
            double timeDiff = Math.max((double)stopWatch.getTime() / 1000.0, 1.0E-6);
            String status = String.format("%s. %s workflows, %.1f workflows / second.", type, workflows, (double)(totalWorkflows += workflows) / timeDiff);
            log.debug("{} Workflow ids: {}.", (Object)status, workflowIds);
            periodicLogger.info(status, new Object[0]);
        }
        log.info("{} finished. Operated on {} workflows in {} seconds.", new Object[]{type, totalWorkflows, stopWatch.getTime() / 1000L});
        return totalWorkflows;
    }
}

