/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.dse.search;

import com.datastax.driver.core.ParseUtils;
import com.datastax.internal.com_google_common.base.Preconditions;
import com.datastax.internal.com_google_common.collect.ImmutableMap;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

public class DateRange {
    private final DateRangeBound lowerBound;
    private final DateRangeBound upperBound;

    public static DateRange parse(String source) throws ParseException {
        Preconditions.checkNotNull(source);
        if (DateRange.isUnbounded(source)) {
            return new DateRange(DateRangeBound.UNBOUNDED);
        }
        if (source.charAt(0) == '[') {
            DateRangeBound upperBound;
            DateRangeBound lowerBound;
            if (source.charAt(source.length() - 1) != ']') {
                throw new ParseException("If date range starts with '[' it must end with ']'; got " + source, source.length() - 1);
            }
            int middle = source.indexOf(" TO ");
            if (middle < 0) {
                throw new ParseException("If date range starts with '[' it must contain ' TO '; got " + source, 0);
            }
            String lowerBoundStr = source.substring(1, middle);
            int upperBoundStart = middle + 4;
            String upperBoundStr = source.substring(upperBoundStart, source.length() - 1);
            try {
                lowerBound = DateRangeBound.parseLowerBound(lowerBoundStr);
            }
            catch (Exception e) {
                throw new ParseException("Cannot parse date range lower bound: " + source, 1);
            }
            try {
                upperBound = DateRangeBound.parseUpperBound(upperBoundStr);
            }
            catch (Exception e) {
                throw new ParseException("Cannot parse date range upper bound: " + source, upperBoundStart);
            }
            return new DateRange(lowerBound, upperBound);
        }
        try {
            return new DateRange(DateRangeBound.parseLowerBound(source));
        }
        catch (Exception e) {
            throw new ParseException("Cannot parse single date range bound: " + source, 0);
        }
    }

    private static boolean isUnbounded(String source) {
        return "*".equals(source);
    }

    public DateRange(DateRangeBound singleBound) {
        Preconditions.checkArgument(singleBound != null, "singleBound cannot be null");
        this.lowerBound = singleBound;
        this.upperBound = null;
    }

    public DateRange(DateRangeBound lowerBound, DateRangeBound upperBound) {
        Preconditions.checkArgument(lowerBound != null, "lowerBound cannot be null");
        Preconditions.checkArgument(upperBound != null, "upperBound cannot be null");
        if (!lowerBound.isUnbounded() && !upperBound.isUnbounded() && lowerBound.timestamp.after(upperBound.timestamp)) {
            throw new IllegalArgumentException(String.format("Lower bound of a date range should be before upper bound, got: [%s TO %s]", lowerBound, upperBound));
        }
        this.lowerBound = lowerBound;
        this.upperBound = upperBound;
    }

    public DateRangeBound getLowerBound() {
        return this.lowerBound;
    }

    public DateRangeBound getUpperBound() {
        return this.upperBound;
    }

    public boolean isSingleBounded() {
        return this.upperBound == null;
    }

    public String toString() {
        if (this.isSingleBounded()) {
            return this.lowerBound.toString();
        }
        return String.format("[%s TO %s]", this.lowerBound, this.upperBound);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        DateRange dateRange = (DateRange)o;
        return this.lowerBound.equals(dateRange.lowerBound) && (this.upperBound != null ? this.upperBound.equals(dateRange.upperBound) : dateRange.upperBound == null);
    }

    public int hashCode() {
        int result = this.lowerBound.hashCode();
        result = 31 * result + (this.upperBound != null ? this.upperBound.hashCode() : 0);
        return result;
    }

    public static class DateRangeBound {
        public static final DateRangeBound UNBOUNDED = new DateRangeBound();
        private final Date timestamp;
        private final Precision precision;

        public static DateRangeBound parseLowerBound(String lowerBound) throws ParseException {
            Preconditions.checkNotNull(lowerBound);
            if (DateRange.isUnbounded(lowerBound)) {
                return UNBOUNDED;
            }
            for (Precision precision : Precision.values()) {
                try {
                    Date timestamp = precision.parse(lowerBound);
                    return DateRangeBound.lowerBound(timestamp, precision);
                }
                catch (Exception exception) {
                }
            }
            throw new ParseException("Cannot parse lower date range bound: " + lowerBound, 0);
        }

        public static DateRangeBound parseUpperBound(String upperBound) throws ParseException {
            Preconditions.checkNotNull(upperBound);
            if (DateRange.isUnbounded(upperBound)) {
                return UNBOUNDED;
            }
            for (Precision precision : Precision.values()) {
                try {
                    Date timestamp = precision.parse(upperBound);
                    return DateRangeBound.upperBound(timestamp, precision);
                }
                catch (Exception exception) {
                }
            }
            throw new ParseException("Cannot parse upper date range bound: " + upperBound, 0);
        }

        public static DateRangeBound lowerBound(Date timestamp, Precision precision) {
            Date roundedLowerBound = precision.roundDown(timestamp);
            return new DateRangeBound(roundedLowerBound, precision);
        }

        public static DateRangeBound upperBound(Date timestamp, Precision precision) {
            Date roundedUpperBound = precision.roundUp(timestamp);
            return new DateRangeBound(roundedUpperBound, precision);
        }

        private DateRangeBound(Date timestamp, Precision precision) {
            Preconditions.checkNotNull(timestamp);
            Preconditions.checkNotNull(precision);
            this.timestamp = timestamp;
            this.precision = precision;
        }

        private DateRangeBound() {
            this.timestamp = null;
            this.precision = null;
        }

        public boolean isUnbounded() {
            return this == UNBOUNDED;
        }

        public Date getTimestamp() {
            return this.timestamp;
        }

        public Precision getPrecision() {
            return this.precision;
        }

        public String toString() {
            if (this.isUnbounded()) {
                return "*";
            }
            return this.precision.format(this.timestamp);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            DateRangeBound that = (DateRangeBound)o;
            if (this.isUnbounded()) {
                return that.isUnbounded();
            }
            return this.timestamp.equals(that.timestamp) && this.precision == that.precision;
        }

        public int hashCode() {
            if (this.isUnbounded()) {
                return 0;
            }
            int result = this.timestamp.hashCode();
            result = 31 * result + this.precision.hashCode();
            return result;
        }

        public static enum Precision {
            MILLISECOND(6, "yyyy-MM-dd'T'HH:mm:ss.SSS", "yyyy-MM-dd'T'HH:mm:ss.SSSZ"),
            SECOND(5, "yyyy-MM-dd'T'HH:mm:ss", "yyyy-MM-dd'T'HH:mm:ssZ"),
            MINUTE(4, "yyyy-MM-dd'T'HH:mm", "yyyy-MM-dd'T'HH:mmZ"),
            HOUR(3, "yyyy-MM-dd'T'HH", "yyyy-MM-dd'T'HHZ"),
            DAY(2, "yyyy-MM-dd", "yyyy-MM-ddZ"),
            MONTH(1, "yyyy-MM", "yyyy-MMZ"),
            YEAR(0, "yyyy", "yyyyZ");

            private static final TimeZone UTC;
            private static final Date MIN_DATE;
            final byte encoding;
            private final String[] patterns;
            private static final Map<Byte, Precision> ENCODINGS;

            private Precision(byte encoding, String ... patterns) {
                this.encoding = encoding;
                this.patterns = patterns;
            }

            static Precision fromEncoding(byte encoding) {
                Precision precision = ENCODINGS.get(encoding);
                if (precision == null) {
                    throw new IllegalArgumentException("Invalid precision encoding: " + encoding);
                }
                return precision;
            }

            public Date roundUp(Date timestamp) {
                GregorianCalendar calendar = Precision.newProlepticCalendar(timestamp);
                switch (this) {
                    case YEAR: {
                        calendar.set(2, 11);
                    }
                    case MONTH: {
                        calendar.set(5, calendar.getActualMaximum(5));
                    }
                    case DAY: {
                        calendar.set(11, 23);
                    }
                    case HOUR: {
                        calendar.set(12, 59);
                    }
                    case MINUTE: {
                        calendar.set(13, 59);
                    }
                    case SECOND: {
                        calendar.set(14, 999);
                    }
                }
                return calendar.getTime();
            }

            public Date roundDown(Date timestamp) {
                GregorianCalendar calendar = Precision.newProlepticCalendar(timestamp);
                switch (this) {
                    case YEAR: {
                        calendar.set(2, 0);
                    }
                    case MONTH: {
                        calendar.set(5, 1);
                    }
                    case DAY: {
                        calendar.set(11, 0);
                    }
                    case HOUR: {
                        calendar.set(12, 0);
                    }
                    case MINUTE: {
                        calendar.set(13, 0);
                    }
                    case SECOND: {
                        calendar.set(14, 0);
                    }
                }
                return calendar.getTime();
            }

            private Date parse(String source) throws ParseException {
                for (String pattern : this.patterns) {
                    try {
                        return ParseUtils.parseDate(source, pattern, true);
                    }
                    catch (ParseException parseException) {
                    }
                }
                throw new ParseException("Unable to parse the date: " + source, 0);
            }

            private String format(Date timestamp) {
                boolean isBeforeChrist;
                GregorianCalendar cal = Precision.newProlepticCalendar(timestamp);
                String pattern = this.patterns[0];
                boolean bl = isBeforeChrist = cal.get(0) == 0;
                if (isBeforeChrist) {
                    if (this == YEAR) {
                        return Precision.formatAsProlepticYear(cal.get(1));
                    }
                    pattern = pattern.substring(5);
                }
                SimpleDateFormat formatter = new SimpleDateFormat(pattern, Locale.ROOT);
                formatter.setCalendar(cal);
                String formatted = formatter.format(timestamp);
                if (isBeforeChrist) {
                    formatted = Precision.formatAsProlepticYear(cal.get(1)) + '-' + formatted;
                }
                if (this == MILLISECOND) {
                    formatted = formatted + "Z";
                }
                return formatted;
            }

            private static String formatAsProlepticYear(int yearBeforeChrist) {
                int yearProleptic = -yearBeforeChrist + 1;
                if (yearProleptic == 0) {
                    return "0000";
                }
                return String.format("%05d", yearProleptic);
            }

            private static GregorianCalendar newProlepticCalendar(Date timestamp) {
                GregorianCalendar cal = new GregorianCalendar(UTC, Locale.ROOT);
                cal.setGregorianChange(MIN_DATE);
                cal.setFirstDayOfWeek(2);
                cal.setMinimalDaysInFirstWeek(4);
                cal.setTime(timestamp);
                return cal;
            }

            static {
                UTC = TimeZone.getTimeZone("UTC");
                MIN_DATE = new Date(Long.MIN_VALUE);
                ImmutableMap.Builder<Byte, Precision> builder = ImmutableMap.builder();
                for (Precision precision : Precision.values()) {
                    builder.put(precision.encoding, precision);
                }
                ENCODINGS = builder.build();
            }
        }
    }
}

