/*
 * Decompiled with CFR 0.152.
 */
package io.opentelemetry.sdk.trace.export;

import io.opentelemetry.api.common.AttributeKey;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.api.metrics.LongCounter;
import io.opentelemetry.api.metrics.Meter;
import io.opentelemetry.api.metrics.MeterProvider;
import io.opentelemetry.context.Context;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.internal.DaemonThreadFactory;
import io.opentelemetry.sdk.internal.ThrowableUtil;
import io.opentelemetry.sdk.trace.ReadWriteSpan;
import io.opentelemetry.sdk.trace.ReadableSpan;
import io.opentelemetry.sdk.trace.SpanProcessor;
import io.opentelemetry.sdk.trace.data.SpanData;
import io.opentelemetry.sdk.trace.export.BatchSpanProcessorBuilder;
import io.opentelemetry.sdk.trace.export.SpanExporter;
import io.opentelemetry.sdk.trace.internal.JcTools;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class BatchSpanProcessor
implements SpanProcessor {
    private static final Logger logger = Logger.getLogger(BatchSpanProcessor.class.getName());
    private static final String WORKER_THREAD_NAME = BatchSpanProcessor.class.getSimpleName() + "_WorkerThread";
    private static final AttributeKey<String> SPAN_PROCESSOR_TYPE_LABEL = AttributeKey.stringKey("spanProcessorType");
    private static final AttributeKey<Boolean> SPAN_PROCESSOR_DROPPED_LABEL = AttributeKey.booleanKey("dropped");
    private static final String SPAN_PROCESSOR_TYPE_VALUE = BatchSpanProcessor.class.getSimpleName();
    private final Worker worker;
    private final AtomicBoolean isShutdown = new AtomicBoolean(false);

    public static BatchSpanProcessorBuilder builder(SpanExporter spanExporter) {
        return new BatchSpanProcessorBuilder(spanExporter);
    }

    BatchSpanProcessor(SpanExporter spanExporter, MeterProvider meterProvider, long scheduleDelayNanos, int maxQueueSize, int maxExportBatchSize, long exporterTimeoutNanos) {
        this.worker = new Worker(spanExporter, meterProvider, scheduleDelayNanos, maxExportBatchSize, exporterTimeoutNanos, JcTools.newFixedSizeQueue((int)maxQueueSize));
        Thread workerThread = new DaemonThreadFactory(WORKER_THREAD_NAME).newThread((Runnable)this.worker);
        workerThread.start();
    }

    @Override
    public void onStart(Context parentContext, ReadWriteSpan span) {
    }

    @Override
    public boolean isStartRequired() {
        return false;
    }

    @Override
    public void onEnd(ReadableSpan span) {
        if (span == null || !span.getSpanContext().isSampled()) {
            return;
        }
        this.worker.addSpan(span);
    }

    @Override
    public boolean isEndRequired() {
        return true;
    }

    @Override
    public CompletableResultCode shutdown() {
        if (this.isShutdown.getAndSet(true)) {
            return CompletableResultCode.ofSuccess();
        }
        return this.worker.shutdown();
    }

    @Override
    public CompletableResultCode forceFlush() {
        return this.worker.forceFlush();
    }

    ArrayList<SpanData> getBatch() {
        return this.worker.batch;
    }

    Queue<ReadableSpan> getQueue() {
        return this.worker.queue;
    }

    public String toString() {
        return "BatchSpanProcessor{spanExporter=" + this.worker.spanExporter + ", scheduleDelayNanos=" + this.worker.scheduleDelayNanos + ", maxExportBatchSize=" + this.worker.maxExportBatchSize + ", exporterTimeoutNanos=" + this.worker.exporterTimeoutNanos + '}';
    }

    private static final class Worker
    implements Runnable {
        private final LongCounter processedSpansCounter;
        private final Attributes droppedAttrs;
        private final Attributes exportedAttrs;
        private final SpanExporter spanExporter;
        private final long scheduleDelayNanos;
        private final int maxExportBatchSize;
        private final long exporterTimeoutNanos;
        private long nextExportTime;
        private final Queue<ReadableSpan> queue;
        private final AtomicInteger spansNeeded = new AtomicInteger(Integer.MAX_VALUE);
        private final BlockingQueue<Boolean> signal;
        private final AtomicReference<CompletableResultCode> flushRequested = new AtomicReference();
        private volatile boolean continueWork = true;
        private final ArrayList<SpanData> batch;

        private Worker(SpanExporter spanExporter, MeterProvider meterProvider, long scheduleDelayNanos, int maxExportBatchSize, long exporterTimeoutNanos, Queue<ReadableSpan> queue) {
            this.spanExporter = spanExporter;
            this.scheduleDelayNanos = scheduleDelayNanos;
            this.maxExportBatchSize = maxExportBatchSize;
            this.exporterTimeoutNanos = exporterTimeoutNanos;
            this.queue = queue;
            this.signal = new ArrayBlockingQueue<Boolean>(1);
            Meter meter = meterProvider.meterBuilder("io.opentelemetry.sdk.trace").build();
            meter.gaugeBuilder("queueSize").ofLongs().setDescription("The number of spans queued").setUnit("1").buildWithCallback(result -> result.record((long)queue.size(), Attributes.of(SPAN_PROCESSOR_TYPE_LABEL, SPAN_PROCESSOR_TYPE_VALUE)));
            this.processedSpansCounter = meter.counterBuilder("processedSpans").setUnit("1").setDescription("The number of spans processed by the BatchSpanProcessor. [dropped=true if they were dropped due to high throughput]").build();
            this.droppedAttrs = Attributes.of(SPAN_PROCESSOR_TYPE_LABEL, SPAN_PROCESSOR_TYPE_VALUE, SPAN_PROCESSOR_DROPPED_LABEL, true);
            this.exportedAttrs = Attributes.of(SPAN_PROCESSOR_TYPE_LABEL, SPAN_PROCESSOR_TYPE_VALUE, SPAN_PROCESSOR_DROPPED_LABEL, false);
            this.batch = new ArrayList(this.maxExportBatchSize);
        }

        private void addSpan(ReadableSpan span) {
            if (!this.queue.offer(span)) {
                this.processedSpansCounter.add(1L, this.droppedAttrs);
            } else if (this.queue.size() >= this.spansNeeded.get()) {
                this.signal.offer(true);
            }
        }

        @Override
        public void run() {
            this.updateNextExportTime();
            while (this.continueWork) {
                if (this.flushRequested.get() != null) {
                    this.flush();
                }
                JcTools.drain(this.queue, (int)(this.maxExportBatchSize - this.batch.size()), span -> this.batch.add(span.toSpanData()));
                if (this.batch.size() >= this.maxExportBatchSize || System.nanoTime() >= this.nextExportTime) {
                    this.exportCurrentBatch();
                    this.updateNextExportTime();
                }
                if (!this.queue.isEmpty()) continue;
                try {
                    long pollWaitTime = this.nextExportTime - System.nanoTime();
                    if (pollWaitTime <= 0L) continue;
                    this.spansNeeded.set(this.maxExportBatchSize - this.batch.size());
                    this.signal.poll(pollWaitTime, TimeUnit.NANOSECONDS);
                    this.spansNeeded.set(Integer.MAX_VALUE);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return;
                }
            }
        }

        private void flush() {
            for (int spansToFlush = this.queue.size(); spansToFlush > 0; --spansToFlush) {
                ReadableSpan span = this.queue.poll();
                assert (span != null);
                this.batch.add(span.toSpanData());
                if (this.batch.size() < this.maxExportBatchSize) continue;
                this.exportCurrentBatch();
            }
            this.exportCurrentBatch();
            CompletableResultCode flushResult = this.flushRequested.get();
            if (flushResult != null) {
                flushResult.succeed();
                this.flushRequested.set(null);
            }
        }

        private void updateNextExportTime() {
            this.nextExportTime = System.nanoTime() + this.scheduleDelayNanos;
        }

        private CompletableResultCode shutdown() {
            CompletableResultCode result = new CompletableResultCode();
            CompletableResultCode flushResult = this.forceFlush();
            flushResult.whenComplete(() -> {
                this.continueWork = false;
                CompletableResultCode shutdownResult = this.spanExporter.shutdown();
                shutdownResult.whenComplete(() -> {
                    if (!flushResult.isSuccess() || !shutdownResult.isSuccess()) {
                        result.fail();
                    } else {
                        result.succeed();
                    }
                });
            });
            return result;
        }

        private CompletableResultCode forceFlush() {
            CompletableResultCode possibleResult;
            CompletableResultCode flushResult = new CompletableResultCode();
            if (this.flushRequested.compareAndSet(null, flushResult)) {
                this.signal.offer(true);
            }
            return (possibleResult = this.flushRequested.get()) == null ? CompletableResultCode.ofSuccess() : possibleResult;
        }

        private void exportCurrentBatch() {
            if (this.batch.isEmpty()) {
                return;
            }
            try {
                CompletableResultCode result = this.spanExporter.export(Collections.unmodifiableList(this.batch));
                result.join(this.exporterTimeoutNanos, TimeUnit.NANOSECONDS);
                if (result.isSuccess()) {
                    this.processedSpansCounter.add((long)this.batch.size(), this.exportedAttrs);
                } else {
                    logger.log(Level.FINE, "Exporter failed");
                }
            }
            catch (Throwable t) {
                ThrowableUtil.propagateIfFatal((Throwable)t);
                logger.log(Level.WARNING, "Exporter threw an Exception", t);
            }
            finally {
                this.batch.clear();
            }
        }
    }
}

