/*
 * Decompiled with CFR 0.152.
 */
package net.oauth.jsontoken;

import com.google.common.base.Preconditions;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.security.SignatureException;
import java.util.List;
import java.util.regex.Pattern;
import net.oauth.jsontoken.Checker;
import net.oauth.jsontoken.Clock;
import net.oauth.jsontoken.JsonToken;
import net.oauth.jsontoken.JsonTokenUtil;
import net.oauth.jsontoken.SystemClock;
import net.oauth.jsontoken.crypto.AsciiStringVerifier;
import net.oauth.jsontoken.crypto.SignatureAlgorithm;
import net.oauth.jsontoken.crypto.Verifier;
import net.oauth.jsontoken.discovery.VerifierProviders;
import org.apache.commons.codec.binary.Base64;
import org.joda.time.Instant;
import org.joda.time.ReadableInstant;

public class JsonTokenParser {
    private final Clock clock;
    private final VerifierProviders verifierProviders;
    private final Checker[] checkers;

    public JsonTokenParser(VerifierProviders verifierProviders, Checker checker) {
        this(new SystemClock(), verifierProviders, checker);
    }

    public JsonTokenParser(Clock clock, VerifierProviders verifierProviders, Checker ... checkers) {
        this.clock = (Clock)Preconditions.checkNotNull((Object)clock);
        this.verifierProviders = verifierProviders;
        this.checkers = checkers;
    }

    public JsonToken deserialize(String tokenString) {
        String[] pieces = this.splitTokenString(tokenString);
        String jwtHeaderSegment = pieces[0];
        String jwtPayloadSegment = pieces[1];
        byte[] signature = Base64.decodeBase64((String)pieces[2]);
        JsonParser parser = new JsonParser();
        JsonObject header = parser.parse(JsonTokenUtil.fromBase64ToJsonString(jwtHeaderSegment)).getAsJsonObject();
        JsonObject payload = parser.parse(JsonTokenUtil.fromBase64ToJsonString(jwtPayloadSegment)).getAsJsonObject();
        JsonToken jsonToken = new JsonToken(header, payload, this.clock, tokenString);
        return jsonToken;
    }

    public void verify(JsonToken jsonToken) throws SignatureException {
        List<Verifier> verifiers = this.provideVerifiers(jsonToken);
        this.verify(jsonToken, verifiers);
    }

    public JsonToken verifyAndDeserialize(String tokenString) throws SignatureException {
        JsonToken jsonToken = this.deserialize(tokenString);
        this.verify(jsonToken);
        return jsonToken;
    }

    public void verify(JsonToken jsonToken, List<Verifier> verifiers) throws SignatureException {
        if (!this.signatureIsValid(jsonToken.getTokenString(), verifiers)) {
            throw new SignatureException("Invalid signature for token: " + jsonToken.getTokenString());
        }
        Instant issuedAt = jsonToken.getIssuedAt();
        Instant expiration = jsonToken.getExpiration();
        if (issuedAt == null && expiration != null) {
            issuedAt = new Instant(0L);
        }
        if (issuedAt != null && expiration == null) {
            expiration = new Instant(Long.MAX_VALUE);
        }
        if (issuedAt != null && expiration != null && (issuedAt.isAfter((ReadableInstant)expiration) || !this.clock.isCurrentTimeInInterval(issuedAt, expiration))) {
            throw new IllegalStateException(String.format("Invalid iat and/or exp. iat: %s exp: %s now: %s", jsonToken.getIssuedAt(), jsonToken.getExpiration(), this.clock.now()));
        }
        if (this.checkers != null) {
            for (Checker checker : this.checkers) {
                checker.check(jsonToken.getPayloadAsJsonObject());
            }
        }
    }

    public boolean signatureIsValid(String tokenString, List<Verifier> verifiers) {
        String[] pieces = this.splitTokenString(tokenString);
        byte[] signature = Base64.decodeBase64((String)pieces[2]);
        String baseString = JsonTokenUtil.toDotFormat(pieces[0], pieces[1]);
        boolean sigVerified = false;
        for (Verifier verifier : verifiers) {
            AsciiStringVerifier asciiVerifier = new AsciiStringVerifier(verifier);
            try {
                asciiVerifier.verifySignature(baseString, signature);
                sigVerified = true;
                break;
            }
            catch (SignatureException e) {
            }
        }
        return sigVerified;
    }

    public boolean expirationIsValid(JsonToken jsonToken, Instant now) {
        Instant expiration = jsonToken.getExpiration();
        return expiration == null || !now.isAfter((ReadableInstant)expiration);
    }

    public boolean issuedAtIsValid(JsonToken jsonToken, Instant now) {
        Instant issuedAt = jsonToken.getIssuedAt();
        return issuedAt == null || !now.isBefore((ReadableInstant)issuedAt);
    }

    private List<Verifier> provideVerifiers(JsonToken jsonToken) throws SignatureException {
        Preconditions.checkNotNull((Object)this.verifierProviders);
        JsonObject header = jsonToken.getHeader();
        JsonElement keyIdJson = header.get("kid");
        String keyId = keyIdJson == null ? null : keyIdJson.getAsString();
        SignatureAlgorithm sigAlg = jsonToken.getSignatureAlgorithm();
        List<Verifier> verifiers = this.verifierProviders.getVerifierProvider(sigAlg).findVerifier(jsonToken.getIssuer(), keyId);
        if (verifiers == null) {
            throw new IllegalStateException("No valid verifier for issuer: " + jsonToken.getIssuer());
        }
        return verifiers;
    }

    private String[] splitTokenString(String tokenString) {
        String[] pieces = tokenString.split(Pattern.quote("."));
        if (pieces.length != 3) {
            throw new IllegalStateException("Expected JWT to have 3 segments separated by '.', but it has " + pieces.length + " segments");
        }
        return pieces;
    }
}

