/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.mutiny.helpers;

import io.smallrye.mutiny.Multi;
import io.smallrye.mutiny.Uni;
import io.smallrye.mutiny.helpers.ParameterValidation;
import java.time.Duration;
import java.util.concurrent.Flow;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;

public class ExponentialBackoff {
    public static final Duration MAX_BACKOFF = Duration.ofMillis(Long.MAX_VALUE);
    public static final double DEFAULT_JITTER = 0.5;

    private ExponentialBackoff() {
    }

    public static Function<Multi<Throwable>, Flow.Publisher<Long>> randomExponentialBackoffFunction(long numRetries, Duration firstBackoff, Duration maxBackoff, double jitterFactor, ScheduledExecutorService executor) {
        ExponentialBackoff.validate(firstBackoff, maxBackoff, jitterFactor, executor);
        AtomicInteger index = new AtomicInteger();
        return t -> t.onItem().transformToUni(failure -> {
            int iteration = index.getAndIncrement();
            if ((long)iteration >= numRetries) {
                failure.addSuppressed(new IllegalStateException("Retries exhausted: " + iteration + "/" + numRetries, (Throwable)failure));
                return Uni.createFrom().failure((Throwable)failure);
            }
            Duration delay = ExponentialBackoff.getNextDelay(firstBackoff, maxBackoff, jitterFactor, iteration);
            return Uni.createFrom().item(Long.valueOf(iteration)).onItem().delayIt().onExecutor(executor).by(delay);
        }).concatenate();
    }

    private static Duration getNextDelay(Duration firstBackoff, Duration maxBackoff, double jitterFactor, int iteration) {
        Duration nextBackoff = ExponentialBackoff.getNextAttemptDelay(firstBackoff, maxBackoff, iteration);
        long jitterOffset = ExponentialBackoff.getJitter(jitterFactor, nextBackoff);
        long lowBound = Math.max(firstBackoff.minus(nextBackoff).toMillis(), -jitterOffset);
        long highBound = Math.min(maxBackoff.minus(nextBackoff).toMillis(), jitterOffset);
        ThreadLocalRandom random = ThreadLocalRandom.current();
        long jitter = highBound == lowBound ? (highBound == 0L ? 0L : random.nextLong(highBound)) : random.nextLong(lowBound, highBound);
        return nextBackoff.plusMillis(jitter);
    }

    private static void validate(Duration firstBackoff, Duration maxBackoff, double jitterFactor, ScheduledExecutorService executor) {
        if (jitterFactor < 0.0 || jitterFactor > 1.0) {
            throw new IllegalArgumentException("jitterFactor must be between 0 and 1 (default 0.5)");
        }
        ParameterValidation.nonNull(firstBackoff, "firstBackoff");
        ParameterValidation.nonNull(maxBackoff, "maxBackoff");
        ParameterValidation.nonNull(executor, "executor");
    }

    public static Function<Multi<Throwable>, Flow.Publisher<Long>> randomExponentialBackoffFunctionExpireAt(long expireAt, Duration firstBackoff, Duration maxBackoff, double jitterFactor, ScheduledExecutorService executor) {
        ExponentialBackoff.validate(firstBackoff, maxBackoff, jitterFactor, executor);
        AtomicInteger index = new AtomicInteger();
        return t -> t.onItem().transformToUni(failure -> {
            int iteration = index.getAndIncrement();
            Duration delay = ExponentialBackoff.getNextDelay(firstBackoff, maxBackoff, jitterFactor, iteration);
            long checkTime = System.currentTimeMillis() + delay.toMillis();
            if (checkTime > expireAt) {
                return Uni.createFrom().failure(new IllegalStateException("Retries exhausted : " + iteration + " attempts against " + checkTime + "/" + expireAt + " expiration", (Throwable)failure));
            }
            return Uni.createFrom().item(Long.valueOf(iteration)).onItem().delayIt().onExecutor(executor).by(delay);
        }).concatenate();
    }

    private static long getJitter(double jitterFactor, Duration nextBackoff) {
        long jitterOffset;
        try {
            jitterOffset = nextBackoff.multipliedBy((long)(100.0 * jitterFactor)).dividedBy(100L).toMillis();
        }
        catch (ArithmeticException ae) {
            jitterOffset = Math.round(9.223372036854776E18 * jitterFactor);
        }
        return jitterOffset;
    }

    private static Duration getNextAttemptDelay(Duration firstBackoff, Duration maxBackoff, int iteration) {
        Duration nextBackoff;
        try {
            nextBackoff = firstBackoff.multipliedBy((long)Math.pow(2.0, iteration));
            if (nextBackoff.compareTo(maxBackoff) > 0) {
                nextBackoff = maxBackoff;
            }
        }
        catch (ArithmeticException overflow) {
            nextBackoff = maxBackoff;
        }
        return nextBackoff;
    }
}

