/**
 * (C) Copyright IBM Corporation 2019.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.openliberty.tools.common.plugins.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.ServerSocket;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.Watchable;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Scanner;
import java.util.Set;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import javax.tools.JavaCompiler;
import javax.tools.JavaFileObject;
import javax.tools.StandardJavaFileManager;
import javax.tools.StandardLocation;
import javax.tools.ToolProvider;

import com.sun.nio.file.SensitivityWatchEventModifier;

import org.apache.commons.io.FileUtils;

import io.openliberty.tools.ant.ServerTask;

/**
 * Utility class for dev mode.
 */
public abstract class DevUtil {

    private static final String START_SERVER_MESSAGE_PREFIX = "CWWKF0011I:";
    private static final String START_APP_MESSAGE_REGEXP = "CWWKZ0001I.*";
    private static final String UPDATED_APP_MESSAGE_REGEXP = "CWWKZ0003I.*";
    private static final String PORT_IN_USE_MESSAGE_PREFIX = "CWWKO0221E:";
    private static final String WEB_APP_AVAILABLE_MESSAGE_PREFIX = "CWWKT0016I:";
    private static final String LISTENING_ON_PORT_MESSAGE_PREFIX = "CWWKO0219I:";
    private static final String HTTP_PREFIX = "http://";
    private static final String HTTPS_PREFIX = "https://";
    private static final String GENERATED_HEADER_REGEX = "# Generated by liberty-.*-plugin";

    private static final String[] IGNORE_DIRECTORY_PREFIXES = new String[] { "." };
    private static final String[] IGNORE_FILE_PREFIXES = new String[] { "." };
    private static final String[] IGNORE_FILE_POSTFIXES = new String[] {
            // core dumps
            ".dmp",
            // vim
            "~",
            // intellij
            "___jb_tmp___", "___jb_old___" };

    /**
     * Log debug
     * 
     * @param msg
     */
    public abstract void debug(String msg);

    /**
     * Log debug
     * 
     * @param msg
     * @param e
     */
    public abstract void debug(String msg, Throwable e);

    /**
     * Log debug
     * 
     * @param e
     */
    public abstract void debug(Throwable e);

    /**
     * Log warning
     * 
     * @param msg
     */
    public abstract void warn(String msg);

    /**
     * Log info
     * 
     * @param msg
     */
    public abstract void info(String msg);

    /**
     * Log error
     * 
     * @param msg
     */
    public abstract void error(String msg);

    /**
     * Log error
     * 
     * @param msg
     * @param e
     */
    public abstract void error(String msg, Throwable e);

    /**
     * Returns whether debug is enabled by the current logger
     * 
     * @return whether debug is enabled
     */
    public abstract boolean isDebugEnabled();

    /**
     * Updates artifacts of current project
     */
    public abstract List<String> getArtifacts();

    /**
     * Recompile the build file
     * 
     * @param buildFile
     * @param artifactPaths
     * @param executor      The thread pool executor
     * @throws PluginExecutionException if there was an error when restarting the server
     * @return true if the build file was recompiled with changes
     */
    public abstract boolean recompileBuildFile(File buildFile, List<String> artifactPaths, ThreadPoolExecutor executor) throws PluginExecutionException;

    /**
     * Run the unit tests
     * 
     * @throws PluginScenarioException  if unit tests failed
     * @throws PluginExecutionException if unit tests could not be run
     */
    public abstract void runUnitTests() throws PluginScenarioException, PluginExecutionException;

    /**
     * Run the integration tests
     * 
     * @throws PluginScenarioException  if integration tests failed
     * @throws PluginExecutionException if integration tests could not be run
     */
    public abstract void runIntegrationTests() throws PluginScenarioException, PluginExecutionException;

    /**
     * Check the configuration file for new features
     * 
     * @param configFile
     * @param serverDir
     */
    public abstract void checkConfigFile(File configFile, File serverDir);

    /**
     * Compile the specified directory
     * 
     * @param dir
     * @return
     */
    public abstract boolean compile(File dir);

    /**
     * Stop the server
     */
    public abstract void stopServer();

    /**
     * Get the ServerTask to start the server, which can be in either "run" or
     * "debug" mode
     * 
     * @return ServerTask the task to start the server
     * @throws Exception if there was an error copying/creating config files
     */
    public abstract ServerTask getServerTask() throws Exception;
    
    /**
     * Redeploy the application
     */
    public abstract void redeployApp() throws PluginExecutionException;

    private File serverDirectory;
    private File sourceDirectory;
    private File testSourceDirectory;
    private File configDirectory;
    private List<File> resourceDirs;
    private boolean hotTests;
    private Path tempConfigPath;
    private boolean skipTests;
    private boolean skipUTs;
    private boolean skipITs;
    private String applicationId;
    private int appStartupTimeout;
    private int appUpdateTimeout;
    private Thread serverThread;
    private AtomicBoolean devStop;
    private String hostName;
    private String httpPort;
    private String httpsPort;
    private final long compileWaitMillis;
    private AtomicBoolean inputUnavailable;
    private int alternativeDebugPort = -1;
    private boolean libertyDebug;
    private int libertyDebugPort;
    private AtomicBoolean detectedAppStarted;
    private long serverStartTimeout;
    private boolean useBuildRecompile;
    private Map<File, Properties> propertyFilesMap;
    private AtomicBoolean calledShutdownHook;
    private boolean gradle;

    public DevUtil(File serverDirectory, File sourceDirectory, File testSourceDirectory, File configDirectory,
            List<File> resourceDirs, boolean hotTests, boolean skipTests, boolean skipUTs, boolean skipITs,
            String applicationId, long serverStartTimeout, int appStartupTimeout, int appUpdateTimeout,
            long compileWaitMillis, boolean libertyDebug, boolean useBuildRecompile, boolean gradle) {
        this.serverDirectory = serverDirectory;
        this.sourceDirectory = sourceDirectory;
        this.testSourceDirectory = testSourceDirectory;
        this.configDirectory = configDirectory;
        this.resourceDirs = resourceDirs;
        this.hotTests = hotTests;
        this.skipTests = skipTests;
        this.skipUTs = skipUTs;
        this.skipITs = skipITs;
        this.applicationId = applicationId;
        this.serverStartTimeout = serverStartTimeout;
        this.appStartupTimeout = appStartupTimeout;
        this.appUpdateTimeout = appUpdateTimeout;
        this.devStop = new AtomicBoolean(false);
        this.compileWaitMillis = compileWaitMillis;
        this.inputUnavailable = new AtomicBoolean(false);
        this.libertyDebug = libertyDebug;
        this.detectedAppStarted = new AtomicBoolean(false);
        this.useBuildRecompile = useBuildRecompile;
        this.calledShutdownHook = new AtomicBoolean(false);
        this.gradle = gradle;
    }

    /**
     * Run unit and/or integration tests
     * 
     * @param waitForApplicationUpdate Whether to wait for the application to update
     *                                 before running integration tests
     * @param messageOccurrences       The previous number of times the application
     *                                 updated message has appeared.
     * @param executor                 The thread pool executor
     * @param forceSkipUTs             Whether to force skip the unit tests
     */
    public void runTests(boolean waitForApplicationUpdate, int messageOccurrences, ThreadPoolExecutor executor,
            boolean forceSkipUTs) {
        if (!skipTests) {
            ServerTask serverTask = null;
            try {
                serverTask = getServerTask();
            } catch (Exception e) {
                // not expected since server should already have been started
                error("Could not get the server task for running tests.", e);
            }
            File logFile = serverTask.getLogFile();
            String regexp = UPDATED_APP_MESSAGE_REGEXP + applicationId;

            try {
                Thread.sleep(500);
            } catch (InterruptedException e) {
                debug("Thread interrupted while waiting to start tests.", e);
            }

            // if queue size >= 1, it means a newer test has been queued so we
            // should skip this and let that run instead
            if (executor.getQueue().size() >= 1) {
                Runnable head = executor.getQueue().peek();
                boolean manualInvocation = ((TestJob) head).isManualInvocation();

                if (manualInvocation) {
                    debug("Tests were re-invoked before previous tests began. Cancelling previous tests and resubmitting them.");
                } else {
                    debug("Changes were detected before tests began. Cancelling tests and resubmitting them.");
                }
                return;
            }

            // skip unit tests if invoked by Gradle
            if (!gradle && !(skipUTs || forceSkipUTs)) {
                info("Running unit tests...");
                try {
                    runUnitTests();
                    info("Unit tests finished.");
                } catch (PluginScenarioException e) {
                    debug(e);
                    error(e.getMessage());
                    // if unit tests failed, don't run integration tests
                    return;
                } catch (PluginExecutionException e) {
                    error(e.getMessage());
                }
            }

            // if queue size >= 1, it means a newer test has been queued so we
            // should skip this and let that run instead
            if (executor.getQueue().size() >= 1) {
                Runnable head = executor.getQueue().peek();
                boolean manualInvocation = ((TestJob) head).isManualInvocation();

                if (manualInvocation) {
                    info("Tests were invoked while previous tests were running. Restarting tests.");
                } else {
                    info("Changes were detected while tests were running. Restarting tests.");
                }
                return;
            }

            if (!skipITs) {

                if (!detectedAppStarted.get()) {
                    if (appStartupTimeout < 0) {
                        warn("The verifyTimeout (verifyAppStartTimeout) value needs to be an integer greater than or equal to 0.  The default value of 30 seconds will be used.");
                        appStartupTimeout = 30;
                    }
                    long timeout = appStartupTimeout * 1000;

                    // Wait for the app started message in messages.log
                    info("Waiting up to " + appStartupTimeout + " seconds for the application to start up...");
                    String startMessage = serverTask.waitForStringInLog(START_APP_MESSAGE_REGEXP, timeout,
                            logFile);
                    if (startMessage == null) {
                        error("Unable to verify if the application was started after " + appStartupTimeout
                                + " seconds.  Consider increasing the verifyTimeout value if this continues to occur.");
                    } else {
                        detectedAppStarted.set(true);
                    }
                }

                if (waitForApplicationUpdate) {
                    // wait until application has been updated
                    if (appUpdateTimeout < 0) {
                        appUpdateTimeout = 5;
                    }
                    long timeout = appUpdateTimeout * 1000;
                    serverTask.waitForUpdatedStringInLog(regexp, timeout, logFile, messageOccurrences);
                }
                if (gradle) {
                    info("Running tests...");
                } else {
                    info("Running integration tests...");
                }
                try {
                    runIntegrationTests();
                    if (gradle) {
                        info("Tests finished.");
                    } else {
                        info("Integration tests finished.");
                    }
                } catch (PluginScenarioException e) {
                    debug(e);
                    error(e.getMessage());
                    // if unit tests failed, don't run integration tests
                    return;
                } catch (PluginExecutionException e) {
                    error(e.getMessage());
                }
            }
        }
    }

    /**
     * Get the number of times the application updated message has appeared in the
     * application log
     * 
     * @return the number of times the application has updated
     */
    public int countApplicationUpdatedMessages() {
        int messageOccurrences = -1;
        if (!(skipTests || skipITs)) {
            try {
                ServerTask serverTask = getServerTask();
                File logFile = serverTask.getLogFile();
                String regexp = UPDATED_APP_MESSAGE_REGEXP + applicationId;
                messageOccurrences = serverTask.countStringOccurrencesInFile(regexp, logFile);
                debug("Message occurrences before compile: " + messageOccurrences);
            } catch (Exception e) {
                debug("Failed to get message occurrences before compile", e);
            }
        }
        return messageOccurrences;
    }

    /**
     * Start the server and keep it running in a background thread.
     * 
     * @throws PluginExecutionException If the server startup could not be verified
     *                                  within the timeout, or server startup
     *                                  failed.
     */
    public void startServer() throws PluginExecutionException {
        try {
            final ServerTask serverTask;
            try {
                serverTask = getServerTask();
            } catch (Exception e) {
                throw new PluginExecutionException("An error occurred while starting the server: " + e.getMessage(), e);
            }

            // Set debug variables in server.env if debug enabled
            enableServerDebug();

            String logsDirectory = serverDirectory.getCanonicalPath() + "/logs";
            File messagesLogFile = new File(logsDirectory + "/messages.log");

            // Watch logs directory if it already exists
            WatchService watchService = FileSystems.getDefault().newWatchService();
            boolean logsExist = new File(logsDirectory).isDirectory();

            if (logsExist) {
                // If the logs directory already exists, then
                // setup a watch service to monitor the directory.
                Paths.get(logsDirectory).register(watchService, StandardWatchEventKinds.ENTRY_CREATE,
                        StandardWatchEventKinds.ENTRY_MODIFY);
            }

            // Start server
            serverThread = new Thread(new Runnable() {

                @Override
                public void run() {
                    try {
                        serverTask.execute();
                    } catch (RuntimeException e) {
                        // If devStop is true server was stopped with Ctl-c, do not throw exception
                        if (devStop.get() == false) {
                            // If a runtime exception occurred in the server task, log and rethrow
                            error("An error occurred while starting the server: " + e.getMessage(), e);
                            throw e;
                        }
                    }
                }

            });

            serverThread.start();

            // If the server thread dies at any point after this, allow the error to say
            // that the server stopped
            setDevStop(false);

            if (logsExist) {
                // If logs already exist, then watch the directory to ensure
                // messages.log is modified before continuing.
                boolean messagesModified = false;
                WatchKey key;
                while (!messagesModified && (key = watchService.take()) != null) {
                    for (WatchEvent<?> event : key.pollEvents()) {
                        if (event.context().toString().equals("messages.log")) {
                            messagesModified = true;
                            debug("messages.log has been changed");
                        }
                    }

                    if (!key.reset()) {
                        break;
                    }
                }
            }

            // Set server start timeout
            if (serverStartTimeout < 0) {
                warn("The serverStartTimeout value needs to be an integer greater than or equal to 0.  The default value of 30 seconds will be used.");
                serverStartTimeout = 30;
            }
            long serverStartTimeoutMillis = serverStartTimeout * 1000;

            // Wait for the server started message in messages.log
            String startMessage = serverTask.waitForStringInLog(START_SERVER_MESSAGE_PREFIX, serverStartTimeoutMillis, messagesLogFile);
            if (startMessage == null) {
                setDevStop(true);
                stopServer();
                throw new PluginExecutionException("Unable to verify if the server was started after " + serverStartTimeout
                        + " seconds.  Consider increasing the serverStartTimeout value if this continues to occur.");
            }

            // Check for port already in use error
            String portError = serverTask.findStringInFile(PORT_IN_USE_MESSAGE_PREFIX, messagesLogFile);
            if (portError != null) {
                error(portError.split(PORT_IN_USE_MESSAGE_PREFIX)[1]);
            }

            // Parse hostname, http, https ports for integration tests to use
            parseHostNameAndPorts(serverTask, messagesLogFile);
        } catch (IOException | InterruptedException e) {
            throw new PluginExecutionException("An error occurred while starting the server: " + e.getMessage(), e);
        }
    }

    public abstract void libertyCreate() throws PluginExecutionException;
    public abstract void libertyDeploy() throws PluginExecutionException;
    public abstract void libertyInstallFeature() throws PluginExecutionException;

    public void restartServer() throws PluginExecutionException {
        info("Restarting server...");
        setDevStop(true);
        stopServer();
        if (serverThread != null) {
            final long threadShutdownTimeoutSeconds = 30;
            try {
                serverThread.join(threadShutdownTimeoutSeconds * 1000);
                if (serverThread.isAlive()) {
                    throw new PluginExecutionException("Could not stop the server after " + threadShutdownTimeoutSeconds + " seconds.  Ensure that the server has been stopped, then start dev mode again.");
                }
            } catch (InterruptedException e) {
                if (serverThread.isAlive()) {
                    throw new PluginExecutionException("Could not stop the server.  Ensure that the server has been stopped, then start dev mode again.", e);
                } else {
                    // the thread was interrupted, but the server thread is already stopped
                    debug(e);
                }
            }
        }
        libertyCreate();
        libertyInstallFeature();
        libertyDeploy();
        startServer();
        setDevStop(false);
        info("The server has been restarted.");
    }

    private void parseHostNameAndPorts(final ServerTask serverTask, File messagesLogFile) throws PluginExecutionException {
        String webAppMessage = serverTask.findStringInFile(WEB_APP_AVAILABLE_MESSAGE_PREFIX, messagesLogFile);
        debug("Web app available message: " + webAppMessage);
        if (webAppMessage != null) {
            int portPrefixIndex = parseHostName(webAppMessage);
            parseHttpPort(webAppMessage, portPrefixIndex);
        }
        List<String> listeningOnPortMessages = serverTask.findStringsInFile(LISTENING_ON_PORT_MESSAGE_PREFIX, messagesLogFile);
        if (listeningOnPortMessages != null) {
            parseHttpsPort(listeningOnPortMessages);
        }
    }

    protected int parseHostName(String webAppMessage) throws PluginExecutionException {
        int protocolIndex = webAppMessage.indexOf(HTTP_PREFIX);
        int hostNameIndex = protocolIndex + HTTP_PREFIX.length();
        if (protocolIndex < 0) {
            protocolIndex = webAppMessage.indexOf(HTTPS_PREFIX);
            hostNameIndex = protocolIndex + HTTPS_PREFIX.length();

            if (protocolIndex < 0) {    
                throw new PluginExecutionException("Could not parse the host name from the log message: " + webAppMessage);
            }
        }

        int portPrefixIndex = webAppMessage.indexOf(":", hostNameIndex);
        if (portPrefixIndex < 0) {
            throw new PluginExecutionException("Could not parse the port number from the log message: " + webAppMessage);
        }
        hostName = webAppMessage.substring(hostNameIndex, portPrefixIndex);
        debug("Parsed host name: " + hostName);
        return portPrefixIndex;
    }

    protected void parseHttpPort(String webAppMessage, int portPrefixIndex) {
        if (!webAppMessage.contains(HTTP_PREFIX)) {
            return;
        }
        int portIndex = portPrefixIndex + 1;
        int portEndIndex = webAppMessage.indexOf("/", portIndex);
        if (portEndIndex < 0) {
            // if no ending slash, the port ends at the end of the message
            portEndIndex = webAppMessage.length();
        }
        httpPort = webAppMessage.substring(portIndex, portEndIndex);
        debug("Parsed http port: " + httpPort);
    }

    protected void parseHttpsPort(List<String> messages) throws PluginExecutionException {
        for (String message : messages) {
            debug("Looking for https port in message: " + message);
            String httpsMessageContents = message.split(LISTENING_ON_PORT_MESSAGE_PREFIX)[1];
            String[] messageTokens = httpsMessageContents.split(" ");
            // Look for endpoint with name containing "-ssl"
            for (String token : messageTokens) {
                if (token.contains("-ssl")) {
                    String parsedHttpsPort = getPortFromMessageTokens(messageTokens);
                    if (parsedHttpsPort != null) {
                        debug("Parsed https port: " + parsedHttpsPort);
                        httpsPort = parsedHttpsPort;
                        return;
                    } else {
                        throw new PluginExecutionException("Could not parse the https port number from the log message: " + message);
                    }
                }
            }
        }
        debug("Could not find https port. The server might not be configured for https.");
    }

    private String getPortFromMessageTokens(String[] messageTokens) throws PluginExecutionException {
        // For each space-separated token, keep only the numeric parts.
        // The port is the last numeric token which is a number <= 65535.
        for (int i = messageTokens.length - 1; i >= 0; i--) {
            String numericToken = messageTokens[i].replaceAll("[^\\d]", "" );
            if (numericToken.length() > 0) {
                try {
                    int parsedPort = Integer.parseInt(numericToken);
                    if (parsedPort <= 65535) {
                        return numericToken;
                    }
                } catch (NumberFormatException e) {
                    // If the token is not parseable for some reason, then it's probably not a port number
                    debug("Could not parse integer from numeric token " + numericToken + " from message token " + messageTokens[i], e);
                }
            }
        }
        return null;
    }

    public void cleanUpServerEnv() {
        // clean up server.env file
        File serverEnvFile;
        File serverEnvBackup;
        try {
            serverEnvBackup = new File(serverDirectory.getCanonicalPath() + "/server.env.bak");
            serverEnvFile = new File(serverDirectory.getCanonicalPath() + "/server.env");
            if (serverEnvBackup.exists()) {
                // Restore original server.env file
                try {
                    Files.copy(serverEnvBackup.toPath(), serverEnvFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                } catch (IOException e) {
                    error("Could not restore server.env: " + e.getMessage());
                }
                serverEnvBackup.delete();
            } else {
                // Delete server.env file
                serverEnvFile.delete();
            }
        } catch (IOException e) {
            error("Could not retrieve server.env: " + e.getMessage());
        }
    }

    public void cleanUpTempConfig() {
        if (this.tempConfigPath != null){
            File tempConfig = this.tempConfigPath.toFile();
            if (tempConfig.exists()){
                try {
                    FileUtils.deleteDirectory(tempConfig);
                    debug("Sucessfully deleted liberty:dev temporary configuration folder");
                } catch (IOException e) {
                    error("Could not delete liberty:dev temporary configuration folder");
                }
            }
        }
    }
    
    /**
     * Whether dev mode intentionally caused the server to stop.
     * 
     * @param devStop If true, stopping the server will not cause dev mode to print
     *                an error message.
     */
    public void setDevStop(boolean devStop) {
        this.devStop.set(devStop);
    }

    public void addShutdownHook(final ThreadPoolExecutor executor) {
        // shutdown hook to stop server when dev mode is terminated
        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                runShutdownHook(executor);
            }
        });
    }

    private void runShutdownHook(final ThreadPoolExecutor executor) {
        if (!calledShutdownHook.getAndSet(true)) {
            debug("Inside Shutdown Hook, shutting down server");
        
            setDevStop(true);
            cleanUpTempConfig();
            cleanUpServerEnv();
    
            if (hotkeyReader != null) {
                hotkeyReader.shutdown();
            }
    
            // shutdown tests
            executor.shutdown();
    
            // stopping server
            stopServer();
        }
    }

    /**
     * Gets a map of the environment variables to set for debug mode.
     * 
     * @param libertyDebugPort the debug port to use
     */
    public Map<String, String> getDebugEnvironmentVariables() throws IOException {
        Map<String, String> map = new HashMap<String, String>();
        map.put("WLP_DEBUG_SUSPEND", "n");
        map.put("WLP_DEBUG_ADDRESS", String.valueOf(findAvailablePort(libertyDebugPort)));
        return map;
    }

    /**
     * Enable server debug variables in server.env, using the user specified debug
     * port if it's available, otherwise uses a random available port.
     * 
     * @throws IOException if there was an IO exception when reading or writing the
     *                     server.env
     */
    public void enableServerDebug() throws IOException {
        enableServerDebug(true);
    }

    private void enableServerDebug(boolean findAvailablePort) throws IOException {
        if (!libertyDebug) {
            return;
        }

        String serverEnvPath = serverDirectory.getCanonicalPath() + "/server.env";
        File serverEnvFile = new File(serverEnvPath);
        StringBuilder sb = new StringBuilder();
        File serverEnvBackup = new File(serverEnvPath + ".bak");
        if (serverEnvFile.exists()) {
            debug("server.env already exists");

            Files.copy(serverEnvFile.toPath(), serverEnvBackup.toPath(), StandardCopyOption.REPLACE_EXISTING);
            boolean deleted = serverEnvFile.delete();
            if (!deleted) {
                error("Could not move existing server.env file");
            }

            BufferedReader reader = new BufferedReader(new FileReader(serverEnvBackup));
            try {
                String line;
                while ((line = reader.readLine()) != null) {
                    sb.append(line);
                    sb.append("\n");
                }
            } finally {
                reader.close();
            }
        } else {
            // if server.env does not exist, clean up any backup file
            serverEnvBackup.delete();
        }
        
        debug("Creating server.env file: " + serverEnvFile.getCanonicalPath());
        sb.append("WLP_DEBUG_SUSPEND=n\n");
        sb.append("WLP_DEBUG_ADDRESS=");
        if (findAvailablePort) {
            sb.append(findAvailablePort(libertyDebugPort));
        } else {
            sb.append(alternativeDebugPort == -1 ? libertyDebugPort : alternativeDebugPort);
        }
        sb.append("\n");

        BufferedWriter writer = new BufferedWriter(new FileWriter(serverEnvFile));
        try {
            writer.write(sb.toString());
        } finally {
            writer.close();
        }

        if (serverEnvFile.exists()) {
            debug("Successfully created liberty:dev server.env file");
        }
    }

    /**
     * Finds an available port. If the preferred port is not available, returns a
     * random available port and caches the result, which will override the
     * preferredPort if this method is called again.
     * 
     * @return An available port.
     * @throws IOException if it could not find any available port, or there was an
     *                     error when opening a server socket regardless of port.
     */
    public int findAvailablePort(int preferredPort) throws IOException {
        int portToTry = preferredPort;
        if (alternativeDebugPort != -1) {
            portToTry = alternativeDebugPort;
        }

        ServerSocket serverSocket = null;
        try {
            serverSocket = new ServerSocket();
            serverSocket.setReuseAddress(false);
            // try binding to the loopback address at the port to try
            serverSocket.bind(new InetSocketAddress(InetAddress.getByName(null), portToTry), 1);
            return serverSocket.getLocalPort();
        } catch (IOException e) {
            if (serverSocket != null) {
                // if binding failed, try binding to a random port
                serverSocket.bind(null, 1);
                int availablePort = serverSocket.getLocalPort();
                if (portToTry == preferredPort) {
                    warn("The debug port " + preferredPort + " is not available.  Using " + availablePort + " as the debug port instead.");
                } else {
                    debug("The previous debug port " + alternativeDebugPort + " is no longer available.  Using " + availablePort + " as the debug port instead.");
                }
                alternativeDebugPort = availablePort;
                return availablePort;
            } else {
                throw new IOException("Could not create a server socket for debugging.", e);
            }
        } finally {
            if (serverSocket != null) {
                serverSocket.close();
            }
        }
    }

    private HotkeyReader hotkeyReader = null;

    /**
     * Run a hotkey reader thread.
     * If the thread is already running, re-prints the message about pressing enter to run tests.
     * 
     * @param executor the test thread executor
     */
    public void runHotkeyReaderThread(ThreadPoolExecutor executor) {
        if (inputUnavailable.get()) {
            return;
        }
        boolean startedNewHotkeyReader = false;
        if (hotkeyReader == null) {
            hotkeyReader = new HotkeyReader(executor);
            new Thread(hotkeyReader).start();
            debug("Started hotkey reader.");
            startedNewHotkeyReader = true;
        }
        if (!skipTests) {
            synchronized(inputUnavailable) {
                try {
                    if (startedNewHotkeyReader) {
                        // if new hotkey reader started, wait for it to try getting the input to see if it's available
                        inputUnavailable.wait(500);
                    }
                    if (!inputUnavailable.get()) {
                        if (hotTests) {
                            info("Tests will run automatically when changes are detected. You can also press the Enter key to run tests on demand.");
                        } else {
                            info("Press the Enter key to run tests on demand. To stop the server and quit dev mode, use Ctrl-C or type 'q' and press the Enter key.");
                        }
                    } else {    
                        debug("Cannot read user input, setting hotTests to true.");
                        info("Tests will run automatically when changes are detected.");
                        hotTests = true;    
                    }
                } catch (InterruptedException e) {
                    debug("Interrupted while waiting to determine whether input can be read", e);
                }
            }
        }
    }

    private class HotkeyReader implements Runnable {
        private Scanner scanner;
        private ThreadPoolExecutor executor;
        private boolean shutdown = false;

        public HotkeyReader(ThreadPoolExecutor executor) {
            this.executor = executor;
        }
    
        @Override
        public void run() {
            debug("Running hotkey reader thread");
            scanner = new Scanner(System.in);
            try {
                readInput();
            } finally {
                scanner.close();
            }
        }

        public void shutdown() {
            shutdown = true;
        }
    
        private void readInput() {
            if (scanner.hasNextLine()) {
                synchronized(inputUnavailable) {
                    inputUnavailable.notify();
                }
                while (!shutdown) {
                    debug("Waiting for Enter key to run tests");
                    if (!scanner.hasNextLine()) {
                        break;
                    }
                    String line = scanner.nextLine();
                    if (line != null && (line.trim().equalsIgnoreCase("q") || line.trim().equalsIgnoreCase("quit")
                            || line.trim().equalsIgnoreCase("exit"))) {
                        debug("Detected exit command");
                        runShutdownHook(executor);
                    } else {
                        debug("Detected Enter key. Running tests...");
                        runTestThread(false, executor, -1, false, true);
                    }    
                }
            } else {
                synchronized(inputUnavailable) {
                    inputUnavailable.set(true);
                    inputUnavailable.notify();
                }
            }
            debug("Hotkey reader thread was shut down");
        }
    }

    // The serverXmlFile parameter can be null when using the server.xml from the configDirectory, which has a default value.
    public void watchFiles(File buildFile, File outputDirectory, File testOutputDirectory,
            final ThreadPoolExecutor executor, List<String> artifactPaths, File serverXmlFile)
            throws Exception {
        try (WatchService watcher = FileSystems.getDefault().newWatchService();) {
            
            File serverXmlFileParent = null;
            if (serverXmlFile != null && serverXmlFile.exists()) {
                serverXmlFileParent = serverXmlFile.getParentFile();
            }
            
            Path srcPath = this.sourceDirectory.getCanonicalFile().toPath();
            Path testSrcPath = this.testSourceDirectory.getCanonicalFile().toPath();
            Path configPath = this.configDirectory.getCanonicalFile().toPath();

            boolean sourceDirRegistered = false;
            boolean testSourceDirRegistered = false;
            boolean configDirRegistered = false;
            boolean serverXmlFileRegistered = false;

            if (this.sourceDirectory.exists()) {
                registerAll(srcPath, watcher);
                sourceDirRegistered = true;
            }

            if (this.testSourceDirectory.exists()) {
                registerAll(testSrcPath,watcher);
                testSourceDirRegistered = true;
            }

            if (this.configDirectory.exists()) {
                registerAll(configPath, watcher);
                configDirRegistered = true;
            }
            
            if (serverXmlFile != null && serverXmlFile.exists() && serverXmlFileParent.exists()){
                Path serverXmlFilePath = serverXmlFileParent.getCanonicalFile().toPath();
                registerAll(serverXmlFilePath, watcher);
                serverXmlFileRegistered = true;
            }
            
            HashMap<File, Boolean> resourceMap = new HashMap<File, Boolean>();
            for (File resourceDir : resourceDirs) {
                resourceMap.put(resourceDir, false);
                if (resourceDir.exists()) {
                    registerAll(resourceDir.getCanonicalFile().toPath(), watcher);
                    resourceMap.put(resourceDir, true);
                }
            }

            buildFile.getParentFile().toPath().register(
                    watcher, new WatchEvent.Kind[] { StandardWatchEventKinds.ENTRY_MODIFY,
                            StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_CREATE },
                    SensitivityWatchEventModifier.HIGH);
            debug("Watching build file directory: " + buildFile.getParentFile().toPath());

            if (propertyFilesMap != null) {
                for (File f : propertyFilesMap.keySet()) {
                    f.getParentFile().toPath().register(
                        watcher, new WatchEvent.Kind[] { StandardWatchEventKinds.ENTRY_MODIFY,
                                StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_CREATE },
                        SensitivityWatchEventModifier.HIGH);
                    debug("Watching property file directory: " + f.getParentFile().toPath());
                }
            }

            Collection<File> recompileJavaSources = new HashSet<File>();
            Collection<File> recompileJavaTests = new HashSet<File>();
            Collection<File> deleteJavaSources = new HashSet<File>();
            Collection<File> deleteJavaTests = new HashSet<File>();
            Collection<File> failedCompilationJavaSources = new HashSet<File>();
            Collection<File> failedCompilationJavaTests = new HashSet<File>();
            long lastJavaSourceChange = System.currentTimeMillis();
            long lastJavaTestChange = System.currentTimeMillis();
            boolean triggerJavaSourceRecompile = false;
            boolean triggerJavaTestRecompile = false;

            // initial source and test compile
            if (this.sourceDirectory.exists()) {
                Collection<File> allJavaSources = FileUtils.listFiles(this.sourceDirectory.getCanonicalFile(),
                        new String[] { "java" }, true);
                recompileJavaSources.addAll(allJavaSources);
            }
            if (this.testSourceDirectory.exists()) {
                Collection<File> allJavaTestSources = FileUtils.listFiles(this.testSourceDirectory.getCanonicalFile(),
                        new String[] { "java" }, true);
                recompileJavaTests.addAll(allJavaTestSources);
            }

            while (true) {
                // stop dev mode if the server has been stopped by another process
                if (serverThread.getState().equals(Thread.State.TERMINATED)) {
                    if (!this.devStop.get()) {
                        // server was stopped outside of dev mode
                        throw new PluginScenarioException("The server has stopped. Exiting dev mode.");
                    } else {
                        // server was stopped by dev mode
                        throw new PluginScenarioException();
                    }
                }

                // process java source files if no changes detected after the compile wait time
                boolean processSources = System.currentTimeMillis() > lastJavaSourceChange + compileWaitMillis;
                boolean processTests = System.currentTimeMillis() > lastJavaTestChange + compileWaitMillis;
                if (processSources) {
                    // delete before recompiling, so if a file is in both lists, its class will be deleted then recompiled
                    if (!deleteJavaSources.isEmpty()) {
                        debug("Deleting Java source files: " + deleteJavaSources);
                        for (File file : deleteJavaSources) {
                            deleteJavaFile(file, outputDirectory, this.sourceDirectory);
                        }
                    }
                    if (!recompileJavaSources.isEmpty() || triggerJavaSourceRecompile) {
                        // try to recompile java files that previously did not compile successfully
                        if (!failedCompilationJavaSources.isEmpty()) {
                            recompileJavaSources.addAll(failedCompilationJavaSources);
                        }
                        if (recompileJavaSource(recompileJavaSources, artifactPaths, executor,
                                outputDirectory, testOutputDirectory)) {
                            // successful compilation so we can clear failedCompilation list
                            failedCompilationJavaSources.clear();
                        } else {
                            failedCompilationJavaSources.addAll(recompileJavaSources);
                        }
                    }
                    // additionally, process java test files if no changes detected after a different timeout
                    // (but source timeout takes precedence i.e. don't recompile tests if someone keeps changing the source)
                    if (processTests) {
                        // delete before recompiling, so if a file is in both lists, its class will be deleted then recompiled
                        if (!deleteJavaTests.isEmpty()) {
                            debug("Deleting Java test files: " + deleteJavaTests);
                            for (File file : deleteJavaTests) {
                                deleteJavaFile(file, testOutputDirectory, this.testSourceDirectory);
                            }
                        }
                        if (!recompileJavaTests.isEmpty() || triggerJavaTestRecompile) {
                            debug("Recompiling Java test files: " + recompileJavaTests);
                            if (!failedCompilationJavaTests.isEmpty()) {
                                recompileJavaTests.addAll(failedCompilationJavaTests);
                            }
                            if (recompileJavaTest(recompileJavaTests, artifactPaths, executor, outputDirectory, testOutputDirectory)) {
                                // successful compilation so we can clear failedCompilation list
                                failedCompilationJavaTests.clear();
                            } else {
                                failedCompilationJavaTests.addAll(recompileJavaTests);
                            }
                        }
                    }

                    // run tests if files were deleted without any other changes, since recompileJavaSource won't run (which normally handles tests)
                    if (!deleteJavaSources.isEmpty() && recompileJavaSources.isEmpty()) {
                        // run tests after waiting for app update since app changed
                        int numApplicationUpdatedMessages = countApplicationUpdatedMessages();
                        runTestThread(true, executor, numApplicationUpdatedMessages, false, false);
                    } else if (processTests && !deleteJavaTests.isEmpty() && recompileJavaTests.isEmpty()) {
                        // run all tests without waiting for app update since only tests changed
                        runTestThread(false, executor, -1, false, false);
                    }

                    deleteJavaSources.clear();
                    recompileJavaSources.clear();
                    triggerJavaTestRecompile = false;
                    triggerJavaSourceRecompile = false;

                    if (processTests) {
                        deleteJavaTests.clear();
                        recompileJavaTests.clear();
                    }
                }

                // check if javaSourceDirectory has been added
                if (!sourceDirRegistered && this.sourceDirectory.exists()
                        && this.sourceDirectory.listFiles().length > 0) {
                    compile(this.sourceDirectory);
                    registerAll(srcPath, watcher);
                    debug("Registering Java source directory: " + this.sourceDirectory);
                    sourceDirRegistered = true;
                } else if (sourceDirRegistered && !this.sourceDirectory.exists()) {
                    cleanTargetDir(outputDirectory);
                    sourceDirRegistered = false;
                }

                // check if testSourceDirectory has been added
                if (!testSourceDirRegistered && this.testSourceDirectory.exists()
                        && this.testSourceDirectory.listFiles().length > 0) {
                    compile(this.testSourceDirectory);
                    registerAll(testSrcPath, watcher);
                    debug("Registering Java test directory: " + this.testSourceDirectory);
                    runTestThread(false, executor, -1, false, false);
                    testSourceDirRegistered = true;

                } else if (testSourceDirRegistered && !this.testSourceDirectory.exists()) {
                    cleanTargetDir(testOutputDirectory);
                    testSourceDirRegistered = false;
                }
                
                // check if configDirectory has been added
                if (!configDirRegistered && this.configDirectory.exists()){
                    configDirRegistered = true;
                    if (serverXmlFile != null && !serverXmlFile.exists()) {
                        registerAll(configPath, watcher);
                        debug("Registering configuration directory: " + this.configDirectory);
                    } else {
                        warn("The server configuration directory " + configDirectory + " has been added. Restart liberty:dev mode for it to take effect.");
                    }
                }
                
                // check if serverXmlFile has been added
                if (!serverXmlFileRegistered && serverXmlFile != null && serverXmlFile.exists()){
                    serverXmlFileRegistered = true;
                    debug("Server configuration file has been added: " + serverXmlFile);
                    warn("The server configuration file " + serverXmlFile + " has been added. Restart liberty:dev mode for it to take effect.");
                }
                
                // check if resourceDirectory has been added
                for (File resourceDir : resourceDirs) {
                    if (!resourceMap.get(resourceDir) && resourceDir.exists()) {
                        // added resource directory
                        registerAll(resourceDir.getCanonicalFile().toPath(), watcher);
                        resourceMap.put(resourceDir, true);
                    } else if (resourceMap.get(resourceDir) && !resourceDir.exists()) {
                        // deleted resource directory
                        warn("The resource directory " + resourceDir
                                + " was deleted.  Restart liberty:dev mode for it to take effect.");
                        resourceMap.put(resourceDir, false);
                    }
                }

                try {
                    final WatchKey wk = watcher.poll(100, TimeUnit.MILLISECONDS);
                    final Watchable watchable = wk.watchable();
                    final Path directory = (Path) watchable;

                    List<WatchEvent<?>> events = wk.pollEvents();

                    for (WatchEvent<?> event : events) {
                        final Path changed = (Path) event.context();
                        debug("Processing events for watched directory: " + directory);

                        File fileChanged = new File(directory.toString(), changed.toString());
                        if (ignoreFileOrDir(fileChanged)) {
                            // skip this file or directory, and continue to the next file or directory
                            continue;
                        }
                        debug("Changed: " + changed + "; " + event.kind());

                        // resource file check
                        File resourceParent = null;
                        for (File resourceDir : resourceDirs) {
                            if (directory.startsWith(resourceDir.getCanonicalFile().toPath())) {
                                resourceParent = resourceDir;
                                break;
                            }
                        }

                        if (fileChanged.isDirectory()) {
                            // if new directory added, watch the entire directory
                            if (event.kind() == StandardWatchEventKinds.ENTRY_CREATE) {
                                registerAll(fileChanged.toPath(), watcher);
                            }
                            // otherwise if a directory was modified, just continue to the next entry
                            // (if delete, can't tell if it was a directory since it doesn't exist anymore)
                            continue;
                        }
                        
                        int numApplicationUpdatedMessages = countApplicationUpdatedMessages();

                        // src/main/java directory
                        if (directory.startsWith(srcPath)) {
                            ArrayList<File> javaFilesChanged = new ArrayList<File>();
                            javaFilesChanged.add(fileChanged);
                            if (fileChanged.exists() && fileChanged.getName().endsWith(".java")
                                    && (event.kind() == StandardWatchEventKinds.ENTRY_MODIFY
                                            || event.kind() == StandardWatchEventKinds.ENTRY_CREATE)) {
                                debug("Java source file modified: " + fileChanged.getName() + ". Adding to list for processing.");
                                lastJavaSourceChange = System.currentTimeMillis();
                                recompileJavaSources.add(fileChanged);
                            } else if (event.kind() == StandardWatchEventKinds.ENTRY_DELETE) {
                                debug("Java file deleted: " + fileChanged.getName() + ". Adding to list for processing.");
                                lastJavaSourceChange = System.currentTimeMillis();
                                deleteJavaSources.add(fileChanged);
                            }
                        } else if (directory.startsWith(testSrcPath)) { // src/main/test
                            ArrayList<File> javaFilesChanged = new ArrayList<File>();
                            javaFilesChanged.add(fileChanged);
                            if (fileChanged.exists() && fileChanged.getName().endsWith(".java")
                                    && (event.kind() == StandardWatchEventKinds.ENTRY_MODIFY
                                            || event.kind() == StandardWatchEventKinds.ENTRY_CREATE)) {
                                debug("Java test file modified: " + fileChanged.getName() + ". Adding to list for processing.");
                                lastJavaTestChange = System.currentTimeMillis();
                                recompileJavaTests.add(fileChanged);
                            } else if (event.kind() == StandardWatchEventKinds.ENTRY_DELETE) {
                                debug("Java test file deleted: " + fileChanged.getName() + ". Adding to list for processing.");
                                lastJavaTestChange = System.currentTimeMillis();
                                deleteJavaTests.add(fileChanged);
                            }
                        } else if (directory.startsWith(configPath) && !isGeneratedConfigFile(fileChanged, configDirectory, serverDirectory)) { // config files
                            if (fileChanged.exists() && (event.kind() == StandardWatchEventKinds.ENTRY_MODIFY
                                    || event.kind() == StandardWatchEventKinds.ENTRY_CREATE)) {
                                copyConfigFolder(fileChanged, configDirectory, null);
                                copyFile(fileChanged, configDirectory, serverDirectory, null);
                                if (event.kind() == StandardWatchEventKinds.ENTRY_CREATE) {
                                    redeployApp();
                                }
                                if (fileChanged.getName().equals("server.env")) {
                                    // re-enable debug variables in server.env
                                    enableServerDebug(false);
                                }
                                runTestThread(true, executor, numApplicationUpdatedMessages, true, false);
                            } else if (event.kind() == StandardWatchEventKinds.ENTRY_DELETE) {
                                info("Config file deleted: " + fileChanged.getName());
                                deleteFile(fileChanged, configDirectory, serverDirectory, null);
                                if (fileChanged.getName().equals("server.env")) {
                                    // re-enable debug variables in server.env
                                    enableServerDebug(false);
                                }
                                runTestThread(true, executor, numApplicationUpdatedMessages, true, false);
                            }
                        } else if (serverXmlFileParent != null && directory.startsWith(serverXmlFileParent.getCanonicalFile().toPath())) {
                            if (fileChanged.exists() && fileChanged.getCanonicalPath().endsWith(serverXmlFile.getName())
                                    && (event.kind() == StandardWatchEventKinds.ENTRY_MODIFY
                                            || event.kind() == StandardWatchEventKinds.ENTRY_CREATE)) {
                                copyConfigFolder(fileChanged, serverXmlFileParent, "server.xml");
                                copyFile(fileChanged, serverXmlFileParent, serverDirectory,
                                        "server.xml");
                                if (event.kind() == StandardWatchEventKinds.ENTRY_CREATE) {
                                    redeployApp();
                                }
                                runTestThread(true, executor, numApplicationUpdatedMessages, true, false);

                            } else if (event.kind() == StandardWatchEventKinds.ENTRY_DELETE
                                    && fileChanged.getCanonicalPath().endsWith(serverXmlFile.getName())) {
                                info("Config file deleted: " + fileChanged.getName());
                                deleteFile(fileChanged, configDirectory, serverDirectory, "server.xml");
                                runTestThread(true, executor, numApplicationUpdatedMessages, true, false);
                            }
                        } else if (resourceParent != null && directory.startsWith(resourceParent.getCanonicalFile().toPath())) { // resources
                            debug("Resource dir: " + resourceParent.toString());
                            if (fileChanged.exists() && (event.kind() == StandardWatchEventKinds.ENTRY_MODIFY
                                    || event.kind() == StandardWatchEventKinds.ENTRY_CREATE)) {
                                copyFile(fileChanged, resourceParent, outputDirectory, null);

                                // run all tests on resource change
                                runTestThread(true, executor, numApplicationUpdatedMessages, false, false);
                            } else if (event.kind() == StandardWatchEventKinds.ENTRY_DELETE) {
                                debug("Resource file deleted: " + fileChanged.getName());
                                deleteFile(fileChanged, resourceParent, outputDirectory, null);
                                // run all tests on resource change
                                runTestThread(true, executor, numApplicationUpdatedMessages, false, false);
                            }
                        } else if (fileChanged.equals(buildFile)
                                && directory.startsWith(buildFile.getParentFile().getCanonicalFile().toPath())
                                && event.kind() == StandardWatchEventKinds.ENTRY_MODIFY) { // pom.xml

                                    boolean recompiledBuild = recompileBuildFile(buildFile, artifactPaths, executor);
                                    // run all tests on build file change
                                    if (recompiledBuild) {
                                        // trigger java source recompile if there are compilation errors
                                        if (!failedCompilationJavaSources.isEmpty()) {
                                            triggerJavaSourceRecompile = true;
                                        }
                                        // trigger java test recompile if there are compilation errors
                                        if (!failedCompilationJavaTests.isEmpty()) {
                                            triggerJavaTestRecompile = true;
                                        }
                                        runTestThread(true, executor, numApplicationUpdatedMessages, false, false);
                                    }
                        } else if (propertyFilesMap != null && propertyFilesMap.keySet().contains(fileChanged)) { // properties file
                            boolean reloadedPropertyFile = reloadPropertyFile(fileChanged);
                            // run all tests on properties file change
                            if (reloadedPropertyFile) {
                                runTestThread(true, executor, numApplicationUpdatedMessages, false, false);
                            }
                        }
                    }
                    // reset the key
                    boolean valid = wk.reset();
                    if (!valid) {
                        debug("WatchService key has been unregistered for " + directory);
                    }
                } catch (InterruptedException | NullPointerException e) {
                    // do nothing let loop continue
                }
            }
        }
    }

    /**
     * Determines if the corresponding target config file was generated by a Liberty plugin
     * 
     * @param fileChanged the file that was changed
     * @param srcDir the directory of the file changed
     * @param targetDir the target directory
     * @throws IOException unable to resolve canonical path
     */
    protected boolean isGeneratedConfigFile(File fileChanged, File srcDir, File targetDir) throws IOException {
        return (fileChanged.getName().equals("bootstrap.properties") || fileChanged.getName().equals("jvm.options"))
                && isGeneratedTargetFile(fileChanged, srcDir, targetDir);
    }

    private boolean isGeneratedTargetFile(File fileChanged, File srcDir, File targetDir) throws IOException {
        File targetFile = getTargetFile(fileChanged, srcDir, targetDir, null);
        try (FileReader fileReader = new FileReader(targetFile);
                BufferedReader bufferedReader = new BufferedReader(fileReader)) {
            String line = bufferedReader.readLine();
            return line.matches(GENERATED_HEADER_REGEX);
        } catch (IOException e) {
            // If the target file could not be read, assume it was not a generated file
            debug("Could not read the target file " + targetFile + ". It will be replaced by the contents of "
                    + fileChanged, e);
        }
        return false;
    }

    /**
     * Reads the file to a String
     * 
     * @param file
     * @return String representation of the file
     * @throws IOException unable to read file to string
     */
    public String readFile(File file) throws IOException {
        return FileUtils.readFileToString(file, StandardCharsets.UTF_8);
    }

    /**
     * Creates a temporary copy of the configuration file and checks the
     * configFile in the temporary directory to avoid install-feature timing
     * issues
     * 
     * @param fileChanged the file that was changed
     * @param srcDir the directory of the file changed
     * @param targetFileName if not null renames the fileChanged to targetFileName in the targetDir
     * @throws IOException creating and copying to tempConfig directory
     */
    public void copyConfigFolder(File fileChanged, File srcDir, String targetFileName)
            throws IOException {
        this.tempConfigPath = Files.createTempDirectory("tempConfig");
        File tempConfig = tempConfigPath.toFile();
        debug("Temporary configuration folder created: " + tempConfig);
        
        FileUtils.copyDirectory(serverDirectory, tempConfig, new FileFilter() {
            public boolean accept(File pathname) {
                String name = pathname.getName();
                // skip:
                // - ignore list
                // - workarea and logs dirs from the server directory, since those can be changing
                boolean skip = ignoreFileOrDir(pathname)
                        || (pathname.isDirectory() && (name.equals("workarea") || name.equals("logs")));
                return !skip;
            }
        }, true);
        copyFile(fileChanged, srcDir, tempConfig, targetFileName);
        checkConfigFile(fileChanged, tempConfig);
        cleanUpTempConfig();
    }

    /**
     * Whether dev mode should ignore a file or directory.
     * 
     * @param file File or directory
     * @return true if the file or directory should be ignored, false otherwise
     */
    private boolean ignoreFileOrDir(File file) {
        String name = file.getName();
        if (file.isDirectory()) {
            for (String prefix : IGNORE_DIRECTORY_PREFIXES) {
                if (name.startsWith(prefix)) {
                    debug("Ignoring " + name);
                    return true;
                }
            }
        } else {
            for (String prefix : IGNORE_FILE_PREFIXES) {
                if (name.startsWith(prefix)) {
                    debug("Ignoring " + name);
                    return true;
                }
            }
            for (String postfix : IGNORE_FILE_POSTFIXES) {
                if (name.endsWith(postfix)) {
                    debug("Ignoring " + name);
                    return true;
                }
            }    
        }
        return false;
    }

    /**
     * Copies the fileChanged from the srcDir to the targetDir.
     * 
     * @param fileChanged the file that was changed
     * @param srcDir the directory of the file changed
     * @param targetDir the target directory
     * @param targetFileName if not null renames the fileChanged to targetFileName in the targetDir
     * @throws IOException unable to resolve canonical path
     */
    public void copyFile(File fileChanged, File srcDir, File targetDir, String targetFileName) throws IOException {
        File targetResource = getTargetFile(fileChanged, srcDir, targetDir, targetFileName);

        try {
            FileUtils.copyFile(fileChanged, targetResource);
            info("Copied file: " + fileChanged.getCanonicalPath() + " to: " + targetResource.getCanonicalPath());
        } catch (FileNotFoundException ex) {
            debug("Failed to copy file: " + fileChanged.getCanonicalPath());
        } catch (Exception ex) {
            debug(ex);
        }
    }

    private File getTargetFile(File fileChanged, File srcDir, File targetDir, String targetFileName) throws IOException {
        String relPath = fileChanged.getCanonicalPath().substring(
                fileChanged.getCanonicalPath().indexOf(srcDir.getCanonicalPath()) + srcDir.getCanonicalPath().length());
        if (targetFileName != null) {
            relPath = relPath.substring(0, relPath.indexOf(fileChanged.getName())) + targetFileName;
        }
        File targetResource = new File(targetDir.getCanonicalPath() + relPath);
        return targetResource;
    }

    /**
     * Deletes the corresponding file in the targetDir.
     * 
     * @param deletedFile the file that was deleted
     * @param dir the directory of the deletedFile
     * @param targetDir the corresponding targetDir of the deletedFile
     * @param targetFileName if not null deletes the targetFile with this name
     * @throws IOException unable to resolve canonical path
     */
    protected void deleteFile(File deletedFile, File dir, File targetDir, String targetFileName) throws IOException {
        File targetFile = getTargetFile(deletedFile, dir, targetDir, targetFileName);
        if (targetFile.exists()) {
            if (targetFile.isDirectory()) {
                try {
                    FileUtils.deleteDirectory(targetFile);
                    info("The directory " + targetFile.getCanonicalPath() + " was deleted.");
                } catch (IllegalArgumentException e) {
                    debug("Could not delete the directory " + targetFile.getCanonicalPath() + ". " + e.getMessage());
                } catch (IOException e) {
                    error("There was an error encountered while deleting the directory " + targetFile.getCanonicalPath()
                            + ". " + e.getMessage());
                }
            } else {
                if (targetFile.delete()){
                    info("The file " + targetFile.getCanonicalPath() + " was deleted.");
                } else {
                    error("Could not delete the file " + targetFile.getCanonicalPath() + ".");
                }
            }
        }
    }

    /**
     * Delete all the Java class files within the specified directory.
     * If the directory is empty, deletes the directory as well.
     *  
     * @param outputDirectory the directory for compiled classes
     */
    protected void cleanTargetDir(File outputDirectory){
        File[] fList = outputDirectory.listFiles();
        if (fList != null) {
            for (File file : fList) {
                if (file.isFile() && file.getName().toLowerCase().endsWith(".class")) {
                   file.delete();
                   info("Deleted Java class file: " + file);
                } else if (file.isDirectory()) {
                    cleanTargetDir(file);
                }
            }
        }
        if (outputDirectory.listFiles().length == 0){
            outputDirectory.delete();
        }
    }

    /**
     * Register the parent directory and all sub-directories with the WatchService
     * 
     * @param start parent directory
     * @param watcher WatchService
     * @throws IOException unable to walk through file tree 
     */
    protected void registerAll(final Path start, final WatchService watcher) throws IOException {
        // register directory and sub-directories
        Files.walkFileTree(start, new SimpleFileVisitor<Path>() {
            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                debug("Watching directory: " + dir.toString());
                dir.register(watcher,
                        new WatchEvent.Kind[] { StandardWatchEventKinds.ENTRY_MODIFY,
                                StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_CREATE },
                        SensitivityWatchEventModifier.HIGH);

                return FileVisitResult.CONTINUE;
            }

        });
    }
    
    /**
     * Get the file from the configDirectory if it exists
     * 
     * @param file 
     * @return file or null if it does not exist
     */
    protected File getFileFromConfigDirectory(String file) {
        File f = new File(configDirectory, file);
        if (configDirectory != null && f.exists()) {
            return f;
        }
        return null;
    }

    /**
     * Given the fileChanged delete the corresponding Java class or directory
     * 
     * @param fileChanged       Java file changed
     * @param classesDir        the directory for compiled classes
     * @param compileSourceRoot the source directory for the Java classes
     * @throws IOException unable to resolve canonical path
     */
    protected void deleteJavaFile(File fileChanged, File classesDir, File compileSourceRoot) throws IOException {
        String fileName = fileChanged.getName();
        File parentFile = fileChanged.getParentFile();

        boolean javaFile = fileName.endsWith(".java");
        String relPath;
        if (javaFile) {
            fileName = fileName.substring(0, fileChanged.getName().indexOf(".java"));
            relPath = parentFile.getCanonicalPath()
                    .substring(parentFile.getCanonicalPath().indexOf(compileSourceRoot.getCanonicalPath())
                            + compileSourceRoot.getCanonicalPath().length())
                    + "/" + fileName + ".class";
        } else {
            relPath = parentFile.getCanonicalPath()
                    .substring(parentFile.getCanonicalPath().indexOf(compileSourceRoot.getCanonicalPath())
                            + compileSourceRoot.getCanonicalPath().length())
                    + "/" + fileName;
        }

        File targetFile = new File(classesDir.getCanonicalPath() + relPath);
        if (targetFile.exists()) {
            if (targetFile.isDirectory()) {
                try {
                    FileUtils.deleteDirectory(targetFile);
                    info("The target directory " + targetFile.getCanonicalPath() + " was deleted.");
                } catch (IllegalArgumentException e) {
                    debug("Could not delete directory " + targetFile.getCanonicalPath() + ". " + e.getMessage());
                } catch (IOException e) {
                    error("There was an error encountered while deleting the directory " + targetFile.getCanonicalPath()
                            + ". " + e.getMessage());
                }
            } else {
                if (targetFile.delete()) {
                    info("The java class " + targetFile.getCanonicalPath() + " was deleted.");
                } else {
                    error("Could not delete the file " + targetFile.getCanonicalPath() + ". ");
                }
            }
        } else {
            warn("File deleted but could not find corresponding file or folder in the target directory: "
                    + fileChanged.getCanonicalPath() + ".");
        }
    }

    /**
     * Recompile Java source files and run tests after application update
     * 
     * @param javaFilesChanged collection of Java files changed
     * @param artifactPaths list of project artifact paths for building the classpath
     * @param executor the test thread executor
     * @param outputDirectory the directory for compiled classes
     * @param testOutputDirectory the directory for compiled test classes
     * @throws PluginExecutionException if the classes output directory doesn't exist and can't be created
     */
    protected boolean recompileJavaSource(Collection<File> javaFilesChanged, List<String> artifactPaths,
            ThreadPoolExecutor executor, File outputDirectory, File testOutputDirectory) throws PluginExecutionException {
        return recompileJava(javaFilesChanged, artifactPaths, executor, false, outputDirectory, testOutputDirectory);
    }

    /**
     * Recompile test source files and run tests immediately
     * 
     * @param javaFilesChanged collection of Java files changed
     * @param artifactPaths list of project artifact paths for building the classpath
     * @param executor the test thread executor
     * @param outputDirectory the directory for compiled classes
     * @param testOutputDirectory the directory for compiled test classes
     * @throws PluginExecutionException if the classes output directory doesn't exist and can't be created
     */
    protected boolean recompileJavaTest(Collection<File> javaFilesChanged, List<String> artifactPaths,
            ThreadPoolExecutor executor, File outputDirectory, File testOutputDirectory) throws PluginExecutionException {
        return recompileJava(javaFilesChanged, artifactPaths, executor, true, outputDirectory, testOutputDirectory);
    }

    /**
     * Recompile source files
     * 
     * @param javaFilesChanged collection of Java files changed
     * @param artifactPaths list of project artifact paths for building the classpath
     * @param executor the test thread executor
     * @param tests indicates whether the files changed were test files
     * @param outputDirectory the directory for compiled classes
     * @param testOutputDirectory the directory for compiled test classes
     * @throws PluginExecutionException if the classes output directory doesn't exist and can't be created
     */
    protected boolean recompileJava(Collection<File> javaFilesChanged, List<String> artifactPaths, ThreadPoolExecutor executor,
            boolean tests, File outputDirectory, File testOutputDirectory) throws PluginExecutionException {
        try {
            int messageOccurrences = countApplicationUpdatedMessages();
            boolean compileResult;
            
            if (useBuildRecompile) {
                compileResult = compile(tests ? testSourceDirectory : sourceDirectory);
            } else {
                // source root is src/main/java or src/test/java
                File classesDir = tests ? testOutputDirectory : outputDirectory;
                if (!classesDir.exists()) {
                    if (!classesDir.mkdirs()) {
                        throw new PluginExecutionException("The classes output directory " + classesDir.getAbsolutePath()
                                + " does not exist and cannot be created.");
                    } else if (classesDir.exists() && Objects.equals(classesDir.getCanonicalFile(), outputDirectory.getCanonicalFile())) {
                        // redeploy application when class directory has been created
                        redeployApp();
                    }
                }

                List<String> optionList = new ArrayList<>();
                List<File> outputDirs = new ArrayList<File>();

                if (tests) {
                    outputDirs.add(outputDirectory);
                    outputDirs.add(testOutputDirectory);
                } else {
                    outputDirs.add(outputDirectory);
                }
                Set<File> classPathElems = getClassPath(artifactPaths, outputDirs);

                JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();
                StandardJavaFileManager fileManager = compiler.getStandardFileManager(null, null, null);

                fileManager.setLocation(StandardLocation.CLASS_PATH, classPathElems);
                fileManager.setLocation(StandardLocation.CLASS_OUTPUT, Collections.singleton(classesDir));

                Collection<JavaFileObject> compilationUnits = new HashSet<JavaFileObject>();
                for (File file : javaFilesChanged) {
                    if (file.exists() && file.isFile()) {
                        for (JavaFileObject o : fileManager.getJavaFileObjects(file)) {
                            compilationUnits.add(o);
                        }    
                    } else {
                        debug("The Java file " + file + " does not exist and will not be compiled.");
                    }
                }

                JavaCompiler.CompilationTask task = compiler.getTask(null, fileManager, null, optionList, null,
                        compilationUnits);

                compileResult = task.call();
            }
            if (compileResult) {
                if (tests) {
                    info("Tests compilation was successful.");
                } else {
                    info("Source compilation was successful.");
                }

                // run tests after successful compile
                if (tests) {
                    // if only tests were compiled, don't need to wait for
                    // app to update
                    runTestThread(false, executor, -1, false, false);
                } else {
                    runTestThread(true, executor, messageOccurrences, false, false);
                }
                return true;
            } else {
                if (tests) {
                    info("Tests compilation had errors.");
                } else {
                    info("Source compilation had errors.");
                }
                return false;
            }
        } catch (Exception e) {
            debug("Error compiling java files", e);
            return false;
        }
    }

    /**
     * Gets the class path for the specified artifactPaths and outputDirs.
     * 
     * @param artifactPaths list of artifacts for the current project
     * @param outputDirs list of output directories for the current project
     * @return set of classpath files
     * @throws IOException unable to resolve canonical path
     */
    protected Set<File> getClassPath(List<String> artifactPaths, List<File> outputDirs) throws IOException {
        List<URL> urls = new ArrayList<>();
        ClassLoader c = Thread.currentThread().getContextClassLoader();
        while (c != null) {
            if (c instanceof URLClassLoader) {
                urls.addAll(Arrays.asList(((URLClassLoader) c).getURLs()));
            }
            c = c.getParent();
        }

        Set<String> parsedFiles = new HashSet<>();
        Deque<String> toParse = new ArrayDeque<>();
        for (URL url : urls) {
            toParse.add(new File(url.getPath()).getCanonicalPath());
        }

        for (String artifactPath : artifactPaths) {
            toParse.add(new File(artifactPath).getCanonicalPath());
        }

        Set<File> classPathElements = new HashSet<>();
        classPathElements.addAll(outputDirs);
        while (!toParse.isEmpty()) {
            String s = toParse.poll();
            if (!parsedFiles.contains(s)) {
                parsedFiles.add(s);
                File file = new File(s);
                if (file.exists() && file.getName().endsWith(".jar")) {
                    classPathElements.add(file);
                    if (!file.isDirectory()) {
                        try (JarFile jar = new JarFile(file)) {
                            Manifest mf = jar.getManifest();
                            if (mf == null || mf.getMainAttributes() == null) {
                                continue;
                            }
                            Object classPath = mf.getMainAttributes().get(Attributes.Name.CLASS_PATH);
                            if (classPath != null) {
                                for (String i : classPath.toString().split(" ")) {
                                    File f;
                                    try {
                                        URL u = new URL(i);
                                        f = new File(u.getPath());
                                    } catch (MalformedURLException e) {
                                        f = new File(file.getParentFile(), i);
                                    }
                                    if (f.exists()) {
                                        toParse.add(f.getCanonicalPath());
                                    }
                                }
                            }
                        } catch (Exception e) {
                            throw new RuntimeException("Failed to open class path file " + file, e);
                        }
                    }
                }
            }
        }
        return classPathElements;
    }

    /**
     * Run tests in a new thread.
     * 
     * @param waitForApplicationUpdate whether it should wait for the application to
     *                                 update before running integration tests
     * @param executor                 the thread pool executor
     * @param messageOccurrences       how many times the application updated
     *                                 message has occurred in the log
     * @param forceSkipUTs             whether to force skip the unit tests
     * @param manualInvocation         whether the tests were manually invoked
     */
    public void runTestThread(boolean waitForApplicationUpdate, ThreadPoolExecutor executor, int messageOccurrences,
            boolean forceSkipUTs, boolean manualInvocation) {
        try {
            if (manualInvocation || hotTests) {
                executor.execute(new TestJob(waitForApplicationUpdate, messageOccurrences, executor, forceSkipUTs,
                        manualInvocation));
            }
        } catch (RejectedExecutionException e) {
            debug("Cannot add thread since max threads reached", e);
        }
    }

    public class TestJob implements Runnable {
        private boolean waitForApplicationUpdate;
        private int messageOccurrences;
        private ThreadPoolExecutor executor;
        private boolean forceSkipUTs;
        private boolean manualInvocation;

        public TestJob(boolean waitForApplicationUpdate, int messageOccurrences, ThreadPoolExecutor executor, boolean forceSkipUTs, boolean manualInvocation) {
            this.waitForApplicationUpdate = waitForApplicationUpdate;
            this.messageOccurrences = messageOccurrences;
            this.executor = executor;
            this.forceSkipUTs = forceSkipUTs;
            this.manualInvocation = manualInvocation;
        }

        @Override
        public void run() {
            try {
                runTests(waitForApplicationUpdate, messageOccurrences, executor, forceSkipUTs);
            } finally {
                // start watching for hotkey presses if not already started, or re-print message if thread already running
                runHotkeyReaderThread(executor);
            }
        }

        public boolean isManualInvocation() {
            return manualInvocation;
        }
    }

    /**
     * Gets the Liberty server's host name.
     * @return hostName the host name, or null if the server is not started
     */
    public String getHostName() {
        return hostName;
    }

    /**
     * Gets the Liberty server's http port.
     * @return httpPort the http port, or null if the server is not started or there is no http port bound
     */
    public String getHttpPort() {
        return httpPort;
    }

    /**
     * Gets the Liberty server's https port.
     * @return httpsPort the https port, or null if the server is not started or there is no https port bound
     */
    public String getHttpsPort() {
        return httpsPort;
    }

    /**
     * Sets the preferred debug port.
     * 
     * @param libertyDebugPort the preferred debug port
     */
    public void setLibertyDebugPort(int libertyDebugPort) {
        this.libertyDebugPort = libertyDebugPort;
    }

    /**
     * Reload the property file by restarting the server if there were changes.
     * 
     * @param propertyFile The property file that was changed.
     * @throws PluginExecutionException if there was an error when reloading the file
     * @return true if the property file was reloaded with changes
     */
    private boolean reloadPropertyFile(File propertyFile) throws PluginExecutionException {
        Properties properties = readPropertiesFromFile(propertyFile);
        if (!Objects.equals(properties, propertyFilesMap.get(propertyFile))) {
            debug("Properties file " + propertyFile.getAbsolutePath() + " has changed. Restarting server...");
            propertyFilesMap.put(propertyFile, properties);    
            restartServer();
            return true;
        } else {
            debug("No changes detected in properties file " + propertyFile.getAbsolutePath());
            return false;
        }
    }

    /**
     * This is needed for Gradle only.
     * 
     * Sets additional property files that may be used by the build.
     * Loads the properties for later comparison of changes.
     * 
     * @param propertyFiles list of property files
     */
    public void setPropertyFiles(List<File> propertyFiles) {
        if (propertyFiles == null) {
            return;
        }
        if (propertyFilesMap == null) {
            propertyFilesMap = new HashMap<File, Properties>(propertyFiles.size());
        }
        for (File propertyFile : propertyFiles) {
            Properties properties = readPropertiesFromFile(propertyFile);
            propertyFilesMap.put(propertyFile, properties);
        }
    }

    /**
     * Read properties from file.  If file does not exist or an IO exception occurred, returns null.
     */
    private Properties readPropertiesFromFile(File propertyFile) {
        Properties properties = null;
        if (propertyFile.exists()) {
            InputStream inputStream = null;
            try {
                debug("Loading properties from file: " + propertyFile);
                inputStream = new FileInputStream(propertyFile);
                properties = new Properties();
                properties.load(inputStream);
            } catch (IOException e) {
                error("Could not read properties file " + propertyFile.getAbsolutePath(), e);
            } finally {
                if (inputStream != null) {
                    try {
                        inputStream.close();
                    } catch (IOException e) {
                        // nothing to do
                    }
                }
            }
        }
        return properties;
    }

}
