/*
 * Decompiled with CFR 0.152.
 */
package io.opentelemetry.contrib.aws.resource;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.api.common.AttributesBuilder;
import io.opentelemetry.contrib.aws.resource.IncubatingAttributes;
import io.opentelemetry.contrib.aws.resource.SimpleHttpClient;
import io.opentelemetry.sdk.resources.Resource;
import io.opentelemetry.semconv.ResourceAttributes;
import java.io.IOException;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;

public final class EcsResource {
    private static final Logger logger = Logger.getLogger(EcsResource.class.getName());
    private static final JsonFactory JSON_FACTORY = new JsonFactory();
    private static final String ECS_METADATA_KEY_V4 = "ECS_CONTAINER_METADATA_URI_V4";
    private static final String ECS_METADATA_KEY_V3 = "ECS_CONTAINER_METADATA_URI";
    private static final Resource INSTANCE = EcsResource.buildResource();

    public static Resource get() {
        return INSTANCE;
    }

    private static Resource buildResource() {
        return EcsResource.buildResource(System.getenv(), new SimpleHttpClient());
    }

    static Resource buildResource(Map<String, String> sysEnv, SimpleHttpClient httpClient) {
        String ecsMetadataUrl = sysEnv.getOrDefault(ECS_METADATA_KEY_V4, sysEnv.getOrDefault(ECS_METADATA_KEY_V3, ""));
        if (!ecsMetadataUrl.isEmpty()) {
            AttributesBuilder attrBuilders = Attributes.builder();
            EcsResource.fetchMetadata(httpClient, ecsMetadataUrl, attrBuilders);
            EcsResource.fetchMetadata(httpClient, ecsMetadataUrl + "/task", attrBuilders);
            return Resource.create((Attributes)attrBuilders.build(), (String)"https://opentelemetry.io/schemas/1.25.0");
        }
        return Resource.empty();
    }

    static void fetchMetadata(SimpleHttpClient httpClient, String url, AttributesBuilder attrBuilders) {
        String json = httpClient.fetchString("GET", url, Collections.emptyMap(), null);
        if (json.isEmpty()) {
            return;
        }
        attrBuilders.put(IncubatingAttributes.CLOUD_PROVIDER, (Object)"aws");
        attrBuilders.put(IncubatingAttributes.CLOUD_PLATFORM, (Object)"aws_ecs");
        try (JsonParser parser = JSON_FACTORY.createParser(json);){
            parser.nextToken();
            LogArnBuilder logArnBuilder = new LogArnBuilder();
            EcsResource.parseResponse(parser, attrBuilders, logArnBuilder);
            logArnBuilder.getLogGroupArn().ifPresent(logGroupArn -> attrBuilders.put(IncubatingAttributes.AWS_LOG_GROUP_ARNS, Collections.singletonList(logGroupArn)));
            logArnBuilder.getLogStreamArn().ifPresent(logStreamArn -> attrBuilders.put(IncubatingAttributes.AWS_LOG_STREAM_ARNS, Collections.singletonList(logStreamArn)));
        }
        catch (IOException e) {
            logger.log(Level.WARNING, "Can't get ECS metadata", e);
        }
    }

    private static Optional<String> getAccountId(@Nullable String arn) {
        return EcsResource.getArnPart(arn, ArnPart.ACCOUNT);
    }

    private static Optional<String> getRegion(@Nullable String arn) {
        return EcsResource.getArnPart(arn, ArnPart.REGION);
    }

    private static Optional<String> getArnPart(@Nullable String arn, ArnPart arnPart) {
        if (arn == null) {
            return Optional.empty();
        }
        String[] arnParts = arn.split(":");
        if (arnPart.partIndex >= arnParts.length) {
            return Optional.empty();
        }
        return Optional.of(arnParts[arnPart.partIndex]);
    }

    static void parseResponse(JsonParser parser, AttributesBuilder attrBuilders, LogArnBuilder logArnBuilder) throws IOException {
        if (!parser.isExpectedStartObjectToken()) {
            logger.log(Level.WARNING, "Couldn't parse ECS metadata, invalid JSON");
            return;
        }
        String arn = null;
        String cluster = null;
        block34: while (parser.nextToken() != JsonToken.END_OBJECT) {
            String value = parser.nextTextValue();
            switch (parser.currentName()) {
                case "AvailabilityZone": {
                    attrBuilders.put(IncubatingAttributes.CLOUD_AVAILABILITY_ZONE, (Object)value);
                    continue block34;
                }
                case "DockerId": {
                    attrBuilders.put(IncubatingAttributes.CONTAINER_ID, (Object)value);
                    continue block34;
                }
                case "DockerName": {
                    attrBuilders.put(IncubatingAttributes.CONTAINER_NAME, (Object)value);
                    continue block34;
                }
                case "Cluster": {
                    cluster = value;
                    continue block34;
                }
                case "ContainerARN": {
                    arn = value;
                    attrBuilders.put(IncubatingAttributes.AWS_ECS_CONTAINER_ARN, (Object)value);
                    attrBuilders.put(IncubatingAttributes.CLOUD_RESOURCE_ID, (Object)value);
                    logArnBuilder.setContainerArn(value);
                    continue block34;
                }
                case "Image": {
                    DockerImage parsedImage = DockerImage.parse(value);
                    if (parsedImage == null) continue block34;
                    attrBuilders.put(IncubatingAttributes.CONTAINER_IMAGE_NAME, (Object)parsedImage.getRepository());
                    attrBuilders.put(ResourceAttributes.CONTAINER_IMAGE_TAG, (Object)parsedImage.getTag());
                    continue block34;
                }
                case "ImageID": {
                    attrBuilders.put("aws.ecs.container.image.id", value);
                    continue block34;
                }
                case "LogOptions": {
                    EcsResource.parseResponse(parser, attrBuilders, logArnBuilder);
                    continue block34;
                }
                case "awslogs-group": {
                    attrBuilders.put(IncubatingAttributes.AWS_LOG_GROUP_NAMES, (Object[])new String[]{value});
                    logArnBuilder.setLogGroupName(value);
                    continue block34;
                }
                case "awslogs-stream": {
                    attrBuilders.put(IncubatingAttributes.AWS_LOG_STREAM_NAMES, (Object[])new String[]{value});
                    logArnBuilder.setLogStreamName(value);
                    continue block34;
                }
                case "awslogs-region": {
                    logArnBuilder.setRegion(value);
                    continue block34;
                }
                case "TaskARN": {
                    arn = value;
                    attrBuilders.put(IncubatingAttributes.AWS_ECS_TASK_ARN, (Object)value);
                    continue block34;
                }
                case "LaunchType": {
                    attrBuilders.put(IncubatingAttributes.AWS_ECS_LAUNCHTYPE, (Object)value.toLowerCase(Locale.ROOT));
                    continue block34;
                }
                case "Family": {
                    attrBuilders.put(IncubatingAttributes.AWS_ECS_TASK_FAMILY, (Object)value);
                    continue block34;
                }
                case "Revision": {
                    attrBuilders.put(IncubatingAttributes.AWS_ECS_TASK_REVISION, (Object)value);
                    continue block34;
                }
            }
            parser.skipChildren();
        }
        String region = EcsResource.getRegion(arn).orElse(null);
        String account = EcsResource.getAccountId(arn).orElse(null);
        if (region != null) {
            attrBuilders.put(IncubatingAttributes.CLOUD_REGION, (Object)region);
        }
        if (account != null) {
            attrBuilders.put(IncubatingAttributes.CLOUD_ACCOUNT_ID, (Object)account);
        }
        if (cluster != null) {
            if (cluster.contains(":")) {
                attrBuilders.put(IncubatingAttributes.AWS_ECS_CLUSTER_ARN, (Object)cluster);
            } else {
                String clusterArn = String.format("arn:aws:ecs:%s:%s:cluster/%s", region, account, cluster);
                attrBuilders.put(IncubatingAttributes.AWS_ECS_CLUSTER_ARN, (Object)clusterArn);
            }
        }
    }

    private EcsResource() {
    }

    private static class LogArnBuilder {
        @Nullable
        String region;
        @Nullable
        String account;
        @Nullable
        String logGroupName;
        @Nullable
        String logStreamName;

        private LogArnBuilder() {
        }

        void setRegion(@Nullable String region) {
            this.region = region;
        }

        void setLogGroupName(@Nullable String logGroupName) {
            this.logGroupName = logGroupName;
        }

        void setLogStreamName(@Nullable String logStreamName) {
            this.logStreamName = logStreamName;
        }

        void setContainerArn(@Nullable String containerArn) {
            this.account = EcsResource.getAccountId(containerArn).orElse(null);
        }

        Optional<String> getLogGroupArn() {
            if (this.region == null || this.account == null || this.logGroupName == null) {
                return Optional.empty();
            }
            return Optional.of("arn:aws:logs:" + this.region + ":" + this.account + ":log-group:" + this.logGroupName);
        }

        Optional<String> getLogStreamArn() {
            if (this.region == null || this.account == null || this.logGroupName == null || this.logStreamName == null) {
                return Optional.empty();
            }
            return Optional.of("arn:aws:logs:" + this.region + ":" + this.account + ":log-group:" + this.logGroupName + ":log-stream:" + this.logStreamName);
        }
    }

    private static enum ArnPart {
        REGION(3),
        ACCOUNT(4);

        final int partIndex;

        private ArnPart(int partIndex) {
            this.partIndex = partIndex;
        }
    }

    private static class DockerImage {
        private static final Pattern imagePattern = Pattern.compile("^(?<repository>([^/\\s]+/)?([^:\\s]+))(:(?<tag>[^@\\s]+))?(@sha256:(?<sha256>\\d+))?$");
        final String repository;
        final String tag;

        private DockerImage(String repository, String tag) {
            this.repository = repository;
            this.tag = tag;
        }

        String getRepository() {
            return this.repository;
        }

        String getTag() {
            return this.tag;
        }

        @Nullable
        static DockerImage parse(@Nullable String image) {
            if (image == null || image.isEmpty()) {
                return null;
            }
            Matcher matcher = imagePattern.matcher(image);
            if (!matcher.matches()) {
                logger.log(Level.WARNING, "Couldn't parse image '" + image + "'");
                return null;
            }
            String repository = matcher.group("repository");
            String tag = matcher.group("tag");
            if (tag == null || tag.isEmpty()) {
                tag = "latest";
            }
            return new DockerImage(repository, tag);
        }
    }
}

