/*
 * Decompiled with CFR 0.152.
 */
package io.opentelemetry.sdk.trace.export;

import com.google.common.annotations.VisibleForTesting;
import io.opentelemetry.api.OpenTelemetry;
import io.opentelemetry.api.common.Labels;
import io.opentelemetry.api.internal.Utils;
import io.opentelemetry.api.metrics.LongCounter;
import io.opentelemetry.api.metrics.LongValueObserver;
import io.opentelemetry.api.metrics.Meter;
import io.opentelemetry.context.Context;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.common.DaemonThreadFactory;
import io.opentelemetry.sdk.common.export.ConfigBuilder;
import io.opentelemetry.sdk.trace.ReadWriteSpan;
import io.opentelemetry.sdk.trace.ReadableSpan;
import io.opentelemetry.sdk.trace.SpanProcessor;
import io.opentelemetry.sdk.trace.data.SpanData;
import io.opentelemetry.sdk.trace.export.SpanExporter;
import java.util.ArrayList;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class BatchSpanProcessor
implements SpanProcessor {
    private static final String WORKER_THREAD_NAME = BatchSpanProcessor.class.getSimpleName() + "_WorkerThread";
    private static final String SPAN_PROCESSOR_TYPE_LABEL = "spanProcessorType";
    private static final String SPAN_PROCESSOR_TYPE_VALUE = BatchSpanProcessor.class.getSimpleName();
    private final Worker worker;
    private final boolean sampled;
    private final AtomicBoolean isShutdown = new AtomicBoolean(false);

    private BatchSpanProcessor(SpanExporter spanExporter, boolean sampled, long scheduleDelayMillis, int maxQueueSize, int maxExportBatchSize, int exporterTimeoutMillis) {
        this.worker = new Worker(spanExporter, scheduleDelayMillis, maxExportBatchSize, exporterTimeoutMillis, new ArrayBlockingQueue(maxQueueSize));
        Thread workerThread = new DaemonThreadFactory(WORKER_THREAD_NAME).newThread((Runnable)this.worker);
        workerThread.start();
        this.sampled = sampled;
    }

    @Override
    public void onStart(Context parentContext, ReadWriteSpan span) {
    }

    @Override
    public boolean isStartRequired() {
        return false;
    }

    @Override
    public void onEnd(ReadableSpan span) {
        if (this.sampled && !span.getSpanContext().isSampled()) {
            return;
        }
        this.worker.addSpan(span);
    }

    @Override
    public boolean isEndRequired() {
        return true;
    }

    @Override
    public CompletableResultCode shutdown() {
        if (this.isShutdown.getAndSet(true)) {
            return CompletableResultCode.ofSuccess();
        }
        return this.worker.shutdown();
    }

    @Override
    public CompletableResultCode forceFlush() {
        return this.worker.forceFlush();
    }

    public static Builder builder(SpanExporter spanExporter) {
        return new Builder(spanExporter);
    }

    public static final class Builder
    extends ConfigBuilder<Builder> {
        private static final String KEY_SCHEDULE_DELAY_MILLIS = "otel.bsp.schedule.delay.millis";
        private static final String KEY_MAX_QUEUE_SIZE = "otel.bsp.max.queue.size";
        private static final String KEY_MAX_EXPORT_BATCH_SIZE = "otel.bsp.max.export.batch.size";
        private static final String KEY_EXPORT_TIMEOUT_MILLIS = "otel.bsp.export.timeout.millis";
        private static final String KEY_SAMPLED = "otel.bsp.export.sampled";
        @VisibleForTesting
        static final long DEFAULT_SCHEDULE_DELAY_MILLIS = 5000L;
        @VisibleForTesting
        static final int DEFAULT_MAX_QUEUE_SIZE = 2048;
        @VisibleForTesting
        static final int DEFAULT_MAX_EXPORT_BATCH_SIZE = 512;
        @VisibleForTesting
        static final int DEFAULT_EXPORT_TIMEOUT_MILLIS = 30000;
        @VisibleForTesting
        static final boolean DEFAULT_EXPORT_ONLY_SAMPLED = true;
        private final SpanExporter spanExporter;
        private long scheduleDelayMillis = 5000L;
        private int maxQueueSize = 2048;
        private int maxExportBatchSize = 512;
        private int exporterTimeoutMillis = 30000;
        private boolean exportOnlySampled = true;

        private Builder(SpanExporter spanExporter) {
            this.spanExporter = Objects.requireNonNull(spanExporter, "spanExporter");
        }

        protected Builder fromConfigMap(Map<String, String> configMap, ConfigBuilder.NamingConvention namingConvention) {
            Boolean boolValue;
            Integer intValue;
            Long longValue = Builder.getLongProperty((String)KEY_SCHEDULE_DELAY_MILLIS, (Map)(configMap = namingConvention.normalize(configMap)));
            if (longValue != null) {
                this.setScheduleDelayMillis(longValue);
            }
            if ((intValue = Builder.getIntProperty((String)KEY_MAX_QUEUE_SIZE, (Map)configMap)) != null) {
                this.setMaxQueueSize(intValue);
            }
            if ((intValue = Builder.getIntProperty((String)KEY_MAX_EXPORT_BATCH_SIZE, (Map)configMap)) != null) {
                this.setMaxExportBatchSize(intValue);
            }
            if ((intValue = Builder.getIntProperty((String)KEY_EXPORT_TIMEOUT_MILLIS, (Map)configMap)) != null) {
                this.setExporterTimeoutMillis(intValue);
            }
            if ((boolValue = Builder.getBooleanProperty((String)KEY_SAMPLED, (Map)configMap)) != null) {
                this.setExportOnlySampled(boolValue);
            }
            return this;
        }

        public Builder setExportOnlySampled(boolean exportOnlySampled) {
            this.exportOnlySampled = exportOnlySampled;
            return this;
        }

        @VisibleForTesting
        boolean getExportOnlySampled() {
            return this.exportOnlySampled;
        }

        public Builder setScheduleDelayMillis(long scheduleDelayMillis) {
            this.scheduleDelayMillis = scheduleDelayMillis;
            return this;
        }

        @VisibleForTesting
        long getScheduleDelayMillis() {
            return this.scheduleDelayMillis;
        }

        public Builder setExporterTimeoutMillis(int exporterTimeoutMillis) {
            this.exporterTimeoutMillis = exporterTimeoutMillis;
            return this;
        }

        @VisibleForTesting
        int getExporterTimeoutMillis() {
            return this.exporterTimeoutMillis;
        }

        public Builder setMaxQueueSize(int maxQueueSize) {
            this.maxQueueSize = maxQueueSize;
            return this;
        }

        @VisibleForTesting
        int getMaxQueueSize() {
            return this.maxQueueSize;
        }

        public Builder setMaxExportBatchSize(int maxExportBatchSize) {
            Utils.checkArgument((maxExportBatchSize > 0 ? 1 : 0) != 0, (String)"maxExportBatchSize must be positive.");
            this.maxExportBatchSize = maxExportBatchSize;
            return this;
        }

        @VisibleForTesting
        int getMaxExportBatchSize() {
            return this.maxExportBatchSize;
        }

        public BatchSpanProcessor build() {
            return new BatchSpanProcessor(this.spanExporter, this.exportOnlySampled, this.scheduleDelayMillis, this.maxQueueSize, this.maxExportBatchSize, this.exporterTimeoutMillis);
        }
    }

    private static final class Worker
    implements Runnable {
        private static final LongCounter.BoundLongCounter droppedSpans;
        private static final LongCounter.BoundLongCounter exportedSpans;
        private static final LongValueObserver spansInQueue;
        private static final Logger logger;
        private final SpanExporter spanExporter;
        private final long scheduleDelayNanos;
        private final int maxExportBatchSize;
        private final int exporterTimeoutMillis;
        private long nextExportTime;
        private final BlockingQueue<ReadableSpan> queue;
        private final AtomicReference<CompletableResultCode> flushRequested = new AtomicReference();
        private volatile boolean continueWork = true;
        private final ArrayList<SpanData> batch;

        private Worker(SpanExporter spanExporter, long scheduleDelayMillis, int maxExportBatchSize, int exporterTimeoutMillis, BlockingQueue<ReadableSpan> queue) {
            this.spanExporter = spanExporter;
            this.scheduleDelayNanos = TimeUnit.MILLISECONDS.toNanos(scheduleDelayMillis);
            this.maxExportBatchSize = maxExportBatchSize;
            this.exporterTimeoutMillis = exporterTimeoutMillis;
            this.queue = queue;
            this.batch = new ArrayList(this.maxExportBatchSize);
            spansInQueue.setCallback(result -> result.observe((long)queue.size(), Labels.of((String)BatchSpanProcessor.SPAN_PROCESSOR_TYPE_LABEL, (String)SPAN_PROCESSOR_TYPE_VALUE)));
        }

        private void addSpan(ReadableSpan span) {
            if (!this.queue.offer(span)) {
                droppedSpans.add(1L);
            }
        }

        @Override
        public void run() {
            this.updateNextExportTime();
            while (this.continueWork) {
                if (this.flushRequested.get() != null) {
                    this.flush();
                }
                try {
                    ReadableSpan lastElement = this.queue.poll(100L, TimeUnit.MILLISECONDS);
                    if (lastElement != null) {
                        this.batch.add(lastElement.toSpanData());
                    }
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return;
                }
                if (this.batch.size() < this.maxExportBatchSize && System.nanoTime() < this.nextExportTime) continue;
                this.exportCurrentBatch();
                this.updateNextExportTime();
            }
        }

        private void flush() {
            for (int spansToFlush = this.queue.size(); spansToFlush > 0; --spansToFlush) {
                ReadableSpan span = (ReadableSpan)this.queue.poll();
                assert (span != null);
                this.batch.add(span.toSpanData());
                if (this.batch.size() < this.maxExportBatchSize) continue;
                this.exportCurrentBatch();
            }
            this.exportCurrentBatch();
            this.flushRequested.get().succeed();
            this.flushRequested.set(null);
        }

        private void updateNextExportTime() {
            this.nextExportTime = System.nanoTime() + this.scheduleDelayNanos;
        }

        private CompletableResultCode shutdown() {
            CompletableResultCode result = new CompletableResultCode();
            CompletableResultCode flushResult = this.forceFlush();
            flushResult.whenComplete(() -> {
                this.continueWork = false;
                CompletableResultCode shutdownResult = this.spanExporter.shutdown();
                shutdownResult.whenComplete(() -> {
                    if (!flushResult.isSuccess() || !shutdownResult.isSuccess()) {
                        result.fail();
                    } else {
                        result.succeed();
                    }
                });
            });
            return result;
        }

        private CompletableResultCode forceFlush() {
            CompletableResultCode flushResult = new CompletableResultCode();
            this.flushRequested.compareAndSet(null, flushResult);
            return this.flushRequested.get();
        }

        private void exportCurrentBatch() {
            if (this.batch.isEmpty()) {
                return;
            }
            try {
                CompletableResultCode result = this.spanExporter.export(this.batch);
                result.join((long)this.exporterTimeoutMillis, TimeUnit.MILLISECONDS);
                if (result.isSuccess()) {
                    exportedSpans.add((long)this.batch.size());
                } else {
                    logger.log(Level.FINE, "Exporter failed");
                }
            }
            catch (Exception e) {
                logger.log(Level.WARNING, "Exporter threw an Exception", e);
            }
            finally {
                this.batch.clear();
            }
        }

        static {
            Meter meter = OpenTelemetry.getGlobalMeter((String)"io.opentelemetry.sdk.trace");
            LongCounter processedSpansCounter = meter.longCounterBuilder("processedSpans").setUnit("1").setDescription("The number of spans processed by the BatchSpanProcessor. [dropped=true if they were dropped due to high throughput]").build();
            droppedSpans = processedSpansCounter.bind(Labels.of((String)BatchSpanProcessor.SPAN_PROCESSOR_TYPE_LABEL, (String)SPAN_PROCESSOR_TYPE_VALUE, (String)"dropped", (String)"true"));
            exportedSpans = processedSpansCounter.bind(Labels.of((String)BatchSpanProcessor.SPAN_PROCESSOR_TYPE_LABEL, (String)SPAN_PROCESSOR_TYPE_VALUE, (String)"dropped", (String)"false"));
            spansInQueue = meter.longValueObserverBuilder("queueSize").setDescription("The number of spans queued").setUnit("1").build();
            logger = Logger.getLogger(Worker.class.getName());
        }
    }
}

