/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.client.security.auth;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.client.security.auth.DelegationToken;
import io.pravega.client.security.auth.DelegationTokenProvider;
import io.pravega.client.stream.impl.Controller;
import io.pravega.common.Exceptions;
import io.pravega.common.LoggerHelpers;
import io.pravega.common.util.ConfigurationOptionsExtractor;
import java.time.Instant;
import java.util.Base64;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JwtTokenProviderImpl
implements DelegationTokenProvider {
    @SuppressFBWarnings(justification="generated code")
    private static final Logger log = LoggerFactory.getLogger(JwtTokenProviderImpl.class);
    @VisibleForTesting
    static final int DEFAULT_REFRESH_THRESHOLD_SECONDS = 5;
    private static final String REFRESH_THRESHOLD_SYSTEM_PROPERTY = "pravega.client.auth.token-refresh.threshold";
    private static final String REFRESH_THRESHOLD_ENV_VARIABLE = "pravega_client_auth_token-refresh.threshold";
    private static final Pattern JWT_EXPIRATION_PATTERN = Pattern.compile("\"exp\":\\s?(\\d+)");
    @VisibleForTesting
    private final int refreshThresholdInSeconds;
    private final Controller controllerClient;
    private final String scopeName;
    private final String streamName;
    private final AtomicReference<DelegationToken> delegationToken = new AtomicReference();
    private final AtomicReference<CompletableFuture<Void>> tokenRefreshFuture = new AtomicReference();

    JwtTokenProviderImpl(Controller controllerClient, String scopeName, String streamName) {
        this(controllerClient, scopeName, streamName, ConfigurationOptionsExtractor.extractInt((String)REFRESH_THRESHOLD_SYSTEM_PROPERTY, (String)REFRESH_THRESHOLD_ENV_VARIABLE, (Integer)5));
    }

    private JwtTokenProviderImpl(Controller controllerClient, String scopeName, String streamName, int refreshThresholdInSeconds) {
        Exceptions.checkNotNullOrEmpty((String)scopeName, (String)"scopeName");
        Preconditions.checkNotNull((Object)controllerClient, (Object)"controllerClient is null");
        Exceptions.checkNotNullOrEmpty((String)streamName, (String)"streamName");
        this.scopeName = scopeName;
        this.streamName = streamName;
        this.controllerClient = controllerClient;
        this.refreshThresholdInSeconds = refreshThresholdInSeconds;
    }

    JwtTokenProviderImpl(String token, Controller controllerClient, String scopeName, String streamName) {
        this(token, controllerClient, scopeName, streamName, ConfigurationOptionsExtractor.extractInt((String)REFRESH_THRESHOLD_SYSTEM_PROPERTY, (String)REFRESH_THRESHOLD_ENV_VARIABLE, (Integer)5));
    }

    JwtTokenProviderImpl(String token, Controller controllerClient, String scopeName, String streamName, int refreshThresholdInSeconds) {
        Exceptions.checkNotNullOrEmpty((String)token, (String)"delegationToken");
        Exceptions.checkNotNullOrEmpty((String)scopeName, (String)"scopeName");
        Preconditions.checkNotNull((Object)controllerClient, (Object)"controllerClient is null");
        Exceptions.checkNotNullOrEmpty((String)streamName, (String)"streamName");
        Long expTime = JwtTokenProviderImpl.extractExpirationTime(token);
        this.delegationToken.set(new DelegationToken(token, expTime));
        this.scopeName = scopeName;
        this.streamName = streamName;
        this.controllerClient = controllerClient;
        this.refreshThresholdInSeconds = refreshThresholdInSeconds;
    }

    @VisibleForTesting
    static Long extractExpirationTime(String token) {
        if (token == null || token.trim().equals("")) {
            return null;
        }
        String[] tokenParts = token.split("\\.");
        if (tokenParts == null || tokenParts.length != 3) {
            return null;
        }
        String encodedBody = tokenParts[1];
        String decodedJsonBody = new String(Base64.getDecoder().decode(encodedBody));
        return JwtTokenProviderImpl.parseExpirationTime(decodedJsonBody);
    }

    @VisibleForTesting
    static Long parseExpirationTime(String jwtBody) {
        String matchedString;
        String[] expiryTimeFieldParts;
        Matcher matcher;
        Long result = null;
        if (jwtBody != null && !jwtBody.trim().equals("") && (matcher = JWT_EXPIRATION_PATTERN.matcher(jwtBody)).find() && (expiryTimeFieldParts = (matchedString = matcher.group()).split(":")) != null && expiryTimeFieldParts.length == 2) {
            try {
                result = Long.parseLong(expiryTimeFieldParts[1].trim());
            }
            catch (NumberFormatException e) {
                log.warn("Encountered this exception when parsing JWT body for expiration time: {}", (Object)e.getMessage());
            }
        }
        return result;
    }

    @Override
    public String retrieveToken() {
        DelegationToken currentToken = this.delegationToken.get();
        if (currentToken == null) {
            return this.refreshToken();
        }
        if (currentToken.getExpiryTime() == null) {
            return currentToken.getValue();
        }
        if (this.isTokenNearingExpiry(currentToken)) {
            log.debug("Token is nearing expiry for scope/stream {}/{}", (Object)this.scopeName, (Object)this.streamName);
            return this.refreshToken();
        }
        return currentToken.getValue();
    }

    @Override
    public boolean populateToken(String token) {
        DelegationToken currentToken = this.delegationToken.get();
        if (token == null || currentToken != null && currentToken.getValue().equals("")) {
            return false;
        }
        return this.delegationToken.compareAndSet(currentToken, new DelegationToken(token, JwtTokenProviderImpl.extractExpirationTime(token)));
    }

    private boolean isTokenNearingExpiry(DelegationToken token) {
        Long currentTokenExpirationTime = token.getExpiryTime();
        return currentTokenExpirationTime != null && this.isWithinRefreshThreshold(currentTokenExpirationTime);
    }

    private boolean isWithinRefreshThreshold(Long expirationTime) {
        assert (expirationTime != null);
        return this.isWithinRefreshThreshold(Instant.now(), Instant.ofEpochSecond(expirationTime));
    }

    @VisibleForTesting
    boolean isWithinRefreshThreshold(Instant currentInstant, Instant expiration) {
        return currentInstant.plusSeconds(this.refreshThresholdInSeconds).getEpochSecond() >= expiration.getEpochSecond();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    String refreshToken() {
        long traceEnterId = LoggerHelpers.traceEnter((Logger)log, (String)"refreshToken", (Object[])new Object[]{this.scopeName, this.streamName});
        CompletableFuture<Void> currentRefreshFuture = this.tokenRefreshFuture.get();
        if (currentRefreshFuture == null) {
            log.debug("Initiated token refresh for scope {} and stream {}", (Object)this.scopeName, (Object)this.streamName);
            currentRefreshFuture = this.recreateToken();
            this.tokenRefreshFuture.compareAndSet(null, currentRefreshFuture);
        } else {
            log.debug("Token is already under refresh for scope {} and stream {}", (Object)this.scopeName, (Object)this.streamName);
        }
        try {
            currentRefreshFuture.join();
        }
        finally {
            this.tokenRefreshFuture.compareAndSet(currentRefreshFuture, null);
        }
        LoggerHelpers.traceLeave((Logger)log, (String)"refreshToken", (long)traceEnterId, (Object[])new Object[]{this.scopeName, this.streamName});
        return this.delegationToken.get().getValue();
    }

    private CompletableFuture<Void> recreateToken() {
        return this.controllerClient.getOrRefreshDelegationTokenFor(this.scopeName, this.streamName).thenAccept(token -> this.delegationToken.set(new DelegationToken((String)token, JwtTokenProviderImpl.extractExpirationTime(token))));
    }

    @SuppressFBWarnings(justification="generated code")
    int getRefreshThresholdInSeconds() {
        return this.refreshThresholdInSeconds;
    }
}

