/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.client.control.impl;

import com.google.auth.Credentials;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.protobuf.ProtocolStringList;
import io.grpc.Channel;
import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import io.grpc.NameResolver;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.grpc.auth.MoreCallCredentials;
import io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.NegotiationType;
import io.grpc.netty.NettyChannelBuilder;
import io.grpc.stub.StreamObserver;
import io.netty.handler.ssl.SslContextBuilder;
import io.pravega.client.admin.KeyValueTableInfo;
import io.pravega.client.control.impl.CancellableRequest;
import io.pravega.client.control.impl.Controller;
import io.pravega.client.control.impl.ControllerFailureException;
import io.pravega.client.control.impl.ControllerImplConfig;
import io.pravega.client.control.impl.ControllerResolverFactory;
import io.pravega.client.control.impl.ModelHelper;
import io.pravega.client.control.impl.PravegaCredentialsWrapper;
import io.pravega.client.control.impl.ReaderGroupConfigRejectedException;
import io.pravega.client.segment.impl.Segment;
import io.pravega.client.stream.InvalidStreamException;
import io.pravega.client.stream.NoSuchScopeException;
import io.pravega.client.stream.PingFailedException;
import io.pravega.client.stream.ReaderGroupConfig;
import io.pravega.client.stream.Stream;
import io.pravega.client.stream.StreamConfiguration;
import io.pravega.client.stream.StreamCut;
import io.pravega.client.stream.Transaction;
import io.pravega.client.stream.TxnFailedException;
import io.pravega.client.stream.impl.SegmentWithRange;
import io.pravega.client.stream.impl.StreamImpl;
import io.pravega.client.stream.impl.StreamSegmentSuccessors;
import io.pravega.client.stream.impl.StreamSegments;
import io.pravega.client.stream.impl.StreamSegmentsWithPredecessors;
import io.pravega.client.stream.impl.TxnSegments;
import io.pravega.client.stream.impl.WriterPosition;
import io.pravega.client.tables.KeyValueTableConfiguration;
import io.pravega.client.tables.impl.KeyValueTableSegments;
import io.pravega.common.Exceptions;
import io.pravega.common.LoggerHelpers;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.function.Callbacks;
import io.pravega.common.hash.RandomFactory;
import io.pravega.common.tracing.RequestTracker;
import io.pravega.common.tracing.TagLogger;
import io.pravega.common.util.AsyncIterator;
import io.pravega.common.util.ContinuationTokenAsyncIterator;
import io.pravega.common.util.Retry;
import io.pravega.controller.stream.api.grpc.v1.Controller;
import io.pravega.controller.stream.api.grpc.v1.ControllerServiceGrpc;
import io.pravega.shared.NameUtils;
import io.pravega.shared.controller.tracing.RPCTracingHelpers;
import io.pravega.shared.protocol.netty.PravegaNodeUri;
import io.pravega.shared.security.auth.AccessOperation;
import java.io.File;
import java.util.AbstractMap;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.net.ssl.SSLException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ControllerImpl
implements Controller {
    private static final TagLogger log = new TagLogger(LoggerFactory.getLogger(ControllerImpl.class));
    private static final long DEFAULT_KEEPALIVE_TIME_MINUTES = 6L;
    private final Retry.RetryAndThrowConditionally retryConfig;
    private final ScheduledExecutorService executor;
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final ManagedChannel channel;
    private final ControllerServiceGrpc.ControllerServiceStub client;
    private final Supplier<Long> requestIdGenerator = RandomFactory.create()::nextLong;
    private final long timeoutMillis;

    public ControllerImpl(ControllerImplConfig config, ScheduledExecutorService executor) {
        this((ManagedChannelBuilder<?>)((NettyChannelBuilder)((NettyChannelBuilder)NettyChannelBuilder.forTarget((String)config.getClientConfig().getControllerURI().toString()).nameResolverFactory((NameResolver.Factory)new ControllerResolverFactory(executor))).defaultLoadBalancingPolicy("round_robin")).keepAliveTime(6L, TimeUnit.MINUTES), config, executor);
        log.info("Controller client connecting to server at {}", (Object)config.getClientConfig().getControllerURI().getAuthority());
    }

    @VisibleForTesting
    public ControllerImpl(ManagedChannelBuilder<?> channelBuilder, ControllerImplConfig config, ScheduledExecutorService executor) {
        Preconditions.checkNotNull(channelBuilder, (Object)"channelBuilder");
        this.executor = executor;
        this.retryConfig = this.createRetryConfig(config);
        if (config.getClientConfig().isEnableTlsToController()) {
            log.debug("Setting up a SSL/TLS channel builder");
            String trustStore = config.getClientConfig().getTrustStore();
            SslContextBuilder sslContextBuilder = GrpcSslContexts.forClient();
            if (!Strings.isNullOrEmpty((String)trustStore)) {
                sslContextBuilder = sslContextBuilder.trustManager(new File(trustStore));
            }
            try {
                channelBuilder = channelBuilder.sslContext(sslContextBuilder.build()).negotiationType(NegotiationType.TLS);
            }
            catch (SSLException e) {
                throw new CompletionException(e);
            }
        } else {
            log.debug("Using a plaintext channel builder");
            channelBuilder = ((NettyChannelBuilder)channelBuilder).negotiationType(NegotiationType.PLAINTEXT);
        }
        channelBuilder = channelBuilder.intercept(new ClientInterceptor[]{RPCTracingHelpers.getClientInterceptor()});
        this.channel = channelBuilder.build();
        this.client = this.getClientWithCredentials(config);
        this.timeoutMillis = config.getTimeoutMillis();
    }

    private ControllerServiceGrpc.ControllerServiceStub getClientWithCredentials(ControllerImplConfig config) {
        ControllerServiceGrpc.ControllerServiceStub client = ControllerServiceGrpc.newStub((Channel)this.channel);
        try {
            io.pravega.shared.security.auth.Credentials credentials = config.getClientConfig().getCredentials();
            if (credentials != null) {
                PravegaCredentialsWrapper wrapper = new PravegaCredentialsWrapper(credentials);
                client = (ControllerServiceGrpc.ControllerServiceStub)client.withCallCredentials(MoreCallCredentials.from((Credentials)wrapper));
            }
        }
        catch (Exception e) {
            log.error("Error while setting credentials to controller client", (Throwable)e);
            this.closeChannel();
            throw e;
        }
        return client;
    }

    private Retry.RetryAndThrowConditionally createRetryConfig(ControllerImplConfig config) {
        return Retry.withExpBackoff((long)config.getInitialBackoffMillis(), (int)config.getBackoffMultiple(), (int)config.getRetryAttempts(), (long)config.getMaxBackoffMillis()).retryWhen(e -> {
            Throwable cause = Exceptions.unwrap((Throwable)e);
            if (cause instanceof StatusRuntimeException) {
                Status.Code code = ((StatusRuntimeException)cause).getStatus().getCode();
                switch (code) {
                    case ABORTED: {
                        return true;
                    }
                    case ALREADY_EXISTS: {
                        return false;
                    }
                    case CANCELLED: {
                        return true;
                    }
                    case DATA_LOSS: {
                        return true;
                    }
                    case DEADLINE_EXCEEDED: {
                        return true;
                    }
                    case FAILED_PRECONDITION: {
                        return true;
                    }
                    case INTERNAL: {
                        return true;
                    }
                    case INVALID_ARGUMENT: {
                        return false;
                    }
                    case NOT_FOUND: {
                        return false;
                    }
                    case OK: {
                        return false;
                    }
                    case OUT_OF_RANGE: {
                        return false;
                    }
                    case PERMISSION_DENIED: {
                        return false;
                    }
                    case RESOURCE_EXHAUSTED: {
                        return true;
                    }
                    case UNAUTHENTICATED: {
                        return false;
                    }
                    case UNAVAILABLE: {
                        return true;
                    }
                    case UNIMPLEMENTED: {
                        return false;
                    }
                    case UNKNOWN: {
                        return true;
                    }
                }
                return true;
            }
            return false;
        });
    }

    @Override
    public CompletableFuture<Boolean> createScope(String scopeName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"createScope", (Object[])new Object[]{scopeName, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.CreateScopeStatus> callback = new RPCAsyncCallback<Controller.CreateScopeStatus>(requestId, "createScope", scopeName);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "createScope", scopeName).createScope(Controller.ScopeInfo.newBuilder().setScope(scopeName).build(), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to create scope: {}", new Object[]{scopeName});
                    throw new ControllerFailureException("Failed to create scope: " + scopeName);
                }
                case INVALID_SCOPE_NAME: {
                    log.warn(requestId, "Illegal scope name: {}", new Object[]{scopeName});
                    throw new IllegalArgumentException("Illegal scope name: " + scopeName);
                }
                case SCOPE_EXISTS: {
                    log.warn(requestId, "Scope already exists: {}", new Object[]{scopeName});
                    return false;
                }
                case SUCCESS: {
                    log.info(requestId, "Scope created successfully: {}", new Object[]{scopeName});
                    return true;
                }
            }
            throw new ControllerFailureException("Unknown return status creating scope " + scopeName + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "createScope {} failed: ", new Object[]{scopeName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"createScope", (long)traceId, (Object[])new Object[]{scopeName, requestId});
        });
    }

    @Override
    public CompletableFuture<Boolean> checkScopeExists(String scopeName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"checkScopeExists", (Object[])new Object[]{scopeName, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.ExistsResponse> callback = new RPCAsyncCallback<Controller.ExistsResponse>(requestId, "checkScopeExists", scopeName);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "checkScopeExists", scopeName).checkScopeExists(Controller.ScopeInfo.newBuilder().setScope(scopeName).build(), callback);
            return callback.getFuture().thenApply(Controller.ExistsResponse::getExists);
        }, this.executor);
        return result.whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "checkScopeExists {} failed: ", new Object[]{scopeName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"checkScopeExists", (long)traceId, (Object[])new Object[]{scopeName, requestId});
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public AsyncIterator<String> listScopes() {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"listScopes", (Object[])new Object[0]);
        long requestId = this.requestIdGenerator.get();
        try {
            Function<Controller.ContinuationToken, CompletableFuture> function = token -> this.retryConfig.runAsync(() -> {
                RPCAsyncCallback<Controller.ScopesResponse> callback = new RPCAsyncCallback<Controller.ScopesResponse>(requestId, "listScopes", new Object[0]);
                new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "listScopes").listScopes(Controller.ScopesRequest.newBuilder().setContinuationToken(token).build(), callback);
                return callback.getFuture().thenApply(x -> {
                    ProtocolStringList result = x.getScopesList();
                    return new AbstractMap.SimpleEntry<Controller.ContinuationToken, ProtocolStringList>(x.getContinuationToken(), result);
                });
            }, this.executor);
            ContinuationTokenAsyncIterator continuationTokenAsyncIterator = new ContinuationTokenAsyncIterator(function, (Object)Controller.ContinuationToken.newBuilder().build());
            return continuationTokenAsyncIterator;
        }
        finally {
            LoggerHelpers.traceLeave((Logger)log, (String)"listStreams", (long)traceId, (Object[])new Object[0]);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public AsyncIterator<Stream> listStreams(String scopeName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"listStreams", (Object[])new Object[]{scopeName});
        long requestId = this.requestIdGenerator.get();
        try {
            Function<Controller.ContinuationToken, CompletableFuture> function = token -> this.retryConfig.runAsync(() -> {
                RPCAsyncCallback<Controller.StreamsInScopeResponse> callback = new RPCAsyncCallback<Controller.StreamsInScopeResponse>(requestId, "listStreams", scopeName);
                Controller.ScopeInfo scopeInfo = Controller.ScopeInfo.newBuilder().setScope(scopeName).build();
                new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "listStreams", scopeName).listStreamsInScope(Controller.StreamsInScopeRequest.newBuilder().setScope(scopeInfo).setContinuationToken(token).build(), callback);
                return callback.getFuture().thenApply(x -> {
                    switch (x.getStatus()) {
                        case SCOPE_NOT_FOUND: {
                            log.warn(requestId, "Scope not found: {}", new Object[]{scopeName});
                            throw new NoSuchScopeException();
                        }
                        case FAILURE: {
                            log.warn(requestId, "Internal Server Error while trying to list streams in scope: {}", new Object[]{scopeName});
                            throw new RuntimeException("Failure while trying to list streams");
                        }
                    }
                    List result = x.getStreamsList().stream().map(y -> new StreamImpl(y.getScope(), y.getStream())).collect(Collectors.toList());
                    return new AbstractMap.SimpleEntry(x.getContinuationToken(), result);
                });
            }, this.executor);
            ContinuationTokenAsyncIterator continuationTokenAsyncIterator = new ContinuationTokenAsyncIterator(function, (Object)Controller.ContinuationToken.newBuilder().build());
            return continuationTokenAsyncIterator;
        }
        finally {
            LoggerHelpers.traceLeave((Logger)log, (String)"listStreams", (long)traceId, (Object[])new Object[0]);
        }
    }

    @Override
    public CompletableFuture<Boolean> deleteScope(String scopeName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"deleteScope", (Object[])new Object[]{scopeName, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.DeleteScopeStatus> callback = new RPCAsyncCallback<Controller.DeleteScopeStatus>(requestId, "deleteScope", scopeName);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "deleteScope", scopeName).deleteScope(Controller.ScopeInfo.newBuilder().setScope(scopeName).build(), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to delete scope: {}", new Object[]{scopeName});
                    throw new ControllerFailureException("Failed to delete scope: " + scopeName);
                }
                case SCOPE_NOT_EMPTY: {
                    log.warn(requestId, "Cannot delete non empty scope: {}", new Object[]{scopeName});
                    throw new IllegalStateException("Scope " + scopeName + " is not empty.");
                }
                case SCOPE_NOT_FOUND: {
                    log.warn(requestId, "Scope not found: {}", new Object[]{scopeName});
                    return false;
                }
                case SUCCESS: {
                    log.info(requestId, "Scope deleted successfully: {}", new Object[]{scopeName});
                    return true;
                }
            }
            throw new ControllerFailureException("Unknown return status deleting scope " + scopeName + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "deleteScope {} failed: ", new Object[]{scopeName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"deleteScope", (long)traceId, (Object[])new Object[]{scopeName, requestId});
        });
    }

    @Override
    public CompletableFuture<Boolean> createStream(String scope, String streamName, StreamConfiguration streamConfig) {
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)streamConfig, (Object)"streamConfig");
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"createStream", (Object[])new Object[]{streamConfig, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.CreateStreamStatus> callback = new RPCAsyncCallback<Controller.CreateStreamStatus>(requestId, "createStream", scope, streamName, streamConfig);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "createStream", scope, streamName).createStream(ModelHelper.decode(scope, streamName, streamConfig), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to create stream: {}", new Object[]{streamName});
                    throw new ControllerFailureException("Failed to create stream: " + streamConfig);
                }
                case INVALID_STREAM_NAME: {
                    log.warn(requestId, "Illegal stream name: {}", new Object[]{streamName});
                    throw new IllegalArgumentException("Illegal stream name: " + streamConfig);
                }
                case SCOPE_NOT_FOUND: {
                    log.warn(requestId, "Scope not found: {}", new Object[]{scope});
                    throw new IllegalArgumentException("Scope does not exist: " + streamConfig);
                }
                case STREAM_EXISTS: {
                    log.warn(requestId, "Stream already exists: {}", new Object[]{streamName});
                    return false;
                }
                case SUCCESS: {
                    log.info(requestId, "Stream created successfully: {}", new Object[]{streamName});
                    return true;
                }
            }
            throw new ControllerFailureException("Unknown return status creating stream " + streamConfig + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "createStream {}/{} failed: ", new Object[]{scope, streamName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"createStream", (long)traceId, (Object[])new Object[]{streamConfig, requestId});
        });
    }

    @Override
    public CompletableFuture<Boolean> checkStreamExists(String scopeName, String streamName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"checkStreamExists", (Object[])new Object[]{scopeName, streamName, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.ExistsResponse> callback = new RPCAsyncCallback<Controller.ExistsResponse>(requestId, "checkStreamExists", scopeName, streamName);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "checkStreamExists", scopeName, streamName).checkStreamExists(Controller.StreamInfo.newBuilder().setScope(scopeName).setStream(streamName).build(), callback);
            return callback.getFuture().thenApply(Controller.ExistsResponse::getExists);
        }, this.executor);
        return result.whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "checkStreamExists {}/{} failed: ", new Object[]{scopeName, streamName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"checkStreamExists", (long)traceId, (Object[])new Object[]{scopeName, streamName, requestId});
        });
    }

    @Override
    public CompletableFuture<Boolean> updateStream(String scope, String streamName, StreamConfiguration streamConfig) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)streamConfig, (Object)"streamConfig");
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"updateStream", (Object[])new Object[]{streamConfig, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.UpdateStreamStatus> callback = new RPCAsyncCallback<Controller.UpdateStreamStatus>(requestId, "updateStream", scope, streamName, streamConfig);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "updateStream", scope, streamName).updateStream(ModelHelper.decode(scope, streamName, streamConfig), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to update stream: {}", new Object[]{streamName});
                    throw new ControllerFailureException("Failed to update stream: " + streamConfig);
                }
                case SCOPE_NOT_FOUND: {
                    log.warn(requestId, "Scope not found: {}", new Object[]{scope});
                    throw new IllegalArgumentException("Scope does not exist: " + streamConfig);
                }
                case STREAM_NOT_FOUND: {
                    log.warn(requestId, "Stream does not exist: {}", new Object[]{streamName});
                    throw new IllegalArgumentException("Stream does not exist: " + streamConfig);
                }
                case SUCCESS: {
                    log.info(requestId, "Successfully updated stream: {}", new Object[]{streamName});
                    return true;
                }
            }
            throw new ControllerFailureException("Unknown return status updating stream " + streamConfig + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "updateStream {}/{} failed: ", new Object[]{scope, streamName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"updateStream", (long)traceId, (Object[])new Object[]{streamConfig, requestId});
        });
    }

    @Override
    public CompletableFuture<List<String>> listSubscribers(String scope, String streamName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)scope, (Object)"scope");
        Preconditions.checkNotNull((Object)streamName, (Object)"stream");
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"listSubscribers", (Object[])new Object[0]);
        long requestId = this.requestIdGenerator.get();
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.SubscribersResponse> callback = new RPCAsyncCallback<Controller.SubscribersResponse>(requestId, "listSubscribers", new Object[0]);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "listSubscribers").listSubscribers(ModelHelper.createStreamInfo(scope, streamName), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to list subscribers for stream {}/{}", new Object[]{scope, streamName});
                    throw new ControllerFailureException("Failed to list subscribers for stream" + streamName);
                }
                case STREAM_NOT_FOUND: {
                    log.warn(requestId, "Stream does not exist: {}", new Object[]{streamName});
                    throw new IllegalArgumentException("Stream does not exist: " + streamName);
                }
                case SUCCESS: {
                    log.info(requestId, "Successfully listed subscribers for stream: {}/{}", new Object[]{scope, streamName});
                    return x.getSubscribersList().stream().collect(Collectors.toList());
                }
            }
            throw new ControllerFailureException("Unknown return status listing subscribers " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "listSubscribers for stream {}/{} failed: ", new Object[]{scope, streamName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"listSubscribers", (long)traceId, (Object[])new Object[]{requestId});
        });
    }

    @Override
    public CompletableFuture<Boolean> updateSubscriberStreamCut(String scope, String streamName, String subscriber, UUID readerGroupId, long generation, StreamCut streamCut) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)scope, (Object)"scope");
        Preconditions.checkNotNull((Object)streamName, (Object)"stream");
        Preconditions.checkNotNull((Object)subscriber, (Object)"subscriber");
        Preconditions.checkNotNull((Object)readerGroupId, (Object)"readerGroupId");
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"updateTruncationStreamCut", (Object[])new Object[]{subscriber, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.UpdateSubscriberStatus> callback = new RPCAsyncCallback<Controller.UpdateSubscriberStatus>(requestId, "updateTruncationStreamCut", scope, streamName, subscriber, streamCut);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "updateTruncationStreamCut", scope, streamName).updateSubscriberStreamCut(ModelHelper.decode(scope, streamName, subscriber, readerGroupId, generation, this.getStreamCutMap(streamCut)), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to update stream cut for Reader Group: {}", new Object[]{subscriber});
                    throw new ControllerFailureException("Failed to update stream cut for Reader Group:" + subscriber);
                }
                case STREAM_NOT_FOUND: {
                    log.warn(requestId, "Stream does not exist: {}", new Object[]{streamName});
                    throw new IllegalArgumentException("Stream does not exist: " + streamName);
                }
                case SUBSCRIBER_NOT_FOUND: {
                    log.warn(requestId, "Subscriber does not exist: {} for stream {}/{}", new Object[]{subscriber, scope, streamName});
                    throw new IllegalArgumentException("Subscriber does not exist: " + subscriber);
                }
                case STREAM_CUT_NOT_VALID: {
                    log.warn(requestId, "StreamCut not valid for stream {}/{} subscriber {}.", new Object[]{scope, streamName, subscriber});
                    throw new IllegalArgumentException("StreamCut not valid for stream " + scope + "/" + streamName + ": subscriber:" + subscriber);
                }
                case GENERATION_MISMATCH: {
                    log.warn(requestId, "Invalid generation for ReaderGroup {}.", new Object[]{subscriber});
                    throw new IllegalArgumentException("Invalid generation for ReaderGroup " + subscriber);
                }
                case SUCCESS: {
                    log.info(requestId, "Successfully updated truncationStreamCut for subscriber {} for stream: {}/{}", new Object[]{subscriber, scope, streamName});
                    return true;
                }
            }
            throw new ControllerFailureException("Unknown return status for updateTruncationStreamCut for Stream :" + scope + "/" + streamName + ": subscriber:" + subscriber + ": status=" + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "updateTruncationStreamCut for Subscriber {} for stream {}/{} failed: ", new Object[]{subscriber, scope, streamName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"updateTruncationStreamCut", (long)traceId, (Object[])new Object[]{subscriber, requestId});
        });
    }

    @Override
    public CompletableFuture<Boolean> truncateStream(String scope, String stream, StreamCut streamCut) {
        return this.truncateStream(scope, stream, this.getStreamCutMap(streamCut));
    }

    private CompletableFuture<Boolean> truncateStream(String scope, String stream, Map<Long, Long> streamCut) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull(streamCut, (Object)"streamCut");
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"truncateStream", (Object[])new Object[]{streamCut, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.UpdateStreamStatus> callback = new RPCAsyncCallback<Controller.UpdateStreamStatus>(requestId, "truncateStream", scope, stream);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "truncateStream", scope, stream).truncateStream(ModelHelper.decode(scope, stream, streamCut), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to truncate stream: {}/{}", new Object[]{scope, stream});
                    throw new ControllerFailureException("Failed to truncate stream: " + scope + "/" + stream);
                }
                case SCOPE_NOT_FOUND: {
                    log.warn(requestId, "Scope not found: {}", new Object[]{scope});
                    throw new IllegalArgumentException("Scope does not exist: " + scope);
                }
                case STREAM_NOT_FOUND: {
                    log.warn(requestId, "Stream does not exist: {}/{}", new Object[]{scope, stream});
                    throw new IllegalArgumentException("Stream does not exist: " + stream);
                }
                case SUCCESS: {
                    log.info(requestId, "Successfully updated stream: {}/{}", new Object[]{scope, stream});
                    return true;
                }
            }
            throw new ControllerFailureException("Unknown return status truncating stream " + scope + "/" + stream + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "truncateStream {}/{} failed: ", new Object[]{scope, stream, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"truncateStream", (long)traceId, (Object[])new Object[]{streamCut, requestId});
        });
    }

    @Override
    public CancellableRequest<Boolean> scaleStream(Stream stream, List<Long> sealedSegments, Map<Double, Double> newKeyRanges, ScheduledExecutorService executor) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        CancellableRequest<Boolean> cancellableRequest = new CancellableRequest<Boolean>();
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"scaleStream", (Object[])new Object[]{stream, requestId});
        this.startScaleInternal(stream, sealedSegments, newKeyRanges, "scaleStream", requestId).whenComplete((startScaleResponse, e) -> {
            if (e != null) {
                log.error(requestId, "Failed to start scale for stream {}", new Object[]{stream, e});
                cancellableRequest.start(() -> Futures.failedFuture((Throwable)e), any -> true, executor);
            } else {
                try {
                    boolean started = this.handleScaleResponse(stream, (Controller.ScaleResponse)startScaleResponse, requestId);
                    cancellableRequest.start(() -> {
                        if (started) {
                            return this.checkScaleStatus(stream, startScaleResponse.getEpoch());
                        }
                        return CompletableFuture.completedFuture(false);
                    }, isDone -> !started || isDone != false, executor);
                    LoggerHelpers.traceLeave((Logger)log, (String)"scaleStream", (long)traceId, (Object[])new Object[]{stream, requestId});
                }
                catch (Exception ex) {
                    log.warn(requestId, "Failed to handle scale response: ", new Object[]{ex});
                    cancellableRequest.start(() -> Futures.failedFuture((Throwable)ex), any -> true, executor);
                }
            }
        });
        return cancellableRequest;
    }

    @Override
    public CompletableFuture<Boolean> startScale(Stream stream, List<Long> sealedSegments, Map<Double, Double> newKeyRanges) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"scaleStream", (Object[])new Object[]{stream, requestId});
        return ((CompletableFuture)this.startScaleInternal(stream, sealedSegments, newKeyRanges, "scaleStream", requestId).thenApply(response -> this.handleScaleResponse(stream, (Controller.ScaleResponse)response, traceId))).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "Failed to start scale of stream: {} ", new Object[]{stream.getStreamName(), e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"scaleStream", (long)traceId, (Object[])new Object[]{stream, requestId});
        });
    }

    private Boolean handleScaleResponse(Stream stream, Controller.ScaleResponse response, long requestId) {
        switch (response.getStatus()) {
            case FAILURE: {
                log.warn(requestId, "Failed to scale stream: {}", new Object[]{stream.getStreamName()});
                throw new ControllerFailureException("Failed to scale stream: " + stream);
            }
            case PRECONDITION_FAILED: {
                log.warn(requestId, "Precondition failed for scale stream: {}", new Object[]{stream.getStreamName()});
                return false;
            }
            case STARTED: {
                log.info(requestId, "Successfully started scale stream: {}", new Object[]{stream.getStreamName()});
                return true;
            }
        }
        throw new ControllerFailureException("Unknown return status scaling stream " + stream + " " + response.getStatus());
    }

    @Override
    public CompletableFuture<Boolean> checkScaleStatus(Stream stream, int scaleEpoch) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)stream, (Object)"stream");
        Preconditions.checkArgument((scaleEpoch >= 0 ? 1 : 0) != 0);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"checkScale", (Object[])new Object[]{stream});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "checkScale", stream, scaleEpoch);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).checkScale(Controller.ScaleStatusRequest.newBuilder().setStreamInfo(ModelHelper.createStreamInfo(stream.getScope(), stream.getStreamName())).setEpoch(scaleEpoch).build(), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(response -> {
            switch (response.getStatus()) {
                case IN_PROGRESS: {
                    return false;
                }
                case SUCCESS: {
                    return true;
                }
                case INVALID_INPUT: {
                    log.warn("Failed to check scale status of stream " + stream.getStreamName() + " because of invalid input");
                    throw new ControllerFailureException("invalid input");
                }
            }
            throw new ControllerFailureException("Unknown return status checking scale of stream " + stream + " " + response.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("checkScaleStatus {} failed: ", (Object)stream.getStreamName(), e);
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"checkScale", (long)traceId, (Object[])new Object[0]);
        });
    }

    private CompletableFuture<Controller.ScaleResponse> startScaleInternal(Stream stream, List<Long> sealedSegments, Map<Double, Double> newKeyRanges, String method, long requestId) {
        Preconditions.checkNotNull((Object)stream, (Object)"stream");
        Preconditions.checkNotNull(sealedSegments, (Object)"sealedSegments");
        Preconditions.checkNotNull(newKeyRanges, (Object)"newKeyRanges");
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.ScaleResponse> callback = new RPCAsyncCallback<Controller.ScaleResponse>(requestId, method, stream);
            long scaleTimestamp = System.currentTimeMillis();
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, method, stream.getScope(), stream.getStreamName(), String.valueOf(scaleTimestamp)).scale(Controller.ScaleRequest.newBuilder().setStreamInfo(ModelHelper.createStreamInfo(stream.getScope(), stream.getStreamName())).addAllSealedSegments((Iterable)sealedSegments).addAllNewKeyRanges((Iterable)newKeyRanges.entrySet().stream().map(x -> Controller.ScaleRequest.KeyRangeEntry.newBuilder().setStart(((Double)x.getKey()).doubleValue()).setEnd(((Double)x.getValue()).doubleValue()).build()).collect(Collectors.toList())).setScaleTimestamp(scaleTimestamp).build(), callback);
            return callback.getFuture();
        }, this.executor);
        return result;
    }

    @Override
    public CompletableFuture<Boolean> sealStream(String scope, String streamName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        Exceptions.checkNotNullOrEmpty((String)streamName, (String)"streamName");
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"sealStream", (Object[])new Object[]{scope, streamName, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.UpdateStreamStatus> callback = new RPCAsyncCallback<Controller.UpdateStreamStatus>(requestId, "sealStream", scope, streamName);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "sealStream", scope, streamName).sealStream(ModelHelper.createStreamInfo(scope, streamName), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to seal stream: {}", new Object[]{streamName});
                    throw new ControllerFailureException("Failed to seal stream: " + streamName);
                }
                case SCOPE_NOT_FOUND: {
                    log.warn(requestId, "Scope not found: {}", new Object[]{scope});
                    throw new InvalidStreamException("Scope does not exist: " + scope);
                }
                case STREAM_NOT_FOUND: {
                    log.warn(requestId, "Stream does not exist: {}", new Object[]{streamName});
                    throw new InvalidStreamException("Stream does not exist: " + streamName);
                }
                case SUCCESS: {
                    log.info(requestId, "Successfully sealed stream: {}", new Object[]{streamName});
                    return true;
                }
            }
            throw new ControllerFailureException("Unknown return status sealing stream " + streamName + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "sealStream {}/{} failed: ", new Object[]{scope, streamName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"sealStream", (long)traceId, (Object[])new Object[]{scope, streamName, requestId});
        });
    }

    @Override
    public CompletableFuture<Boolean> deleteStream(String scope, String streamName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        Exceptions.checkNotNullOrEmpty((String)streamName, (String)"streamName");
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"deleteStream", (Object[])new Object[]{scope, streamName, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.DeleteStreamStatus> callback = new RPCAsyncCallback<Controller.DeleteStreamStatus>(requestId, "deleteStream", scope, streamName);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "deleteStream", scope, streamName).deleteStream(ModelHelper.createStreamInfo(scope, streamName), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to delete stream: {}", new Object[]{streamName});
                    throw new ControllerFailureException("Failed to delete stream: " + streamName);
                }
                case STREAM_NOT_FOUND: {
                    log.warn(requestId, "Stream does not exist: {}", new Object[]{streamName});
                    return false;
                }
                case STREAM_NOT_SEALED: {
                    log.warn(requestId, "Stream is not sealed: {}", new Object[]{streamName});
                    throw new IllegalArgumentException("Stream is not sealed: " + streamName);
                }
                case SUCCESS: {
                    log.info(requestId, "Successfully deleted stream: {}", new Object[]{streamName});
                    return true;
                }
            }
            throw new ControllerFailureException("Unknown return status deleting stream " + streamName + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "deleteStream {}/{} failed: ", new Object[]{scope, streamName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"deleteStream", (long)traceId, (Object[])new Object[]{scope, streamName, requestId});
        });
    }

    @Override
    public CompletableFuture<Map<Segment, Long>> getSegmentsAtTime(Stream stream, long timestamp) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)stream, (Object)"stream");
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"getSegmentsAtTime", (Object[])new Object[]{stream, timestamp});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "getSegmentsAtTime", stream, timestamp);
            Controller.StreamInfo streamInfo = ModelHelper.createStreamInfo(stream.getScope(), stream.getStreamName(), AccessOperation.NONE);
            Controller.GetSegmentsRequest request = Controller.GetSegmentsRequest.newBuilder().setStreamInfo(streamInfo).setTimestamp(timestamp).build();
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).getSegments(request, callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(segments -> {
            log.debug("Received the following data from the controller {}", (Object)segments.getSegmentsList());
            return segments.getSegmentsList().stream().collect(Collectors.toMap(location -> ModelHelper.encode(location.getSegmentId()), location -> location.getOffset()));
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("get Segments of {} at time {} failed: ", new Object[]{stream.getStreamName(), timestamp, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"getSegmentsAtTime", (long)traceId, (Object[])new Object[0]);
        });
    }

    @Override
    public CompletableFuture<StreamSegmentsWithPredecessors> getSuccessors(Segment segment) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"getSuccessors", (Object[])new Object[]{segment});
        CompletableFuture resultFuture = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "getSuccessors", segment);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).getSegmentsImmediatelyFollowing(ModelHelper.decode(segment), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)resultFuture.thenApply(successors -> {
            log.debug("Received the following data from the controller {}", (Object)successors.getSegmentsList());
            HashMap<SegmentWithRange, List<Long>> result = new HashMap<SegmentWithRange, List<Long>>();
            for (Controller.SuccessorResponse.SegmentEntry entry : successors.getSegmentsList()) {
                result.put(ModelHelper.encode(entry.getSegment()), entry.getValueList());
            }
            return new StreamSegmentsWithPredecessors(result, successors.getDelegationToken());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("getSuccessors of segment {} failed: ", (Object)segment.getSegmentId(), e);
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"getSuccessors", (long)traceId, (Object[])new Object[0]);
        });
    }

    @Override
    public CompletableFuture<StreamSegmentSuccessors> getSuccessors(StreamCut from) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Stream stream = from.asImpl().getStream();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"getSuccessors", (Object[])new Object[]{stream});
        return this.getSegmentsBetweenStreamCuts(from, StreamCut.UNBOUNDED).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("getSuccessorsFromCut for {} failed: ", (Object)stream.getStreamName(), e);
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"getSuccessors", (long)traceId, (Object[])new Object[0]);
        });
    }

    @Override
    public CompletableFuture<StreamSegmentSuccessors> getSegments(StreamCut fromStreamCut, StreamCut toStreamCut) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)fromStreamCut, (Object)"fromStreamCut");
        Preconditions.checkNotNull((Object)toStreamCut, (Object)"toStreamCut");
        Preconditions.checkArgument((boolean)fromStreamCut.asImpl().getStream().equals(toStreamCut.asImpl().getStream()), (Object)"Ensure streamCuts for the same stream is passed");
        Stream stream = fromStreamCut.asImpl().getStream();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"getSegments", (Object[])new Object[]{stream});
        return this.getSegmentsBetweenStreamCuts(fromStreamCut, toStreamCut).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("getSegments for {} failed: ", (Object)stream.getStreamName(), e);
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"getSegments", (long)traceId, (Object[])new Object[0]);
        });
    }

    private CompletableFuture<StreamSegmentSuccessors> getSegmentsBetweenStreamCuts(StreamCut fromStreamCut, StreamCut toStreamCut) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Stream stream = fromStreamCut.asImpl().getStream();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"getSegments", (Object[])new Object[]{stream});
        CompletableFuture resultFuture = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "getSuccessorsFromCut", new Object[0]);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).getSegmentsBetween(ModelHelper.decode(stream.getScope(), stream.getStreamName(), this.getStreamCutMap(fromStreamCut), this.getStreamCutMap(toStreamCut)), callback);
            return callback.getFuture();
        }, this.executor);
        return resultFuture.thenApply(response -> {
            log.debug("Received the following data from the controller {}", (Object)response.getSegmentsList());
            return new StreamSegmentSuccessors(response.getSegmentsList().stream().map(ModelHelper::encode).collect(Collectors.toSet()), response.getDelegationToken());
        });
    }

    private Map<Long, Long> getStreamCutMap(StreamCut streamCut) {
        if (streamCut.equals(StreamCut.UNBOUNDED)) {
            return Collections.emptyMap();
        }
        return streamCut.asImpl().getPositions().entrySet().stream().collect(Collectors.toMap(x -> ((Segment)x.getKey()).getSegmentId(), Map.Entry::getValue));
    }

    @Override
    public CompletableFuture<StreamSegments> getCurrentSegments(String scope, String stream) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        Exceptions.checkNotNullOrEmpty((String)stream, (String)"stream");
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"getCurrentSegments", (Object[])new Object[]{scope, stream});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "getCurrentSegments", scope, stream);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).getCurrentSegments(ModelHelper.createStreamInfo(scope, stream, AccessOperation.NONE), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(this::getStreamSegments)).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("getCurrentSegments for {}/{} failed: ", new Object[]{scope, stream, e});
            }
            if (x.getNumberOfSegments() == 0) {
                log.warn("getCurrentSegments for {}/{} returned zero segments since the Stream is sealed", (Object)scope, (Object)stream);
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"getCurrentSegments", (long)traceId, (Object[])new Object[0]);
        });
    }

    @Override
    public CompletableFuture<StreamSegments> getEpochSegments(String scope, String stream, int epoch) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        Exceptions.checkNotNullOrEmpty((String)stream, (String)"stream");
        Exceptions.checkArgument((epoch >= 0 ? 1 : 0) != 0, (String)"epoch", (String)"Should be a positive integer", (Object[])new Object[0]);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"getEpochSegments", (Object[])new Object[]{scope, stream});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "getEpochSegments", scope, stream);
            Controller.GetEpochSegmentsRequest request = Controller.GetEpochSegmentsRequest.newBuilder().setStreamInfo(ModelHelper.createStreamInfo(scope, stream)).setEpoch(epoch).build();
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).getEpochSegments(request, callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(this::getStreamSegments)).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("getEpochSegments for {}/{} with for epoch {} failed: ", new Object[]{scope, stream, epoch, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"getEpochSegments", (long)traceId, (Object[])new Object[0]);
        });
    }

    private StreamSegments getStreamSegments(Controller.SegmentRanges ranges) {
        log.debug("Received the following data from the controller {}", (Object)ranges.getSegmentRangesList());
        TreeMap<Double, SegmentWithRange> rangeMap = new TreeMap<Double, SegmentWithRange>();
        for (Controller.SegmentRange r : ranges.getSegmentRangesList()) {
            Preconditions.checkState((r.getMinKey() <= r.getMaxKey() ? 1 : 0) != 0, (String)"Min keyrange %s was not less than maximum keyRange %s for segment %s", (Object)r.getMinKey(), (Object)r.getMaxKey(), (Object)r.getSegmentId());
            rangeMap.put(r.getMaxKey(), new SegmentWithRange(ModelHelper.encode(r.getSegmentId()), r.getMinKey(), r.getMaxKey()));
        }
        return new StreamSegments(rangeMap);
    }

    @Override
    public CompletableFuture<PravegaNodeUri> getEndpointForSegment(String qualifiedSegmentName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Exceptions.checkNotNullOrEmpty((String)qualifiedSegmentName, (String)"qualifiedSegmentName");
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"getEndpointForSegment", (Object[])new Object[]{qualifiedSegmentName});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "getEndpointForSegment", qualifiedSegmentName);
            Segment segment = Segment.fromScopedName(qualifiedSegmentName);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).getURI(ModelHelper.createSegmentId(segment.getScope(), segment.getStreamName(), segment.getSegmentId()), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(ModelHelper::encode)).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("getEndpointForSegment {} failed: ", (Object)qualifiedSegmentName, e);
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"getEndpointForSegment", (long)traceId, (Object[])new Object[0]);
        });
    }

    @Override
    public CompletableFuture<Boolean> isSegmentOpen(Segment segment) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"isSegmentOpen", (Object[])new Object[]{segment});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "isSegmentOpen", segment);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).isSegmentValid(ModelHelper.createSegmentId(segment.getScope(), segment.getStreamName(), segment.getSegmentId()), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(Controller.SegmentValidityResponse::getResponse)).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("isSegmentOpen for segment {} failed: ", (Object)segment, e);
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"isSegmentOpen", (long)traceId, (Object[])new Object[0]);
        });
    }

    @Override
    public CompletableFuture<TxnSegments> createTransaction(Stream stream, long lease) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)stream, (Object)"stream");
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"createTransaction", (Object[])new Object[]{stream, lease});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "createTransaction", stream, lease);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).createTransaction(Controller.CreateTxnRequest.newBuilder().setStreamInfo(ModelHelper.createStreamInfo(stream.getScope(), stream.getStreamName())).setLease(lease).build(), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(this::convert)).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("createTransaction on stream {} failed: ", (Object)stream.getStreamName(), e);
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"createTransaction", (long)traceId, (Object[])new Object[0]);
        });
    }

    private TxnSegments convert(Controller.CreateTxnResponse response) {
        TreeMap<Double, SegmentWithRange> rangeMap = new TreeMap<Double, SegmentWithRange>();
        for (Controller.SegmentRange r : response.getActiveSegmentsList()) {
            Preconditions.checkState((r.getMinKey() <= r.getMaxKey() ? 1 : 0) != 0);
            rangeMap.put(r.getMaxKey(), new SegmentWithRange(ModelHelper.encode(r.getSegmentId()), r.getMinKey(), r.getMaxKey()));
        }
        StreamSegments segments = new StreamSegments(rangeMap);
        return new TxnSegments(segments, ModelHelper.encode(response.getTxnId()));
    }

    @Override
    public CompletableFuture<Transaction.PingStatus> pingTransaction(Stream stream, UUID txId, long lease) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"pingTransaction", (Object[])new Object[]{stream, txId, lease});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "pingTransaction", txId, lease);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).pingTransaction(Controller.PingTxnRequest.newBuilder().setStreamInfo(ModelHelper.createStreamInfo(stream.getScope(), stream.getStreamName())).setTxnId(ModelHelper.decode(txId)).setLease(lease).build(), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(status -> {
            try {
                return ModelHelper.encode(status.getStatus(), stream + " " + txId);
            }
            catch (PingFailedException ex) {
                throw new CompletionException(ex);
            }
        })).whenComplete((s, e) -> {
            if (e != null) {
                log.warn("PingTransaction {} failed:", (Object)txId, e);
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"pingTransaction", (long)traceId, (Object[])new Object[0]);
        });
    }

    @Override
    public CompletableFuture<Void> commitTransaction(Stream stream, String writerId, Long timestamp, UUID txId) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)stream, (Object)"stream");
        Preconditions.checkNotNull((Object)txId, (Object)"txId");
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"commitTransaction", (Object[])new Object[]{stream, txId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "commitTransaction", stream, writerId, timestamp, txId);
            Controller.TxnRequest.Builder txnRequest = Controller.TxnRequest.newBuilder().setStreamInfo(ModelHelper.createStreamInfo(stream.getScope(), stream.getStreamName())).setWriterId(writerId).setTxnId(ModelHelper.decode(txId));
            if (timestamp != null) {
                txnRequest.setTimestamp(timestamp.longValue());
            } else {
                txnRequest.setTimestamp(Long.MIN_VALUE);
            }
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).commitTransaction(txnRequest.build(), callback);
            return callback.getFuture();
        }, this.executor);
        return result.thenApply(txnStatus -> {
            if (txnStatus.getStatus().equals((Object)Controller.TxnStatus.Status.STREAM_NOT_FOUND)) {
                log.warn("Stream not found: {}", (Object)stream.getStreamName());
                throw new InvalidStreamException("Stream no longer exists: " + stream);
            }
            if (txnStatus.getStatus().equals((Object)Controller.TxnStatus.Status.TRANSACTION_NOT_FOUND)) {
                log.warn("transaction not found: {}", (Object)txId);
                throw Exceptions.sneakyThrow((Throwable)new TxnFailedException("Transaction was already either committed or aborted"));
            }
            if (txnStatus.getStatus().equals((Object)Controller.TxnStatus.Status.SUCCESS)) {
                return null;
            }
            log.warn("Unable to commit transaction {} commit status is {}", (Object)txId, (Object)txnStatus.getStatus());
            throw Exceptions.sneakyThrow((Throwable)new TxnFailedException("Commit transaction failed with status: " + txnStatus.getStatus()));
        });
    }

    @Override
    public CompletableFuture<Void> abortTransaction(Stream stream, UUID txId) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)stream, (Object)"stream");
        Preconditions.checkNotNull((Object)txId, (Object)"txId");
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"abortTransaction", (Object[])new Object[]{stream, txId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "abortTransaction", stream, txId);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).abortTransaction(Controller.TxnRequest.newBuilder().setStreamInfo(ModelHelper.createStreamInfo(stream.getScope(), stream.getStreamName())).setTxnId(ModelHelper.decode(txId)).build(), callback);
            return callback.getFuture();
        }, this.executor);
        return result.thenApply(txnStatus -> {
            LoggerHelpers.traceLeave((Logger)log, (String)"abortTransaction", (long)traceId, (Object[])new Object[0]);
            if (txnStatus.getStatus().equals((Object)Controller.TxnStatus.Status.STREAM_NOT_FOUND)) {
                log.warn("Stream not found: {}", (Object)stream.getStreamName());
                throw new InvalidStreamException("Stream no longer exists: " + stream);
            }
            if (txnStatus.getStatus().equals((Object)Controller.TxnStatus.Status.TRANSACTION_NOT_FOUND)) {
                log.warn("transaction not found: {}", (Object)txId);
                throw Exceptions.sneakyThrow((Throwable)new TxnFailedException("Transaction was already either committed or aborted"));
            }
            if (txnStatus.getStatus().equals((Object)Controller.TxnStatus.Status.SUCCESS)) {
                return null;
            }
            log.warn("Unable to abort transaction {} abort status is {} ", (Object)txId, (Object)txnStatus.getStatus());
            throw new RuntimeException("Error aborting transaction: " + txnStatus.getStatus());
        });
    }

    @Override
    public CompletableFuture<Transaction.Status> checkTransactionStatus(Stream stream, UUID txId) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)stream, (Object)"stream");
        Preconditions.checkNotNull((Object)txId, (Object)"txId");
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"checkTransactionStatus", (Object[])new Object[]{stream, txId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "checkTransactionStatus", stream, txId);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).checkTransactionState(Controller.TxnRequest.newBuilder().setStreamInfo(ModelHelper.createStreamInfo(stream.getScope(), stream.getStreamName())).setTxnId(ModelHelper.decode(txId)).build(), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(status -> ModelHelper.encode(status.getState(), stream + " " + txId))).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("checkTransactionStatus on " + stream + " " + txId + " failed: ", e);
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"checkTransactionStatus", (long)traceId, (Object[])new Object[0]);
        });
    }

    @Override
    public CompletableFuture<Void> noteTimestampFromWriter(String writer, Stream stream, long timestamp, WriterPosition lastWrittenPosition) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)stream, (Object)"stream");
        Preconditions.checkNotNull((Object)writer, (Object)"writer");
        Preconditions.checkNotNull((Object)lastWrittenPosition, (Object)"lastWrittenPosition");
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"noteTimestampFromWriter", (Object[])new Object[]{writer, stream});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "lastWrittenPosition", writer, stream, timestamp);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).noteTimestampFromWriter(Controller.TimestampFromWriter.newBuilder().setWriter(writer).setTimestamp(timestamp).setPosition(ModelHelper.createStreamCut(stream, lastWrittenPosition)).build(), callback);
            return callback.getFuture();
        }, this.executor);
        return result.thenApply(response -> {
            LoggerHelpers.traceLeave((Logger)log, (String)"noteTimestampFromWriter", (long)traceId, (Object[])new Object[0]);
            if (response.getResult().equals((Object)Controller.TimestampResponse.Status.SUCCESS)) {
                return null;
            }
            log.warn("Writer " + writer + " failed to note time because: " + response.getResult() + " time was: " + timestamp + " position=" + lastWrittenPosition);
            throw new RuntimeException("failed to note time because: " + response.getResult());
        });
    }

    @Override
    public CompletableFuture<Void> removeWriter(String writerId, Stream stream) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)stream, (Object)"stream");
        Preconditions.checkNotNull((Object)writerId, (Object)"writerId");
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"writerShutdown", (Object[])new Object[]{writerId, stream});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "writerShutdown", writerId, stream);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).removeWriter(Controller.RemoveWriterRequest.newBuilder().setWriter(writerId).setStream(ModelHelper.createStreamInfo(stream.getScope(), stream.getStreamName())).build(), callback);
            return callback.getFuture();
        }, this.executor);
        return result.thenApply(response -> {
            LoggerHelpers.traceLeave((Logger)log, (String)"writerShutdown", (long)traceId, (Object[])new Object[0]);
            if (response.getResult().equals((Object)Controller.RemoveWriterResponse.Status.SUCCESS)) {
                return null;
            }
            log.warn("Notifying the controller of writer shutdown failed for writer: " + writerId + " because of " + response.getResult());
            throw new RuntimeException("Unable to remove writer due to: " + response.getResult());
        });
    }

    @Override
    public void close() {
        if (!this.closed.getAndSet(true)) {
            Callbacks.invokeSafely(this::closeChannel, ex -> log.error("Error while closing ControllerImpl.", ex));
        }
    }

    private void closeChannel() {
        this.channel.shutdownNow();
        Exceptions.handleInterrupted(() -> {
            boolean shutdownStatus = this.channel.awaitTermination(20L, TimeUnit.SECONDS);
            log.debug("Controller client shutdown has been initiated. Channel status: channel.isTerminated():{}", (Object)shutdownStatus);
        });
    }

    @Override
    public CompletableFuture<String> getOrRefreshDelegationTokenFor(String scope, String streamName, AccessOperation accessOperation) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        Exceptions.checkNotNullOrEmpty((String)streamName, (String)"stream");
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"getOrRefreshDelegationTokenFor", (Object[])new Object[]{scope, streamName});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "getOrRefreshDelegationTokenFor", scope, streamName);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).getDelegationToken(ModelHelper.createStreamInfo(scope, streamName, accessOperation), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(token -> token.getDelegationToken())).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("getOrRefreshDelegationTokenFor {}/{} failed: ", new Object[]{scope, streamName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"getOrRefreshDelegationTokenFor", (long)traceId, (Object[])new Object[0]);
        });
    }

    @Override
    public CompletableFuture<Boolean> createKeyValueTable(String scope, String kvtName, KeyValueTableConfiguration kvtConfig) {
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)kvtConfig, (Object)"KeyValueTableConfig");
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"createKeyValueTable", (Object[])new Object[]{kvtConfig, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.CreateKeyValueTableStatus> callback = new RPCAsyncCallback<Controller.CreateKeyValueTableStatus>(requestId, "createKeyValueTable", scope, kvtName, kvtConfig);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "createKeyValueTable", scope, kvtName).createKeyValueTable(ModelHelper.decode(scope, kvtName, kvtConfig), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to create KeyValueTable: {}", new Object[]{kvtName});
                    throw new ControllerFailureException("Failed to create KeyValueTable: " + kvtConfig);
                }
                case INVALID_TABLE_NAME: {
                    log.warn(requestId, "Illegal KeyValueTable name: {}", new Object[]{kvtName});
                    throw new IllegalArgumentException("Illegal KeyValueTable name: " + kvtName);
                }
                case SCOPE_NOT_FOUND: {
                    log.warn(requestId, "Scope not found: {}", new Object[]{scope});
                    throw new IllegalArgumentException("Scope does not exist: " + scope);
                }
                case TABLE_EXISTS: {
                    log.warn(requestId, "KeyValueTable already exists: {}", new Object[]{kvtName});
                    return false;
                }
                case SUCCESS: {
                    log.info(requestId, "KeyValueTable created successfully: {}/{}", new Object[]{scope, kvtName});
                    return true;
                }
            }
            throw new ControllerFailureException("Unknown return status creating KeyValueTable " + kvtConfig + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "createKeyValueTable {}/{} failed: ", new Object[]{scope, kvtName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"createKeyValueTable", (long)traceId, (Object[])new Object[]{kvtName, requestId});
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public AsyncIterator<KeyValueTableInfo> listKeyValueTables(String scopeName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"listKeyValueTables", (Object[])new Object[]{scopeName});
        long requestId = this.requestIdGenerator.get();
        try {
            Function<Controller.ContinuationToken, CompletableFuture> function = token -> this.retryConfig.runAsync(() -> {
                RPCAsyncCallback<Controller.KVTablesInScopeResponse> callback = new RPCAsyncCallback<Controller.KVTablesInScopeResponse>(requestId, "listKeyValueTables", scopeName);
                Controller.ScopeInfo scopeInfo = Controller.ScopeInfo.newBuilder().setScope(scopeName).build();
                new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "listKeyValueTables", scopeName).listKeyValueTables(Controller.KVTablesInScopeRequest.newBuilder().setScope(scopeInfo).setContinuationToken(token).build(), callback);
                return callback.getFuture().thenApply(x -> {
                    switch (x.getStatus()) {
                        case SCOPE_NOT_FOUND: {
                            log.warn(requestId, "Scope not found: {}", new Object[]{scopeName});
                            throw new NoSuchScopeException();
                        }
                        case FAILURE: {
                            log.warn(requestId, "Internal Server Error while trying to list streams in scope: {}", new Object[]{scopeName});
                            throw new RuntimeException("Failure while trying to list streams");
                        }
                    }
                    List kvtList = x.getKvtablesList().stream().map(y -> new KeyValueTableInfo(y.getScope(), y.getKvtName())).collect(Collectors.toList());
                    return new AbstractMap.SimpleEntry(x.getContinuationToken(), kvtList);
                });
            }, this.executor);
            ContinuationTokenAsyncIterator continuationTokenAsyncIterator = new ContinuationTokenAsyncIterator(function, (Object)Controller.ContinuationToken.newBuilder().build());
            return continuationTokenAsyncIterator;
        }
        finally {
            LoggerHelpers.traceLeave((Logger)log, (String)"listKeyValueTables", (long)traceId, (Object[])new Object[0]);
        }
    }

    @Override
    public CompletableFuture<Boolean> deleteKeyValueTable(String scope, String kvtName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        Exceptions.checkNotNullOrEmpty((String)kvtName, (String)"KeyValueTableName");
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"deleteKeyValueTable", (Object[])new Object[]{scope, kvtName, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.DeleteKVTableStatus> callback = new RPCAsyncCallback<Controller.DeleteKVTableStatus>(requestId, "deleteKeyValueTable", scope, kvtName);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "deleteKeyValueTable", scope, kvtName).deleteKeyValueTable(ModelHelper.createKeyValueTableInfo(scope, kvtName), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to delete KeyValueTable: {}", new Object[]{kvtName});
                    throw new ControllerFailureException("Failed to delete KeyValueTable: " + kvtName);
                }
                case TABLE_NOT_FOUND: {
                    log.warn(requestId, "KeyValueTable does not exist: {}", new Object[]{kvtName});
                    return false;
                }
                case SUCCESS: {
                    log.info(requestId, "Successfully deleted KeyValueTable: {}", new Object[]{kvtName});
                    return true;
                }
            }
            throw new ControllerFailureException("Unknown return status deleting KeyValueTable " + kvtName + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "deleteKeyValueTable {}/{} failed: ", new Object[]{scope, kvtName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"deleteKeyValueTable", (long)traceId, (Object[])new Object[]{scope, kvtName, requestId});
        });
    }

    @Override
    public CompletableFuture<KeyValueTableSegments> getCurrentSegmentsForKeyValueTable(String scope, String kvtName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        Exceptions.checkNotNullOrEmpty((String)kvtName, (String)"stream");
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"getCurrentSegmentsForKeyValueTable", (Object[])new Object[]{scope, kvtName});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback callback = new RPCAsyncCallback(traceId, "getCurrentSegmentsKeyValueTable", scope, kvtName);
            ((ControllerServiceGrpc.ControllerServiceStub)this.client.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).getCurrentSegmentsKeyValueTable(ModelHelper.createKeyValueTableInfo(scope, kvtName), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(ranges -> {
            log.debug("Received the following data from the controller {}", (Object)ranges.getSegmentRangesList());
            TreeMap<Double, SegmentWithRange> rangeMap = new TreeMap<Double, SegmentWithRange>();
            for (Controller.SegmentRange r : ranges.getSegmentRangesList()) {
                Preconditions.checkState((r.getMinKey() <= r.getMaxKey() ? 1 : 0) != 0, (String)"Min keyrange %s was not less than maximum keyRange %s for segment %s", (Object)r.getMinKey(), (Object)r.getMaxKey(), (Object)r.getSegmentId());
                rangeMap.put(r.getMaxKey(), new SegmentWithRange(ModelHelper.encode(r.getSegmentId()), r.getMinKey(), r.getMaxKey()));
            }
            return new KeyValueTableSegments(rangeMap);
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn("getCurrentSegmentsForKeyValueTable for {}/{} failed: ", new Object[]{scope, kvtName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"getCurrentSegmentsForKeyValueTable", (long)traceId, (Object[])new Object[0]);
        });
    }

    @Override
    public CompletableFuture<ReaderGroupConfig> createReaderGroup(String scope, String rgName, ReaderGroupConfig rgConfig) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        Exceptions.checkNotNullOrEmpty((String)rgName, (String)"rgName");
        Preconditions.checkNotNull((Object)rgConfig, (Object)"rgConfig");
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"createReaderGroup", (Object[])new Object[]{rgConfig, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.CreateReaderGroupResponse> callback = new RPCAsyncCallback<Controller.CreateReaderGroupResponse>(requestId, "createReaderGroup", scope, rgName, rgConfig);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "createReaderGroup", scope, rgName).createReaderGroup(ModelHelper.decode(scope, rgName, rgConfig), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to create reader group: {}", new Object[]{rgName});
                    throw new ControllerFailureException("Failed to create reader group: " + rgName);
                }
                case INVALID_RG_NAME: {
                    log.warn(requestId, "Illegal Reader Group Name: {}", new Object[]{rgName});
                    throw new IllegalArgumentException("Illegal readergroup name: " + rgName);
                }
                case SCOPE_NOT_FOUND: {
                    log.warn(requestId, "Scope not found: {}", new Object[]{scope});
                    throw new IllegalArgumentException("Scope does not exist: " + scope);
                }
                case SUCCESS: {
                    log.info(requestId, "ReaderGroup created successfully: {}", new Object[]{rgName});
                    return ModelHelper.encode(x.getConfig());
                }
            }
            throw new ControllerFailureException("Unknown return status creating reader group " + rgName + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "createReaderGroup {}/{} failed: ", new Object[]{scope, rgName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"createReaderGroup", (long)traceId, (Object[])new Object[]{rgConfig, requestId});
        });
    }

    @Override
    public CompletableFuture<Long> updateReaderGroup(String scope, String rgName, ReaderGroupConfig rgConfig) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        Exceptions.checkNotNullOrEmpty((String)rgName, (String)"rgName");
        Preconditions.checkNotNull((Object)rgConfig, (Object)"rgConfig");
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"updateReaderGroup", (Object[])new Object[]{rgConfig, requestId});
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.UpdateReaderGroupResponse> callback = new RPCAsyncCallback<Controller.UpdateReaderGroupResponse>(requestId, "updateReaderGroup", scope, rgName, rgConfig);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "updateReaderGroup", scope, rgName).updateReaderGroup(ModelHelper.decode(scope, rgName, rgConfig), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to create reader group: {}", new Object[]{rgName});
                    throw new ControllerFailureException("Failed to create readergroup: " + rgName);
                }
                case INVALID_CONFIG: {
                    log.warn(requestId, "Illegal Reader Group Config for reader group {}: {}", new Object[]{rgName, rgConfig});
                    throw new ReaderGroupConfigRejectedException("Invalid Reader Group Config: " + rgConfig.toString());
                }
                case RG_NOT_FOUND: {
                    log.warn(requestId, "Scope not found: {}", new Object[]{scope});
                    throw new IllegalArgumentException("Scope does not exist: " + scope);
                }
                case SUCCESS: {
                    log.info(requestId, "ReaderGroup created successfully: {}", new Object[]{rgName});
                    return x.getGeneration();
                }
            }
            throw new ControllerFailureException("Unknown return status creating reader group " + rgName + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "createReaderGroup {}/{} failed: ", new Object[]{scope, rgName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"createReaderGroup", (long)traceId, (Object[])new Object[]{rgConfig, requestId});
        });
    }

    @Override
    public CompletableFuture<ReaderGroupConfig> getReaderGroupConfig(String scope, String rgName) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        String emptyUUID = "";
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"getReaderGroupConfig", (Object[])new Object[]{scope, rgName, requestId});
        String scopedRGName = NameUtils.getScopedReaderGroupName((String)scope, (String)rgName);
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.ReaderGroupConfigResponse> callback = new RPCAsyncCallback<Controller.ReaderGroupConfigResponse>(requestId, "getReaderGroupConfig", scope, rgName);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "getReaderGroupConfig", scope, rgName).getReaderGroupConfig(ModelHelper.createReaderGroupInfo(scope, rgName, "", 0L), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to get config for reader group: {}", new Object[]{scopedRGName});
                    throw new ControllerFailureException("Failed to get config for reader group: " + scopedRGName);
                }
                case RG_NOT_FOUND: {
                    log.warn(requestId, "ReaderGroup not found: {}", new Object[]{scopedRGName});
                    throw new IllegalArgumentException("ReaderGroup does not exist: " + scopedRGName);
                }
                case SUCCESS: {
                    log.info(requestId, "Successfully got config for Reader Group: {}", new Object[]{scopedRGName});
                    return ModelHelper.encode(x.getConfig());
                }
            }
            throw new ControllerFailureException("Unknown return status getting config for ReaderGroup " + scopedRGName + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "getReaderGroupConfig failed for Reader Group: ", new Object[]{scopedRGName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"getReaderGroupConfig", (long)traceId, (Object[])new Object[]{scope, rgName, requestId});
        });
    }

    @Override
    public CompletableFuture<Boolean> deleteReaderGroup(String scope, String rgName, UUID readerGroupId) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Exceptions.checkNotNullOrEmpty((String)scope, (String)"scope");
        Exceptions.checkNotNullOrEmpty((String)rgName, (String)"rgName");
        Preconditions.checkNotNull((Object)readerGroupId, (Object)"rgId");
        long requestId = this.requestIdGenerator.get();
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"deleteReaderGroup", (Object[])new Object[]{scope, rgName, requestId});
        String scopedRGName = NameUtils.getScopedReaderGroupName((String)scope, (String)rgName);
        CompletableFuture result = this.retryConfig.runAsync(() -> {
            RPCAsyncCallback<Controller.DeleteReaderGroupStatus> callback = new RPCAsyncCallback<Controller.DeleteReaderGroupStatus>(requestId, "deleteReaderGroup", scope, rgName);
            new ControllerClientTagger(this.client, this.timeoutMillis).withTag(requestId, "deleteReaderGroup", scope, rgName).deleteReaderGroup(ModelHelper.createReaderGroupInfo(scope, rgName, readerGroupId.toString(), 0L), callback);
            return callback.getFuture();
        }, this.executor);
        return ((CompletableFuture)result.thenApply(x -> {
            switch (x.getStatus()) {
                case FAILURE: {
                    log.warn(requestId, "Failed to delete reader group: {}", new Object[]{scopedRGName});
                    throw new ControllerFailureException("Failed to delete reader group: " + scopedRGName);
                }
                case RG_NOT_FOUND: {
                    log.warn(requestId, "ReaderGroup not found: {}", new Object[]{scopedRGName});
                    throw new IllegalArgumentException("ReaderGroup does not exist: " + scopedRGName);
                }
                case SUCCESS: {
                    log.info(requestId, "Successfully deleted Reader Group: {}", new Object[]{scopedRGName});
                    return true;
                }
            }
            throw new ControllerFailureException("Unknown return status getting config for ReaderGroup " + scopedRGName + " " + x.getStatus());
        })).whenComplete((x, e) -> {
            if (e != null) {
                log.warn(requestId, "deleteReaderGroup failed for Reader Group: ", new Object[]{scopedRGName, e});
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"deleteReaderGroup", (long)traceId, (Object[])new Object[]{scope, rgName, requestId});
        });
    }

    private static class ControllerClientTagger {
        private ControllerServiceGrpc.ControllerServiceStub clientStub;
        private final long timeoutMillis;

        ControllerClientTagger(ControllerServiceGrpc.ControllerServiceStub clientStub, long timeoutMillis) {
            this.clientStub = clientStub;
            this.timeoutMillis = timeoutMillis;
        }

        ControllerClientTagger withTag(long requestId, String ... requestInfo) {
            String requestDescriptor = RequestTracker.buildRequestDescriptor((String[])requestInfo);
            log.info(requestId, "Tagging client request ({}).", new Object[]{requestDescriptor});
            this.clientStub = (ControllerServiceGrpc.ControllerServiceStub)((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withOption(RPCTracingHelpers.REQUEST_DESCRIPTOR_CALL_OPTION, (Object)requestDescriptor)).withOption(RPCTracingHelpers.REQUEST_ID_CALL_OPTION, (Object)String.valueOf(requestId));
            return this;
        }

        public void createScope(Controller.ScopeInfo scopeInfo, RPCAsyncCallback<Controller.CreateScopeStatus> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).createScope(scopeInfo, callback);
        }

        public void listScopes(Controller.ScopesRequest request, RPCAsyncCallback<Controller.ScopesResponse> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).listScopes(request, callback);
        }

        public void checkScopeExists(Controller.ScopeInfo request, RPCAsyncCallback<Controller.ExistsResponse> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).checkScopeExists(request, callback);
        }

        public void checkStreamExists(Controller.StreamInfo request, RPCAsyncCallback<Controller.ExistsResponse> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).checkStreamExists(request, callback);
        }

        public void listStreamsInScope(Controller.StreamsInScopeRequest request, RPCAsyncCallback<Controller.StreamsInScopeResponse> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).listStreamsInScope(request, callback);
        }

        public void deleteScope(Controller.ScopeInfo scopeInfo, RPCAsyncCallback<Controller.DeleteScopeStatus> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).deleteScope(scopeInfo, callback);
        }

        public void createStream(Controller.StreamConfig streamConfig, RPCAsyncCallback<Controller.CreateStreamStatus> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).createStream(streamConfig, callback);
        }

        public void scale(Controller.ScaleRequest scaleRequest, RPCAsyncCallback<Controller.ScaleResponse> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).scale(scaleRequest, callback);
        }

        public void updateStream(Controller.StreamConfig streamConfig, RPCAsyncCallback<Controller.UpdateStreamStatus> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).updateStream(streamConfig, callback);
        }

        public void truncateStream(Controller.StreamCut streamCut, RPCAsyncCallback<Controller.UpdateStreamStatus> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).truncateStream(streamCut, callback);
        }

        public void sealStream(Controller.StreamInfo streamInfo, RPCAsyncCallback<Controller.UpdateStreamStatus> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).sealStream(streamInfo, callback);
        }

        public void deleteStream(Controller.StreamInfo streamInfo, RPCAsyncCallback<Controller.DeleteStreamStatus> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).deleteStream(streamInfo, callback);
        }

        public void updateSubscriberStreamCut(Controller.SubscriberStreamCut subscriberStreamCut, RPCAsyncCallback<Controller.UpdateSubscriberStatus> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).updateSubscriberStreamCut(subscriberStreamCut, callback);
        }

        public void listSubscribers(Controller.StreamInfo request, RPCAsyncCallback<Controller.SubscribersResponse> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).listSubscribers(request, callback);
        }

        public void createKeyValueTable(Controller.KeyValueTableConfig kvtConfig, RPCAsyncCallback<Controller.CreateKeyValueTableStatus> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).createKeyValueTable(kvtConfig, callback);
        }

        void listKeyValueTables(Controller.KVTablesInScopeRequest request, RPCAsyncCallback<Controller.KVTablesInScopeResponse> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).listKeyValueTablesInScope(request, callback);
        }

        void deleteKeyValueTable(Controller.KeyValueTableInfo kvtInfo, RPCAsyncCallback<Controller.DeleteKVTableStatus> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).deleteKeyValueTable(kvtInfo, callback);
        }

        void createReaderGroup(Controller.ReaderGroupConfiguration rgConfig, RPCAsyncCallback<Controller.CreateReaderGroupResponse> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).createReaderGroup(rgConfig, callback);
        }

        void getReaderGroupConfig(Controller.ReaderGroupInfo readerGroupInfo, RPCAsyncCallback<Controller.ReaderGroupConfigResponse> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).getReaderGroupConfig(readerGroupInfo, callback);
        }

        void deleteReaderGroup(Controller.ReaderGroupInfo readerGroupInfo, RPCAsyncCallback<Controller.DeleteReaderGroupStatus> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).deleteReaderGroup(readerGroupInfo, callback);
        }

        void updateReaderGroup(Controller.ReaderGroupConfiguration rgConfig, RPCAsyncCallback<Controller.UpdateReaderGroupResponse> callback) {
            ((ControllerServiceGrpc.ControllerServiceStub)this.clientStub.withDeadlineAfter(this.timeoutMillis, TimeUnit.MILLISECONDS)).updateReaderGroup(rgConfig, callback);
        }
    }

    private static final class RPCAsyncCallback<T>
    implements StreamObserver<T> {
        private final long traceId;
        private final String method;
        private final Object[] parameters;
        private T result = null;
        private final CompletableFuture<T> future = new CompletableFuture();

        RPCAsyncCallback(long traceId, String method, Object ... args) {
            this.traceId = traceId;
            this.method = method;
            this.parameters = args;
        }

        public void onNext(T value) {
            this.result = value;
        }

        public void onError(Throwable t) {
            log.warn("gRPC call for {} with trace id {} and parameters {} failed with server error.", new Object[]{this.method, this.traceId, this.parameters, t});
            if (t instanceof RuntimeException) {
                this.future.completeExceptionally(t);
            } else {
                this.future.completeExceptionally(new RuntimeException(t));
            }
        }

        public void onCompleted() {
            this.future.complete(this.result);
        }

        public CompletableFuture<T> getFuture() {
            return this.future;
        }
    }
}

