/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.keycloak.client;

import io.pravega.common.util.Retry;
import io.pravega.keycloak.client.KeycloakAuthenticationException;
import io.pravega.keycloak.client.KeycloakAuthorizationException;
import io.pravega.keycloak.client.KeycloakConfigResolver;
import io.pravega.keycloak.client.KeycloakConfigurationException;
import io.pravega.keycloak.org.keycloak.authorization.client.AuthzClient;
import io.pravega.keycloak.org.keycloak.authorization.client.ClientAuthenticator;
import io.pravega.keycloak.org.keycloak.authorization.client.Configuration;
import io.pravega.keycloak.org.keycloak.authorization.client.util.HttpResponseException;
import io.pravega.keycloak.org.keycloak.common.util.Time;
import io.pravega.keycloak.org.keycloak.jose.jws.JWSInput;
import io.pravega.keycloak.org.keycloak.jose.jws.JWSInputException;
import io.pravega.keycloak.org.keycloak.representations.AccessToken;
import io.pravega.keycloak.org.keycloak.representations.AccessTokenResponse;
import io.pravega.keycloak.org.keycloak.representations.idm.authorization.AuthorizationRequest;
import io.pravega.keycloak.org.keycloak.representations.idm.authorization.AuthorizationResponse;
import io.pravega.keycloak.org.keycloak.util.BasicAuthHelper;
import java.io.IOException;
import java.net.ConnectException;
import java.net.UnknownHostException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.Predicate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeycloakAuthzClient {
    private static final Logger LOG = LoggerFactory.getLogger(KeycloakAuthzClient.class);
    public static final String DEFAULT_PRAVEGA_CONTROLLER_CLIENT_ID = "pravega-controller";
    private static final int DEFAULT_TOKEN_MIN_TIME_TO_LIVE_SECS = 60;
    private static final int DEFAULT_HTTP_MAX_RETRIES = 20;
    private static final int DEFAULT_HTTP_INITIAL_DELAY_MS = 100;
    private final AuthzClient client;
    private final TokenCache tokenCache;
    private final int httpMaxRetries;
    private final int httpInitialDelayMs;

    public static Builder builder() {
        return new Builder();
    }

    public KeycloakAuthzClient(AuthzClient client, TokenCache tokenCache) {
        this.client = client;
        this.tokenCache = tokenCache;
        this.httpMaxRetries = 20;
        this.httpInitialDelayMs = 100;
    }

    public KeycloakAuthzClient(AuthzClient client, TokenCache tokenCache, int httpMaxRetries, int httpInitialDelayMs) {
        this.client = client;
        this.tokenCache = tokenCache;
        this.httpMaxRetries = httpMaxRetries;
        this.httpInitialDelayMs = httpInitialDelayMs;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String getRPT() {
        AuthorizationResponse token;
        TokenCache tokenCache = this.tokenCache;
        synchronized (tokenCache) {
            token = this.tokenCache.getIfValid();
        }
        if (token == null) {
            AccessTokenResponse accResponse;
            try {
                accResponse = (AccessTokenResponse)Retry.withExpBackoff((long)this.httpInitialDelayMs, (int)2, (int)this.httpMaxRetries).retryWhen(KeycloakAuthzClient.isRetryable()).run(() -> this.client.obtainAccessToken());
                LOG.debug("Obtained access token from Keycloak");
            }
            catch (HttpResponseException e) {
                LOG.error("Failed to obtain access token from Keycloak", (Throwable)e);
                throw new KeycloakAuthenticationException(e);
            }
            AuthorizationRequest request = new AuthorizationRequest();
            try {
                token = (AuthorizationResponse)Retry.withExpBackoff((long)this.httpInitialDelayMs, (int)2, (int)this.httpMaxRetries).retryWhen(KeycloakAuthzClient.isRetryable()).run(() -> this.client.authorization(accResponse.getToken()).authorize(request));
                LOG.debug("Obtained RPT from Keycloak");
            }
            catch (HttpResponseException e) {
                LOG.error("Failed to obtain RPT from Keycloak", (Throwable)e);
                throw new KeycloakAuthorizationException(e);
            }
            TokenCache tokenCache2 = this.tokenCache;
            synchronized (tokenCache2) {
                this.tokenCache.update(token);
            }
        }
        return token.getToken();
    }

    private static Predicate<Throwable> isRetryable() {
        return e -> {
            Throwable rootCause = KeycloakAuthzClient.unwrap(e);
            if (rootCause instanceof HttpResponseException) {
                int statusCode = ((HttpResponseException)e).getStatusCode();
                if (statusCode == 400 || statusCode == 401 || statusCode == 403) {
                    LOG.error("Non retryable HttpResponseException with HTTP code: {}", (Object)statusCode);
                    return false;
                }
                LOG.warn("Retryable HttpResponseException with HTTP code:  {}", (Object)statusCode);
                return true;
            }
            if (rootCause instanceof ConnectException || rootCause instanceof UnknownHostException) {
                LOG.warn("Retryable connection exception", rootCause);
                return true;
            }
            LOG.error("Other non retryable exception", rootCause);
            return false;
        };
    }

    private static Throwable unwrap(Throwable e) {
        if (e.getCause() != null) {
            return KeycloakAuthzClient.unwrap(e.getCause());
        }
        return e;
    }

    static AccessToken toAccessToken(String rawToken) {
        AccessToken accessToken;
        try {
            accessToken = new JWSInput(rawToken).readJsonContent(AccessToken.class);
        }
        catch (JWSInputException cause) {
            throw new IllegalArgumentException("Failed to deserialize token", cause);
        }
        return accessToken;
    }

    public static class Builder {
        private String audience = "pravega-controller";
        private String configFile;
        private String configString;
        private BiFunction<Configuration, ClientAuthenticator, AuthzClient> clientSupplier = AuthzClient::create;
        private int httpMaxRetries = 20;
        private int httpInitialDelayMs = 100;

        public Builder withConfigFile(String path) {
            this.configFile = path;
            return this;
        }

        public Builder withConfigString(String configString) {
            this.configString = configString;
            return this;
        }

        public Builder withAudience(String audience) {
            this.audience = audience;
            return this;
        }

        public Builder withCustomRetries(int httpMaxRetries, int httpInitialDelayMs) {
            this.httpInitialDelayMs = httpInitialDelayMs;
            this.httpMaxRetries = httpMaxRetries;
            return this;
        }

        Builder withAuthzClientSupplier(BiFunction<Configuration, ClientAuthenticator, AuthzClient> clientSupplier) {
            this.clientSupplier = clientSupplier;
            return this;
        }

        public KeycloakAuthzClient build() {
            Configuration configuration;
            try {
                String errorMessage = "Unable to resolve a Keycloak client configuration for Pravega authentication purposes.\n\nUse one of the following approaches to provide a client configuration (in Keycloak OIDC JSON format):\n1. Use a builder method to set the keycloak OIDC config as a JSON string.\n2. Use a builder method to set the path to a file.\n3. Set the environment variable 'KEYCLOAK_SERVICE_ACCOUNT_FILE' to the path to a file.\n4. Update the classpath to contain a resource named 'keycloak.json'.\n";
                configuration = this.configString != null ? KeycloakConfigResolver.resolveFromString(this.configString).orElseThrow(() -> new KeycloakConfigurationException(errorMessage)) : KeycloakConfigResolver.resolve(this.configFile).orElseThrow(() -> new KeycloakConfigurationException(errorMessage));
            }
            catch (IOException e) {
                throw new KeycloakConfigurationException("Unexpected error in resolving or loading the Keycloak client configuration", e);
            }
            if (configuration.getTokenMinimumTimeToLive() == 0) {
                configuration.setTokenMinimumTimeToLive(60);
            }
            ClientAuthenticator authenticator = this.createClientAuthenticator(configuration.getResource(), (String)configuration.getCredentials().get("secret"));
            AuthzClient client = (AuthzClient)Retry.withExpBackoff((long)this.httpInitialDelayMs, (int)2, (int)this.httpMaxRetries).retryWhen(KeycloakAuthzClient.isRetryable()).run(() -> this.clientSupplier.apply(configuration, authenticator));
            configuration.setResource(this.audience);
            return new KeycloakAuthzClient(client, new TokenCache(configuration.getTokenMinimumTimeToLive()), this.httpMaxRetries, this.httpInitialDelayMs);
        }

        private ClientAuthenticator createClientAuthenticator(final String clientId, final String clientSecret) {
            return new ClientAuthenticator(){

                @Override
                public void configureClientCredentials(Map<String, List<String>> requestParams, Map<String, String> requestHeaders) {
                    Objects.requireNonNull(clientId, "Client ID not provided.");
                    Objects.requireNonNull(clientSecret, "Client secret not provided.");
                    requestHeaders.put("Authorization", BasicAuthHelper.createHeader(clientId, clientSecret));
                }
            };
        }
    }

    static class TokenCache {
        private final int tokenMinimumTimeToLiveSecs;
        AuthorizationResponse response;
        AccessToken token;

        public TokenCache(int tokenMinimumTimeToLiveSecs) {
            this.tokenMinimumTimeToLiveSecs = tokenMinimumTimeToLiveSecs;
        }

        public AuthorizationResponse getIfValid() {
            if (this.response != null && this.token != null && this.isTokenTimeToLiveSufficient()) {
                return this.response;
            }
            return null;
        }

        public void update(AuthorizationResponse response) {
            Objects.requireNonNull(response);
            this.token = KeycloakAuthzClient.toAccessToken(response.getToken());
            this.response = response;
        }

        boolean isTokenTimeToLiveSufficient() {
            return this.token.getExpiration() - this.tokenMinimumTimeToLiveSecs > Time.currentTime();
        }
    }
}

