/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.host;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.auth.JKSHelper;
import io.pravega.common.auth.ZKTLSUtils;
import io.pravega.common.cluster.Host;
import io.pravega.segmentstore.contracts.StreamSegmentStore;
import io.pravega.segmentstore.contracts.tables.TableStore;
import io.pravega.segmentstore.server.host.StorageLoader;
import io.pravega.segmentstore.server.host.ZKSegmentContainerManager;
import io.pravega.segmentstore.server.host.delegationtoken.TokenVerifierImpl;
import io.pravega.segmentstore.server.host.handler.PravegaConnectionListener;
import io.pravega.segmentstore.server.host.stat.AutoScaleMonitor;
import io.pravega.segmentstore.server.host.stat.AutoScalerConfig;
import io.pravega.segmentstore.server.store.ServiceBuilder;
import io.pravega.segmentstore.server.store.ServiceBuilderConfig;
import io.pravega.segmentstore.server.store.ServiceConfig;
import io.pravega.segmentstore.storage.ConfigSetup;
import io.pravega.segmentstore.storage.impl.bookkeeper.BookKeeperConfig;
import io.pravega.segmentstore.storage.impl.bookkeeper.BookKeeperLogFactory;
import io.pravega.segmentstore.storage.impl.rocksdb.RocksDBCacheFactory;
import io.pravega.segmentstore.storage.impl.rocksdb.RocksDBConfig;
import io.pravega.segmentstore.storage.mocks.InMemoryDurableDataLogFactory;
import io.pravega.shared.metrics.MetricsConfig;
import io.pravega.shared.metrics.MetricsProvider;
import io.pravega.shared.metrics.StatsProvider;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.curator.RetryPolicy;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.ExponentialBackoffRetry;
import org.apache.curator.utils.ZookeeperFactory;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ServiceStarter {
    @SuppressFBWarnings(justification="generated code")
    private static final Logger log = LoggerFactory.getLogger(ServiceStarter.class);
    private final ServiceBuilderConfig builderConfig;
    private final ServiceConfig serviceConfig;
    private final ServiceBuilder serviceBuilder;
    private StatsProvider statsProvider;
    private PravegaConnectionListener listener;
    private AutoScaleMonitor autoScaleMonitor;
    private CuratorFramework zkClient;
    private boolean closed;

    public ServiceStarter(ServiceBuilderConfig config) {
        this.builderConfig = config;
        this.serviceConfig = (ServiceConfig)this.builderConfig.getConfig(ServiceConfig::builder);
        this.serviceBuilder = this.createServiceBuilder();
    }

    private ServiceBuilder createServiceBuilder() {
        ServiceBuilder builder = ServiceBuilder.newInMemoryBuilder((ServiceBuilderConfig)this.builderConfig);
        this.attachDataLogFactory(builder);
        this.attachRocksDB(builder);
        this.attachStorage(builder);
        this.attachZKSegmentManager(builder);
        return builder;
    }

    public void start() throws Exception {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        log.info("Initializing metrics provider ...");
        MetricsProvider.initialize((MetricsConfig)((MetricsConfig)this.builderConfig.getConfig(MetricsConfig::builder)));
        this.statsProvider = MetricsProvider.getMetricsProvider();
        this.statsProvider.start();
        log.info("Initializing ZooKeeper Client ...");
        this.zkClient = this.createZKClient();
        log.info("Initializing Service Builder ...");
        this.serviceBuilder.initialize();
        log.info("Creating StreamSegmentService ...");
        StreamSegmentStore service = this.serviceBuilder.createStreamSegmentService();
        log.info("Creating TableStoreService ...");
        TableStore tableStoreService = this.serviceBuilder.createTableStoreService();
        log.info("Creating Segment Stats recorder ...");
        this.autoScaleMonitor = new AutoScaleMonitor(service, (AutoScalerConfig)this.builderConfig.getConfig(AutoScalerConfig::builder));
        TokenVerifierImpl tokenVerifier = new TokenVerifierImpl((AutoScalerConfig)this.builderConfig.getConfig(AutoScalerConfig::builder));
        log.info(this.serviceConfig.toString());
        log.info(((AutoScalerConfig)this.builderConfig.getConfig(AutoScalerConfig::builder)).toString());
        this.listener = new PravegaConnectionListener(this.serviceConfig.isEnableTls(), this.serviceConfig.isEnableTlsReload(), this.serviceConfig.getListeningIPAddress(), this.serviceConfig.getListeningPort(), service, tableStoreService, this.autoScaleMonitor.getStatsRecorder(), this.autoScaleMonitor.getTableSegmentStatsRecorder(), tokenVerifier, this.serviceConfig.getCertFile(), this.serviceConfig.getKeyFile(), this.serviceConfig.isReplyWithStackTraceOnError());
        this.listener.startListening();
        log.info("PravegaConnectionListener started successfully.");
        log.info("StreamSegmentService started.");
    }

    public void shutdown() {
        if (!this.closed) {
            this.serviceBuilder.close();
            log.info("StreamSegmentService shut down.");
            if (this.listener != null) {
                this.listener.close();
                log.info("PravegaConnectionListener closed.");
            }
            if (this.statsProvider != null) {
                this.statsProvider.close();
                this.statsProvider = null;
                log.info("Metrics statsProvider is now closed.");
            }
            if (this.zkClient != null) {
                this.zkClient.close();
                this.zkClient = null;
                log.info("ZooKeeper Client shut down.");
            }
            if (this.autoScaleMonitor != null) {
                this.autoScaleMonitor.close();
                this.autoScaleMonitor = null;
                log.info("AutoScaleMonitor shut down.");
            }
            if (this.serviceConfig.isSecureZK()) {
                ZKTLSUtils.unsetSecureZKClientProperties();
            }
            this.closed = true;
        }
    }

    private void attachDataLogFactory(ServiceBuilder builder) {
        builder.withDataLogFactory(setup -> {
            switch (this.serviceConfig.getDataLogTypeImplementation()) {
                case BOOKKEEPER: {
                    return new BookKeeperLogFactory((BookKeeperConfig)setup.getConfig(BookKeeperConfig::builder), this.zkClient, setup.getCoreExecutor());
                }
                case INMEMORY: {
                    return new InMemoryDurableDataLogFactory(setup.getCoreExecutor());
                }
            }
            throw new IllegalStateException("Unsupported storage implementation: " + this.serviceConfig.getDataLogTypeImplementation());
        });
    }

    private void attachRocksDB(ServiceBuilder builder) {
        builder.withCacheFactory(setup -> new RocksDBCacheFactory((RocksDBConfig)setup.getConfig(RocksDBConfig::builder)));
    }

    private void attachStorage(ServiceBuilder builder) {
        builder.withStorageFactory(setup -> {
            StorageLoader loader = new StorageLoader();
            return loader.load((ConfigSetup)setup, this.serviceConfig.getStorageImplementation().toString(), setup.getStorageExecutor());
        });
    }

    private void attachZKSegmentManager(ServiceBuilder builder) {
        builder.withContainerManager(setup -> new ZKSegmentContainerManager(setup.getContainerRegistry(), this.zkClient, new Host(this.serviceConfig.getPublishedIPAddress(), this.serviceConfig.getPublishedPort(), null), setup.getCoreExecutor()));
    }

    @VisibleForTesting
    public CuratorFramework createZKClient() {
        if (this.serviceConfig.isSecureZK()) {
            ZKTLSUtils.setSecureZKClientProperties((String)this.serviceConfig.getZkTrustStore(), (String)JKSHelper.loadPasswordFrom((String)this.serviceConfig.getZkTrustStorePasswordPath()));
        }
        CuratorFramework zkClient = CuratorFrameworkFactory.builder().connectString(this.serviceConfig.getZkURL()).namespace("pravega/" + this.serviceConfig.getClusterName()).zookeeperFactory((ZookeeperFactory)new ZKClientFactory()).retryPolicy((RetryPolicy)new ExponentialBackoffRetry(this.serviceConfig.getZkRetrySleepMs(), this.serviceConfig.getZkRetryCount())).sessionTimeoutMs(this.serviceConfig.getZkSessionTimeoutMs()).build();
        zkClient.start();
        return zkClient;
    }

    public static void main(String[] args) throws Exception {
        AtomicReference<ServiceStarter> serviceStarter = new AtomicReference<ServiceStarter>();
        try {
            System.err.println(System.getProperty("pravega.configurationFile", "config.properties"));
            ServiceBuilderConfig config = ServiceBuilderConfig.builder().include(System.getProperty("pravega.configurationFile", "config.properties")).include(System.getProperties()).build();
            log.info("Segment store configuration:");
            config.forEach((key, value) -> log.info("{} = {}", key, value));
            serviceStarter.set(new ServiceStarter(config));
        }
        catch (Throwable e) {
            log.error("Could not create a Service with default config, Aborting.", e);
            System.exit(1);
        }
        try {
            ((ServiceStarter)serviceStarter.get()).start();
        }
        catch (Throwable e) {
            log.error("Could not start the Service, Aborting.", e);
            System.exit(1);
        }
        try {
            Runtime.getRuntime().addShutdownHook(new Thread(() -> {
                log.info("Caught interrupt signal...");
                ((ServiceStarter)serviceStarter.get()).shutdown();
            }));
            Thread.sleep(Long.MAX_VALUE);
        }
        catch (InterruptedException ex) {
            log.info("Caught interrupt signal...");
        }
        finally {
            ((ServiceStarter)serviceStarter.get()).shutdown();
            System.exit(0);
        }
    }

    @ThreadSafe
    static class ZKClientFactory
    implements ZookeeperFactory {
        @GuardedBy(value="this")
        private ZooKeeper client;
        @GuardedBy(value="this")
        private String connectString;
        @GuardedBy(value="this")
        private int sessionTimeout;
        @GuardedBy(value="this")
        private boolean canBeReadOnly;

        ZKClientFactory() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public ZooKeeper newZooKeeper(String connectString, int sessionTimeout, Watcher watcher, boolean canBeReadOnly) throws Exception {
            Exceptions.checkNotNullOrEmpty((String)connectString, (String)"connectString");
            Preconditions.checkArgument((sessionTimeout > 0 ? 1 : 0) != 0, (Object)"sessionTimeout should be a positive integer");
            ZKClientFactory zKClientFactory = this;
            synchronized (zKClientFactory) {
                if (this.client == null) {
                    this.connectString = connectString;
                    this.sessionTimeout = sessionTimeout;
                    this.canBeReadOnly = canBeReadOnly;
                }
                log.info("Creating new Zookeeper client with arguments: {}, {}, {}.", new Object[]{this.connectString, this.sessionTimeout, this.canBeReadOnly});
                this.client = new ZooKeeper(this.connectString, this.sessionTimeout, watcher, this.canBeReadOnly);
                return this.client;
            }
        }
    }
}

