/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.host.handler;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.pravega.auth.AuthHandler;
import io.pravega.auth.TokenException;
import io.pravega.auth.TokenExpiredException;
import io.pravega.common.Exceptions;
import io.pravega.common.LoggerHelpers;
import io.pravega.common.Timer;
import io.pravega.common.function.Callbacks;
import io.pravega.common.io.StreamHelpers;
import io.pravega.common.tracing.TagLogger;
import io.pravega.common.util.ArrayView;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.segmentstore.contracts.AttributeUpdate;
import io.pravega.segmentstore.contracts.AttributeUpdateType;
import io.pravega.segmentstore.contracts.Attributes;
import io.pravega.segmentstore.contracts.BadAttributeUpdateException;
import io.pravega.segmentstore.contracts.BadOffsetException;
import io.pravega.segmentstore.contracts.ContainerNotFoundException;
import io.pravega.segmentstore.contracts.MergeStreamSegmentResult;
import io.pravega.segmentstore.contracts.ReadResult;
import io.pravega.segmentstore.contracts.ReadResultEntry;
import io.pravega.segmentstore.contracts.ReadResultEntryContents;
import io.pravega.segmentstore.contracts.ReadResultEntryType;
import io.pravega.segmentstore.contracts.StreamSegmentExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentMergedException;
import io.pravega.segmentstore.contracts.StreamSegmentNotExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentSealedException;
import io.pravega.segmentstore.contracts.StreamSegmentStore;
import io.pravega.segmentstore.contracts.StreamSegmentTruncatedException;
import io.pravega.segmentstore.contracts.tables.BadKeyVersionException;
import io.pravega.segmentstore.contracts.tables.KeyNotExistsException;
import io.pravega.segmentstore.contracts.tables.TableEntry;
import io.pravega.segmentstore.contracts.tables.TableKey;
import io.pravega.segmentstore.contracts.tables.TableSegmentNotEmptyException;
import io.pravega.segmentstore.contracts.tables.TableStore;
import io.pravega.segmentstore.server.IllegalContainerStateException;
import io.pravega.segmentstore.server.host.delegationtoken.DelegationTokenVerifier;
import io.pravega.segmentstore.server.host.delegationtoken.PassingTokenVerifier;
import io.pravega.segmentstore.server.host.handler.ServerConnection;
import io.pravega.segmentstore.server.host.stat.SegmentStatsRecorder;
import io.pravega.segmentstore.server.host.stat.TableSegmentStatsRecorder;
import io.pravega.shared.protocol.netty.FailingRequestProcessor;
import io.pravega.shared.protocol.netty.RequestProcessor;
import io.pravega.shared.protocol.netty.WireCommand;
import io.pravega.shared.protocol.netty.WireCommands;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PravegaRequestProcessor
extends FailingRequestProcessor
implements RequestProcessor {
    static final Duration TIMEOUT = Duration.ofMinutes(1L);
    private static final TagLogger log = new TagLogger(LoggerFactory.getLogger(PravegaRequestProcessor.class));
    private static final int MAX_READ_SIZE = 0x200000;
    private static final ByteBuffer EMPTY_BYTE_BUFFER = ByteBuffer.wrap(new byte[0]);
    private static final String EMPTY_STACK_TRACE = "";
    private final StreamSegmentStore segmentStore;
    private final TableStore tableStore;
    private final ServerConnection connection;
    private final SegmentStatsRecorder statsRecorder;
    private final TableSegmentStatsRecorder tableStatsRecorder;
    private final DelegationTokenVerifier tokenVerifier;
    private final boolean replyWithStackTraceOnError;

    @VisibleForTesting
    public PravegaRequestProcessor(StreamSegmentStore segmentStore, TableStore tableStore, ServerConnection connection) {
        this(segmentStore, tableStore, connection, SegmentStatsRecorder.noOp(), TableSegmentStatsRecorder.noOp(), new PassingTokenVerifier(), false);
    }

    PravegaRequestProcessor(StreamSegmentStore segmentStore, TableStore tableStore, ServerConnection connection, SegmentStatsRecorder statsRecorder, TableSegmentStatsRecorder tableStatsRecorder, DelegationTokenVerifier tokenVerifier, boolean replyWithStackTraceOnError) {
        this.segmentStore = (StreamSegmentStore)Preconditions.checkNotNull((Object)segmentStore, (Object)"segmentStore");
        this.tableStore = (TableStore)Preconditions.checkNotNull((Object)tableStore, (Object)"tableStore");
        this.connection = (ServerConnection)Preconditions.checkNotNull((Object)connection, (Object)"connection");
        this.tokenVerifier = (DelegationTokenVerifier)Preconditions.checkNotNull((Object)tokenVerifier, (Object)"tokenVerifier");
        this.statsRecorder = (SegmentStatsRecorder)Preconditions.checkNotNull((Object)statsRecorder, (Object)"statsRecorder");
        this.tableStatsRecorder = (TableSegmentStatsRecorder)Preconditions.checkNotNull((Object)tableStatsRecorder, (Object)"tableStatsRecorder");
        this.replyWithStackTraceOnError = replyWithStackTraceOnError;
    }

    public void readSegment(WireCommands.ReadSegment readSegment) {
        Timer timer = new Timer();
        String segment = readSegment.getSegment();
        String operation = "readSegment";
        if (!this.verifyToken(segment, readSegment.getOffset(), readSegment.getDelegationToken(), "readSegment")) {
            return;
        }
        int readSize = Math.min(0x200000, Math.max(8, readSegment.getSuggestedLength()));
        long trace = LoggerHelpers.traceEnter((Logger)log, (String)"readSegment", (Object[])new Object[]{readSegment});
        ((CompletableFuture)this.segmentStore.read(segment, readSegment.getOffset(), readSize, TIMEOUT).thenAccept(readResult -> {
            LoggerHelpers.traceLeave((Logger)log, (String)"readSegment", (long)trace, (Object[])new Object[]{readResult});
            this.handleReadResult(readSegment, (ReadResult)readResult);
            this.statsRecorder.readComplete(timer.getElapsed());
        })).exceptionally(ex -> this.handleException(readSegment.getRequestId(), segment, readSegment.getOffset(), "readSegment", this.wrapCancellationException((Throwable)ex)));
    }

    private boolean verifyToken(String segment, long requestId, String delegationToken, String operation) {
        boolean isTokenValid = false;
        try {
            this.tokenVerifier.verifyToken(segment, delegationToken, AuthHandler.Permissions.READ);
            isTokenValid = true;
        }
        catch (TokenException e) {
            this.handleException(requestId, segment, operation, e);
        }
        return isTokenValid;
    }

    private void handleReadResult(WireCommands.ReadSegment request, ReadResult result) {
        boolean atTail;
        String segment = request.getSegment();
        ArrayList<ReadResultEntryContents> cachedEntries = new ArrayList<ReadResultEntryContents>();
        ReadResultEntry nonCachedEntry = this.collectCachedEntries(request.getOffset(), result, cachedEntries);
        String operation = "readSegment";
        boolean truncated = nonCachedEntry != null && nonCachedEntry.getType() == ReadResultEntryType.Truncated;
        boolean endOfSegment = nonCachedEntry != null && nonCachedEntry.getType() == ReadResultEntryType.EndOfStreamSegment;
        boolean bl = atTail = nonCachedEntry != null && nonCachedEntry.getType() == ReadResultEntryType.Future;
        if (!cachedEntries.isEmpty() || endOfSegment) {
            ByteBuffer data = this.copyData(cachedEntries);
            WireCommands.SegmentRead reply = new WireCommands.SegmentRead(segment, request.getOffset(), atTail, endOfSegment, data, request.getRequestId());
            this.connection.send((WireCommand)reply);
            this.statsRecorder.read(segment, reply.getData().array().length);
        } else if (truncated) {
            ((CompletableFuture)this.segmentStore.getStreamSegmentInfo(segment, TIMEOUT).thenAccept(info -> this.connection.send((WireCommand)new WireCommands.SegmentIsTruncated(request.getRequestId(), segment, info.getStartOffset(), EMPTY_STACK_TRACE, nonCachedEntry.getStreamSegmentOffset())))).exceptionally(e -> this.handleException(request.getRequestId(), segment, nonCachedEntry.getStreamSegmentOffset(), "readSegment", this.wrapCancellationException((Throwable)e)));
        } else {
            Preconditions.checkState((nonCachedEntry != null ? 1 : 0) != 0, (Object)"No ReadResultEntries returned from read!?");
            nonCachedEntry.requestContent(TIMEOUT);
            ((CompletableFuture)((CompletableFuture)nonCachedEntry.getContent().thenAccept(contents -> {
                ByteBuffer data = this.copyData(Collections.singletonList(contents));
                WireCommands.SegmentRead reply = new WireCommands.SegmentRead(segment, nonCachedEntry.getStreamSegmentOffset(), false, endOfSegment, data, request.getRequestId());
                this.connection.send((WireCommand)reply);
                this.statsRecorder.read(segment, reply.getData().array().length);
            })).exceptionally(e -> {
                if (Exceptions.unwrap((Throwable)e) instanceof StreamSegmentTruncatedException) {
                    String clientReplyStackTrace = this.replyWithStackTraceOnError ? e.getMessage() : EMPTY_STACK_TRACE;
                    this.connection.send((WireCommand)new WireCommands.SegmentIsTruncated(request.getRequestId(), segment, nonCachedEntry.getStreamSegmentOffset(), clientReplyStackTrace, nonCachedEntry.getStreamSegmentOffset()));
                } else {
                    this.handleException(request.getRequestId(), segment, nonCachedEntry.getStreamSegmentOffset(), "readSegment", this.wrapCancellationException((Throwable)e));
                }
                return null;
            })).exceptionally(e -> this.handleException(request.getRequestId(), segment, nonCachedEntry.getStreamSegmentOffset(), "readSegment", this.wrapCancellationException((Throwable)e)));
        }
    }

    private Throwable wrapCancellationException(Throwable u) {
        Throwable wrappedException = null;
        if (u != null && (wrappedException = Exceptions.unwrap((Throwable)u)) instanceof CancellationException) {
            wrappedException = new ReadCancellationException(wrappedException);
        }
        return wrappedException;
    }

    private ReadResultEntry collectCachedEntries(long initialOffset, ReadResult readResult, ArrayList<ReadResultEntryContents> cachedEntries) {
        long expectedOffset = initialOffset;
        while (readResult.hasNext()) {
            ReadResultEntry entry = (ReadResultEntry)readResult.next();
            if (entry.getType() == ReadResultEntryType.Cache) {
                Preconditions.checkState((entry.getStreamSegmentOffset() == expectedOffset ? 1 : 0) != 0, (Object)"Data returned from read was not contiguous.");
                ReadResultEntryContents content = entry.getContent().getNow(null);
                expectedOffset += (long)content.getLength();
                cachedEntries.add(content);
                continue;
            }
            return entry;
        }
        return null;
    }

    private ByteBuffer copyData(List<ReadResultEntryContents> contents) {
        int totalSize = contents.stream().mapToInt(ReadResultEntryContents::getLength).sum();
        ByteBuffer data = ByteBuffer.allocate(totalSize);
        int bytesCopied = 0;
        for (ReadResultEntryContents content : contents) {
            int copied = StreamHelpers.readAll((InputStream)content.getData(), (byte[])data.array(), (int)bytesCopied, (int)(totalSize - bytesCopied));
            Preconditions.checkState((copied == content.getLength() ? 1 : 0) != 0, (Object)"Read fewer bytes than available.");
            bytesCopied += copied;
        }
        return data;
    }

    public void updateSegmentAttribute(WireCommands.UpdateSegmentAttribute updateSegmentAttribute) {
        long requestId = updateSegmentAttribute.getRequestId();
        String segmentName = updateSegmentAttribute.getSegmentName();
        UUID attributeId = updateSegmentAttribute.getAttributeId();
        long newValue = updateSegmentAttribute.getNewValue();
        long expectedValue = updateSegmentAttribute.getExpectedValue();
        String operation = "updateSegmentAttribute";
        if (!this.verifyToken(segmentName, updateSegmentAttribute.getRequestId(), updateSegmentAttribute.getDelegationToken(), "updateSegmentAttribute")) {
            return;
        }
        long trace = LoggerHelpers.traceEnter((Logger)log, (String)"updateSegmentAttribute", (Object[])new Object[]{updateSegmentAttribute});
        AttributeUpdate update = new AttributeUpdate(attributeId, AttributeUpdateType.ReplaceIfEquals, newValue, expectedValue);
        this.segmentStore.updateAttributes(segmentName, Collections.singletonList(update), TIMEOUT).whenComplete((v, e) -> {
            LoggerHelpers.traceLeave((Logger)log, (String)"updateSegmentAttribute", (long)trace, (Object[])new Object[]{e});
            Consumer<Throwable> failureHandler = t -> {
                log.error(requestId, "Error (Segment = '{}', Operation = '{}')", new Object[]{segmentName, "handling result of updateSegmentAttribute", t});
                this.connection.close();
            };
            if (e == null) {
                Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.SegmentAttributeUpdated(requestId, true), failureHandler);
            } else if (Exceptions.unwrap((Throwable)e) instanceof BadAttributeUpdateException) {
                log.debug("Updating segment attribute {} failed due to: {}", (Object)update, (Object)e.getMessage());
                Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.SegmentAttributeUpdated(requestId, false), failureHandler);
            } else {
                this.handleException(requestId, segmentName, "updateSegmentAttribute", (Throwable)e);
            }
        });
    }

    public void getSegmentAttribute(WireCommands.GetSegmentAttribute getSegmentAttribute) {
        long requestId = getSegmentAttribute.getRequestId();
        String segmentName = getSegmentAttribute.getSegmentName();
        UUID attributeId = getSegmentAttribute.getAttributeId();
        String operation = "getSegmentAttribute";
        if (!this.verifyToken(segmentName, getSegmentAttribute.getRequestId(), getSegmentAttribute.getDelegationToken(), "getSegmentAttribute")) {
            return;
        }
        long trace = LoggerHelpers.traceEnter((Logger)log, (String)"getSegmentAttribute", (Object[])new Object[]{getSegmentAttribute});
        ((CompletableFuture)this.segmentStore.getStreamSegmentInfo(segmentName, TIMEOUT).thenAccept(properties -> {
            LoggerHelpers.traceLeave((Logger)log, (String)"getSegmentAttribute", (long)trace, (Object[])new Object[]{properties});
            if (properties == null) {
                this.connection.send((WireCommand)new WireCommands.NoSuchSegment(requestId, segmentName, EMPTY_STACK_TRACE, -1L));
            } else {
                Map attributes = properties.getAttributes();
                Long value = (Long)attributes.get(attributeId);
                if (value == null) {
                    value = Long.MIN_VALUE;
                }
                this.connection.send((WireCommand)new WireCommands.SegmentAttribute(requestId, value.longValue()));
            }
        })).exceptionally(e -> this.handleException(requestId, segmentName, "getSegmentAttribute", (Throwable)e));
    }

    public void getStreamSegmentInfo(WireCommands.GetStreamSegmentInfo getStreamSegmentInfo) {
        String segmentName = getStreamSegmentInfo.getSegmentName();
        String operation = "getStreamSegmentInfo";
        if (!this.verifyToken(segmentName, getStreamSegmentInfo.getRequestId(), getStreamSegmentInfo.getDelegationToken(), "getStreamSegmentInfo")) {
            return;
        }
        ((CompletableFuture)this.segmentStore.getStreamSegmentInfo(segmentName, TIMEOUT).thenAccept(properties -> {
            if (properties != null) {
                WireCommands.StreamSegmentInfo result = new WireCommands.StreamSegmentInfo(getStreamSegmentInfo.getRequestId(), properties.getName(), true, properties.isSealed(), properties.isDeleted(), properties.getLastModified().getTime(), properties.getLength(), properties.getStartOffset());
                log.trace("Read stream segment info: {}", (Object)result);
                this.connection.send((WireCommand)result);
            } else {
                log.trace("getStreamSegmentInfo could not find segment {}", (Object)segmentName);
                this.connection.send((WireCommand)new WireCommands.StreamSegmentInfo(getStreamSegmentInfo.getRequestId(), segmentName, false, true, true, 0L, 0L, 0L));
            }
        })).exceptionally(e -> this.handleException(getStreamSegmentInfo.getRequestId(), segmentName, "getStreamSegmentInfo", (Throwable)e));
    }

    public void createSegment(WireCommands.CreateSegment createStreamSegment) {
        Timer timer = new Timer();
        String operation = "createSegment";
        List<AttributeUpdate> attributes = Arrays.asList(new AttributeUpdate(Attributes.SCALE_POLICY_TYPE, AttributeUpdateType.Replace, Byte.valueOf(createStreamSegment.getScaleType()).longValue()), new AttributeUpdate(Attributes.SCALE_POLICY_RATE, AttributeUpdateType.Replace, Integer.valueOf(createStreamSegment.getTargetRate()).longValue()), new AttributeUpdate(Attributes.CREATION_TIME, AttributeUpdateType.None, System.currentTimeMillis()));
        if (!this.verifyToken(createStreamSegment.getSegment(), createStreamSegment.getRequestId(), createStreamSegment.getDelegationToken(), "createSegment")) {
            return;
        }
        log.info(createStreamSegment.getRequestId(), "Creating stream segment {}.", new Object[]{createStreamSegment});
        ((CompletableFuture)this.segmentStore.createStreamSegment(createStreamSegment.getSegment(), attributes, TIMEOUT).thenAccept(v -> this.connection.send((WireCommand)new WireCommands.SegmentCreated(createStreamSegment.getRequestId(), createStreamSegment.getSegment())))).whenComplete((res, e) -> {
            if (e == null) {
                this.statsRecorder.createSegment(createStreamSegment.getSegment(), createStreamSegment.getScaleType(), createStreamSegment.getTargetRate(), timer.getElapsed());
            } else {
                this.handleException(createStreamSegment.getRequestId(), createStreamSegment.getSegment(), "createSegment", (Throwable)e);
            }
        });
    }

    public void mergeSegments(WireCommands.MergeSegments mergeSegments) {
        String operation = "mergeSegments";
        if (!this.verifyToken(mergeSegments.getSource(), mergeSegments.getRequestId(), mergeSegments.getDelegationToken(), "mergeSegments")) {
            return;
        }
        log.info(mergeSegments.getRequestId(), "Merging Segments {} ", new Object[]{mergeSegments});
        ((CompletableFuture)this.segmentStore.mergeStreamSegment(mergeSegments.getTarget(), mergeSegments.getSource(), TIMEOUT).thenAccept(mergeResult -> {
            this.recordStatForTransaction((MergeStreamSegmentResult)mergeResult, mergeSegments.getTarget());
            this.connection.send((WireCommand)new WireCommands.SegmentsMerged(mergeSegments.getRequestId(), mergeSegments.getTarget(), mergeSegments.getSource(), mergeResult.getTargetSegmentLength()));
        })).exceptionally(e -> {
            if (Exceptions.unwrap((Throwable)e) instanceof StreamSegmentMergedException) {
                log.info(mergeSegments.getRequestId(), "Stream segment is already merged '{}'.", new Object[]{mergeSegments.getSource()});
                this.segmentStore.getStreamSegmentInfo(mergeSegments.getTarget(), TIMEOUT).thenAccept(properties -> this.connection.send((WireCommand)new WireCommands.SegmentsMerged(mergeSegments.getRequestId(), mergeSegments.getTarget(), mergeSegments.getSource(), properties.getLength())));
                return null;
            }
            return this.handleException(mergeSegments.getRequestId(), mergeSegments.getSource(), "mergeSegments", (Throwable)e);
        });
    }

    public void sealSegment(WireCommands.SealSegment sealSegment) {
        String segment = sealSegment.getSegment();
        String operation = "sealSegment";
        if (!this.verifyToken(segment, sealSegment.getRequestId(), sealSegment.getDelegationToken(), "sealSegment")) {
            return;
        }
        log.info(sealSegment.getRequestId(), "Sealing segment {} ", new Object[]{sealSegment});
        ((CompletableFuture)this.segmentStore.sealStreamSegment(segment, TIMEOUT).thenAccept(size -> this.connection.send((WireCommand)new WireCommands.SegmentSealed(sealSegment.getRequestId(), segment)))).whenComplete((r, e) -> {
            if (e != null) {
                this.handleException(sealSegment.getRequestId(), segment, "sealSegment", (Throwable)e);
            } else {
                this.statsRecorder.sealSegment(sealSegment.getSegment());
            }
        });
    }

    public void truncateSegment(WireCommands.TruncateSegment truncateSegment) {
        String segment = truncateSegment.getSegment();
        String operation = "truncateSegment";
        if (!this.verifyToken(segment, truncateSegment.getRequestId(), truncateSegment.getDelegationToken(), "truncateSegment")) {
            return;
        }
        long offset = truncateSegment.getTruncationOffset();
        log.info(truncateSegment.getRequestId(), "Truncating segment {} at offset {}.", new Object[]{segment, offset});
        ((CompletableFuture)this.segmentStore.truncateStreamSegment(segment, offset, TIMEOUT).thenAccept(v -> this.connection.send((WireCommand)new WireCommands.SegmentTruncated(truncateSegment.getRequestId(), segment)))).exceptionally(e -> this.handleException(truncateSegment.getRequestId(), segment, offset, "truncateSegment", (Throwable)e));
    }

    public void deleteSegment(WireCommands.DeleteSegment deleteSegment) {
        String segment = deleteSegment.getSegment();
        String operation = "deleteSegment";
        if (!this.verifyToken(segment, deleteSegment.getRequestId(), deleteSegment.getDelegationToken(), "deleteSegment")) {
            return;
        }
        log.info(deleteSegment.getRequestId(), "Deleting segment {} ", new Object[]{deleteSegment});
        ((CompletableFuture)this.segmentStore.deleteStreamSegment(segment, TIMEOUT).thenRun(() -> {
            this.connection.send((WireCommand)new WireCommands.SegmentDeleted(deleteSegment.getRequestId(), segment));
            this.statsRecorder.deleteSegment(segment);
        })).exceptionally(e -> this.handleException(deleteSegment.getRequestId(), segment, "deleteSegment", (Throwable)e));
    }

    public void updateSegmentPolicy(WireCommands.UpdateSegmentPolicy updateSegmentPolicy) {
        String operation = "updateSegmentPolicy";
        if (!this.verifyToken(updateSegmentPolicy.getSegment(), updateSegmentPolicy.getRequestId(), updateSegmentPolicy.getDelegationToken(), "updateSegmentPolicy")) {
            return;
        }
        List<AttributeUpdate> attributes = Arrays.asList(new AttributeUpdate(Attributes.SCALE_POLICY_TYPE, AttributeUpdateType.Replace, (long)updateSegmentPolicy.getScaleType()), new AttributeUpdate(Attributes.SCALE_POLICY_RATE, AttributeUpdateType.Replace, (long)updateSegmentPolicy.getTargetRate()));
        log.info(updateSegmentPolicy.getRequestId(), "Updating segment policy {} ", new Object[]{updateSegmentPolicy});
        ((CompletableFuture)this.segmentStore.updateAttributes(updateSegmentPolicy.getSegment(), attributes, TIMEOUT).thenRun(() -> this.connection.send((WireCommand)new WireCommands.SegmentPolicyUpdated(updateSegmentPolicy.getRequestId(), updateSegmentPolicy.getSegment())))).whenComplete((r, e) -> {
            if (e != null) {
                this.handleException(updateSegmentPolicy.getRequestId(), updateSegmentPolicy.getSegment(), "updateSegmentPolicy", (Throwable)e);
            } else {
                this.statsRecorder.policyUpdate(updateSegmentPolicy.getSegment(), updateSegmentPolicy.getScaleType(), updateSegmentPolicy.getTargetRate());
            }
        });
    }

    public void createTableSegment(WireCommands.CreateTableSegment createTableSegment) {
        String operation = "createTableSegment";
        if (!this.verifyToken(createTableSegment.getSegment(), createTableSegment.getRequestId(), createTableSegment.getDelegationToken(), "createTableSegment")) {
            return;
        }
        log.info(createTableSegment.getRequestId(), "Creating table segment {}.", new Object[]{createTableSegment});
        Timer timer = new Timer();
        ((CompletableFuture)this.tableStore.createSegment(createTableSegment.getSegment(), TIMEOUT).thenAccept(v -> {
            this.connection.send((WireCommand)new WireCommands.SegmentCreated(createTableSegment.getRequestId(), createTableSegment.getSegment()));
            this.tableStatsRecorder.createTableSegment(createTableSegment.getSegment(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(createTableSegment.getRequestId(), createTableSegment.getSegment(), "createTableSegment", (Throwable)e));
    }

    public void deleteTableSegment(WireCommands.DeleteTableSegment deleteTableSegment) {
        String segment = deleteTableSegment.getSegment();
        String operation = "deleteTableSegment";
        if (!this.verifyToken(segment, deleteTableSegment.getRequestId(), deleteTableSegment.getDelegationToken(), "deleteTableSegment")) {
            return;
        }
        log.info(deleteTableSegment.getRequestId(), "Deleting table segment {}.", new Object[]{deleteTableSegment});
        Timer timer = new Timer();
        ((CompletableFuture)this.tableStore.deleteSegment(segment, deleteTableSegment.isMustBeEmpty(), TIMEOUT).thenRun(() -> {
            this.connection.send((WireCommand)new WireCommands.SegmentDeleted(deleteTableSegment.getRequestId(), segment));
            this.tableStatsRecorder.deleteTableSegment(segment, timer.getElapsed());
        })).exceptionally(e -> this.handleException(deleteTableSegment.getRequestId(), segment, "deleteTableSegment", (Throwable)e));
    }

    public void mergeTableSegments(WireCommands.MergeTableSegments mergeTableSegments) {
        String operation = "mergeTableSegments";
        if (!this.verifyToken(mergeTableSegments.getSource(), mergeTableSegments.getRequestId(), mergeTableSegments.getDelegationToken(), "mergeTableSegments")) {
            return;
        }
        log.info(mergeTableSegments.getRequestId(), "Merging table segments {}.", new Object[]{mergeTableSegments});
        ((CompletableFuture)this.tableStore.merge(mergeTableSegments.getTarget(), mergeTableSegments.getSource(), TIMEOUT).thenRun(() -> this.connection.send((WireCommand)new WireCommands.SegmentsMerged(mergeTableSegments.getRequestId(), mergeTableSegments.getTarget(), mergeTableSegments.getSource(), -1L)))).exceptionally(e -> this.handleException(mergeTableSegments.getRequestId(), mergeTableSegments.getSource(), "mergeTableSegments", (Throwable)e));
    }

    public void sealTableSegment(WireCommands.SealTableSegment sealTableSegment) {
        String segment = sealTableSegment.getSegment();
        String operation = "sealTableSegment";
        if (!this.verifyToken(segment, sealTableSegment.getRequestId(), sealTableSegment.getDelegationToken(), "sealTableSegment")) {
            return;
        }
        log.info(sealTableSegment.getRequestId(), "Sealing table segment {}.", new Object[]{sealTableSegment});
        ((CompletableFuture)this.tableStore.seal(segment, TIMEOUT).thenRun(() -> this.connection.send((WireCommand)new WireCommands.SegmentSealed(sealTableSegment.getRequestId(), segment)))).exceptionally(e -> this.handleException(sealTableSegment.getRequestId(), segment, "sealTableSegment", (Throwable)e));
    }

    public void updateTableEntries(WireCommands.UpdateTableEntries updateTableEntries) {
        String segment = updateTableEntries.getSegment();
        String operation = "updateTableEntries";
        if (!this.verifyToken(segment, updateTableEntries.getRequestId(), updateTableEntries.getDelegationToken(), "updateTableEntries")) {
            return;
        }
        log.info(updateTableEntries.getRequestId(), "Updating table segment {}.", new Object[]{updateTableEntries});
        ArrayList<TableEntry> entries = new ArrayList<TableEntry>(updateTableEntries.getTableEntries().getEntries().size());
        AtomicBoolean conditional = new AtomicBoolean(false);
        for (Map.Entry e2 : updateTableEntries.getTableEntries().getEntries()) {
            TableEntry v = TableEntry.versioned((ArrayView)this.getArrayView(((WireCommands.TableKey)e2.getKey()).getData()), (ArrayView)this.getArrayView(((WireCommands.TableValue)e2.getValue()).getData()), (long)((WireCommands.TableKey)e2.getKey()).getKeyVersion());
            entries.add(v);
            if (!v.getKey().hasVersion()) continue;
            conditional.set(true);
        }
        Timer timer = new Timer();
        ((CompletableFuture)this.tableStore.put(segment, entries, TIMEOUT).thenAccept(versions -> {
            this.connection.send((WireCommand)new WireCommands.TableEntriesUpdated(updateTableEntries.getRequestId(), versions));
            this.tableStatsRecorder.updateEntries(updateTableEntries.getSegment(), entries.size(), conditional.get(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(updateTableEntries.getRequestId(), segment, "updateTableEntries", (Throwable)e));
    }

    public void removeTableKeys(WireCommands.RemoveTableKeys removeTableKeys) {
        String segment = removeTableKeys.getSegment();
        String operation = "removeTableKeys";
        if (!this.verifyToken(segment, removeTableKeys.getRequestId(), removeTableKeys.getDelegationToken(), "removeTableKeys")) {
            return;
        }
        log.info(removeTableKeys.getRequestId(), "Removing table keys {}.", new Object[]{removeTableKeys});
        ArrayList<TableKey> keys = new ArrayList<TableKey>(removeTableKeys.getKeys().size());
        AtomicBoolean conditional = new AtomicBoolean(false);
        for (WireCommands.TableKey k : removeTableKeys.getKeys()) {
            TableKey v = TableKey.versioned((ArrayView)this.getArrayView(k.getData()), (long)k.getKeyVersion());
            keys.add(v);
            if (!v.hasVersion()) continue;
            conditional.set(true);
        }
        Timer timer = new Timer();
        ((CompletableFuture)this.tableStore.remove(segment, keys, TIMEOUT).thenRun(() -> {
            this.connection.send((WireCommand)new WireCommands.TableKeysRemoved(removeTableKeys.getRequestId(), segment));
            this.tableStatsRecorder.removeKeys(removeTableKeys.getSegment(), keys.size(), conditional.get(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(removeTableKeys.getRequestId(), segment, "removeTableKeys", (Throwable)e));
    }

    public void readTable(WireCommands.ReadTable readTable) {
        String segment = readTable.getSegment();
        String operation = "readTable";
        if (!this.verifyToken(segment, readTable.getRequestId(), readTable.getDelegationToken(), "readTable")) {
            return;
        }
        log.info(readTable.getRequestId(), "Reading from table {}.", new Object[]{readTable});
        List keys = readTable.getKeys().stream().map(k -> this.getArrayView(k.getData())).collect(Collectors.toList());
        Timer timer = new Timer();
        ((CompletableFuture)this.tableStore.get(segment, keys, TIMEOUT).thenAccept(values -> {
            this.connection.send((WireCommand)new WireCommands.TableRead(readTable.getRequestId(), segment, this.getTableEntriesCommand(keys, (List<TableEntry>)values)));
            this.tableStatsRecorder.getKeys(readTable.getSegment(), keys.size(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(readTable.getRequestId(), segment, "readTable", (Throwable)e));
    }

    public void readTableKeys(WireCommands.ReadTableKeys readTableKeys) {
        String segment = readTableKeys.getSegment();
        String operation = "readTableKeys";
        if (!this.verifyToken(segment, readTableKeys.getRequestId(), readTableKeys.getDelegationToken(), "readTableKeys")) {
            return;
        }
        log.info(readTableKeys.getRequestId(), "Fetching keys from {}.", new Object[]{readTableKeys});
        int suggestedKeyCount = readTableKeys.getSuggestedKeyCount();
        ByteBuf token = readTableKeys.getContinuationToken();
        byte[] state = null;
        if (!token.equals((Object)Unpooled.EMPTY_BUFFER)) {
            state = token.array();
        }
        AtomicInteger msgSize = new AtomicInteger(0);
        AtomicReference<ByteBuf> continuationToken = new AtomicReference<ByteBuf>(Unpooled.EMPTY_BUFFER);
        ArrayList keys = new ArrayList();
        Timer timer = new Timer();
        ((CompletableFuture)((CompletableFuture)this.tableStore.keyIterator(segment, state, TIMEOUT).thenCompose(itr -> itr.collectRemaining(e -> {
            List list = keys;
            synchronized (list) {
                if (keys.size() < suggestedKeyCount && msgSize.get() < 0x200000) {
                    Collection tableKeys = e.getEntries();
                    ArrayView lastState = e.getState();
                    keys.addAll(tableKeys);
                    continuationToken.set(Unpooled.wrappedBuffer((byte[])lastState.array(), (int)lastState.arrayOffset(), (int)lastState.getLength()));
                    msgSize.addAndGet(this.getTableKeyBytes(segment, tableKeys, lastState.getLength()));
                    return true;
                }
                return false;
            }
        }))).thenAccept(v -> {
            List wireCommandKeys;
            List list = keys;
            synchronized (list) {
                log.debug(readTableKeys.getRequestId(), "{} keys obtained for ReadTableKeys request.", new Object[]{keys.size()});
                wireCommandKeys = keys.stream().map(k -> {
                    ArrayView keyArray = k.getKey();
                    return new WireCommands.TableKey(Unpooled.wrappedBuffer((byte[])keyArray.array(), (int)keyArray.arrayOffset(), (int)keyArray.getLength()), k.getVersion());
                }).collect(Collectors.toList());
            }
            this.connection.send((WireCommand)new WireCommands.TableKeysRead(readTableKeys.getRequestId(), segment, wireCommandKeys, (ByteBuf)continuationToken.get()));
            this.tableStatsRecorder.iterateKeys(readTableKeys.getSegment(), keys.size(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(readTableKeys.getRequestId(), segment, "readTableKeys", (Throwable)e));
    }

    public void readTableEntries(WireCommands.ReadTableEntries readTableEntries) {
        String segment = readTableEntries.getSegment();
        String operation = "readTableEntries";
        if (!this.verifyToken(segment, readTableEntries.getRequestId(), readTableEntries.getDelegationToken(), "readTableEntries")) {
            return;
        }
        log.info(readTableEntries.getRequestId(), "Fetching keys from {}.", new Object[]{readTableEntries});
        int suggestedEntryCount = readTableEntries.getSuggestedEntryCount();
        ByteBuf token = readTableEntries.getContinuationToken();
        byte[] state = null;
        if (!token.equals((Object)Unpooled.EMPTY_BUFFER)) {
            state = token.array();
        }
        AtomicInteger msgSize = new AtomicInteger(0);
        AtomicReference<ByteBuf> continuationToken = new AtomicReference<ByteBuf>(Unpooled.EMPTY_BUFFER);
        ArrayList entries = new ArrayList();
        Timer timer = new Timer();
        ((CompletableFuture)((CompletableFuture)this.tableStore.entryIterator(segment, state, TIMEOUT).thenCompose(itr -> itr.collectRemaining(e -> {
            List list = entries;
            synchronized (list) {
                if (entries.size() < suggestedEntryCount && msgSize.get() < 0x200000) {
                    Collection tableEntries = e.getEntries();
                    ArrayView lastState = e.getState();
                    entries.addAll(tableEntries);
                    continuationToken.set(Unpooled.wrappedBuffer((byte[])lastState.array(), (int)lastState.arrayOffset(), (int)lastState.getLength()));
                    msgSize.addAndGet(this.getTableEntryBytes(segment, tableEntries, lastState.getLength()));
                    return true;
                }
                return false;
            }
        }))).thenAccept(v -> {
            List wireCommandEntries;
            List list = entries;
            synchronized (list) {
                log.debug(readTableEntries.getRequestId(), "{} entries obtained for ReadTableEntries request.", new Object[]{entries.size()});
                wireCommandEntries = entries.stream().map(e -> {
                    TableKey k = e.getKey();
                    WireCommands.TableKey keyWireCommand = new WireCommands.TableKey(Unpooled.wrappedBuffer((byte[])k.getKey().array(), (int)k.getKey().arrayOffset(), (int)k.getKey().getLength()), k.getVersion());
                    ArrayView value = e.getValue();
                    WireCommands.TableValue valueWireCommand = new WireCommands.TableValue(Unpooled.wrappedBuffer((byte[])value.array(), (int)value.arrayOffset(), (int)value.getLength()));
                    return new AbstractMap.SimpleImmutableEntry<WireCommands.TableKey, WireCommands.TableValue>(keyWireCommand, valueWireCommand);
                }).collect(Collectors.toList());
            }
            this.connection.send((WireCommand)new WireCommands.TableEntriesRead(readTableEntries.getRequestId(), segment, new WireCommands.TableEntries(wireCommandEntries), (ByteBuf)continuationToken.get()));
            this.tableStatsRecorder.iterateEntries(readTableEntries.getSegment(), entries.size(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(readTableEntries.getRequestId(), segment, "readTableEntries", (Throwable)e));
    }

    private int getTableKeyBytes(String segment, Collection<TableKey> keys, int continuationTokenLength) {
        int headerLength = (Integer)WireCommands.TableKeysRead.GET_HEADER_BYTES.apply(keys.size());
        int segmentLength = segment.getBytes().length;
        int dataLength = keys.stream().mapToInt(value -> value.getKey().getLength() + 8).sum();
        return continuationTokenLength + headerLength + segmentLength + dataLength;
    }

    private int getTableEntryBytes(String segment, Collection<TableEntry> items, int continuationTokenLength) {
        int headerLength = (Integer)WireCommands.TableEntriesRead.GET_HEADER_BYTES.apply(items.size());
        int segmentLength = segment.getBytes().length;
        int dataLength = items.stream().mapToInt(value -> value.getKey().getKey().getLength() + 8 + value.getValue().getLength()).sum();
        return headerLength + segmentLength + dataLength + continuationTokenLength;
    }

    private ArrayView getArrayView(ByteBuf buf) {
        int length = buf.readableBytes();
        if (buf.hasArray()) {
            return new ByteArraySegment(buf.array(), buf.readerIndex(), length);
        }
        byte[] bytes = new byte[length];
        buf.getBytes(buf.readerIndex(), bytes);
        return new ByteArraySegment(bytes, 0, length);
    }

    private WireCommands.TableEntries getTableEntriesCommand(List<ArrayView> inputKeys, List<TableEntry> resultEntries) {
        Preconditions.checkArgument((resultEntries.size() == inputKeys.size() ? 1 : 0) != 0, (Object)"Number of input keys should match result entry count.");
        List entries = IntStream.range(0, resultEntries.size()).mapToObj(i -> {
            TableEntry resultTableEntry = (TableEntry)resultEntries.get(i);
            if (resultTableEntry == null) {
                ArrayView k = (ArrayView)inputKeys.get(i);
                WireCommands.TableKey keyWireCommand = new WireCommands.TableKey(Unpooled.wrappedBuffer((byte[])k.array(), (int)k.arrayOffset(), (int)k.getLength()), -1L);
                return new AbstractMap.SimpleImmutableEntry<WireCommands.TableKey, WireCommands.TableValue>(keyWireCommand, WireCommands.TableValue.EMPTY);
            }
            TableEntry te = (TableEntry)resultEntries.get(i);
            TableKey k = te.getKey();
            WireCommands.TableKey keyWireCommand = new WireCommands.TableKey(Unpooled.wrappedBuffer((byte[])k.getKey().array(), (int)k.getKey().arrayOffset(), (int)k.getKey().getLength()), k.getVersion());
            ArrayView v = te.getValue();
            WireCommands.TableValue valueWireCommand = new WireCommands.TableValue(Unpooled.wrappedBuffer((byte[])v.array(), (int)v.arrayOffset(), (int)v.getLength()));
            return new AbstractMap.SimpleImmutableEntry<WireCommands.TableKey, WireCommands.TableValue>(keyWireCommand, valueWireCommand);
        }).collect(Collectors.toList());
        return new WireCommands.TableEntries(entries);
    }

    private Void handleException(long requestId, String segment, String operation, Throwable u) {
        return this.handleException(requestId, segment, -1L, operation, u);
    }

    private Void handleException(long requestId, String segment, long offset, String operation, Throwable u) {
        if (u == null) {
            IllegalStateException exception = new IllegalStateException("No exception to handle.");
            log.error(requestId, "Error (Segment = '{}', Operation = '{}')", new Object[]{segment, operation, exception});
            throw exception;
        }
        u = Exceptions.unwrap((Throwable)u);
        String clientReplyStackTrace = this.replyWithStackTraceOnError ? Throwables.getStackTraceAsString((Throwable)u) : EMPTY_STACK_TRACE;
        Consumer<Throwable> failureHandler = t -> {
            log.error(requestId, "Error (Segment = '{}', Operation = '{}')", new Object[]{segment, "handling result of " + operation, t});
            this.connection.close();
        };
        if (u instanceof StreamSegmentExistsException) {
            log.info(requestId, "Segment '{}' already exists and cannot perform operation '{}'.", new Object[]{segment, operation});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.SegmentAlreadyExists(requestId, segment, clientReplyStackTrace), failureHandler);
        } else if (u instanceof StreamSegmentNotExistsException) {
            log.warn(requestId, "Segment '{}' does not exist and cannot perform operation '{}'.", new Object[]{segment, operation});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.NoSuchSegment(requestId, segment, clientReplyStackTrace, offset), failureHandler);
        } else if (u instanceof StreamSegmentSealedException) {
            log.info(requestId, "Segment '{}' is sealed and cannot perform operation '{}'.", new Object[]{segment, operation});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.SegmentIsSealed(requestId, segment, clientReplyStackTrace, offset), failureHandler);
        } else if (u instanceof ContainerNotFoundException) {
            int containerId = ((ContainerNotFoundException)u).getContainerId();
            log.warn(requestId, "Wrong host. Segment = '{}' (Container {}) is not owned. Operation = '{}').", new Object[]{segment, containerId, operation});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.WrongHost(requestId, segment, EMPTY_STACK_TRACE, clientReplyStackTrace), failureHandler);
        } else if (u instanceof ReadCancellationException) {
            log.info(requestId, "Closing connection {} while reading segment {} due to CancellationException.", new Object[]{this.connection, segment});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.SegmentRead(segment, offset, true, false, EMPTY_BYTE_BUFFER, requestId), failureHandler);
        } else if (u instanceof CancellationException) {
            log.info(requestId, "Closing connection {} while performing {} due to {}.", new Object[]{this.connection, operation, u.toString()});
            this.connection.close();
        } else if (u instanceof TokenExpiredException) {
            log.warn(requestId, "Expired token during operation {}", new Object[]{operation});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.AuthTokenCheckFailed(requestId, clientReplyStackTrace, WireCommands.AuthTokenCheckFailed.ErrorCode.TOKEN_EXPIRED), failureHandler);
        } else if (u instanceof TokenException) {
            log.warn(requestId, "Token exception encountered during operation {}.", new Object[]{operation, u});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.AuthTokenCheckFailed(requestId, clientReplyStackTrace, WireCommands.AuthTokenCheckFailed.ErrorCode.TOKEN_CHECK_FAILED), failureHandler);
        } else if (u instanceof UnsupportedOperationException) {
            log.warn(requestId, "Unsupported Operation '{}'.", new Object[]{operation, u});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.OperationUnsupported(requestId, operation, clientReplyStackTrace), failureHandler);
        } else if (u instanceof BadOffsetException) {
            BadOffsetException badOffset = (BadOffsetException)u;
            log.info(requestId, "Segment '{}' is truncated and cannot perform operation '{}' at offset '{}'", new Object[]{segment, operation, offset});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.SegmentIsTruncated(requestId, segment, badOffset.getExpectedOffset(), clientReplyStackTrace, offset), failureHandler);
        } else if (u instanceof TableSegmentNotEmptyException) {
            log.warn(requestId, "Table segment '{}' is not empty to perform '{}'.", new Object[]{segment, operation});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.TableSegmentNotEmpty(requestId, segment, clientReplyStackTrace), failureHandler);
        } else if (u instanceof KeyNotExistsException) {
            log.warn(requestId, "Conditional update on Table segment '{}' failed as the key does not exist.", new Object[]{segment});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.TableKeyDoesNotExist(requestId, segment, clientReplyStackTrace), failureHandler);
        } else if (u instanceof BadKeyVersionException) {
            log.warn(requestId, "Conditional update on Table segment '{}' failed due to bad key version.", new Object[]{segment});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.TableKeyBadVersion(requestId, segment, clientReplyStackTrace), failureHandler);
        } else {
            this.logError(requestId, segment, operation, u);
            this.connection.close();
            throw new IllegalStateException("Unknown exception.", u);
        }
        return null;
    }

    private void logError(long requestId, String segment, String operation, Throwable u) {
        if (u instanceof IllegalContainerStateException) {
            log.warn(requestId, "Error (Segment = '{}', Operation = '{}'): {}", new Object[]{segment, operation, u.toString()});
        } else {
            log.error(requestId, "Error (Segment = '{}', Operation = '{}')", new Object[]{segment, operation, u});
        }
    }

    private void recordStatForTransaction(MergeStreamSegmentResult mergeResult, String targetSegmentName) {
        try {
            if (mergeResult != null && mergeResult.getAttributes().containsKey(Attributes.CREATION_TIME) && mergeResult.getAttributes().containsKey(Attributes.EVENT_COUNT)) {
                long creationTime = (Long)mergeResult.getAttributes().get(Attributes.CREATION_TIME);
                int numOfEvents = ((Long)mergeResult.getAttributes().get(Attributes.EVENT_COUNT)).intValue();
                long len = mergeResult.getMergedDataLength();
                this.statsRecorder.merge(targetSegmentName, len, numOfEvents, creationTime);
            }
        }
        catch (Exception ex) {
            log.error("exception while computing stats while merging txn into {}", (Object)targetSegmentName, (Object)ex);
        }
    }

    private static class ReadCancellationException
    extends RuntimeException {
        ReadCancellationException(Throwable wrappedException) {
            super("CancellationException during operation Read segment", wrappedException);
        }
    }
}

