/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.common.util;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.util.RetriesExhaustedException;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class Retry {
    @SuppressFBWarnings(justification="generated code")
    private static final Logger log = LoggerFactory.getLogger(Retry.class);
    private static final long DEFAULT_RETRY_INIT_DELAY = 100L;
    private static final int DEFAULT_RETRY_MULTIPLIER = 2;
    private static final long DEFAULT_RETRY_MAX_DELAY = Duration.ofSeconds(5L).toMillis();

    private Retry() {
    }

    public static RetryWithBackoff withExpBackoff(long initialMillis, int multiplier, int attempts) {
        return Retry.withExpBackoff(initialMillis, multiplier, attempts, Long.MAX_VALUE);
    }

    public static RetryWithBackoff withExpBackoff(long initialMillis, int multiplier, int attempts, long maxDelay) {
        Preconditions.checkArgument((initialMillis >= 1L ? 1 : 0) != 0, (Object)"InitialMillis must be a positive integer.");
        Preconditions.checkArgument((multiplier >= 1 ? 1 : 0) != 0, (Object)"multiplier must be a positive integer.");
        Preconditions.checkArgument((attempts >= 1 ? 1 : 0) != 0, (Object)"attempts must be a positive integer.");
        Preconditions.checkArgument((maxDelay >= 1L ? 1 : 0) != 0, (Object)"maxDelay must be a positive integer.");
        return new RetryWithBackoff(initialMillis, multiplier, attempts, maxDelay);
    }

    public static RetryUnconditionally indefinitelyWithExpBackoff(long initialMillis, int multiplier, long maxDelay, Consumer<Throwable> consumer) {
        Preconditions.checkArgument((initialMillis >= 1L ? 1 : 0) != 0, (Object)"InitialMillis must be a positive integer.");
        Preconditions.checkArgument((multiplier >= 1 ? 1 : 0) != 0, (Object)"multiplier must be a positive integer.");
        Preconditions.checkArgument((maxDelay >= 1L ? 1 : 0) != 0, (Object)"maxDelay must be a positive integer.");
        RetryWithBackoff params = new RetryWithBackoff(initialMillis, multiplier, Integer.MAX_VALUE, maxDelay);
        return new RetryUnconditionally(consumer, params);
    }

    public static RetryUnconditionally indefinitelyWithExpBackoff(String failureMessage) {
        Exceptions.checkNotNullOrEmpty(failureMessage, "failureMessage");
        RetryWithBackoff params = new RetryWithBackoff(100L, 2, Integer.MAX_VALUE, DEFAULT_RETRY_MAX_DELAY);
        Consumer<Throwable> consumer = e -> {
            if (log.isDebugEnabled()) {
                log.debug(failureMessage);
            } else {
                log.warn(failureMessage);
            }
        };
        return new RetryUnconditionally(consumer, params);
    }

    public static final class RetryUnconditionally
    extends RetryAndThrowBase<RuntimeException> {
        private final Consumer<Throwable> consumer;

        RetryUnconditionally(Consumer<Throwable> consumer, RetryWithBackoff params) {
            super(params);
            this.consumer = consumer;
        }

        @Override
        boolean canRetry(Throwable e) {
            this.consumer.accept(e);
            return true;
        }
    }

    public static final class RetryAndThrowConditionally
    extends RetryAndThrowBase<RuntimeException> {
        private final Predicate<Throwable> predicate;

        private RetryAndThrowConditionally(Predicate<Throwable> predicate, RetryWithBackoff params) {
            super(params);
            this.predicate = predicate;
        }

        @Override
        boolean canRetry(Throwable e) {
            return this.predicate.test(e);
        }
    }

    public static final class RetryAndThrowExceptionally<RetryT extends Exception, ThrowsT extends Exception>
    extends RetryAndThrowBase<ThrowsT> {
        private final Class<RetryT> retryType;
        private final Class<ThrowsT> throwType;

        private RetryAndThrowExceptionally(Class<RetryT> retryType, Class<ThrowsT> throwType, RetryWithBackoff params) {
            super(params);
            this.retryType = retryType;
            this.throwType = throwType;
        }

        @Override
        boolean canRetry(Throwable e) {
            Class<Throwable> type = this.getErrorType(e);
            if (this.throwType.isAssignableFrom(type) && this.retryType.isAssignableFrom(this.throwType)) {
                return false;
            }
            return this.retryType.isAssignableFrom(type);
        }

        private Class<? extends Throwable> getErrorType(Throwable e) {
            if (Exceptions.shouldUnwrap(this.retryType) || Exceptions.shouldUnwrap(this.throwType)) {
                return e.getClass();
            }
            return Exceptions.unwrap(e).getClass();
        }
    }

    public static abstract class RetryAndThrowBase<ThrowsT extends Exception> {
        final RetryWithBackoff params;

        private RetryAndThrowBase(RetryWithBackoff params) {
            this.params = params;
        }

        public <RetryT extends Exception, ReturnT> ReturnT run(Retryable<ReturnT, RetryT, ThrowsT> r) throws ThrowsT {
            Preconditions.checkNotNull(r);
            long delay = this.params.initialMillis;
            Exception last = null;
            for (int attemptNumber = 1; attemptNumber <= this.params.attempts; ++attemptNumber) {
                try {
                    return r.attempt();
                }
                catch (Exception e) {
                    if (!this.canRetry(e)) {
                        if (e instanceof RuntimeException) {
                            throw (RuntimeException)e;
                        }
                        throw e;
                    }
                    last = e;
                    if (attemptNumber >= this.params.attempts) continue;
                    long sleepFor = delay;
                    Exceptions.handleInterrupted(() -> Thread.sleep(sleepFor));
                    delay = Math.min(this.params.maxDelay, (long)this.params.multiplier * delay);
                    log.debug("Retrying command. Retry #{}, timestamp={}", (Object)attemptNumber, (Object)Instant.now());
                    continue;
                }
            }
            throw new RetriesExhaustedException(last);
        }

        public CompletableFuture<Void> runInExecutor(Runnable task, ScheduledExecutorService executorService) {
            Preconditions.checkNotNull((Object)task);
            AtomicBoolean isDone = new AtomicBoolean();
            AtomicInteger attemptNumber = new AtomicInteger(1);
            AtomicLong delay = new AtomicLong(0L);
            return Futures.loop(() -> !isDone.get(), () -> ((CompletableFuture)((CompletableFuture)Futures.delayedFuture(Duration.ofMillis(delay.get()), executorService).thenRunAsync(task, executorService)).thenRun(() -> isDone.set(true))).exceptionally(ex -> {
                if (!this.canRetry((Throwable)ex)) {
                    isDone.set(true);
                } else {
                    if (attemptNumber.get() + 1 > this.params.attempts) {
                        isDone.set(true);
                        throw new RetriesExhaustedException((Throwable)ex);
                    }
                    delay.set(attemptNumber.get() == 1 ? this.params.initialMillis : Math.min(this.params.maxDelay, (long)this.params.multiplier * delay.get()));
                    attemptNumber.incrementAndGet();
                    log.debug("Retrying command. Retry #{}, timestamp={}", (Object)attemptNumber, (Object)Instant.now());
                }
                return null;
            }), (Executor)executorService);
        }

        public <ReturnT> CompletableFuture<ReturnT> runAsync(Supplier<CompletableFuture<ReturnT>> r, ScheduledExecutorService executorService) {
            Preconditions.checkNotNull(r);
            CompletableFuture result = new CompletableFuture();
            AtomicInteger attemptNumber = new AtomicInteger(1);
            AtomicLong delay = new AtomicLong(0L);
            Futures.loop(() -> !result.isDone(), () -> ((CompletableFuture)Futures.delayedFuture(r, delay.get(), executorService).thenAccept(result::complete)).exceptionally(ex -> {
                if (!this.canRetry((Throwable)ex)) {
                    result.completeExceptionally((Throwable)ex);
                } else if (attemptNumber.get() + 1 > this.params.attempts) {
                    result.completeExceptionally(new RetriesExhaustedException((Throwable)ex));
                } else {
                    delay.set(attemptNumber.get() == 1 ? this.params.initialMillis : Math.min(this.params.maxDelay, (long)this.params.multiplier * delay.get()));
                    attemptNumber.incrementAndGet();
                    log.debug("Retrying command. Retry #{}, timestamp={}", (Object)attemptNumber, (Object)Instant.now());
                }
                return null;
            }), (Executor)executorService);
            return result;
        }

        abstract boolean canRetry(Throwable var1);
    }

    @FunctionalInterface
    public static interface Retryable<ReturnT, RetryableET extends Exception, NonRetryableET extends Exception> {
        public ReturnT attempt() throws RetryableET, NonRetryableET;
    }

    public static final class RetryExceptionally<RetryT extends Exception> {
        private final Class<RetryT> retryType;
        private final RetryWithBackoff params;

        private RetryExceptionally(Class<RetryT> retryType, RetryWithBackoff params) {
            this.retryType = retryType;
            this.params = params;
        }

        public <ThrowsT extends Exception> RetryAndThrowExceptionally<RetryT, ThrowsT> throwingOn(Class<ThrowsT> throwType) {
            Preconditions.checkNotNull(throwType);
            return new RetryAndThrowExceptionally(this.retryType, throwType, this.params);
        }
    }

    public static final class RetryWithBackoff {
        private final long initialMillis;
        private final int multiplier;
        private final int attempts;
        private final long maxDelay;

        private RetryWithBackoff(long initialMillis, int multiplier, int attempts, long maxDelay) {
            this.initialMillis = initialMillis;
            this.multiplier = multiplier;
            this.attempts = attempts;
            this.maxDelay = maxDelay;
        }

        public <RetryT extends Exception> RetryExceptionally<RetryT> retryingOn(Class<RetryT> retryType) {
            Preconditions.checkNotNull(retryType);
            return new RetryExceptionally(retryType, this);
        }

        public RetryAndThrowConditionally retryWhen(Predicate<Throwable> predicate) {
            Preconditions.checkNotNull(predicate);
            return new RetryAndThrowConditionally(predicate, this);
        }

        @SuppressFBWarnings(justification="generated code")
        public long getInitialMillis() {
            return this.initialMillis;
        }

        @SuppressFBWarnings(justification="generated code")
        public RetryWithBackoff withInitialMillis(long initialMillis) {
            return this.initialMillis == initialMillis ? this : new RetryWithBackoff(initialMillis, this.multiplier, this.attempts, this.maxDelay);
        }

        @SuppressFBWarnings(justification="generated code")
        public int getMultiplier() {
            return this.multiplier;
        }

        @SuppressFBWarnings(justification="generated code")
        public RetryWithBackoff withMultiplier(int multiplier) {
            return this.multiplier == multiplier ? this : new RetryWithBackoff(this.initialMillis, multiplier, this.attempts, this.maxDelay);
        }

        @SuppressFBWarnings(justification="generated code")
        public int getAttempts() {
            return this.attempts;
        }

        @SuppressFBWarnings(justification="generated code")
        public RetryWithBackoff withAttempts(int attempts) {
            return this.attempts == attempts ? this : new RetryWithBackoff(this.initialMillis, this.multiplier, attempts, this.maxDelay);
        }

        @SuppressFBWarnings(justification="generated code")
        public long getMaxDelay() {
            return this.maxDelay;
        }

        @SuppressFBWarnings(justification="generated code")
        public RetryWithBackoff withMaxDelay(long maxDelay) {
            return this.maxDelay == maxDelay ? this : new RetryWithBackoff(this.initialMillis, this.multiplier, this.attempts, maxDelay);
        }
    }
}

