/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.common.util.btree;

import com.google.common.base.Preconditions;
import io.pravega.common.TimeoutTimer;
import io.pravega.common.util.AsyncIterator;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.common.util.btree.BTreePage;
import io.pravega.common.util.btree.ByteArrayComparator;
import io.pravega.common.util.btree.PageCollection;
import io.pravega.common.util.btree.PageEntry;
import io.pravega.common.util.btree.PageWrapper;
import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import lombok.NonNull;

class EntryIterator
implements AsyncIterator<List<PageEntry>> {
    private static final ByteArrayComparator KEY_COMPARATOR = new ByteArrayComparator();
    private final ByteArraySegment firstKey;
    private final boolean firstKeyInclusive;
    private final ByteArraySegment lastKey;
    private final boolean lastKeyInclusive;
    private final LocatePage locatePage;
    private final Duration fetchTimeout;
    private final AtomicBoolean finished;
    private final PageCollection pageCollection;
    private final AtomicReference<PageWrapper> lastPage;
    private final AtomicInteger processedPageCount;

    EntryIterator(@NonNull ByteArraySegment firstKey, boolean firstKeyInclusive, @NonNull ByteArraySegment lastKey, boolean lastKeyInclusive, @NonNull LocatePage locatePage, long indexLength, @NonNull Duration fetchTimeout) {
        if (firstKey == null) {
            throw new NullPointerException("firstKey is marked @NonNull but is null");
        }
        if (lastKey == null) {
            throw new NullPointerException("lastKey is marked @NonNull but is null");
        }
        if (locatePage == null) {
            throw new NullPointerException("locatePage is marked @NonNull but is null");
        }
        if (fetchTimeout == null) {
            throw new NullPointerException("fetchTimeout is marked @NonNull but is null");
        }
        int c = KEY_COMPARATOR.compare(firstKey, lastKey);
        if (firstKeyInclusive && lastKeyInclusive) {
            Preconditions.checkArgument((c <= 0 ? 1 : 0) != 0, (Object)"firstKey must be smaller than or equal to lastKey.");
        } else {
            Preconditions.checkArgument((c < 0 ? 1 : 0) != 0, (Object)"firstKey must be smaller than lastKey.");
        }
        this.firstKey = firstKey;
        this.firstKeyInclusive = firstKeyInclusive;
        this.lastKey = lastKey;
        this.lastKeyInclusive = lastKeyInclusive;
        this.locatePage = locatePage;
        this.fetchTimeout = fetchTimeout;
        this.pageCollection = new PageCollection(indexLength);
        this.lastPage = new AtomicReference<Object>(null);
        this.finished = new AtomicBoolean();
        this.processedPageCount = new AtomicInteger();
    }

    @Override
    public CompletableFuture<List<PageEntry>> getNext() {
        if (this.finished.get()) {
            return CompletableFuture.completedFuture(null);
        }
        TimeoutTimer timer = new TimeoutTimer(this.fetchTimeout);
        return this.locateNextPage(timer).thenApply(pageWrapper -> {
            this.lastPage.set((PageWrapper)pageWrapper);
            List<PageEntry> result = null;
            if (pageWrapper != null) {
                result = this.extractFromPage((PageWrapper)pageWrapper);
                this.processedPageCount.incrementAndGet();
            }
            if (result == null) {
                this.finished.set(true);
            }
            return result;
        });
    }

    private CompletableFuture<PageWrapper> locateNextPage(TimeoutTimer timer) {
        if (this.lastPage.get() == null) {
            return this.locatePage.apply(this.firstKey, this.pageCollection, timer);
        }
        return this.getNextLeafPage(timer);
    }

    private CompletableFuture<PageWrapper> getNextLeafPage(TimeoutTimer timer) {
        PageWrapper parentPage;
        int pageKeyPos;
        PageWrapper lastPage = this.lastPage.get();
        assert (lastPage != null);
        do {
            if ((parentPage = lastPage.getParent()) == null) {
                return CompletableFuture.completedFuture(null);
            }
            ByteArraySegment pageKey = lastPage.getPointer().getKey();
            BTreePage.SearchResult pos = parentPage.getPage().search(pageKey, 0);
            assert (pos.isExactMatch()) : "expecting exact match";
            pageKeyPos = pos.getPosition() + 1;
            this.pageCollection.remove(lastPage);
        } while (pageKeyPos == (lastPage = parentPage).getPage().getCount());
        ByteArraySegment referenceKey = lastPage.getPage().getKeyAt(pageKeyPos);
        return this.locatePage.apply(referenceKey, this.pageCollection, timer);
    }

    private List<PageEntry> extractFromPage(PageWrapper pageWrapper) {
        int firstIndex;
        BTreePage page = pageWrapper.getPage();
        assert (!page.getConfig().isIndexPage()) : "expecting leaf page";
        if (this.processedPageCount.get() == 0) {
            BTreePage.SearchResult startPos = page.search(this.firstKey, 0);
            firstIndex = startPos.getPosition();
            if (startPos.isExactMatch() && !this.firstKeyInclusive) {
                ++firstIndex;
            }
        } else {
            firstIndex = 0;
        }
        BTreePage.SearchResult endPos = page.search(this.lastKey, 0);
        int lastIndex = endPos.getPosition();
        if (!endPos.isExactMatch() || endPos.isExactMatch() && !this.lastKeyInclusive) {
            --lastIndex;
        }
        if (firstIndex > lastIndex) {
            return Collections.emptyList();
        }
        if (lastIndex < 0) {
            return null;
        }
        return page.getEntries(firstIndex, lastIndex);
    }

    @FunctionalInterface
    static interface LocatePage {
        public CompletableFuture<PageWrapper> apply(ByteArraySegment var1, PageCollection var2, TimeoutTimer var3);
    }
}

