/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.storage.impl.rocksdb;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.Timer;
import io.pravega.common.function.Callbacks;
import io.pravega.common.io.FileHelpers;
import io.pravega.common.util.BufferView;
import io.pravega.segmentstore.storage.Cache;
import io.pravega.segmentstore.storage.CacheException;
import io.pravega.segmentstore.storage.impl.rocksdb.RocksDBConfig;
import io.pravega.segmentstore.storage.impl.rocksdb.RocksDBMetrics;
import java.io.File;
import java.nio.file.Paths;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.rocksdb.BlockBasedTableConfig;
import org.rocksdb.Env;
import org.rocksdb.Options;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksMemEnv;
import org.rocksdb.TableFormatConfig;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RocksDBCache
implements Cache {
    @SuppressFBWarnings(justification="generated code")
    private static final Logger log = LoggerFactory.getLogger(RocksDBCache.class);
    private static final String FILE_PREFIX = "cache_";
    private static final String DB_LOG_DIR = "log";
    private static final int MAX_WRITE_BUFFER_NUMBER = 4;
    private static final int MIN_WRITE_BUFFER_NUMBER_TO_MERGE = 2;
    private static final int INTERNAL_ROCKSDB_PARALLELISM = 8;
    private final String id;
    private final Options databaseOptions;
    private final WriteOptions writeOptions;
    private final AtomicReference<RocksDB> database;
    private final AtomicBoolean closed;
    private final String dbDir;
    private final String logId;
    private final Consumer<String> closeCallback;
    private final int writeBufferSizeMB;
    private final int readCacheSizeMB;
    private final int cacheBlockSizeKB;
    private final boolean directReads;
    private final boolean memoryOnly;

    RocksDBCache(String id, RocksDBConfig config, Consumer<String> closeCallback) {
        Exceptions.checkNotNullOrEmpty((String)id, (String)"id");
        Preconditions.checkNotNull((Object)config, (Object)"config");
        this.id = id;
        this.logId = String.format("RocksDBCache[%s]", id);
        this.dbDir = Paths.get(config.getDatabaseDir(), FILE_PREFIX + this.id).toString();
        this.closeCallback = closeCallback;
        this.closed = new AtomicBoolean();
        this.database = new AtomicReference();
        this.writeBufferSizeMB = config.getWriteBufferSizeMB() / 4;
        this.readCacheSizeMB = config.getReadCacheSizeMB();
        this.cacheBlockSizeKB = config.getCacheBlockSizeKB();
        this.directReads = config.isDirectReads();
        this.memoryOnly = config.isMemoryOnly();
        try {
            this.databaseOptions = this.createDatabaseOptions();
            this.writeOptions = this.createWriteOptions();
        }
        catch (Exception ex) {
            try {
                this.close();
            }
            catch (Exception closeEx) {
                ex.addSuppressed(closeEx);
            }
            throw ex;
        }
    }

    void initialize() {
        Preconditions.checkState((this.database.get() == null ? 1 : 0) != 0, (String)"%s has already been initialized.", (Object)this.logId);
        try {
            this.clear(true);
            this.database.set(this.openDatabase());
        }
        catch (Exception ex) {
            try {
                this.close();
            }
            catch (Exception closeEx) {
                ex.addSuppressed(closeEx);
            }
            throw ex;
        }
        log.info("{}: Initialized.", (Object)this.logId);
    }

    public void close() {
        if (this.closed.compareAndSet(false, true)) {
            RocksDB db = this.database.get();
            if (db != null) {
                db.close();
                this.database.set(null);
            }
            if (this.writeOptions != null) {
                this.writeOptions.close();
            }
            if (this.databaseOptions != null) {
                this.databaseOptions.close();
            }
            this.clear(false);
            log.info("{}: Closed.", (Object)this.logId);
            Consumer<String> callback = this.closeCallback;
            if (callback != null) {
                Callbacks.invokeSafely(callback, (Object)this.id, null);
            }
        }
    }

    public void insert(Cache.Key key, byte[] data) {
        this.ensureInitializedAndNotClosed();
        Timer timer = new Timer();
        byte[] serializedKey = key.serialize();
        try {
            this.database.get().put(this.writeOptions, serializedKey, data);
        }
        catch (RocksDBException ex) {
            throw this.convert(ex, "insert key '%s'", key);
        }
        RocksDBMetrics.insert(timer.getElapsedMillis(), serializedKey.length + (data != null ? data.length : 0));
    }

    public void insert(Cache.Key key, BufferView data) {
        this.insert(key, data.getCopy());
    }

    public byte[] get(Cache.Key key) {
        byte[] result;
        this.ensureInitializedAndNotClosed();
        Timer timer = new Timer();
        byte[] serializedKey = key.serialize();
        try {
            result = this.database.get().get(serializedKey);
        }
        catch (RocksDBException ex) {
            throw this.convert(ex, "get key '%s'", key);
        }
        RocksDBMetrics.get(timer.getElapsedMillis(), result != null ? (long)result.length : 0L);
        return result;
    }

    public void remove(Cache.Key key) {
        this.ensureInitializedAndNotClosed();
        Timer timer = new Timer();
        try {
            this.database.get().delete(this.writeOptions, key.serialize());
        }
        catch (RocksDBException ex) {
            throw this.convert(ex, "remove key '%s'", key);
        }
        RocksDBMetrics.delete(timer.getElapsedMillis());
    }

    private WriteOptions createWriteOptions() {
        return new WriteOptions().setDisableWAL(true).setSync(false);
    }

    private RocksDB openDatabase() {
        try {
            return RocksDB.open((Options)this.databaseOptions, (String)this.dbDir);
        }
        catch (RocksDBException ex) {
            throw this.convert(ex, "initialize RocksDB instance", new Object[0]);
        }
    }

    private Options createDatabaseOptions() {
        BlockBasedTableConfig tableFormatConfig = new BlockBasedTableConfig().setBlockSize((long)this.cacheBlockSizeKB * 1024L).setBlockCacheSize((long)this.readCacheSizeMB * 1024L * 1024L).setCacheIndexAndFilterBlocks(true);
        Options options = new Options().setCreateIfMissing(true).setDbLogDir(Paths.get(this.dbDir, DB_LOG_DIR).toString()).setWalTtlSeconds(0L).setWalBytesPerSync(0L).setWalSizeLimitMB(0L).setWriteBufferSize((long)this.writeBufferSizeMB * 1024L * 1024L).setMaxWriteBufferNumber(4).setMinWriteBufferNumberToMerge(2).setTableFormatConfig((TableFormatConfig)tableFormatConfig).setOptimizeFiltersForHits(true).setUseDirectReads(this.directReads).setSkipStatsUpdateOnDbOpen(true).setIncreaseParallelism(8).setMaxBackgroundFlushes(4).setMaxBackgroundJobs(8).setMaxBackgroundCompactions(8);
        if (this.memoryOnly) {
            RocksMemEnv env = new RocksMemEnv();
            options.setEnv((Env)env);
        }
        return options;
    }

    private void clear(boolean recreateDirectory) {
        File dbDir = new File(this.dbDir);
        if (FileHelpers.deleteFileOrDirectory((File)dbDir)) {
            log.debug("{}: Deleted existing database directory '{}'.", (Object)this.logId, (Object)dbDir.getAbsolutePath());
        }
        if (recreateDirectory && dbDir.mkdirs()) {
            log.info("{}: Created empty database directory '{}'.", (Object)this.logId, (Object)dbDir.getAbsolutePath());
        }
    }

    private RuntimeException convert(RocksDBException exception, String message, Object ... messageFormatArgs) {
        String exceptionMessage = String.format("Unable to %s (CacheId=%s).", String.format(message, messageFormatArgs), this.id);
        throw new CacheException(exceptionMessage, (Throwable)exception);
    }

    private void ensureInitializedAndNotClosed() {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkState((this.database.get() != null ? 1 : 0) != 0, (String)"%s has not been initialized.", (Object)this.logId);
    }

    @SuppressFBWarnings(justification="generated code")
    public String getId() {
        return this.id;
    }
}

