/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.storage.mocks;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.util.ArrayView;
import io.pravega.common.util.CloseableIterator;
import io.pravega.common.util.SequencedItemList;
import io.pravega.segmentstore.storage.DataLogDisabledException;
import io.pravega.segmentstore.storage.DataLogInitializationException;
import io.pravega.segmentstore.storage.DataLogWriterNotPrimaryException;
import io.pravega.segmentstore.storage.DurableDataLog;
import io.pravega.segmentstore.storage.DurableDataLogException;
import io.pravega.segmentstore.storage.LogAddress;
import io.pravega.segmentstore.storage.QueueStats;
import io.pravega.segmentstore.storage.ThrottleSourceListener;
import io.pravega.segmentstore.storage.WriteSettings;
import io.pravega.segmentstore.storage.WriteTooLongException;
import java.beans.ConstructorProperties;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.time.Duration;
import java.util.Iterator;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
class InMemoryDurableDataLog
implements DurableDataLog {
    static final Supplier<Duration> DEFAULT_APPEND_DELAY_PROVIDER = () -> Duration.ZERO;
    private final EntryCollection entries;
    private final String clientId;
    private final ScheduledExecutorService executorService;
    private final Supplier<Duration> appendDelayProvider;
    @GuardedBy(value="entries")
    private long offset;
    @GuardedBy(value="entries")
    private long epoch;
    private boolean closed;
    private boolean initialized;

    InMemoryDurableDataLog(EntryCollection entries, ScheduledExecutorService executorService) {
        this(entries, DEFAULT_APPEND_DELAY_PROVIDER, executorService);
    }

    InMemoryDurableDataLog(EntryCollection entries, Supplier<Duration> appendDelayProvider, ScheduledExecutorService executorService) {
        this.entries = (EntryCollection)Preconditions.checkNotNull((Object)entries, (Object)"entries");
        this.appendDelayProvider = (Supplier)Preconditions.checkNotNull(appendDelayProvider, (Object)"appendDelayProvider");
        this.executorService = (ScheduledExecutorService)Preconditions.checkNotNull((Object)executorService, (Object)"executorService");
        this.offset = Long.MIN_VALUE;
        this.epoch = Long.MIN_VALUE;
        this.clientId = UUID.randomUUID().toString();
    }

    @Override
    public void close() {
        if (!this.closed) {
            try {
                this.entries.releaseLock(this.clientId);
            }
            catch (DataLogWriterNotPrimaryException dataLogWriterNotPrimaryException) {
                // empty catch block
            }
            this.closed = true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void initialize(Duration timeout) throws DataLogInitializationException {
        long newEpoch = this.entries.acquireLock(this.clientId);
        EntryCollection entryCollection = this.entries;
        synchronized (entryCollection) {
            this.epoch = newEpoch;
            Entry last = this.entries.getLast();
            this.offset = last == null ? 0L : last.sequenceNumber + (long)last.data.length;
        }
        this.initialized = true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void enable() {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Preconditions.checkState((!this.initialized ? 1 : 0) != 0, (Object)"InMemoryDurableDataLog is initialized; cannot enable.");
        EntryCollection entryCollection = this.entries;
        synchronized (entryCollection) {
            this.entries.enable();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void disable() throws DurableDataLogException {
        this.ensurePreconditions();
        EntryCollection entryCollection = this.entries;
        synchronized (entryCollection) {
            this.entries.disable(this.clientId);
        }
        this.close();
    }

    @Override
    public WriteSettings getWriteSettings() {
        return new WriteSettings(this.entries.getMaxAppendSize(), Duration.ofMinutes(1L), Integer.MAX_VALUE);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long getEpoch() {
        this.ensurePreconditions();
        EntryCollection entryCollection = this.entries;
        synchronized (entryCollection) {
            return this.epoch;
        }
    }

    @Override
    public QueueStats getQueueStatistics() {
        return QueueStats.DEFAULT;
    }

    @Override
    public void registerQueueStateChangeListener(ThrottleSourceListener listener) {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public CompletableFuture<LogAddress> append(ArrayView data, Duration timeout) {
        CompletableFuture<LogAddress> result;
        this.ensurePreconditions();
        if (data.getLength() > this.getWriteSettings().getMaxWriteLength()) {
            return Futures.failedFuture((Throwable)((Object)new WriteTooLongException(data.getLength(), this.getWriteSettings().getMaxWriteLength())));
        }
        try {
            Entry entry = new Entry(data);
            EntryCollection entryCollection = this.entries;
            synchronized (entryCollection) {
                entry.sequenceNumber = this.offset;
                this.entries.add(entry, this.clientId);
                this.offset += (long)entry.data.length;
            }
            result = CompletableFuture.completedFuture(new InMemoryLogAddress(entry.sequenceNumber));
        }
        catch (Throwable ex) {
            return Futures.failedFuture((Throwable)ex);
        }
        Duration delay = this.appendDelayProvider.get();
        if (delay.compareTo(Duration.ZERO) <= 0) {
            return result;
        }
        return result.thenComposeAsync(logAddress -> Futures.delayedFuture((Duration)delay, (ScheduledExecutorService)this.executorService).thenApply(ignored -> logAddress), (Executor)this.executorService);
    }

    @Override
    public CompletableFuture<Void> truncate(LogAddress upToAddress, Duration timeout) {
        this.ensurePreconditions();
        return CompletableFuture.runAsync(() -> {
            try {
                EntryCollection entryCollection = this.entries;
                synchronized (entryCollection) {
                    this.entries.truncate(upToAddress.getSequence(), this.clientId);
                }
            }
            catch (DataLogWriterNotPrimaryException ex) {
                throw new CompletionException((Throwable)((Object)ex));
            }
        }, this.executorService);
    }

    @Override
    public CloseableIterator<DurableDataLog.ReadItem, DurableDataLogException> getReader() throws DurableDataLogException {
        this.ensurePreconditions();
        return new ReadResultIterator(this.entries.iterator());
    }

    private void ensurePreconditions() {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Preconditions.checkState((boolean)this.initialized, (Object)"InMemoryDurableDataLog is not initialized.");
    }

    static class InMemoryLogAddress
    extends LogAddress {
        InMemoryLogAddress(long sequence) {
            super(sequence);
        }

        public int hashCode() {
            return Long.hashCode(this.getSequence());
        }

        public boolean equals(Object other) {
            if (other instanceof InMemoryLogAddress) {
                return this.getSequence() == ((InMemoryLogAddress)other).getSequence();
            }
            return false;
        }
    }

    static class Entry
    implements SequencedItemList.Element {
        long sequenceNumber = -1L;
        final byte[] data;

        Entry(ArrayView inputData) {
            this.data = new byte[inputData.getLength()];
            System.arraycopy(inputData.array(), inputData.arrayOffset(), this.data, 0, this.data.length);
        }

        public String toString() {
            return String.format("SequenceNumber = %d, Length = %d", this.sequenceNumber, this.data.length);
        }

        @SuppressFBWarnings(justification="generated code")
        public long getSequenceNumber() {
            return this.sequenceNumber;
        }
    }

    static class EntryCollection {
        private final SequencedItemList<Entry> entries = new SequencedItemList();
        private final AtomicReference<String> writeLock = new AtomicReference();
        private final AtomicLong epoch = new AtomicLong();
        private final AtomicBoolean enabled;
        private final int maxAppendSize;

        EntryCollection() {
            this(1040384);
        }

        EntryCollection(int maxAppendSize) {
            this.maxAppendSize = maxAppendSize;
            this.enabled = new AtomicBoolean(true);
        }

        void enable() {
            if (!this.enabled.compareAndSet(false, true)) {
                throw new IllegalStateException("Log already enabled.");
            }
        }

        void disable(String clientId) throws DataLogWriterNotPrimaryException {
            this.ensureLock(clientId);
            if (!this.enabled.compareAndSet(true, false)) {
                throw new IllegalStateException("Log already disabled.");
            }
        }

        public void add(Entry entry, String clientId) throws DataLogWriterNotPrimaryException {
            this.ensureLock(clientId);
            this.ensureEnabled();
            this.entries.add((SequencedItemList.Element)entry);
        }

        int getMaxAppendSize() {
            return this.maxAppendSize;
        }

        Entry getLast() {
            return (Entry)this.entries.getLast();
        }

        void truncate(long upToSequence, String clientId) throws DataLogWriterNotPrimaryException {
            this.ensureLock(clientId);
            this.ensureEnabled();
            this.entries.truncate(upToSequence);
        }

        Iterator<Entry> iterator() {
            this.ensureEnabled();
            return this.entries.read(Long.MIN_VALUE, Integer.MAX_VALUE);
        }

        long acquireLock(String clientId) throws DataLogDisabledException {
            Exceptions.checkNotNullOrEmpty((String)clientId, (String)"clientId");
            if (!this.enabled.get()) {
                throw new DataLogDisabledException("Log is disabled; cannot acquire lock.");
            }
            this.writeLock.set(clientId);
            return this.epoch.incrementAndGet();
        }

        void releaseLock(String clientId) throws DataLogWriterNotPrimaryException {
            Exceptions.checkNotNullOrEmpty((String)clientId, (String)"clientId");
            if (!this.writeLock.compareAndSet(clientId, null)) {
                throw new DataLogWriterNotPrimaryException("Unable to release exclusive write lock because the current client does not own it. Current owner: " + clientId);
            }
        }

        private void ensureLock(String clientId) throws DataLogWriterNotPrimaryException {
            Exceptions.checkNotNullOrEmpty((String)clientId, (String)"clientId");
            String existingLockOwner = this.writeLock.get();
            if (existingLockOwner != null && !existingLockOwner.equals(clientId)) {
                throw new DataLogWriterNotPrimaryException("Unable to perform operation because the write lock is owned by a different client " + clientId);
            }
        }

        private void ensureEnabled() {
            Preconditions.checkState((boolean)this.enabled.get(), (Object)"Log not enabled.");
        }
    }

    private static class ReadResultItem
    implements DurableDataLog.ReadItem {
        private final byte[] payload;
        private final LogAddress address;

        ReadResultItem(Entry entry) {
            this.payload = entry.data;
            this.address = new InMemoryLogAddress(entry.sequenceNumber);
        }

        @Override
        public InputStream getPayload() {
            return new ByteArrayInputStream(this.payload);
        }

        @Override
        public int getLength() {
            return this.payload.length;
        }

        public String toString() {
            return String.format("Address = %s, Length = %d", this.address, this.payload.length);
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        public LogAddress getAddress() {
            return this.address;
        }
    }

    private static class ReadResultIterator
    implements CloseableIterator<DurableDataLog.ReadItem, DurableDataLogException> {
        private final Iterator<Entry> entryIterator;

        public DurableDataLog.ReadItem getNext() throws DurableDataLogException {
            if (this.entryIterator.hasNext()) {
                return new ReadResultItem(this.entryIterator.next());
            }
            return null;
        }

        public void close() {
        }

        @ConstructorProperties(value={"entryIterator"})
        @SuppressFBWarnings(justification="generated code")
        public ReadResultIterator(Iterator<Entry> entryIterator) {
            this.entryIterator = entryIterator;
        }
    }
}

